#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>

#include "kdddllparser.h"

#include <qfile.h>
#include <qstack.h>
#include <qwindowdefs.h>


bool isspace( char c )
{
    return c == ' ' || c == '\t' || c == '\n';
}

typedef QStack<QString> QStringStack;

/* Attribute item preparation*/
typedef int (*convfunc)( const char * );

/** attribute value conversion function char* -> int
  */
static int convInt( const char * );

/** attribute value conversion function char* -> bool
  * true is returned for true,yes,on, 1, t, y
  * all other values return false
  */
static int convBool( const char * );

/** parses a htmlcolordef #rrggbb to the corresponding int
  */
static int convColor( const char * );

/** parses a align from left, center, right to 1, 2, 4
  */
static int convHAlign( const char * );

/** parses a align from top, center, bottom to 1, 2, 4
  */
static int convVAlign( const char * );


/** A structure for holding a attribute type declaration.
  * The name is not really necessary, but it's
  * cute to have one if you wannt do debug.
  */
typedef struct tag_typeitem
{
    const char *name;
    const char *regexp;
    const convfunc convert;
}   typeitem;

/** list of all used attribute types
  */
static const
typeitem typelist[ ATT_T_COUNT ] =
{
    { "NONE",   "()",                            (convfunc)(0) },
    { "INT",    "(-[0-9]+|[0-9]+)",              convInt       },
    { "BOOL",   "(true|false)",                  convBool      },
    { "ANY",    "([^\"]*)",                      (convfunc)(0) },
    { "NAME",   "([_a-zA-Z][_a-zA-Z0-9]{0,31})", (convfunc)(0) },
    { "COLOR",  "(\\#[0-9a-fA-F]{6})",           convColor     },
    { "HALIGN", "(left|center|right)",           convHAlign    },
    { "VALIGN", "(top|center|bottom)",           convVAlign    },
    { "TYPE",   "(int|text)",                    (convfunc)(0) }
};

/** A structure for holding a attribute declaration.
  * The name is not really necessary, but it's
  * cute to have one if you wannt do debug.
  */
typedef struct  tag_attritem
{
    const char *name;
    const int  typeindex;
    regex_t    compiled;
} attritem;

const char *attrPatternFmt = "%s *= *\"{0,1}%s\"{0,1}";

static
attritem attrlist[ ATT_COUNT ] =
{
    { "NAME",      ATT_T_NAME   },
    { "TYPE",      ATT_T_TYPE   },
    { "TITLE",     ATT_T_ANY    },
    { "BORDER",    ATT_T_BOOL   },
    { "MAXLENGTH", ATT_T_INT    },
    { "COLSPAN",   ATT_T_INT    },
    { "ROWSPAN",   ATT_T_INT    },
    { "MINROWS",   ATT_T_INT    },
    { "CHECKED",   ATT_T_BOOL   },
    { "ID",        ATT_T_INT    },
    { "GROUP",     ATT_T_NAME   },
    { "TEXTCOLOR", ATT_T_COLOR  },
    { "BGCOLOR",   ATT_T_HALIGN },
    { "ALIGN",     ATT_T_VALIGN },
    { "TEXT",      ATT_T_ANY    },
    { "INDEX",     ATT_T_INT    }
};

typedef struct tag_tagitem
{
    const char       *name;

    /** Decides weather between the start and end tag there may
      * occur a text we have to read an also
      * decides where the tag has to be closed.
      */
    const int        closetag;

    /* the attribues ar a integer array in the form
     * {
     * attrindex_1, attrscope_1,
     * attrindex_2, attrscope_2,
     * ...
     * -1
     * }
     */
    const int        *attribute;

} tagitem;

/** Now we will declare all possible attribute lists.
  *
  * If you declare new tags, try to reuse existing attribute lists.
  * The reason why some combinations of attributes wich are really 
  * valid for nearly every tag -like name or export- is that in the list
  * a tag-scope declaration take exactly -2 * sizeof int- while I do not
  * want to make a guess who many bytes a exection handling in the parser
  * might take.
  * Ok, the sourcecode night get a little bit longish, but
  * the resulting binary will be smaller.
  *
  * Do not mix the names of the lists to the similar named tags.
  * They are independent.
  * I only named some attribute lists like the tags using them because 
  * I am lazy.
  * So a name like 
  * --attrListMustNameMayIndexMayColspanMayRowspan--
  * would be more correct as attrListCombo but you see why...
  */
static const int attrListEmpty[ 1 ] =
{
    -1
};
 
static const int attrListName[ 3 ] =
{
    ATT_NAME,      ATT_S_MUST,
    -1
};

static const int attrListPage[ 5 ] =
{
    ATT_NAME,      ATT_S_MUST,
    ATT_TITLE,     ATT_S_MAY,
    -1
};

static const int attrListGroup[ 7 ]  =
{
    ATT_NAME,      ATT_S_MUST,
    ATT_TITLE,     ATT_S_MAY,
    ATT_BORDER,    ATT_S_MAY,
    -1
};

static const int attrListLabel[ 7 ] =
{
    ATT_NAME,      ATT_S_MUST,
    ATT_COLSPAN,   ATT_S_MAY,
    ATT_ROWSPAN,   ATT_S_MAY,
    -1
};

/* this is a example of reusing a list */
static const int *attrListButton = (const int*)attrListLabel;
                                        
static const int attrListCombo[ 9 ] =
{
    ATT_NAME,      ATT_S_MUST,
    ATT_INDEX,     ATT_S_MAY,
    ATT_COLSPAN,   ATT_S_MAY,
    ATT_ROWSPAN,   ATT_S_MAY,
    -1
};

static const int attrListCheck[ 9 ] =
{
    ATT_NAME,      ATT_S_MUST,
    ATT_CHECKED,   ATT_S_MAY,
    ATT_COLSPAN,   ATT_S_MAY,
    ATT_ROWSPAN,   ATT_S_MAY,
    -1
};

static const int attrListList[ 11 ]  =
{
    ATT_NAME,      ATT_S_MUST,
    ATT_INDEX ,    ATT_S_MAY,
    ATT_COLSPAN,   ATT_S_MAY,
    ATT_ROWSPAN,   ATT_S_MAY,
    ATT_MINROWS,   ATT_S_MAY,
    -1
};

static const int attrListLineed[ 11 ] =
{
    ATT_NAME,      ATT_S_MUST,
    ATT_TYPE,      ATT_S_MAY,
    ATT_COLSPAN,   ATT_S_MAY,
    ATT_ROWSPAN,   ATT_S_MAY,
    ATT_MAXLENGTH, ATT_S_MAY,
    -1
};

static const int attrListRadio[ 13 ] =
{
    ATT_NAME,      ATT_S_MUST,
    ATT_CHECKED,   ATT_S_MAY,
    ATT_COLSPAN,   ATT_S_MAY,
    ATT_ROWSPAN,   ATT_S_MAY,
    ATT_GROUP,     ATT_S_MUST,
    ATT_ID,        ATT_S_MAY,
    -1
};

/** Now to declare wich tag has wich attrList
  */
tagitem taglist[ TAG_COUNT ] =
{
    { "KDDL",        TAG_C_ANYWHERE, (const int*)( 0              ) },
    { "/KDDL",       TAG_C_CLOSE,    (const int*)( 0              ) },
    { "GROUP",       TAG_C_ANYWHERE, (const int*)( attrListGroup  ) },
    { "/GROUP",      TAG_C_CLOSE,    (const int*)( 0              ) },
    { "BOOK",        TAG_C_ANYWHERE, (const int*)( attrListName   ) },
    { "/BOOK",       TAG_C_CLOSE,    (const int*)( 0              ) },
    { "PAGE",        TAG_C_ANYWHERE, (const int*)( attrListPage   ) },
    { "/PAGE",       TAG_C_CLOSE,    (const int*)( 0              ) },
    { "LABEL",       TAG_C_TEXT,     (const int*)( attrListLabel  ) },
    { "BUTTON",      TAG_C_TEXT,     (const int*)( attrListButton ) },
    { "CHECKBOX",    TAG_C_TEXT,     (const int*)( attrListCheck  ) },
    { "RADIOBUTTON", TAG_C_TEXT,     (const int*)( attrListRadio  ) },
    { "LINEEDIT",    TAG_C_TEXT,     (const int*)( attrListLineed ) },
    { "LISTBOX",     TAG_C_ANYWHERE, (const int*)( attrListList   ) },
    { "/LISTBOX",    TAG_C_CLOSE,    (const int*)( 0              ) },
    { "COMBOBOX",    TAG_C_ANYWHERE, (const int*)( attrListCombo  ) },
    { "/COMBOBOX",   TAG_C_CLOSE,    (const int*)( 0              ) },
    { "LI",          TAG_C_TEXT,     (const int*)( 0              ) },
    { "BR",          TAG_C_IMPLICIT, (const int*)( 0              ) },
    { "SKIP",        TAG_C_IMPLICIT, (const int*)( 0              ) }
};

/* You already know: ... the distant future ...
const colorConvertList[] =
{
    { "black",   0x000000 },
    { "maroon",  0x800000 },
    { "green",   0x008000 },
    { "olive",   0x808000 },
    { "navy",    0x000080 },
    { "purple",  0x800080 },
    { "teal",    0x008080 },
    { "gray",    0x808080 },
    { "silver",  0xC0C0C0 },
    { "red",     0xFF0000 },
    { "lime",    0x00FF00 },
    { "yellow",  0xFFFF00 },
    { "blue",    0x0000FF },
    { "fuchsia", 0xFF00FF },
    { "aqua",    0x00FFFF },
    { "white",   0xFFFFFF }
}
*/

inline
bool intRead( QIODevice *in, int &val )
{
    return in->readBlock( (char *)( &val ), sizeof( int ) ) == sizeof( int );
}


inline
bool intWrite( QIODevice *out, int val )
{
    return out->writeBlock( (char *)( &val ), sizeof( int ) ) == sizeof( int );
}


KDDDLLAttrValue::KDDDLLAttrValue( const QString &_value, int _index ) :
    value( _value ),
    index( _index )
{
};

KDDDLLAttrValue::KDDDLLAttrValue( QIODevice *in ) :
    value( "" ),
    index( -1 )
{
    if( in )
    {
        if( intRead( in, index ) )
        {
            int len = 0;
            if( intRead( in, len ) )
            {
                char *str = new char[ len + 1 ];
                int readDone = in->readBlock( str, len );
                str[ readDone ] = 0;
                value = str;
                delete str;
            }
        }
    }
};

const char* KDDDLLAttrValue::name() const
{
    return attrlist[ index ].name;
}

bool KDDDLLAttrValue::write( QIODevice *out )
{
    bool ok = intWrite( out, index );
    int len = value.length();
    ok &= intWrite( out, len );
    char *data = value.data();
    ok &= ( out->writeBlock( data, len ) == len );
    return ok;
}


const char *KDDDLLAttrValueList::getVal( const char *attr, const char *def ) const
{
    KDDDLLAttrValue *atv = find( attr );
    if( !atv )
        return def;
    return atv->value.data();
}

int KDDDLLAttrValueList::getVal( const char *attr, int def ) const
{
    KDDDLLAttrValue *atv = find( attr );
    if( !atv )
        return def;
    const char *cr = atv->value.data();
    if( typelist[ attrlist[ atv->index ].typeindex ].convert )
        return (typelist[ attrlist[ atv->index ].typeindex ].convert)( cr );
    return atoi( cr );
}

void KDDDLLAttrValueList::read( QIODevice *in )
{
    int countin = 0;
    if( intRead( in, countin ) )
    {
        for( int i = 0; i < countin; i++ )
        {
            int len = 0;
            intRead( in, len );
            char *key = new char[ len + 1 ];
            in->readBlock( key, len );
            key[ len ] = 0;
            insert( key, new KDDDLLAttrValue( in ) );
            delete key;
        }
    }
}

bool KDDDLLAttrValueList::write( QIODevice *out )
{
    int countout =  count();
    bool ok = intWrite( out, countout );

    for( QDictIterator<KDDDLLAttrValue> it( *this ); ok && it.current(); ++it )
    {
        const char *key = it.currentKey();
        int len = strlen( key );
        ok = intWrite( out, len );
        ok = ok && ( out->writeBlock( key, len ) == len );
        ok = ok && it.current()->write( out );
    }
    return ok;
}



KDDDLLTag::KDDDLLTag( int _index ) :
    attr(),
    index( _index )
{
}

KDDDLLTag::KDDDLLTag( QIODevice *in ) :
    attr(),
    index( -1 )
{
    if( intRead( in, index ) )
    {
        attr.read( in );
    }

}

const char* KDDDLLTag::name() const
{
    return taglist[ index ].name;
}

bool KDDDLLTag::write( QIODevice *out )
{
    bool ok = intWrite( out, index );
    ok &= attr.write( out );
    return ok;
}



KDDDLLTagList::KDDDLLTagList( QIODevice *in ) :
    QVector<KDDDLLTag>()
{
    if( in )
    {
        int countin;
        if( intRead( in, countin ) )
        {
            for( int i = 0; i < countin; i++ )
            {
                add( new KDDDLLTag( in ) );
            }
        }
    }
}

bool KDDDLLTagList::add( KDDDLLTag *tag )
{
    if( size() == count() )
    {
        resize( size() + 10 );
    }
    return insert( count(), tag );
}

bool KDDDLLTagList::write( QIODevice *out )
{
    int countout =  count();
    bool ok = intWrite( out, countout );
    for( int i = 0; ok && i < countout; i++ )
    {
        ok = at( i )->write( out );
    }
    return ok;
}


/*
KDDDLLData::KDDDLLData( const char *filename )
{
    FILE *ofile = 0;
    if( filename )
    {
        ofile = fopen( filename, "r" );
    }
    iface = new KDDDLLTagList( ofile );
    tags  = new KDDDLLTagList( ofile );
}


bool KDDDLLData::save( const char *filename )
{
    int hasIface = ( iface->count() != 0 );
    FILE *ofile = fopen( filename, "w" );
    bool ok = ( fwrite( &hasIface, 1, sizeof( int ), ofile ) == sizeof( int ) );
    ok = ok & iface->write( ofile );
    ok = ok & tags->write( ofile );
    ok = ok && fclose( ofile );
    return ok;
}
*/


int convInt( const char *v )
{
    return atoi( v );
}

int convBool( const char *v )
{
    return ( !strcasecmp( v, "true" ) ||
             !strcasecmp( v, "yes" )  ||
             !strcasecmp( v, "on" )   ||
             !strcmp( v, "1" )        ||
             !strcasecmp( v, "t" )    ||
             !strcasecmp( v, "y" ) );
}

int convColor( const char *v )
{
    int r = 0x00000000;
    if( !sscanf( "#%x", v, &r ) )
    {
    }
    return r;
}

int convHAlign( const char *v )
{
    return ( ( !strcasecmp( v, "left" )   << 0 ) |
             ( !strcasecmp( v, "center" ) << 1 ) |
             ( !strcasecmp( v, "right" )  << 2 ) );
}

int convVAlign( const char *v )
{
    return ( ( !strcasecmp( v, "top" )    << 0 ) |
             ( !strcasecmp( v, "center" ) << 1 ) |
             ( !strcasecmp( v, "bottom" ) << 2 ) );
}

int convStyle( const char *v )
{
    if( strcasecmp( v, "windows" ) )
        return WindowsStyle;
    return MotifStyle;
}



const int maxMatch = 10;
const char *tagAttrMatch = "^ *<([^ 0-9][^ >]*)(( +[^>]*)*)>";
const char *tagTextMatch = "([^<]*)</([^>]*)>";

void reset( regmatch_t* match )
{
    int i;
    for( i = 0; i < maxMatch; i++ )
    {
        match[ i ].rm_so = match[ i ].rm_eo = -1;
    }
}

void kmsg( QTextStream *msg, int line, const char *fmt, ... )
{
    if( msg )
    {
        char * fmtd;
        asprintf( &fmtd, "%i: %s\n", line, fmt );
        char * out;
        va_list ap;
        va_start( ap, fmt );
        vasprintf( &out, fmtd, ap );
        va_end( ap );
        (*msg) << out;
        free( out );
        free( fmtd );
    }
}



#define OUT( lv, ms ) if( (lv) >= msglevel ) kmsg( msg, lineno, (ms) );
#define OUT1( lv, ms, p1 ) if( (lv) >= msglevel ) kmsg( msg, lineno, (ms), (p1) );
#define OUT2( lv, ms, p1, p2 ) if( (lv) >= msglevel ) kmsg( msg, lineno, (ms), (p1), (p2) );
#define OUT3( lv, ms, p1, p2, p3 ) if( (lv) >= msglevel ) kmsg( msg, lineno, (ms), (p1), (p2), (p3) );
#define OUT4( lv, ms, p1, p2, p3, p4 ) if( (lv) >= msglevel ) kmsg( msg, lineno, (ms), (p1), (p2), (p3), (p4) );


bool KDDDLLparse( KDDDLLTagList *tagList, QTextStream *source, QTextStream *msg, int msglevel )
{
    regmatch_t matches[ maxMatch ];
    regex_t    compiledtagtextmatch;
    regex_t    compiledtagattrmatch;

    int        lineno = 0;
    bool       ret = true;

    QStringStack tagStack;

    QString line;
    int i; // we will reuse this

    /** combine and compile the patterns for attributes and their type
      */
    for( i = 0; i < ATT_COUNT; i++ )
    {
        char *tmp;
        asprintf( &tmp,
                  attrPatternFmt,
                  attrlist[ i ].name,
                  typelist[ attrlist[ i ].typeindex ].regexp );
        regcomp( &(attrlist[ i ].compiled),
                 tmp,
                 REG_ICASE | REG_EXTENDED );
        free( tmp );
    }
    /** The pattern for the first tag in the line.
      * We allways parse a tag until anotherone
      * or a closing tag is starting, so always thisone should fit.
      */
    regcomp( &(compiledtagattrmatch), tagAttrMatch, REG_ICASE | REG_EXTENDED );
    /** if a tag has tagged text, this pattern should fit
      * the tagged text plus the end tag.
      */
    regcomp( &(compiledtagtextmatch), tagTextMatch, REG_ICASE | REG_EXTENDED );


    while( !source->eof() )
    {
        lineno++;
        line = source->readLine();
        while( !line.isEmpty() )
        {
            reset( matches );
            if( !regexec( &(compiledtagattrmatch), line.data(), maxMatch, matches, 0 ) )
            {
                QString tag( line.mid( matches[ 1 ].rm_so, matches[ 1 ].rm_eo - matches[ 1 ].rm_so ) );
                QString attr( line.mid( matches[ 2 ].rm_so, matches[ 2 ].rm_eo - matches[ 2 ].rm_so ) );
                OUT2( OUT_INFO, "read tag '%s' attr '%s'", tag.data(), attr.data() );
                line = line.mid( matches[ 0 ].rm_eo, line.length() - matches[ 0 ].rm_eo );
                /** look through taglist weather we know that tag
                  */
                for( i = 0; i < TAG_COUNT && strcasecmp( taglist[ i ].name, tag.data() ); i++ );
                if( i == TAG_COUNT )
                {
                    OUT1( OUT_FATAL, "unknown tag '%s'", tag.data() );
                    ret = false;
                    goto clean;
                }

                const tagitem *acttag = &(taglist[ i ]);
                KDDDLLTag *theTag = new KDDDLLTag( i );
                tagList->add( theTag );

                /** TODO: Check here weather at top of tagStzack is limited in value
                  * e.g. a <LI> nees either <COMBOBOX> or <LISTBOX>
                  * Do it general, not in style : if( strcasecmp( stack.top.data(), "COMBOBOX" ) )
                  */

                /** look through the attributes defined in taglist for this tag
                  * weather we can find them
                  */
                if( acttag->attribute ) /* i.e. we have attributes we need to parse */
                {
                    for( i = 0; acttag->attribute[ i * 2 ] != -1; i++ )
                    {
                        attritem *actattr = &(attrlist[ acttag->attribute[ i * 2 ] ]);
                        bool done = !regexec( &(actattr->compiled), attr.data(), maxMatch, matches, 0 );
                        if( !done && acttag->attribute[ i * 2 + 1 ] & ATT_S_MUST )
                        {
                            OUT1( OUT_ERR, "attr '%s' ommited but it's mandatory", actattr->name );
                        }
                        if( done )  // ok, attribute found
                        {
                            KDDDLLAttrValue *theAttr = new KDDDLLAttrValue( attr.mid( matches[ 1 ].rm_so, matches[ 1 ].rm_eo - matches[ 1 ].rm_so ), acttag->attribute[ i * 2 ] );
                            theTag->attr.insert( actattr->name, theAttr );
                            attr.remove( matches[ 0 ].rm_so, matches[ 0 ].rm_eo - matches[ 0 ].rm_so );
                            OUT2( OUT_INFO, "attr '%s' = '%s' added", theAttr->name(), theAttr->value.data() );
                        }
                    }
                    /** let's see weather still somethin in attr,
                      * telling we didn't parse it completely
                      */
                    attr = attr.stripWhiteSpace();
                    if( attr.length() )
                    {
                        OUT2( OUT_ERR, "error tag '%s' doesn't know '%s'", theTag->name(), attr.data() );
                        OUT1( OUT_ERR, "'%s' stripped", attr.data() );
                        attr = "";
                    }
                }
                /** handle tagged text, opening and closing of tags according
                  * to tag definition
                  */
                switch( acttag->closetag )
                {
                    case TAG_C_TEXT: // telling we should expect text an an endtag
                        { // to allow block scoped variables
                            /** read lines until endtag, eof or error
                              */
                            int startline = lineno;
                            // while no match we extend the line
                            while( regexec( &(compiledtagtextmatch), line.data(), maxMatch, matches, 0 ) && !source->eof() )
                            {
                                line += source->readLine(); // for good allready the \n is cutted
                                lineno++;
                            } // maybe we also should cut doubled whitespaces. What do you think?

                            if( regexec( &(compiledtagtextmatch), line.data(), maxMatch, matches, 0 ) && !source->eof() )
                            {
                                OUT1( OUT_FATAL, "tag '%s' not closed, trying to insert closing tag", theTag->name() );
                                ret = false;
                                goto clean;
                            }

                            /** check weather the close tag fits the open tag
                              */
                            QString tmp = line.mid( matches[ 2 ].rm_so, matches[ 2 ].rm_eo - matches[ 2 ].rm_so );
                            if( strcasecmp( tmp.data(), acttag->name ) )
                            {
                                if( OUT_FATAL >= msglevel )
                                    kmsg( msg, startline, "tag '%s' was not closed at postition", acttag->name );
                                OUT1( OUT_FATAL, "of tag '%s'", tmp.data() );
                                ret = false;
                                goto clean;
                            }

                            /** place the tagged text as attribure TEXT
                              */
                            KDDDLLAttrValue *theAttr = new KDDDLLAttrValue( line.mid( matches[ 1 ].rm_so,matches[ 1 ].rm_eo - matches[ 1 ].rm_so ), ATT_TEXT );
                            theTag->attr.insert( "TEXT", theAttr );
                            OUT2( OUT_INFO, "attr '%s' = '%s' added", theAttr->name(), theAttr->value.data() );
                            line.remove( 0, matches[ 0 ].rm_eo );
                        }
                        break;

                    case TAG_C_IMPLICIT:
                        /** nice, we don't need to do anything, cause there is no
                          * closing tag ( BR, SKIP et aliae )
                          */
                        break;

                    case TAG_C_ANYWHERE:
                        /** Push tagtype to tagstack.
                          * This happens for GROUP, BOOK.. in general for every tag
                          * where there's also a "/TAG" declaration in taglist, telling
                          * that we later must find the end-tag
                          */
                        tagStack.push( new QString( tag ) );
                        break;

                    case TAG_C_CLOSE:
                        {
                            /** Pop tagtype from tagstack, compare and report if different.
                              * This is an endtag
                              */
                            if( tagStack.isEmpty() )
                            {
                                OUT1( OUT_ERR, "no more open tags, but '/%s' found", tag.data() );
                            }
                            else
                            {
                                if( strcasecmp( tagStack.top()->data(), &(tag.data()[ 1 ] ) ) )
                                {
                                    OUT2( OUT_ERR, "tag '/%s' found but expecting tag '/%s'", tag.data(), tagStack.top()->data() );
                                }
                                else
                                {
                                    tagStack.pop();
                                }
                            }
                        }
                        break;

                    default: // weird: we shouldn't land here
                        OUT( OUT_FATAL, "panic because somebody messed up kdddll itself -> so sorry !" );
                        break;
                }
            }
            else
            {
                // should only happen if somebody wrote before <KDDL> tag
                OUT1( OUT_FATAL, "nasty error parsing '%s'", line.data() );
                ret = false;
                goto clean;
            }
            line.stripWhiteSpace();
        }
    }

clean:
    for( i = 0; i < ATT_COUNT; i++ )
    {
        regfree( &(attrlist[ i ].compiled) );
    }
    regfree( &(compiledtagattrmatch) );
    regfree( &(compiledtagtextmatch) );

    if( !tagStack.isEmpty() )
    {
         while( !tagStack.isEmpty() )
         {
             OUT1( OUT_ERR, "unclosed tag '%s' at eof", tagStack.top()->data() );
             tagStack.pop();
         }
     }
    return ret;
}

