/***************************************************************************
                          kduviewitem.cpp  -  description
                             -------------------
    begin                : Mon Jan 24 2000
    copyright            : (C) 2000 by David Beattie
    email                : dbeattie@softhome.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kduviewitem.h"
#include "kduprocess.h"

KduViewItem::KduViewItem(KduView *parent, char const *const name, int const size)
	:QListViewItem((QListView *) parent,
	               computedText(name,(size == -1),(size == -1) ? 0 : size)),
	 linkname(name),
	 sizeunknown(size == -1),
	 dusize(sizeunknown ? 0 : size),
	 duprocess(0)
{}

KduViewItem::KduViewItem(KduViewItem *parent, char const *const name, int const size)
	:QListViewItem((QListViewItem *) parent,
	                computedText(name,(size == -1),(size == -1) ? 0 : size)),
	 linkname(name),
	 sizeunknown(size == -1),
	 dusize(sizeunknown ? 0 : size),
	 duprocess(0)
{}

KduViewItem::KduViewItem(KduViewItem *parent, KduViewItem *after, char const *const name, int const size)
	:QListViewItem((QListViewItem *) parent, (QListViewItem *) after,
	                computedText(name,(size == -1),(size == -1) ? 0 : size)),
	 linkname(name),
	 sizeunknown(size == -1),
	 dusize(sizeunknown ? 0 : size),
	 duprocess(0)
{}

KduViewItem::~KduViewItem()
{
	if (duprocess)
		delete duprocess;
}

/**  */
QString KduViewItem::name(void) const
{
	return linkname;
}

/** register (or unregister if dp is NULL) a DuProcess with this KduViewItem,
so that it will be killed when this item is deleted.
 */
void KduViewItem::setDuProcess(KDuProcess *dp)
{
	duprocess = dp;
	setText(0,computedText(linkname,(sizeunknown || dp),dusize));
}

/** DuProcesses call this function to communicate
to this KduViewItem what the size of the directory
or file is as reported by "du". */
void KduViewItem::setSize(int size)
{
	// if size is known, then we are modifying it...and we should propogate
	// the size change up the tree
	if (sizeunknown)
	{
		dusize += size;
		sizeunknown = FALSE;
 		setText(0,computedText(linkname,(sizeunknown || duprocess),dusize));
	}
	else
		adjustSizeBy(size-dusize);
}
/** adds or subtracts an amount from the listed
size of the directory or file.  Also propagates this change up the tree.*/
void KduViewItem::adjustSizeBy(int delta)
{
	KduViewItem *p = parent();
	dusize += delta;
	setText(0,computedText(linkname,(sizeunknown || duprocess),dusize));
	if (p)
		p->adjustSizeBy(delta);
}

/** returns the KDuProcess rooted at this
KduViewItem, if any. */
KDuProcess * KduViewItem::getDuProcess() const
{
	return duprocess;
}
/** returns whether the size of the item
is known.  If it is not, then surely, a
DuProcess is running to determine the
size. */
bool KduViewItem::isSizeUnknown(void) const
{
	return sizeunknown;
}
/** returns the "du"-computed directory or
file size as stored in this KduViewItem. */
int KduViewItem::size(void) const
{
	return dusize;
}
/** typecasting function */
KduViewItem *KduViewItem::parent(void) const
{
	return (KduViewItem *) QListViewItem::parent();
}
/** typecasting function */
KduViewItem *KduViewItem::firstChild(void) const
{
	return (KduViewItem *) QListViewItem::firstChild();
}
/** typecasting function */
KduViewItem *KduViewItem::nextSibling(void) const
{
	return (KduViewItem *) QListViewItem::nextSibling();
}
/** translates a name, sizeunknown, size triplet
into displayable text */
QString KduViewItem::computedText(char const *const name, bool const sizeunknown, int const size)
{
	return QString(name) + " (" + (sizeunknown ? (QString) "?" : QString().setNum(size)) + " K)";
}

/** returns whether this item does not lie
within a closed subtree. */
bool KduViewItem::isVisible(void) const
{
	KduViewItem *p = this->parent();
	while (p)
		if (p->isOpen())
			p = p->parent();
		else
			return FALSE;
	return TRUE;
}

/** display optimization: don't invalidate height
unless this item is open.  This will avoid
unnecessary "triggerUpdate" calls. */
void KduViewItem::invalidateHeight(void)
{
	if (isOpen())
		QListViewItem::invalidateHeight();
}
/** counteract display optimization of
"invalidateHeight".... if we're changing the
isOpen() status of the element, then we
really do need to invalidate the height,
no matter what. */
void KduViewItem::setOpen(bool o){
	QListViewItem::setOpen(o);
//	o = listView()->isUpdatesEnabled();
//	listView()->setUpdatesEnabled(FALSE);
	QListViewItem::invalidateHeight();
//	listView()->setUpdatesEnabled(o);
}
/**  */
const char * KduViewItem::key(int column, bool) const
{
	switch (column) {
		case 0: return name();
		case 1: return QString().sprintf("%010d",sizeunknown ? 0 : size());
		default: return 0;
	}
}
