/***********************************************************************
**
**   airspace.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  2000 by Heiner Lamprecht, Florian Ehinger
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include "airspace.h"

#include <mapcalc.h>

#include <kapp.h>
#include <kiconloader.h>
#include <qdatastream.h>
#include <qfile.h>
#include <qtextstream.h>

QRegion* Airspace::drawRegion(QPainter* targetPainter,
        double dX, double dY, long mapCenterLon)
{
  extern double _currentScale;
  double lat, lon, projX, projY, drawX, drawY;

  QPointArray pointArray(locLength);

  switch(typeID) {
    case AirC:
      targetPainter->setBrush(NoBrush);
      targetPainter->setPen(QPen(QColor(0,120,0), 4));
      break;
    case AirCtemp:
      targetPainter->setBrush(QBrush(QColor(0,200,0),Dense6Pattern));
      targetPainter->setPen(QPen(QColor(0,120,0), 4));
      break;
    case AirD:
      targetPainter->setBrush(NoBrush);
      targetPainter->setPen(QPen(QColor(0,180,0), 3, DashLine));
      break;
    case AirDtemp:
      targetPainter->setBrush(QBrush(QColor(120,255,120), FDiagPattern));
      targetPainter->setPen(QPen(QColor(0,180,0), 3, DashLine));
      break;
    case AirElow:
      targetPainter->setBrush(NoBrush);
      targetPainter->setPen(QPen(QColor(200,100,100), 4));
      break;
    case AirEhigh:
      targetPainter->setBrush(NoBrush);
      targetPainter->setPen(QPen(QColor(100,100,175), 4));
      break;
    case AirF:
      targetPainter->setBrush(QBrush(QColor(100,100,175), Dense5Pattern));
      targetPainter->setPen(QPen(QColor(50,50,125), 2));
      break;
    case ControlD:
      targetPainter->setBrush(QBrush(QColor(200,50,50), Dense5Pattern));
      targetPainter->setPen(QPen(QColor(0,0,150), 2, DashLine));
      break;
    case Restricted:
      targetPainter->setBrush(QBrush(QColor(100,100,175), BDiagPattern));
      targetPainter->setPen(QPen(QColor(50,50,125), 2));
      break;
    case Danger:
      targetPainter->setBrush(QBrush(QColor(100,100,175), BDiagPattern));
      targetPainter->setPen(QPen(QColor(50,50,125), 2));
      break;
    case LowFlight:
      targetPainter->setBrush(QBrush(QColor(255,150,150), FDiagPattern));
      targetPainter->setPen(QPen(QColor(150,0,0), 3, DashLine));
      break;
  }
  for(unsigned int i = 0; i < locLength; i++) {
    lat = numToRad(latitude[i]);
    lon = numToRad(longitude[i] - mapCenterLon);
    projX = calc_X_Lambert(lat, lon);
    projY = calc_Y_Lambert(lat, lon);
    drawX = (projX * RADIUS / _currentScale) + dX;
    drawY = (projY * RADIUS / _currentScale) + dY;

    pointArray.setPoint(i, drawX, drawY);
  }
  targetPainter->drawPolygon(pointArray);

  return (new QRegion(pointArray));
}
/*
void Airspace::drawMapElement(QPainter* targetPainter,
        double dX, double dY, long mapCenterLon)
{
  extern double _scale[];
  extern double _currentScale;
  double lat, lon, projX, projY, drawX, drawY;

  QPointArray pointArray(locLength);

  switch(typeID) {
    case AirC:
      targetPainter->setBrush(NoBrush);
      targetPainter->setPen(QPen(QColor(0,120,0), 4));
      break;
    case AirCtemp:
      targetPainter->setBrush(QBrush(QColor(0,200,0),Dense6Pattern));
      targetPainter->setPen(QPen(QColor(0,120,0), 4));
      break;
    case AirD:
      targetPainter->setBrush(NoBrush);
      targetPainter->setPen(QPen(QColor(0,180,0), 3, DashLine));
      break;
    case AirDtemp:
      targetPainter->setBrush(QBrush(QColor(120,255,120), FDiagPattern));
      targetPainter->setPen(QPen(QColor(0,180,0), 3, DashLine));
      break;
    case AirElow:
      targetPainter->setBrush(NoBrush);
      targetPainter->setPen(QPen(QColor(200,100,100), 4));
      break;
    case AirEhigh:
      targetPainter->setBrush(NoBrush);
      targetPainter->setPen(QPen(QColor(100,100,175), 4));
      break;
    case AirF:
      targetPainter->setBrush(QBrush(QColor(100,100,175), Dense5Pattern));
      targetPainter->setPen(QPen(QColor(50,50,125), 2));
      break;
    case ControlD:
      targetPainter->setBrush(QBrush(QColor(200,50,50), Dense5Pattern));
      targetPainter->setPen(QPen(QColor(0,0,150), 2, DashLine));
      break;
    case Restricted:
      targetPainter->setBrush(QBrush(QColor(100,100,175), BDiagPattern));
      targetPainter->setPen(QPen(QColor(50,50,125), 2));
      break;
    case Danger:
      targetPainter->setBrush(QBrush(QColor(100,100,175), BDiagPattern));
      targetPainter->setPen(QPen(QColor(50,50,125), 2));
      break;
    case LowFlight:
      targetPainter->setBrush(QBrush(QColor(255,150,150), FDiagPattern));
      targetPainter->setPen(QPen(QColor(150,0,0), 3, DashLine));
      break;
  }
  for(unsigned int i = 0; i < locLength; i++) {
    lat = numToRad(latitude[i]);
    lon = numToRad(longitude[i] - mapCenterLon);
    projX = calc_X_Lambert(lat, lon);
    projY = calc_Y_Lambert(lat, lon);
    drawX = (projX * RADIUS / _currentScale) + dX;
    drawY = (projY * RADIUS / _currentScale) + dY;

    pointArray.setPoint(i, drawX, drawY);
  }
  targetPainter->drawPolygon(pointArray);
}
*/
bool Airspace::writeElementBin(const char* fileName)
{
  QFile outputFile(fileName);
  outputFile.open(IO_WriteOnly | IO_Append);
  QDataStream out(&outputFile);

  out << (Q_UINT8) typeID;
  out << (const char*) name;

  switch (typeID) {
    case AirC:
    case AirCtemp:
    case AirD:
    case AirDtemp:
    case AirF:
    case Restricted:
    case Danger:
      out << (Q_UINT16) lLimit;
      out << (Q_UINT8) lLimitType;
      out << (Q_UINT16) uLimit;
      out << (Q_UINT8) uLimitType;
      break;
    case ControlD:
      out << (Q_UINT16) uLimit;
      out << (Q_UINT8) uLimitType;
      break;
    case AirElow:
    case AirEhigh:
    case LowFlight:
      break;
  }
  out << (Q_UINT16) locLength;
  for(unsigned int i = 0; i < locLength; i++) {
    out << (Q_INT32) latitude[i];
    out << (Q_INT32) longitude[i];
  }

  return true;
}

bool Airspace::writeElementAscii(const char* fileName)
{
  QFile mapFile(fileName);
  if(!mapFile.open(IO_ReadWrite)) {
    return false;
  }
  mapFile.at(mapFile.size());
  QTextStream mapOut(&mapFile);

  /*************************************************************************
   *                                                                       *
   *  NOTE:                                                                *
   *  In the ascii-file, the name is written even if it is not used in the *
   *  map. This is, because with the name it is pretty much easier to      *
   *  find something in the ascii-file.                                    *
   *                                                                       *
   *************************************************************************/

  mapOut << "[NEW]\n"
         << "TYPE=" << typeID << endl
         << "NAME=" << name << endl;
  switch (typeID) {
    case AirC:
    case AirCtemp:
    case AirD:
    case AirDtemp:
    case AirF:
    case Restricted:
    case Danger:
      mapOut << "LOWER=" << lLimit << endl
             << "LTYPE=" << lLimitType << endl
             << "UPPER=" << uLimit << endl
             << "UTYPE=" << uLimitType << endl;
      break;
    case ControlD:
      mapOut << "UPPER=" << uLimit << endl
             << "UTYPE=" << uLimitType << endl;
      break;
    case AirElow:
    case AirEhigh:
    case LowFlight:
      break;
  }
  for(unsigned int i = 0; i < locLength; i++) {
    mapOut << latitude[i] << " " << (Q_INT32) longitude[i] << endl;
  }
  mapOut << "[END]\n";

  mapFile.close();

  return true;
}

Airspace::Airspace(QString n, unsigned int t, unsigned int length,
            long* latList, long* lonList,
            unsigned int upper = 0, unsigned int upperType = 0,
            unsigned int lower = 0, unsigned int lowerType = 0)
      : AreaElement(n, t, length, latList, lonList)
{
  lLimit = lower;
  lLimitType = lowerType;
  uLimit = upper;
  uLimitType = upperType;
}

Airspace::~Airspace()
{

}
