/***********************************************************************
**
**   intairport.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  2000 by Heiner Lamprecht, Florian Ehinger
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include "intairport.h"

#include <mapcalc.h>

#include <kapp.h>
#include <kiconloader.h>
#include <qdatastream.h>
#include <qfile.h>
#include <qtextstream.h>

QRegion* IntAirport::drawRegion(QPainter* targetPainter,
        double dX, double dY, long mapCenterLon)
{
  extern double _currentScale;
  double lat, lon, projX, projY, drawX, drawY, drawX2, drawY2;

  QPointArray pointArray(locLength);

  for(unsigned int i = 0; i < locLength; i++) {
    lat = numToRad(latitude[i]);
    lon = numToRad(longitude[i] - mapCenterLon);
    projX = calc_X_Lambert(lat, lon);
    projY = calc_Y_Lambert(lat, lon);
    drawX = (projX * RADIUS / _currentScale) + dX;
    drawY = (projY * RADIUS / _currentScale) + dY;

    pointArray.setPoint(i, drawX, drawY);
  }

  targetPainter->setBrush(QBrush(QColor(0,0,120),SolidPattern));
  targetPainter->setPen(QPen(QColor(0,0,0), 1));

  targetPainter->drawPolygon(pointArray);

  for(unsigned int i = 0; i < rwNum; i++) {
    lat = numToRad(rwData[i].latitudeA);
    lon = numToRad(rwData[i].longitudeA - mapCenterLon);
    projX = calc_X_Lambert(lat, lon);
    projY = calc_Y_Lambert(lat, lon);
    drawX = (projX * RADIUS / _currentScale) + dX;
    drawY = (projY * RADIUS / _currentScale) + dY;

    lat = numToRad(rwData[i].latitudeB);
    lon = numToRad(rwData[i].longitudeB - mapCenterLon);
    projX = calc_X_Lambert(lat, lon);
    projY = calc_Y_Lambert(lat, lon);
    drawX2 = (projX * RADIUS / _currentScale) + dX;
    drawY2 = (projY * RADIUS / _currentScale) + dY;

    targetPainter->setBrush(NoBrush);
    targetPainter->setPen(QPen(QColor(255, 0, 0), 3));
    targetPainter->drawLine(drawX, drawY, drawX2, drawY2);
  }
  return (new QRegion(pointArray));
}

bool IntAirport::writeElementBin(const char* fileName)
{
  QFile outputFile(fileName);
  if(!outputFile.open(IO_WriteOnly | IO_Append)) {
    return false;
  }
  QDataStream out(&outputFile);

  out << (Q_UINT8) typeID;
  out << (const char*) name;
  out << (const char*) alias;
  out << (Q_UINT16) elevation;
  out << (const char*) frequency;
  out << (Q_UINT8) vdf;
  out << (Q_UINT8) rwNum;
  for(unsigned int i = 0; i < rwNum; i++) {
    out << (Q_UINT8) rwData[i].direction;
    out << (Q_UINT16) rwData[i].length;
    out << (Q_UINT8) rwData[i].surface;
    out << (Q_INT32) rwData[i].latitudeA;
    out << (Q_INT32) rwData[i].longitudeA;
    out << (Q_INT32) rwData[i].latitudeB;
    out << (Q_INT32) rwData[i].longitudeB;
  }
  out << (Q_UINT16) locLength;
  for(unsigned int i = 0; i < locLength; i++) {
    out << (Q_INT32) latitude[i];
    out << (Q_INT32) longitude[i];
  }


  outputFile.close();

  return true;
}

bool IntAirport::writeElementAscii(const char* fileName)
{
  QFile mapFile(fileName);
  if(!mapFile.open(IO_ReadWrite)) {
    return false;
  }
  mapFile.at(mapFile.size());
  QTextStream mapOut(&mapFile);

  mapOut << "[NEW]\n"
         << "TYPE=" << typeID << endl
         << "NAME=" << name << endl
         << "ALIAS=" << alias << endl
//         << "AT=" << latitude << " " << longitude << endl
         << "ELEVATION=" << elevation << endl
         << "FREQUENCY=" << frequency << endl;
  for(unsigned int i = 0; i < rwNum; i++) {
    mapOut << "RUNWAY=" << rwData[i].direction << " "
           << rwData[i].length << " "
           << rwData[i].surface << " "
           << rwData[i].latitudeA << " "
           << rwData[i].longitudeA << " "
           << rwData[i].latitudeB << " "
           << rwData[i].longitudeB << endl;
  }
  for(unsigned int i = 0; i < locLength; i++) {
    mapOut << latitude[i] << " " << longitude[i] << endl;
  }
  mapOut << "[END]\n";
  mapFile.close();

  return true;
}

IntAirport::IntAirport(QString n, QString a,
              unsigned int length, long* latList, long* lonList,
              unsigned int elev, QString f, bool v = false,
              struct intrunway* rwDat = 0, unsigned int rwN = 0)
    : AreaElement(n, BaseMapElement::IntAirport, length, latList, lonList)
{
  alias = a;
  elevation = elev;
  frequency = f;
  vdf = v;
  rwData = rwDat;
  rwNum = rwN;
}

IntAirport::~IntAirport(){

}
