#include "khost.h"
#include <kapp.h>
#include <qmessagebox.h>
#include <kiconloader.h>
#include <StringListEditDialog.h>
#include <StringListSelectAndReorderSet.h>
#include <iostream.h>

extern "C" {
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
}

#define CONFIG_SECTION "config"
#define DATA_SECTION "data"
#define ENABLE_ACCESS_CONTROL "EnableAccessControl"
#define RESTORE_ON_STARTUP "RestoreOnStartup"
#define HOST_LIST "Hosts"
#define ALLOWED_HOSTS "Allowed"
#define USE_TILE "UseTile"

// #############################################################################
// MOC OUTPUT FILES:
#include "khost.moc"
// #############################################################################

KHost::KHost(const char* name)
  : KTMainWindow(name),
    control(false),
    restore(true)
{
  // ############################################################################
  KToolBar* toolbar=toolBar();
  KStatusBar* statusbar=statusBar();
  QSize size;
  // ----- create set of hosts:
  set=new StringListSelectAndReorderSet(this);
  size=set->sizeHint();
  if(size.width()<200) size.setWidth(200);
  if(size.height()<150) size.setHeight(150);
  // set->setMinimumSize(size);
  set->setFixedSize(size);
  setFrameBorderWidth(2);
  set->enableReordering(false);
  setView(set);
  set->setHeadlines(i18n("Used Hosts"), i18n("Hosts Allowed To Access"));
  connect(set, SIGNAL(itemSelected(const QString&)),
	  SLOT(add(const QString&)));
  connect(set, SIGNAL(itemUnselected(const QString&)),
	  SLOT(remove(const QString&)));
  // ----- construct toolbar:
  toolbar=toolBar();
  toolbar->insertButton(Icon("exit.xpm"),
			Exit, SIGNAL(clicked()),
			this, SLOT(exit()), true,
			i18n("Exit khost"));
  toolbar->insertButton(Icon("lock.xpm"),
			Control, SIGNAL(toggled(int)),
			this, SLOT(enableControl(int)), true,
			i18n("Enable host access control"));
  toolbar->setToggle(Control, true);
  toolbar->insertButton(Icon("idea.xpm"),
			NewHost, SIGNAL(clicked()),
			this, SLOT(editHostList()), true,
			i18n("Edit list of hosts"));
  /*
    toolbar->insertButton(Icon("tick.xpm"),
    AddHost, SIGNAL(clicked()),
    this, SLOT(add()), true,
    i18n("Allow access from this host"));
    toolbar->insertButton(Icon("delete.xpm"),
    RemoveHost, SIGNAL(clicked()),
    this, SLOT(remove()), true,
    i18n("Do not allow access from this host"));
  */
  toolbar->insertSeparator();
  toolbar->insertButton(Icon("configure.xpm"),
			Configure, SIGNAL(clicked()),
			this, SLOT(configure()), false,
			i18n("Configure khost"));
  toolbar->insertButton(Icon("help.xpm"),
			Help, SIGNAL(clicked()),
			this, SLOT(help()), false,
			i18n("Help"));
  toolbar->insertButton(Icon("mini-display.xpm"),
			About, SIGNAL(clicked()),
			this, SLOT(about()), false,
			i18n("About khost"));
  // ----- construct status bar:
  statusbar->insertItem("", 0);
  connect(&timer, SIGNAL(timeout()), SLOT(timeout()));
  // ----- other stuff:
  xhost=new XHostInterface;
  connect(kapp, SIGNAL(saveYourself()), SLOT(save()));
  connect(xhost, SIGNAL(setStatus(const QString&)),
	  SLOT(setStatus(const QString&)));
  // ----- check data file and reload last settings:
  checkDataDirectory();
  load();
  if(restore)
    {
      apply();
    } else {
      allowed.clear(); // only show known hosts
      control=false;
    }
  toolbar->setButton(Control, control);
  // ----- set allowed hosts:
  set->setValues(hosts);
  list<int> temp=allowed; // "allowed" changes when a host is selected
  set->select(temp);
  set->setEnabled(control);
  // -----  handle geometry:
  updateRects();
  // ############################################################################
}

KHost::~KHost()
{
  // ############################################################################
  delete xhost;
  // ############################################################################
}

void
KHost::exit()
{
  // ############################################################################
  if(QMessageBox::information
     (this, i18n("khost: Really quit?"),
      i18n("Really quit khost?\n"
	   "Use iconify to dock it in the panel."),
      i18n("Yes"), i18n("No"))==0)
    {
      save();
      kapp->quit();
    }
  // ############################################################################
}
  
void
KHost::enableControl(int)
{
  // ############################################################################
  control=toolBar()->isButtonOn(Control);
  if(xhost->enableHostAccessControl(control))
    {
      if(control)
	{
	  emit(setStatus(i18n("Access control enabled.")));
	} else {
	  emit(setStatus(i18n("Access control disabled.")));
	}
    } else {
      emit(setStatus(i18n("Error calling xhost.")));
    } 
  set->setEnabled(control);
  // ############################################################################
}
  
void
KHost::editHostList()
{
  // ############################################################################
  StringListEditDialog *dialog;
  QStrList selected;
  list<int> indices;
  int count;
  // -----
  set->getSelection(selected);
  dialog=new StringListEditDialog(this);
  dialog->setStrings(hosts);
  dialog->setCaption(i18n("khost: Edit list of used hosts"));
  if(dialog->exec())
    {
      dialog->getStrings(hosts);
      // ----- update the selector
      for(count=selected.count()-1; count>=0; --count)
	{
	  if(hosts.find(selected.at(count))==-1)
	    {
	      selected.remove(count);
	    } else {
	      indices.push_back(count);
	    }
	}
      allowed=indices;
      set->setValues(hosts);
      set->select(allowed);
    }
  // ############################################################################
}
  
void
KHost::add(const QString& host)
{
  // ############################################################################
  QString temp;
  // -----
  if(xhost->addAllowedHost(host))
    {
      temp.sprintf(i18n("Enabled access from host %s."), host.data());
    } else {
      temp=i18n("Error calling xhost.");
    }
  set->getSelection(allowed);
  emit(setStatus(temp));
  // ############################################################################
}

void
KHost::remove(const QString& host)
{
  // ############################################################################
  QString temp;
  // -----
  if(xhost->removeHostname(host))
    {
      temp.sprintf(i18n("Disabled access from host %s."), host.data());
    } else {
      temp=i18n("Error calling xhost.");
    }
  set->getSelection(allowed);
  emit(setStatus(temp));
  // ############################################################################
}
  
void
KHost::configure()
{
  // ############################################################################
  // ############################################################################
}
  
void
KHost::help()
{
  // ############################################################################
  // ############################################################################
}
  
void
KHost::about()
{
  // ############################################################################
  // ############################################################################
}
  
bool
KHost::load()
{
  // ############################################################################
  KeyValueMap *config;
  KeyValueMap *settings;
  // -----
  if(!data.load())
    {
      QMessageBox::information
	(this, i18n("khost: Intern Error"),
	 i18n("khost could not read its own data file.\n"
	      "khost will exit now."));
      kapp->quit();
    }
  // -----
  if(!data.get(CONFIG_SECTION, config))
    {
      return false;
    }
  if(!config->get(RESTORE_ON_STARTUP, restore))
    {
      return false;
    }
  if(!config->get(USE_TILE, usetile))
    {
      return false;
    }
  if(!data.get(DATA_SECTION, settings))
    {
      return false;
    }
  if(!settings->get(ENABLE_ACCESS_CONTROL, control))
    {
      return false;
    }
  if(!settings->get(HOST_LIST, hosts))
    {
      return false;
    }
  if(!settings->get(ALLOWED_HOSTS, allowed))
    {
      return false;
    }
  // -----
  return true;
  // ############################################################################
}

bool
KHost::save()
{
  // ############################################################################
  KeyValueMap *config;
  KeyValueMap *settings;
  // -----
  if(!data.get(CONFIG_SECTION, config))
    {
      return false;
    }
  if(!config->insert(RESTORE_ON_STARTUP, restore, true))
    {
      return false;
    }
  if(!config->insert(USE_TILE, usetile, true))
    {
      return false;
    }
  if(!data.get(DATA_SECTION, settings))
    {
      return false;
    }
  if(!settings->insert(ENABLE_ACCESS_CONTROL, control, true))
    {
      return false;
    }
  if(!settings->insert(HOST_LIST, hosts, true))
    {
      return false;
    }
  if(!settings->insert(ALLOWED_HOSTS, allowed, true))
    {
      return false;
    }
  // -----
  return data.save(i18n("Data file for khost."), true);
  // ############################################################################
}

bool
KHost::apply()
{
  // ############################################################################
  return true;
  // ############################################################################
}
  
void
KHost::setStatus(const QString& text)
{
  // ############################################################################
  messages.push_back(text);
  if(!timer.isActive())
    {
      timer.start(0);
    } else {
      timer.start(500);
    }
  // ############################################################################
}
  
void
KHost::timeout()
{
  // ############################################################################
  if(!messages.empty())
    {
      statusBar()->changeItem(messages.front(), 0);
      messages.pop_front();
      if(messages.empty())
	{
	  timer.start(5000, true);
	} else {
	  timer.start(1000, true);
	}
    } else {
      statusBar()->changeItem("", 0);
    }
  // ############################################################################
}
  
bool
KHost::checkDataDirectory()
{
  // ############################################################################
  QString dir=kapp->localkdedir()+"/share/apps/";
  QString mydir="khost/";
  QString filename="khost-settings";
  // -----
  if(::access(dir, F_OK | W_OK | X_OK)!=0)
    {
      QMessageBox::critical
	(this, i18n("khost: Environment error"),
	 i18n("khost could not find your local KDE directory.\n"
	      "Run kfm (the KDE filemanager) once to create it.\n"
	      "khost will exit now."));
      kapp->quit();
    }
  // -----
  mydir=dir+mydir;
  if(::access(mydir, F_OK | W_OK | X_OK)!=0)
    {
      if(mkdir(mydir, 0755)==-1)
	{
	  QMessageBox::information
	    (this, i18n("khost: Critical error"),
	     i18n("khost could not create your local data directory.\n"
		  "Something is wrong with your local KDE directory,\n"
		  "possibly you do not have writing permission.\n"
		  "khost will exit now."));
	  kapp->quit();
	} else {
	  QMessageBox::information
	    (this, i18n("khost: Directory created"),
	     i18n("khost has created the directory\n    ")
	     +mydir+
	     i18n("\nwhere you local data will be stored."));
	}
    }
  if(!data.setFileName(mydir+filename, true, false))
    {
      if(!createDataFile())
	{
	  QMessageBox::information
	    (this, i18n("khost: Critical error"),
	     i18n("khost could not create your data file.directory.\n"
		  "This might be an intern error.\n"
		  "khost will exit now."));
	  kapp->quit();
	}
    }
  return true;
  // ############################################################################
}
  

bool
KHost::createDataFile()
{
  // ############################################################################
  const QString dir=kapp->localkdedir()+"/share/apps/";
  const QString mydir="khost/";
  const QString filename="khost-settings";
  // -----
  if(!data.setFileName(dir+mydir+filename, false, false))
    {
      return false;
    }
  // -----
  data.clear();
  if(!data.createSection(CONFIG_SECTION)
     || !data.createSection(DATA_SECTION))
    {
      return false;
    }
  // -----
  return restoreDefaults();
  // ############################################################################
}

bool
KHost::restoreDefaults()
{ // this method does not reset the host list
  // ############################################################################
  const bool EnableAccessControl=true, 
    RestoreOnStartup=true,
    UseTile=true; // the defaults
  KeyValueMap *config;
  KeyValueMap *settings;
  // -----   
  if(!data.get(CONFIG_SECTION, config))
    {
      return false;
    }
  if(!config->insert(RESTORE_ON_STARTUP, RestoreOnStartup, true))
    {
      return false;
    }
  if(!config->insert(USE_TILE, UseTile, true))
    {
      return false;
    }
  if(!data.get(DATA_SECTION, settings))
    {
      return false;
    }
  if(!settings->insert(ENABLE_ACCESS_CONTROL, EnableAccessControl, true))
    {
      return false;
    }
  // -----
  return data.save(i18n("Data file for khost."), true);
  // ############################################################################
}
  
int main(int argc, char** argv)
{
  // ############################################################################
  KApplication app(argc, argv);
  KHost *khost=new KHost;
  int rc;
  // -----
  app.setMainWidget(khost);
  khost->show();
  rc=app.exec();
  // -----
  delete khost;
  return rc;
  // ############################################################################
}

