/****************************************************************
**
** Implementation of PagePanel class
**
****************************************************************/

#include <assert.h>
#include <kapp.h>
#include <qdatetime.h>

#include "pagepanel.h"

extern "C" {
    #include <stdio.h>
}

const char *PagePanel::statName = "/proc/stat";

// Constructor f. Page Panel
PagePanel::PagePanel(QWidget *parent, const char *name) :
    LPanel(2,parent,name)
{
    setIncremental(true);
    setFilled(true);
    setToolTip(i18n("Paging activity (In/Out); scale = 1000/sec, dynamic"));
    setDescription(i18n("&Paging activity"));
    setName("Paging");

    setYScale(1000); // Initially, but will be overridden with next redraw event
    setYLines(1000);  // Denote dynamic Scaling
    
    setGraphColor(0,QColor(green));     // In
    setGraphColor(1,QColor(blue));      // Out
    
    setBackgroundColor(QColor(darkGray));
    
    getNewData(0);
    // Start timer 
    //startTimer(getRedrawInterval());
}


// Destructor f. CPU Panel
PagePanel::~PagePanel()
{
}


// read another Line from virtual Paging Statistics file
// page 550044 797364
// swap 51005 358009
bool PagePanel::getNewData(FArray *row) 
{
    bool retval;
    char line[255]; // Should be enough
    FILE *inpfile;
    unsigned long pIn = 0, pOut = 0,x1,x2;
    QDateTime now;
    static unsigned long oldpIn = 0,oldpOut = 0;
    static QDateTime lastTime;
    
    inpfile = fopen(statName,"r");
    if (!inpfile) {
        cerr << "PagePanel::PagePanel: Could not open file " << statName << endl;
        exit(-1);
    }
    while (fgets(line,sizeof(line),inpfile)) {
        if (!strncmp(line,"page",4)) {
            if (2 == sscanf(line + 5, "%lu %lu",&x1,&x2)) {
                pIn += x1; pOut += x2;
            }
            else {
                cerr << "Unparseable Page information in " << line;
                return false;
            }
        }
        else if (!strncmp(line,"swap",4)) {
            if (2 == sscanf(line + 5, "%lu %lu",&x1,&x2)) {
                pIn += x1; pOut += x2;
            }
            else {
                cerr << "Unparseable Swap information in " << line;
                return false;
            }
        }
    }
    fclose(inpfile);

    now = QDateTime::currentDateTime();
    
    // Compute derivatives
    // Beware: Redraw Inteval must be at least ONE SECOND!!!
    if (oldpIn != 0) {
        int deltaSec = lastTime.secsTo(now);
        if (deltaSec < 1) deltaSec = 1; // FixMe: higher time resolution!
        assert(row); 
        (*row)[0] = (pIn  - oldpIn)  / deltaSec;
        (*row)[1] = (pOut - oldpOut) / deltaSec;
        
        retval = true;
    }    
    else {
        retval = false;
    }
    
    oldpIn = pIn;
    oldpOut = pOut;
    lastTime = now;
    
    return retval;
}
