/*
	This file is part of `klp', a KDE Line Printer queue manager

	Copyright (C) 1998
	Frans van Dorsselaer
	<dorssel@MolPhys.LeidenUniv.nl>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "lpr_common.h"
#include "inter.h"
#include "server.h"

#define i18n(s)	s


static int check_common(char *reply, char *errors, int retval)
{
	if (strstr(errors, "root privileges required")) {
		iwrite_fail(i18n("Root privileges required"));
		return 0;
	}
	if (strstr(errors, "cannot connect")) {
		iwrite_fail(i18n("Unable to connect to remote print server"));
		return 0;
	}
	if (strstr(reply, "unknown printer")) {
		iwrite_fail(i18n("Unknown printer"));
		return 0;
	}
	if (retval) {
		fprintf(stderr, "klp: child exited with status %d\n", retval);
		iwrite_fail(i18n("Error executing child process"));
		return 0;
	}
	return 1;
}


/*
 *	parse_next_job
 *
 *	parse somthing like:
 *
 *	1st   username   123      stupid filename of 27 bytes       9876 bytes
 *
 *      \_/   \______/   \_/      \_________________________/       \________/
 *
 *      rank  user       jobid    jobname                           size
 *
 *	Note that rank `active' counts as rank 0.
 *	returns 0 on fail, 1 on success.
 */
static struct queue_item *parse_next_job(const char *line)
{
	struct queue_item *item = NULL;
	char *line_cp;
	char *l;
	char *p;
	int job_number;

	item = xmalloc(sizeof(struct queue_item));
	*item = empty_queue_item;
	while (isspace(*line))
		++line;
	l = line_cp = xstrdup(line);
	if (isdigit(*l)) {
		item->rank = atoi(l);
		while (isalnum(*l))
			++l;
	} else if (!strncmp(l, "active", 6)) {
		l += 6;
		item->rank = 0;
	} else
		goto fail;
	if (!isspace(*l))
		goto fail;

	while (isspace(*l))
		++l;
	if (!isgraph(*l))
		goto fail;
	p = l;
	while(isgraph(*l))
		++l;
	if (!isspace(*l))
		goto fail;
	*(l++) = '\0';
	item->user = xstrdup(p);
	
	while (isspace(*l))
		++l;
	if (!isdigit(*l))
		goto fail;
	job_number = atoi(l);
	while (isdigit(*l))
		++l;
	if (!isspace(*l))
		goto fail;

	while (isspace(*l))
		++l;

	p = l + strlen(l);
	while ((p - l > 0) && isspace(*(p - 1)))
		--p;
	*p = '\0';
	while ((p - l > 0) && isgraph(*(p - 1)))
		--p;
	if (strcmp(p, "bytes") || !isspace(*(p - 1)))
		goto fail;
	while ((p - l > 0) && isspace(*(p - 1)))
		--p;
	*p = '\0';
	while ((p - l > 0) && isgraph(*(p - 1)))
		--p;
	if (!isspace(*(p - 1)))
		goto fail;
	item->size = 0;
	item->size = atoi(p);
	
	while ((p - l > 0) && isspace(*(p - 1)))
		--p;
	*p = '\0';

	item->job_name = xstrdup(l);
	item->job_id = xmalloc(4);
	sprintf(item->job_id, "%d", job_number % 1000);
	free(line_cp);
	return item;

fail:
	free(line_cp);
	free_queue_item(item);
	return NULL;
}


struct queue *lpr_q_parse(char *reply, char *errors, int retval)
{
	struct queue *queue;
	char *this_line;
	char *next_line;
	struct queue_item *item;

	if (!check_common(reply, errors, retval))
		return NULL;
	queue = xmalloc(sizeof(struct queue));
	*queue = empty_queue;
	next_line = reply;
	do {
		this_line = next_line;
		next_line = strchr(this_line, '\n');
		if (next_line)
			*(next_line++) = '\0';
		if ((item = parse_next_job(this_line))) {
			queue->items = xrealloc(queue->items, (queue->item_count + 1) * sizeof(struct queue_item *));
			queue->items[queue->item_count++] = item;
		} else if (strstr(this_line, "is down"))
			queue->no_print = 1;
		else if (strstr(this_line, "queue is turned off"))
			queue->no_queue = 1;
	} while (next_line);
	return queue;
}


void lpr_p_parse(const char * const files[], char *reply, char *errors, int retval)
{
	if (!check_common(reply, errors, retval))
		return;
	iwrite_char('p');
}


void lpr_r_parse(const char * const jobids[], char *reply, char *errors, int retval)
{
	if (!check_common(reply, errors, retval))
		return;
	iwrite_char('r');
}
