/*
	This file is part of `klp', a KDE Line Printer queue manager

	Copyright (C) 1998
	Frans van Dorsselaer
	<dorssel@MolPhys.LeidenUniv.nl>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "options.h"


#ifdef HAVE_GETOPT_H
static struct option long_options[] = {
	{ "dock", 0, NULL, 'd' },
	{ "no-dock", 0, NULL, 'n' },
	{ "hide", 0, NULL, 'h' },
	{ "show", 0, NULL, 's' },
	{ "printer", 1, NULL, 'p' },
	{ "help", 0, NULL, '?' },
	{ NULL, 0, NULL, 0 }
};
#endif


static void version(void)
{
	printf("klp " VERSION "\n");
}


static void usage_and_exit(void)
{
	version();
	printf(	"\n"
		"klp lets you manage your print queue.  It uses a graphical frontend\n"
		"designed for use with the K Desktop Environment.\n"
		"\n"
		"Usage: klp [OPTION]...\n"
		"\n"
		"Options override the settings as saved in the configuration file.\n"
		"\n"
		"Options:\n"
#ifdef HAVE_GETOPT_H
		"  -d, --dock          Dock in the panel\n"
		"  -h, --hide          Hide the main window (implies --dock)\n"
		"  -n, --no-dock       Do not dock in the panel (implies --show)\n"
		"  -p, --printer=NUM   Set printer NUM as the active printer (0 for none)\n"
		"  -s, --show          Show the main window\n"
		"  -?, --help          Displays this usage information\n"
#else
		"  -d       Dock in the panel\n"
		"  -h       Hide the main window (implies --dock)\n"
		"  -n       Do not dock in the panel (implies --show)\n"
		"  -p NUM   Set printer NUM as the active printer (0 for none)\n"
		"  -s       Show the main window\n"
		"  -?       Displays this usage information\n"
#endif
		"\n"
		"Please report bugs to <dorssel@MolPhys.LeidenUniv.nl>\n");
	exit(EXIT_FAILURE);
}


void parse_options(struct options *opts, int argc, char *argv[])
{
	int o;
	char *p;

	opts->set.all = 0;
#ifdef HAVE_GETOPT_H
	while ((o = getopt_long(argc, argv, "dhnp:s?", long_options, NULL)) != EOF) {
#else
	while ((o = getopt(argc, argv, "dhnp:s?")) != EOF) {
#endif
		switch (o) {
			case 'd':
				opts->set.opt.dock = 1;
				opts->dock = 1;
				break;
			case 'h':
				opts->set.opt.dock = 1;
				opts->set.opt.hide = 1;
				opts->dock = 1;
				opts->hide = 1;
				break;
			case 'n':
				opts->set.opt.dock = 1;
				opts->set.opt.hide = 1;
				opts->dock = 0;
				opts->hide = 0;
				break;
			case 'p':
				opts->set.opt.active_printer = 1;
				opts->active_printer = strtoul(optarg, &p, 10);
				if (p != optarg + strlen(optarg)) {
#ifdef HAVE_GETOPT_H
					fprintf(stderr, "%s: option `-p, --printer' requires an unsigned number argument\n", argv[0]);
#else
					fprintf(stderr, "%s: option `-p' requires an unsigned number argument\n", argv[0]);
#endif
					usage_and_exit();
				}
				break;
			case 's':
				opts->set.opt.hide = 1;
				opts->hide = 0;
				break;
			case '?':
				usage_and_exit();
				break;
			default:
				abort();
		}
	}
	if (optind < argc) {
		fprintf(stderr, "%s: non-option argument `%s'\n", argv[0], argv[optind]);
		usage_and_exit();
	}
}
