/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <o.gantz@tu-bs.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include "preview.h"
#include "preview.moc"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#undef TrueColor	// Avoid Qt <-> X11 conflict

#include <qglobal.h>
#include <qwidget.h>
#include <qpixmap.h>
#include <qevent.h>
#include <qkeycode.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qscrollbar.h>

#include <kmenubar.h>
#include <ktoolbar.h>
#include <kstatusbar.h>
#include <kapp.h>
#include <kiconloader.h>
#include <kmsgbox.h>

#include "mlofile.h"
#include "printdlg.h"
#include "maildlg.h"
#include "global.h"


#define TOOL_ID_PREV    0
#define TOOL_ID_NEXT    1
#define TOOL_ID_FIRST   2
#define TOOL_ID_LAST    3
#define TOOL_ID_ZOOMIN  5
#define TOOL_ID_ZOOMOUT 6



FaxWidget * fw = 0;


char * shrinkPage( int w, int h, char * data )
{
	static const uchar shrink_tab[128] = {
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f
	};
	char * ndata;
	uchar * line, * nline;
	int nw, nh, i, j, k, bpl, nbpl;

	nw = w >> 1;
	nh = h >> 1;

	bpl = (((w-1) >> 4) + 1) << 1;
	nbpl = (((nw-1) >> 4) + 1) << 1;
	ndata = (char *)malloc(nbpl * nh);

	line = (uchar *)data;
	nline = (uchar *)ndata;

	for (i=0; i < nh; i++) {
		if (i & 1)
			line += bpl;
		for (j=0; j < nbpl; j++) {
			k = j << 1;
			if (j & 1)
				nline[j] = shrink_tab[line[k] >> 1] | (shrink_tab[line[k|1] >> 1] << 4);
			else
				nline[j] = shrink_tab[line[k] & 0x7f] | (shrink_tab[line[k|1] & 0x7f] << 4);
		}
		line += bpl;
		if (!(i & 1))
			line += bpl;
		nline += nbpl;
	}

	return ndata;
}



FaxWidget::FaxWidget( QWidget * parent, const char * name, WFlags f ) : QWidget( parent, name, f )
{
	int screen;

	scrollx = 0;
	scrolly = 0;

	display = x11Display();
	screen = x11Screen();

	window = XCreateSimpleWindow(display, winId(), 0, 0, 1, 1, 0,
		 BlackPixel(display, screen),
		 WhitePixel(display, screen));
	gc = XCreateGC(display, window, 0L, (XGCValues *)0);
	XSetForeground(display, gc, BlackPixel(display, screen));
	XSetBackground(display, gc, WhitePixel(display, screen));
	XSetFunction(display, gc, GXcopy);
	XSelectInput(display, window, ButtonPressMask | ButtonReleaseMask | Button1MotionMask | ExposureMask);
	XFlush(display);

	move_cursor = XCreateFontCursor(display, XC_fleur);
	moving = FALSE;

	image = 0;
}


FaxWidget::~FaxWidget()
{
	deleteImage();
	XFreeCursor(display, move_cursor);
	XFreeGC(display, gc);
	XDestroyWindow(display, window);
}


void FaxWidget::setImage( int w, int h, char * data )
{
	deleteImage();

	im_width = w;
	im_height = h;

	if ((w) && (h)) {
		image = XCreateImage(display, DefaultVisual(display, x11Screen()),
			1, XYBitmap, 0, data, im_width, im_height, 16, 0);
		XMapRaised(display, window);
	} else {
		XUnmapWindow(display, window);
		XResizeWindow(display, window, 1, 1);
	}
}


void FaxWidget::deleteImage()
{
	if (image) {
		image->data = 0;
		XDestroyImage(image);
		image = 0;
	}
}


void FaxWidget::x11EventFilter( XEvent * ev )
{
	bool refresh = FALSE;
	bool move_it = FALSE;
	int mv_x = 0, mv_y = 0;
	int x1 = INT_MAX, y1 = INT_MAX, x2 = 0, y2 = 0, xb, yb;

	do {
		switch( ev->type ) {
			case ButtonPress:
				if (ev->xbutton.button == Button1) {
					XDefineCursor(display, window, move_cursor);
					moving = TRUE;
					move_x = ev->xbutton.x;
					move_y = ev->xbutton.y;
				}
				break;
			case ButtonRelease:
				if (ev->xbutton.button == Button1) {
					XUndefineCursor(display, window);
					moving = FALSE;
				}
				break;
			case MotionNotify:
				if (moving) {
					move_it = TRUE;
					mv_x = ev->xmotion.x;
					mv_y = ev->xmotion.y;
				}
				break;
			case Expose:
				refresh = TRUE;
				xb = ev->xexpose.x + ev->xexpose.width;
				yb = ev->xexpose.y + ev->xexpose.height;
				if (ev->xexpose.x < x1)
					x1 = ev->xexpose.x;
				if (ev->xexpose.y < y1)
					y1 = ev->xexpose.y;
				if (xb > x2)
					x2 = xb;
				if (yb > y2)
					y2 = yb;
			default:
				break;
		}
	} while (XCheckWindowEvent(display, window, ButtonPressMask | ButtonReleaseMask | Button1MotionMask | ExposureMask, ev));

	if (refresh && image)
		XPutImage(display, window, gc, image, scrollx + x1, scrolly + y1, x1, y1, x2-x1, y2-y1);

	if (move_it) {
		emit move( mv_x - move_x, mv_y - move_y );
		move_x = mv_x;
		move_y = mv_y;
	}
}	


void FaxWidget::setXScroll( int x )
{
	scrollx = x;
	repaint( FALSE );
}


void FaxWidget::setYScroll( int y )
{
	scrolly = y;
	repaint( FALSE );
}


void FaxWidget::paintEvent( QPaintEvent * )
{
	int w, h, x = 0, y = 0;

	if ((w = width()) > im_width) {
		x = (w - im_width) >> 1;
		w = im_width;
	}
	if ((h = height()) > im_height) {
		y = (h - im_height) >> 1;
		h = im_height;
	}
	if (image) {
		XMoveResizeWindow(display, window, x, y, w, h);
		XPutImage(display, window, gc, image, scrollx, scrolly, 0, 0, w, h);
	}
}



Preview::Preview( QWidget * parent, const char * name, WFlags f ) : QWidget( parent, name, f )
{
	int i;
	QPixmap pix;
	KIconLoader * kil = mykapp->getIconLoader();

	fname = 0;
	salias = 0;
	for (i=0; i < 3; i++)
		page_datas[i] = 0;
	page = pages = 0;
	page_width = page_height = 0;
	zoom_width = zoom_height = 0;
	zoom = 1;

	QPopupMenu * file_menu = new QPopupMenu( 0, "prefilemenu" );
	CHECK_PTR( file_menu );
	file_menu->insertItem( i18n("&Print..."), this, SLOT(printFile()), CTRL+Key_P );
	file_menu->insertItem( i18n("&Mail..."), this, SLOT(mailFile()), CTRL+Key_M );
	file_menu->insertSeparator();
	file_menu->insertItem( i18n("&Close"), this, SLOT(hide()), CTRL+Key_W );

	QPopupMenu * view_menu = new QPopupMenu( 0, "previewmenu" );
	CHECK_PTR( view_menu );
	menu_item_prev = view_menu->insertItem( i18n("&Previous Page"), this, SLOT(prevPage()), ALT+Key_Left );
	menu_item_next = view_menu->insertItem( i18n("&Next Page"), this, SLOT(nextPage()), ALT+Key_Right );
	menu_item_first = view_menu->insertItem( i18n("&First Page"), this, SLOT(firstPage()), ALT+Key_Home );
	menu_item_last = view_menu->insertItem( i18n("&Last Page"), this, SLOT(lastPage()), ALT+Key_End );
	view_menu->insertSeparator();
	menu_item_zoomin = view_menu->insertItem( i18n("Zoom &In"), this, SLOT(zoomIn()),  ALT+Key_Plus );
	menu_item_zoomout = view_menu->insertItem( i18n("Zoom &Out"), this, SLOT(zoomOut()),  ALT+Key_Minus );

	QPopupMenu * opt_menu = new QPopupMenu( 0, "preoptmenu" );
	CHECK_PTR( opt_menu );
	menu_item_tool = opt_menu->insertItem(i18n("Show &Toolbar"), this, SLOT(toggleToolbar()), CTRL+Key_T );
	opt_menu->setItemChecked( menu_item_tool, TRUE );
	menu_item_status = opt_menu->insertItem(i18n("Show Status&bar"), this, SLOT(toggleStatusbar()), CTRL+Key_S );
	opt_menu->setItemChecked( menu_item_status, TRUE );

	menuBar = new KMenuBar( this, "premenubar" );
	CHECK_PTR( menuBar );

	menuBar->insertItem( i18n("&File"), file_menu );
	menuBar->insertItem( i18n("&View"), view_menu );
	menuBar->insertItem( i18n("&Options"), opt_menu );

	menuBar->show();


	toolBar = new KToolBar( this, "pretoolbar" );
	CHECK_PTR( toolBar );

	pix = kil->loadIcon( "back.xpm" );
	toolBar->insertButton( pix, TOOL_ID_PREV, SIGNAL(clicked()), this, SLOT(prevPage()), TRUE, i18n("Previous Page") );
	pix = kil->loadIcon( "forward.xpm" );
	toolBar->insertButton( pix, TOOL_ID_NEXT, SIGNAL(clicked()), this, SLOT(nextPage()), TRUE, i18n("Next Page") );
	pix = kil->loadIcon( "start.xpm" );
	toolBar->insertButton( pix, TOOL_ID_FIRST, SIGNAL(clicked()), this, SLOT(firstPage()), TRUE, i18n("First Page") );
	pix = kil->loadIcon( "finish.xpm" );
	toolBar->insertButton( pix, TOOL_ID_LAST, SIGNAL(clicked()), this, SLOT(lastPage()), TRUE, i18n("Last Page") );
	toolBar->insertSeparator( 4 );
	pix = kil->loadIcon( "viewmag+.xpm" );
	toolBar->insertButton( pix, TOOL_ID_ZOOMIN, SIGNAL(clicked()), this, SLOT(zoomIn()), TRUE, i18n("Zoom In") );
	pix = kil->loadIcon( "viewmag-.xpm" );
	toolBar->insertButton( pix, TOOL_ID_ZOOMOUT, SIGNAL(clicked()), this, SLOT(zoomOut()), TRUE, i18n("Zoom Out") );
	toolBar->insertSeparator( 7 );
	pix = kil->loadIcon( "fileprint.xpm" );
	toolBar->insertButton( pix, 8, SIGNAL(clicked()), this, SLOT(printFile()), TRUE, i18n("Print Facsimile") );
	pix = kil->loadIcon( "send.xpm" );
	toolBar->insertButton( pix, 9, SIGNAL(clicked()), this, SLOT(mailFile()), TRUE, i18n("Mail Facsimile") );
	pix = kil->loadIcon( "exit.xpm" );
	toolBar->insertButton( pix, 10, SIGNAL(clicked()), this, SLOT(hide()), TRUE, i18n("Close") );
	toolBar->alignItemRight( 10, TRUE );

	toolBar->show();
	toolBarOn = TRUE;


	statusBar = new KStatusBar( this, "prestatusbar" );
	CHECK_PTR( statusBar );
	
	statusBar->insertItem( "00000000000000.FAX", 0 );
	statusBar->insertItem( i18n("000000000, Normal"), 1 );
	statusBar->setAlignment( 1, AlignHCenter );
	statusBar->insertItem( i18n("Page 000 of 000"), 2 );
	statusBar->setAlignment( 2, AlignHCenter );
	statusBar->insertItem( "", 3 );
	statusBar->changeItem( "", 0 );
	statusBar->changeItem( "", 1 );
	statusBar->changeItem( "", 2 );

	statusBar->show();
	statusBarOn = TRUE;

	view = new FaxWidget( this, "faxwidget" );
	CHECK_PTR( view );
	fw = view;

	hscroll = new QScrollBar( QScrollBar::Horizontal, this, "prehscroll" );
	CHECK_PTR( hscroll );
	hscroll->setFixedHeight( hscroll->sizeHint().height() );
	vscroll = new QScrollBar( QScrollBar::Vertical, this, "prevscroll" );
	CHECK_PTR( vscroll );
	vscroll->setFixedWidth( vscroll->sizeHint().width() );

	connect( menuBar, SIGNAL(moved(menuPosition)), SLOT(updateRects()) );
	connect( toolBar, SIGNAL(moved(BarPosition)), SLOT(updateRects()) );
	connect( view, SIGNAL(move(int, int)), SLOT(moveSlot(int, int)) );
	connect( hscroll, SIGNAL(valueChanged(int)), view, SLOT(setXScroll(int)) );
	connect( vscroll, SIGNAL(valueChanged(int)), view, SLOT(setYScroll(int)) );

	setCaption( i18n("KMLOFax Preview") );
}


Preview::~Preview()
{
	if (fname)
		free(fname);
	if (salias)
		free(salias);
	deletePages();
}


void Preview::toggleToolbar()
{
	toolBarOn = !toolBarOn;

	menuBar->setItemChecked( menu_item_tool, toolBarOn );
	if (toolBarOn)
		toolBar->show();
	else
		toolBar->hide();
	updateRects();
}


void Preview::toggleStatusbar()
{
	statusBarOn = !statusBarOn;

	menuBar->setItemChecked( menu_item_status, statusBarOn );
	if (statusBarOn)
		statusBar->show();
	else
		statusBar->hide();
	updateRects();
}


void Preview::showFax( const char * name, const char * alias )
{
	statusBar->changeItem( "", 0 );
	statusBar->changeItem( "", 3 );

	file.setName( expand_path(name) );
	if (!file.open()) {
		KMsgBox::message( 0, i18n("File Error"), i18n("Cannot open facsimile file."), KMsgBox::EXCLAMATION);
		return;
	}
	pages = file.pages();
	file.close();

	if (fname)
		free(fname);
	fname = strdup(name);

	if (salias)
		free(salias);
	salias = strdup(alias);

	statusBar->changeItem( fname, 0 );
	statusBar->changeItem( salias, 3 );

	loadPage( 1 );

	if (isVisible())
		raise();
	else
		show();
}


void Preview::show()
{
	if (menuBar->menuBarPos() == KMenuBar::Floating)
		menuBar->show();
	if (toolBar->barPos() == KToolBar::Floating)
		toolBar->show();
	QWidget::show();
	updateRects();
}


void Preview::hide()
{
	if (menuBar->menuBarPos() == KMenuBar::Floating)
		menuBar->hide();
	if (toolBar->barPos() == KToolBar::Floating)
		toolBar->hide();
	QWidget::hide();
	deletePages();
}


void Preview::deletePages()
{
	int i;

	page_width = page_height = 0;

	for (i=0; i < 3; i++)
		if (page_datas[i]) {
			free(page_datas[i]);
			page_datas[i] = 0;
		}
	view->deleteImage();
}


void Preview::showPage()
{
	menuBar->setItemEnabled( menu_item_zoomin, zoom > 0 );
	menuBar->setItemEnabled( menu_item_zoomout, zoom < 2 );
	toolBar->setItemEnabled( TOOL_ID_ZOOMIN, zoom > 0 );
	toolBar->setItemEnabled( TOOL_ID_ZOOMOUT, zoom < 2 );

	zoom_width = page_width >> zoom;
	zoom_height = page_height >> zoom;
	
	view->setImage( zoom_width, zoom_height, page_datas[zoom] );
	
	updateRects();
	view->repaint( FALSE );
}


void Preview::loadPage( int p )
{
	page_info_t info;
	char buff[32], * pd;
	int i, bpl;

	statusBar->changeItem( "", 1 );
	statusBar->changeItem( "", 2 );

	deletePages();

	page = p;

	menuBar->setItemEnabled( menu_item_prev, page != 1 );
	menuBar->setItemEnabled( menu_item_next, page != pages );
	menuBar->setItemEnabled( menu_item_first, page != 1 );
	menuBar->setItemEnabled( menu_item_last, page != pages );
	toolBar->setItemEnabled( TOOL_ID_FIRST, page != 1 );
	toolBar->setItemEnabled( TOOL_ID_PREV, page != 1 );
	toolBar->setItemEnabled( TOOL_ID_NEXT, page != pages );
	toolBar->setItemEnabled( TOOL_ID_LAST, page != pages );

	if (!file.open()) {
		KMsgBox::message( 0, i18n("File Error"), i18n("Cannot open facsimile file."), KMsgBox::EXCLAMATION);
		showPage();
		return;
	}
	if (!file.readPageInfo( p, &info )) {
		KMsgBox::message( 0, i18n("File Error"), i18n("Cannot read facsimile page info."), KMsgBox::EXCLAMATION);
		file.close();
		showPage();
		return;
	}

	page_width = info.width;
	page_height = file.pageHeight( page );

	sprintf(buff, "%d%d, %s", page_width, page_height, (info.lpi == 98) ? i18n("Normal") : i18n("Fine"));
	statusBar->changeItem( buff, 1 );

	sprintf(buff, i18n("Page %d of %d"), page, pages);
	statusBar->changeItem( buff, 2 );
	
	mykapp->setOverrideCursor( waitCursor );
	mykapp->processEvents( 100 );

	bpl = (((page_width-1) >> 4) + 1) << 1;
	if (info.lpi == 98)
		page_datas[0] = (char *)malloc(bpl * page_height * 2);
	else
		page_datas[0] = (char *)malloc(bpl * page_height);
	pd = page_datas[0];

	file.gotoPage( page );
	if (info.lpi == 98) {
		for (i=0; i < page_height; i++) {
			file.readImgLine( (uchar *)pd );
			memcpy((void *)(pd + bpl), (void *)pd, bpl);
			pd += (bpl << 1);
		}
		page_height <<= 1;
	} else {
		for (i=0; i < page_height; i++) {
			file.readImgLine( (uchar *)pd );
			pd += bpl;
		}
	}
	file.close();

	page_datas[1] = shrinkPage( page_width, page_height, page_datas[0] );
	page_datas[2] = shrinkPage( page_width >> 1, page_height >> 1, page_datas[1] );

	mykapp->restoreOverrideCursor();

	showPage();
}


void Preview::prevPage()
{
	if (page > 1)
		loadPage( page-1 );
}


void Preview::nextPage()
{
	if (page < pages)
		loadPage( page+1 );
}


void Preview::firstPage()
{
	if (page > 1)
		loadPage( 1 );
}


void Preview::lastPage()
{
	if (page < pages)
		loadPage( pages );
}


void Preview::zoomIn()
{
	if (zoom) {
		zoom--;
		showPage();
	}
}


void Preview::zoomOut()
{
	if (zoom < 2) {
		zoom++;
		showPage();
	}
}


void Preview::printFile()
{
	printdlg->printFax( fname, salias );
}


void Preview::mailFile()
{
	maildlg->mailFax( fname, salias );
}


void Preview::moveSlot( int x, int y )
{
	hscroll->setValue( hscroll->value() - x );
	vscroll->setValue( vscroll->value() - y );
}


void Preview::updateRects()
{
#define vSpace 2
	int x, y, w, h, t;
	bool need_hscroll, need_vscroll;

	x = y = 0;
	w = width();
	h = height();

	if (menuBar->menuBarPos() == KMenuBar::Top) {
		t = menuBar->heightForWidth( w );
		menuBar->setGeometry( 0, 0, w, t);
		y += t;
		h -= t;
	} else if (menuBar->menuBarPos() == KMenuBar::Bottom) {
		t = menuBar->heightForWidth( w );
		h -= t;
		menuBar->setGeometry( 0, y+h, w, t);
	}

	if (statusBar->isVisible()) {
		t = statusBar->height();
		h -= t;
		statusBar->setGeometry( 0, y+h, w, t );
	}

	if (toolBar->isVisible()) {
		if (toolBar->barPos() == KToolBar::Left) {
			toolBar->setMaxHeight( h );
			toolBar->updateRects( TRUE );
			t = toolBar->width();
			toolBar->move( x, y );
			x += t;
			w -= t;
		}
		if (toolBar->barPos() == KToolBar::Right) {
			toolBar->setMaxHeight( h );
			toolBar->updateRects( TRUE );
			t = toolBar->width();
			w -= t;
			toolBar->move( x+w, y );
		}
		if (toolBar->barPos() == KToolBar::Top) {
			toolBar->setMaxWidth( w );
			toolBar->updateRects( TRUE );
			t = toolBar->height();
			toolBar->move( x, y );
			y += t;
			h -= t;
		}
		if (toolBar->barPos() == KToolBar::Bottom) {
			toolBar->setMaxWidth( w );
			toolBar->updateRects( TRUE );
			t = toolBar->height();
			h -= t;
			toolBar->move( x, y+h );
		}
	}

	if ((need_hscroll = (zoom_width > w)))
		h -= hscroll->height();
	if ((need_vscroll = (zoom_height > h))) {
		w -= vscroll->width();
		if (!need_hscroll && (zoom_width > w)) {
			need_hscroll = TRUE;
			h -= hscroll->height();
		}
	}
	
	if (need_hscroll) {
		hscroll->setRange( 0, zoom_width - w );
		hscroll->setSteps( 12, w );
		hscroll->setGeometry( x, y+h, w, hscroll->height() );
		hscroll->show();
	} else {
		hscroll->hide();
		hscroll->setRange( 0, 0 );
		hscroll->setValue( 0 );
	}
	
	if (need_vscroll) {
		vscroll->setRange( 0, zoom_height - h );
		vscroll->setSteps( 12, h );
		vscroll->setGeometry( x+w, y, vscroll->width(), h );
		vscroll->show();
	} else {
		vscroll->hide();
		vscroll->setRange( 0, 0 );
		vscroll->setValue(0);
	}
	
	view->setGeometry( x, y, w, h );
}


void Preview::keyPressEvent( QKeyEvent * e )
{
	switch (e->key()) {
		case Key_Left:
			hscroll->setValue( hscroll->value() - hscroll->lineStep() );
			break;
		case Key_Right:
			hscroll->setValue( hscroll->value() + hscroll->lineStep() );
			break;
		case Key_Up:
			vscroll->setValue( vscroll->value() - vscroll->lineStep() );
			break;
		case Key_Down:
			vscroll->setValue( vscroll->value() + vscroll->lineStep() );
			break;
		case Key_PageUp:
			vscroll->setValue( vscroll->value() - vscroll->pageStep() );
			break;
		case Key_PageDown:
			vscroll->setValue( vscroll->value() + vscroll->pageStep() );
			break;
		case Key_Home:
			vscroll->setValue( vscroll->minValue() );
			break;
		case Key_End:
			vscroll->setValue( vscroll->maxValue() );
			break;
		default:
			e->ignore();
	}
}


void Preview::resizeEvent( QResizeEvent * )
{
	updateRects();
}
