/**************************************************************************
                          headerview.cpp  -  description                              
                             -------------------                                         
    begin                : Tue May 18 1999                                           
    copyright            : (C) 1999 by Norbert Weuster                         
    email                : weuster@uni-duisburg.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include "kheaderview.h"

KHeaderView::KHeaderView(QWidget *parent, KnetdumpDoc* doc, const char *name ) : QWidget(parent,name), pdoc(doc), pParent(parent) {
    // initialize all layout-pointer
    pLinkLayout=0;
    pOldLinkLayout=0;
    pNetwLayout=0;
    pTranspLayout=0;
    pEtherLayout=0;
    pEther802_2Layout=0;
    pTokenLayout=0;
    pIPLayout=0;
    pARPLayout=0;
    pTCPLayout=0;
    pUDPLayout=0; 
    pICMPLayout=0;
    pIPXLayout=0;
    pSPXLayout=0;
    pNETBIOSLayout=0;
    pDataLayout=0;
    // initialize pointer to the scroller
    pPacketScroller=0;

    drawing = false;

    // set the scroller
    if (pdoc->isStopped()) {
        slotDocIsPaused();
    }
}

KHeaderView::~KHeaderView () {
}

void KHeaderView::startslot () {
    //
    // view each received packet as headers
    // 
    draw();

    // finished work on received packet, inform doc
    emit gotData();
}

void KHeaderView::drawLinkHeader (QPainter& p) {
    //
    // draws the link header
    //

  // which linkheader
  // Note: do not set any pointer to Link-/Netw -header, they have should been set in DOC.
  switch (pdoc->getLinkLayer()) {
    case ARPHRD_ETHER:
    case ARPHRD_IEEE802: {
        static QString etherText[4];
        register QString tmp;
        struct ether802hdr *addr = (struct ether802hdr *)pdoc->getpLinkLayer(); 
        // set the addresses
        etherText[0].sprintf ("%02x:%02x:%02x:%02x:%02x:%02x\0",
                              addr->ether_dhost[0], addr->ether_dhost[1],
                              addr->ether_dhost[2], addr->ether_dhost[3],
                              addr->ether_dhost[4], addr->ether_dhost[5]);
        etherText[1].sprintf ("%02x:%02x:%02x:%02x:%02x:%02x\0",
                              addr->ether_shost[0], addr->ether_shost[1],
                              addr->ether_shost[2], addr->ether_shost[3],
                              addr->ether_shost[4], addr->ether_shost[5]);
        // the third field is length for IEEE802_2 or the ethertype for a normal etherhdr
        etherText[2].sprintf ("0x%04x\0", ntohs(addr->length));

        // differ between 802_2 and ether_header
        if (ntohs(addr->length) <= ETHERMTU) {
            if (!pEther802_2Layout) {
                pEther802_2Layout = new Ether802_2Layout (this, "Ether802_2Layout");
                QObject::connect(pEther802_2Layout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
            }
            //LLC
            etherText[3].sprintf("%02x:%02x:%02x\0", addr->dsap, addr->ssap, addr->cntl);
	    //*** be carefull, bytesRemain will be only valid, if the last if-clause was true!
            bytesRemain= ntohs(addr->length);
            if (addr->dsap == EXTENDED_SAP && addr->ssap == EXTENDED_SAP && addr->cntl == UI_CMD) {
                //SNAP
                printf("SNAP found\n");
                tmp.sprintf("/%02x:%02x:%02x|%04x\0", addr->org_code[0], addr->org_code[1], addr->org_code[2], addr->ether_type);
                etherText[3]+=tmp;
                bytesRemain -= sizeof(struct ether802hdr);
            }
            else {
                // no SNAP found
                bytesRemain -= (3 * sizeof(u_int8_t) + sizeof(struct ether_header));
            }

            pEther802_2Layout->setEther802_2Header(etherText);
            hmax=pEther802_2Layout->height()+5;
            pLinkLayout = pEther802_2Layout;
        } else {
	    // we have an ARPHRD_ETHER header, so reset bytesRemain
	    bytesRemain = pdoc->byterec();
            if (!pEtherLayout) {
                pEtherLayout = new EtherLayout (this, "EtherLayout");
                QObject::connect(pEtherLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
            }
	    bytesRemain -= sizeof(struct ether_header);
	    // set the text of buttons
            pEtherLayout->setEtherHeader(etherText);
            // set the height (y-coordinate)
            hmax=pEtherLayout->height()+5;
            pLinkLayout = pEtherLayout;
        }            
        if (pLinkLayout) pLinkLayout->move(10,0);
    }
    break;
    case ARPHRD_PRONET: {
        static QString tokenText[7];
        register QString tmp;
        const struct trh_hdr *pTokenHdr = (struct trh_hdr *)pdoc->getpLinkLayer(); 
	const struct trllc *tokenllc = 0;
	// set access control field
	tokenText[0].sprintf("%02x", pTokenHdr->ac) ;
	// set frame control field
	tokenText[1].sprintf("%02x", pTokenHdr->fc) ;
	// set the addresses
        tokenText[2].sprintf ("%02x:%02x:%02x:%02x:%02x:%02x\0",
                              pTokenHdr->daddr[0], pTokenHdr->daddr[1],
                              pTokenHdr->daddr[2], pTokenHdr->daddr[3],
                              pTokenHdr->daddr[4], pTokenHdr->daddr[5]);
        tokenText[3].sprintf ("%02x:%02x:%02x:%02x:%02x:%02x\0",
                              pTokenHdr->saddr[0], pTokenHdr->saddr[1],
                              pTokenHdr->saddr[2], pTokenHdr->saddr[3],
                              pTokenHdr->saddr[4], pTokenHdr->saddr[5]);
        if (!pTokenLayout) {
	    pTokenLayout = new TokenLayout (this, "TokenRing-Layout");
	    QObject::connect(pTokenLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
	}
	// if RIF Routing Information Field in ON, RC Routing Control follows
	if (pTokenHdr->saddr[0] & TR_RII) {
	  register int rii_len=  (TR_RCF_LEN_MASK & ntohs(pTokenHdr->rcf)) >> 8;
	  tokenText[4].sprintf("%04x",pTokenHdr->rcf);
	  if (  rii_len > 16 )
	    tokenText[5].sprintf ("RD");
	  else
	    tokenText[5].sprintf ("No RD");
	  bytesRemain-= ( rii_len + ETH_HLEN);
	  tokenllc=(const struct trllc *) ((char *) pTokenHdr + rii_len + ETH_HLEN);
	}
	else {
	  tokenllc=(const struct trllc *) ((char *) pTokenHdr + ETH_HLEN);
	  tokenText[4].sprintf("No RC");
	  tokenText[5].sprintf ("No RD");
	  bytesRemain-= ETH_HLEN;
	}
	
	// now working on the llc-header
	tokenText[6].sprintf("%02x:%02x:%02x\0", tokenllc->dsap, tokenllc->ssap, tokenllc->llc);
	if (tokenllc->dsap == EXTENDED_SAP && tokenllc->ssap == EXTENDED_SAP && tokenllc->llc == UI_CMD) { 
	    // SNAP found
	    printf("SNAP found\n");
	    tmp.sprintf("%02x:%02x:%02x|%04x\0", tokenllc->protid[0], tokenllc->protid[1], tokenllc->protid[2], tokenllc->ethertype);
	    tokenText[6]+=tmp;
	    bytesRemain -= sizeof(struct trllc);
	}
	else {
	  // no SNAP found
	  bytesRemain -= 3 * sizeof(char) ;
	}
	
	pTokenLayout->setTokenRingHeader(tokenText);
	hmax=pTokenLayout->height()+5;
	pLinkLayout = pTokenLayout;
	if (pLinkLayout) pLinkLayout->move(10,0);
    }
    break;
  case ARPHRD_LOOPBACK:
	pLinkLayout = 0;
	hmax=0;
	break;
  case ARPHRD_PPP:
  case ARPHRD_SLIP:
  default:
        cerr << "got a Linklayer of " << pdoc->getLinkLayer() << endl;
        pLinkLayout = 0;
        hmax= 0;
        break;
    }
    if (pLinkLayout) pLinkLayout->show();
    // hide layout, if plinklayout is another than before
    if (pLinkLayout && pOldLinkLayout && pOldLinkLayout!=pLinkLayout) {
        pOldLinkLayout->hide();
    }
    pOldLinkLayout= pLinkLayout;
}

void KHeaderView::drawNetwHeader (QPainter& p) {
    //
    // draws the network header (IP, ARP, ...)
    //
    static u_int16_t oldNetwHeader;

    // hide layouts, if old 
    if (oldNetwHeader!=pdoc->getNetwProtocol() && pNetwLayout) {
        pNetwLayout->hide();
        if (pTranspLayout) pTranspLayout->hide();
        if (pDataLayout)   pDataLayout->hide();
    }
    switch (oldNetwHeader=pdoc->getNetwProtocol()) {
    case 0xf0: // netbios
        drawNETBIOSHeader(p);
        break;
    case 0xe0: // novell netware
    case ETH_P_IPX:
        drawIPXHeader(p);
        break;
    case ETHERTYPE_IP:
        drawIPHeader(p);
        break;
    case ETHERTYPE_ARP:
        drawARPHeader(p);
        break;
    case ETHERTYPE_REVARP:
        drawARPHeader(p);
        break;
    default:
        pData=(const u_char *) pdoc->getpNetwLayer();
        cerr << "network protocol " << oldNetwHeader << " not implemented, so couldn't draw"<<endl;
        cerr << "the network protocol "  << oldNetwHeader << " should be found in RFC1700" << endl;
        break;
    }
}

void KHeaderView::drawIPHeader (QPainter& p) {
    //
    // draws IP header
    //
    static u_int8_t oldTransHeader;
    struct iphdr *ip= (struct iphdr *) pdoc->getpNetwLayer();
    register short unsigned int count;
    static QString ipText[13];
    register QString addr;
    register const u_char* pp;
    register QString str;

    if (!pIPLayout) {
        pIPLayout = new IPLayout (this, "IPLayout");
        QObject::connect(pIPLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
    }
       
    // get text of buttons
    ipText[0].sprintf ("Ver: %u", ip->version);
    ipText[1].sprintf ("ihl: %u", ip->ihl);
    ipText[2].sprintf ("tos: %u", ip->tos);
    ipText[3].sprintf ("length: %u", ntohs(ip->tot_len));
    ipText[4].sprintf ("id: %u", ntohs(ip->id));
    ipText[5].sprintf ("frag_off: %u", ntohs(ip->frag_off));
    ipText[6].sprintf ("ttl: %u", ip->ttl);
    ipText[7].sprintf ("protocol: %u", ip->protocol);
    ipText[8].sprintf ("check: %u", ntohs(ip->check));
    pdoc->gethostname_from(addr);
    ipText[9] = "source IP address: ";
    ipText[9]+= addr;
    pdoc->gethostname_to(addr);
    ipText[10] = "destination IP address: ";
    ipText[10]+= addr;

    // options
    pp= (const u_char *) pdoc->getpNetwLayer() + sizeof(struct iphdr);
    count= ip->ihl*4-sizeof(struct iphdr);
    ipText[11]= "options: 0x";
    for ( ; count > 0 ; count--) {
        str.sprintf("%02x:", *pp++);
        ipText[11] += str;
    }

    // data
    count= ntohs(ip->tot_len) - ip->ihl*4;
    if (count > 20) {
        anymore="...";
        count=20;
    } else anymore=0;
    ipText[12]= "data: 0x";
    for ( ; count > 0 ; count--) {
        str.sprintf("%02x:", *pp++);
        ipText[12] += str;       
    }
    ipText[12]+= anymore;
   
#ifdef DEBUG_OUT
    if (bytesRemain != ntohs(ip->tot_len)) {
      printf("bytesRemain=%u ; ", bytesRemain);
      printf("ip->tot_len=%u ; ", ntohs(ip->tot_len));
      printf("ip->tot_len - ip->ihl*4 == %u\n", ntohs(ip->tot_len)-ip->ihl*4);
    }
#endif
    // bytesRemain = ntohs(ip->tot_len);
    // bytesRemain -= sizeof(struct iphdr); 
    bytesRemain-= ntohs(ip->tot_len) - ip->ihl*4;

    // set text of buttons
    pIPLayout->setIPHeader(ipText);
    // show layout and set height
    pIPLayout->move(10, hmax);
    pNetwLayout = pIPLayout;
    pNetwLayout->show();
    hmax += pIPLayout->height()+5;

    if (oldTransHeader != ip->protocol && pTranspLayout) {
        pTranspLayout->hide();
        // if (pDataLayout)   pDataLayout->hide();
    }
    
    // switch to the transport header, if TCP/UDP
    switch(oldTransHeader=ip->protocol) {
    case IPPROTO_TCP:
        drawTCPHeader(p);
        break;
    case IPPROTO_UDP:
        drawUDPHeader(p);
        break;
    case IPPROTO_ICMP:
        drawICMPHeader(p);
        break;
    default:
        pData = (const u_char *) pdoc->getpTranspLayer();
        break;
    }
}

void KHeaderView::drawARPHeader (QPainter& p) {
    //
    // draws ARP header
    //
    struct arphdr *arp= (struct arphdr *) pdoc->getpNetwLayer();
    register u_char *pp= (u_char *) arp + sizeof(struct arphdr);
    static QString arpText[9];
    register QString str;
    register u_int32_t addr;
    register unsigned short int count;

    if (!pARPLayout) {
        pARPLayout= new ARPLayout (this, "ARPLayout");
        QObject::connect(pARPLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
                }
    bytesRemain -= sizeof(struct arphdr);
        
    // get text of buttons
    arpText[0].sprintf("%04x", ntohs(arp->ar_hrd));
    arpText[1].sprintf("%04x", ntohs(arp->ar_pro));
    arpText[2].sprintf("%u", arp->ar_hln);
    arpText[3].sprintf("%u", arp->ar_pln);
    
    // operation
    switch (ntohs(arp->ar_op)) {
    case 1:
        arpText[4]="request";
        break;
    case 2:
        arpText[4]="reply";
        break;
    case 3:
        arpText[4]="R-request";
        break;
    case 4:
        arpText[4]="R-reply";
        break;
    default:
        arpText[4].sprintf("op:", ntohs(arp->ar_op));
        break;
    }
    // set addresses
    if (ntohs(arp->ar_pro) == ETHERTYPE_IP) {
        arpText[5]= "";
        for (count=arp->ar_hln; count>0; count--) {
            str.sprintf("%02x:", *pp++);
            arpText[5] += str;
        }
        arpText[6]= "";
        addr= *((u_int32_t *) pp);
        register in_addr inaddr= {addr};
        //inaddr.s_addr= addr;
        arpText[6] += inet_ntoa(inaddr);
        pp += arp->ar_pln;
        
        arpText[7]= "";
        for (count=arp->ar_hln; count>0; count--) {
            str.sprintf("%02x:", *pp++);
            arpText[7] += str;
        }
        arpText[8]= "";
        addr= *((u_int32_t *) pp);
        inaddr.s_addr=addr;
        arpText[8] += inet_ntoa(inaddr);
        pp += arp->ar_pln;

        // bytesRemain -= (2*arp->ar_hln + 2*arp->ar_pln);
    }

    pData = (const u_char *) pp; 

    // set text of buttons
    pARPLayout->setARPHeader(arpText);
    // show layout and set height
    pARPLayout->move(10, hmax);
    pNetwLayout = pARPLayout;
    pNetwLayout->show();
    hmax += pARPLayout->height()+5;
}

void KHeaderView::drawTCPHeader (QPainter& p) {
    //
    // draws TCP header
    //
    struct tcphdr *tcp= (struct tcphdr *) pdoc->getpTranspLayer();
    register short unsigned int count;
    static QString tcpText[12];
    register QString str;
    register const u_char* pp;

    if (!pTCPLayout) {
        pTCPLayout = new TCPLayout (this, "TCPLayout");
        QObject::connect(pTCPLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
    }
    
    // get text of buttons
    tcpText[0].sprintf("source port: %u", ntohs(tcp->source));
    tcpText[1].sprintf("destination port: %u", ntohs(tcp->dest));
    tcpText[2].sprintf("seq: %u", ntohl(tcp->seq));
    tcpText[3].sprintf("ack: %u", ntohl(tcp->ack_seq));
    tcpText[4].sprintf("hlength: %u", tcp->doff);
    // reserved tcpText[5].sprintf (": %u", ptcp->th_x2); // reserved
    tcpText[6] ="flags:";
    if (tcp->fin)
        tcpText[6] += 'F';
    if (tcp->syn)
        tcpText[6] += 'S';
    if (tcp->rst)
        tcpText[6] += 'R';
    if (tcp->psh)
        tcpText[6] += 'P';
    if (tcp->ack)
        tcpText[6] += 'A';		
    if (tcp->urg)
        tcpText[6] += 'U';
    tcpText[7].sprintf ("window size: %u", ntohs(tcp->window));
    tcpText[8].sprintf ("checksum: %u", ntohs(tcp->check));
    tcpText[9].sprintf ("urgent: %u", ntohs(tcp->urg_ptr));
    // options
    pp= (const u_char *) pdoc->getpTranspLayer() + sizeof(struct tcphdr);
    tcpText[10]= "options: 0x";
    for (count=tcp->doff*4 - sizeof(struct tcphdr) ; count >0 ; count--) {
        str.sprintf("%02x:", *pp++);
        tcpText[10] += str;
    }

    bytesRemain -= tcp->doff*4;
    // data
    pData  = (const u_char *) pp;

    count= bytesRemain;
    if ( count > 20) {
        count=20;
        anymore="...";
    } else anymore=0;

    tcpText[11]= "data: 0x";
    for ( ; count > 0 ; count--) {
        str.sprintf("%02x:", *pp++);
        tcpText[11] +=str;
    }
    tcpText[11]+=anymore;

    // set text of buttons
    pTCPLayout->setTCPHeader(tcpText);
    // show layout and set height
    pTCPLayout->move(10, hmax);
    pTranspLayout = pTCPLayout;
    pTranspLayout->show();
    hmax += pTCPLayout->height()+5;
}

void KHeaderView::drawUDPHeader (QPainter& p) {
    //
    // draws UDP header
    //
    struct udphdr *udp= (struct udphdr *) pdoc->getpTranspLayer();
    register const u_char* pp;
    register QString str;
    static QString udpText[5];
    register short unsigned int count;

    if (!pUDPLayout) {
        pUDPLayout = new UDPLayout (this, "UDPLayout");
        QObject::connect(pUDPLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
    }
    
    // get text of buttons
    udpText[0].sprintf ("source port: %u", ntohs(udp->source));
    udpText[1].sprintf ("destination port: %u", ntohs(udp->dest));
    udpText[2].sprintf ("length: %u", ntohs(udp->len));
    udpText[3].sprintf ("checksum: %u", ntohs(udp->check));
    // data
    pData  = (u_char *) pdoc->getpTranspLayer() + sizeof(struct udphdr);
    pp = pData;
    bytesRemain -= sizeof(struct udphdr);
    count= bytesRemain;
    if ( count > 20 ) {
        count = 20 ;
        anymore="...";
    } else anymore=0;
    udpText[4]= "data: 0x";
    for ( ; count > 0 ; count--) {
        str.sprintf("%02x:", *pp++);
        udpText[4] += str;
    }
    udpText[4] += anymore;

    // set text of buttons
    pUDPLayout->setUDPHeader(udpText);
    // show layout and set height
    pUDPLayout->move(10, hmax);
    pTranspLayout = pUDPLayout;
    pTranspLayout->show();
    hmax += pUDPLayout->height()+5;
}

void KHeaderView::drawICMPHeader (QPainter& p) {
    //
    // draws ICMP header
    //
    struct icmphdr *icmp= (struct icmphdr *) pdoc->getpTranspLayer();
    const u_char* pp;
    register QString str;
    static QString icmpText[4];
    register short unsigned int count;

    if (!pICMPLayout) {
        pICMPLayout = new ICMPLayout (this, "ICMPLayout");
        QObject::connect(pICMPLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
    }

    // get text of buttons
    icmpText[0].sprintf ("type: %u", icmp->type);
    icmpText[1].sprintf ("code: %u", icmp->code);
    icmpText[2].sprintf ("check: %u", ntohs(icmp->checksum));
    // data
    pp = (const u_char *)icmp + sizeof(icmp->type) + sizeof(icmp->code) + sizeof
(icmp->checksum);
    count= sizeof(icmp->un);   // should be 32 Bits == 4 Bytes
    icmpText[3]= "data: 0x";
    for ( ; count > 0 ; count--) {
        str.sprintf("%02x:", *pp++);
        icmpText[3] += str;
    }

    // anything else
    pData  = (u_char *) pdoc->getpTranspLayer() + sizeof(struct icmphdr);
    bytesRemain -= sizeof(struct icmphdr);

    // set text of buttons
    pICMPLayout->setICMPHeader(icmpText);
    // show layout and set height
    pICMPLayout->move(10, hmax);
    pTranspLayout = pICMPLayout;
    pTranspLayout->show();
    hmax += pICMPLayout->height()+5;
}
                                   
void KHeaderView::drawIPXHeader (QPainter& p) {
    //
    // draws ipx header
    //
    struct ipxhdr *ipx= (struct ipxhdr *) pdoc->getpNetwLayer();
    const u_char* pp;
    register QString str;
    static QString ipxText[11];
    register short unsigned int count;

    if (!pIPXLayout) {
        pIPXLayout = new IPXLayout (this, "IPXLayout");
        QObject::connect(pIPXLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
    }

    // get text of buttons
    ipxText[0].sprintf ("check: %u", ntohs(ipx->check));
    ipxText[1].sprintf ("length: %u", ntohs(ipx->length));
    ipxText[2].sprintf ("transpctrl: %u", ipx->transpctrl);
    ipxText[3].sprintf ("packettype: %u", ipx->packettype);
    ipxText[4].sprintf ("Dest. Network: %u", ntohs(ipx->dnetwork));
    ipxText[5].sprintf ("Dest. Host:%02x:%02x:%02x:%02x:%02x:%02x\0",
                              ipx->dhost[0], ipx->dhost[1],
                              ipx->dhost[2], ipx->dhost[3],
                              ipx->dhost[4], ipx->dhost[5]); 
    ipxText[6].sprintf ("Dest. Socket: %u", ntohs(ipx->dsocket));
    ipxText[7].sprintf ("Src. Network: %u", ntohs(ipx->snetwork));
    ipxText[8].sprintf ("Src. Host:%02x:%02x:%02x:%02x:%02x:%02x\0",
                              ipx->shost[0], ipx->shost[1],
                              ipx->shost[2], ipx->shost[3],
                              ipx->shost[4], ipx->shost[5]);
    ipxText[9].sprintf ("Src. Socket: %u", ntohs(ipx->ssocket));
    // data
    pp = (const u_char *)ipx + sizeof(struct ipxhdr);
    count= ipx->length - sizeof(struct ipxhdr);  
    if ( count > 20 ) {
        count = 20 ;
        anymore="...";
    } else anymore=0;
    ipxText[10]= "data: 0x";
    for ( ; count > 0 ; count--) {
        str.sprintf("%02x:", *pp++);
        ipxText[10] += str;
    } 
    ipxText[10] +=anymore;

    // anything else
    pData  = (u_char *) pdoc->getpTranspLayer();
    bytesRemain -= sizeof(struct ipxhdr);
    // set text of buttons
    pIPXLayout->setIPXHeader(ipxText);
    // show layout and set height
    pIPXLayout->move(10, hmax);
    pTranspLayout = pIPXLayout;
    pTranspLayout->show();
    hmax += pIPXLayout->height()+5;

    // switch to the transport header, if TCP/UDP
    if (ipx->packettype==6) 
        drawSPXHeader(p);
    else
        pData = (const u_char *) pdoc->getpTranspLayer();
}

void KHeaderView::drawSPXHeader (QPainter& p) {
    //
    // draws spx header
    //
    struct spxhdr *spx= (struct spxhdr *) pdoc->getpTranspLayer();
    const u_char* pp;
    register QString str;
    static QString spxText[8];
    register short unsigned int count;

    if (!pSPXLayout) {
        pSPXLayout = new SPXLayout (this, "SPXLayout");
        QObject::connect(pSPXLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
    }

    // get text of buttons
    spxText[0].sprintf ("connection ctrl: %u", spx->connctrl);
    spxText[1].sprintf ("data stream type: %u", spx->datastreamtype);
    spxText[2].sprintf ("src connection ID: %u", ntohs(spx->sconnid));
    spxText[3].sprintf ("dest connection ID: %u", ntohs(spx->dconnid));
    spxText[4].sprintf ("seq: %u", ntohs(spx->seq));
    spxText[5].sprintf ("ack: %u", ntohs(spx->ack));
    spxText[6].sprintf ("allocation: %u", ntohs(spx->allocation));

    // data
    pp = (const u_char *)spx + sizeof(struct spxhdr);
    count= bytesRemain;
    spxText[7]= "data: 0x";
    for ( ; count > 0 ; count--) {
        str.sprintf("%02x:", *pp++);
        spxText[7] += str;
    }

    // anything else
    pData  = (u_char *) pdoc->getpTranspLayer() + sizeof(struct spxhdr);
    bytesRemain -= sizeof(struct spxhdr);

    // set text of buttons
    pSPXLayout->setSPXHeader(spxText);
    // show layout and set height
    pSPXLayout->move(10, hmax);
    pTranspLayout = pSPXLayout;
    pTranspLayout->show();
    hmax += pSPXLayout->height()+5;
}


void KHeaderView::drawNETBIOSHeader (QPainter& p) {
    //
    // draws NETBIOS header
    //
    struct netbioshdr *netbios= (struct netbioshdr *) pdoc->getpNetwLayer();
    const u_char* pp;
    register QString str;
    static QString netbiosText[9];
    register short unsigned int count, addrlen;
    
    if (!pNETBIOSLayout) {
        pNETBIOSLayout = new NETBIOSLayout (this, "NETBIOSLayout");
        QObject::connect(pNETBIOSLayout, SIGNAL(htmlHelp(char*, char*)), pParent, SLOT(slotHTMLHelp(char*, char*)));
    }

    // get text of buttons
    netbiosText[0].sprintf ("length: %u", (netbios->length));
    netbiosText[1].sprintf ("efff: %u", (netbios->efff));
    netbiosText[2].sprintf ("cmd: %u", netbios->cmd);
    netbiosText[3].sprintf ("opt1: %u", netbios->opt1);
    netbiosText[4].sprintf ("opt2: %u", ntohs(netbios->opt2));
    netbiosText[5].sprintf ("xr: %u", ntohl(netbios->xr));
    // TODO
    addrlen=(netbios->length - sizeof(struct netbioshdr))/2;
    pp = (const u_char *)netbios + sizeof(struct netbioshdr);
    netbiosText[6]= "Dest N.: " ;
    netbiosText[7]= "Source N.: " ;
    for (count=addrlen; count>0; count--) {
        str.sprintf("%02x:", *pp++);
        netbiosText[6] += str;
    }
    for (count=addrlen; count>0; count--) {
        str.sprintf("%02x:", *pp++);
        netbiosText[7] += str;
    }
    bytesRemain-= 2*addrlen ;

    // data
    pp = (const u_char *)netbios + sizeof(struct netbioshdr) + addrlen*2;
    pData  = (const u_char *) pp;
    count= bytesRemain;
    if (count > 20) {
      anymore="...";
      count=20;
    } else anymore=0;
    netbiosText[8]= "data: 0x";
    for ( ; count > 0 ; count--) {
        str.sprintf("%02x:", *pp++);
        netbiosText[8] += str;
    }
    netbiosText[8] += anymore;

    // set text of buttons
    pNETBIOSLayout->setNETBIOSHeader(netbiosText);
    // show layout and set height
    pNETBIOSLayout->move(10, hmax);
    pTranspLayout = pNETBIOSLayout;
    pTranspLayout->show();
    hmax += pNETBIOSLayout->height()+5;
}

void KHeaderView::drawData (QPainter& p) {
    //
    // draws the inlayed data in hex
    //
    if (!pData) return;

    register short unsigned int count;
    static QString data;
    register QString str;

    if (!pDataLayout) {
        pDataLayout= new QLabel(this, "pDataLayout");
    }

    count= bytesRemain;

    if ( count > 20 ) {
        count = 20 ;
        anymore="...";
    } else anymore=0;
    
    data= "data: 0x";
    for ( ; count > 0 ; count--) {
        str.sprintf("%02x:", *pData++);
        data += str;
    }
    data+= anymore;

    // set text of data-button
    pDataLayout->setText(data);
    // show layout and set height
    pDataLayout->resize(600,30);
    pDataLayout->setFrameStyle( QFrame::WinPanel | QFrame::Raised);
    pDataLayout->move(10, hmax);
    pDataLayout->show();
}

void KHeaderView::draw() { 
    //
    // Just draws the headers.
    //
    if (drawing || !pdoc->getPacketNum()) return;
    drawing = true;

    setUpdatesEnabled( FALSE );

    QPainter p;
    p.begin(this);
    
    // set counter of received bytes
    bytesRemain = pdoc->byterec();
    
    // draw the headers, 
    // start with the link header
    drawLinkHeader(p);
    // the network header should also be present and will decide further drawings
    drawNetwHeader(p);
    // draw the data
    drawData(p);
    
    // set scroller
    if (pPacketScroller)
        pPacketScroller->setGeometry( width()-30, 0, 15, height()-20 );

    p.end();
    drawing = false;
    setUpdatesEnabled( TRUE );
}

void KHeaderView::slotDocIsStarted () {
    if (pPacketScroller) pPacketScroller->hide();
}

void KHeaderView::slotDocIsPaused  () {
    unsigned short int packetNum= pdoc->getPacketNum();
    //QPainter p;

    if (!pPacketScroller) {
        pPacketScroller = new QScrollBar(QScrollBar::Vertical, this, "PacketScroller");
        QObject::connect(pPacketScroller,SIGNAL(valueChanged(int)), pdoc, SLOT(getPacket(int)));
    }

    if (packetNum<1) {
        pPacketScroller->hide();
        return;
    }
    
    // setup scroller
    pPacketScroller->setRange( 0, packetNum-1); 
    pPacketScroller->setSteps( 1, 10 );
    pPacketScroller->setValue(packetNum-1); 
    pPacketScroller->setGeometry( width()-30, 0, 15, height()-20 );
    // pPacketScroller->move(width()-30, 0);
   
    pPacketScroller->show();
}

void KHeaderView::slotErase() {  // if packets are deleted, erase and hide widgets
    if (pLinkLayout)
        pLinkLayout->hide();
    if (pNetwLayout)
        pNetwLayout->hide();
    if (pTranspLayout)
        pTranspLayout->hide();
    if (pDataLayout)
        pDataLayout->hide();

    erase();
}

