// -*- C++ -*-

//
//  konline
//
//  Copyright (C) 1998 Christoph Neerfeld
//  email:  Christoph.Neerfeld@home.ivm.de or chris@kde.org
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

extern "C" {
#include <string.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <stdio.h>
#include <errno.h>
}

#include "server.h"

int Server::init( const char *h, unsigned short p, const char *pass, unsigned long id )
{
  strncpy(host, h, 120);
  host[120] = '\0';
  port = p;
  if( pass != 0 )
    {
      strncpy( passwd, pass, 10 );
      passwd[10] = '\0';
    }
  else
    passwd[0] = '\0';
  my_id = id;

  /**
    If host contains the IP-address in standard numbers and dots notation,
    inet_aton() converts this notation into a binary representation.
    It returns FALSE if the conversion was not successful.
    */
  if( !inet_aton( host, &servername.sin_addr ) )
    {
      /**
	host does not contain the IP-address, so try to translate the hostname
	into the IP-address.
	*/
      struct hostent *hostinfo = 0;
      hostinfo = gethostbyname(host);
      if( hostinfo == 0 ) // the address to the given hostname could not be found
	return -1;
      servername.sin_addr = *(struct in_addr *) hostinfo->h_addr;
    }
  servername.sin_family = AF_INET;    // set connection type to IP
  servername.sin_port = htons(port);  // set the portnumber of the server

  /**
    The TCP connection to the server is taken down after any command. So KOnline must
    listen on its own socket for any messages coming from KOnserver; e.g. whenever a new
    user goes online. This socket is bound to the variable listensock.
    The portnumber of this socket is choosen by the OS and must be transmitted to KOnserver
    when a connection is established.
   */
  // create client listen socket 
  if( listensock < 0 )
    {
      listensock = socket(PF_INET, SOCK_STREAM, 0);
      if (listensock < 0)
	{
	  fprintf(stderr,"Server::init: unable to create listen socket");
	  return -1;
	}
      listenname.sin_family = AF_INET;
      // listenname.sin_port is not set; this is done by the OS
      listenname.sin_addr.s_addr = htonl (INADDR_ANY);
      if (bind(listensock, (struct sockaddr *) &listenname, sizeof(listenname)) < 0)
	{
	  fprintf(stderr,"Server::init: can't bind listen socket\n");
	  close(listensock);
	  return -1;
	}

      //Get the portnumber that was choosen by the OS.
      struct sockaddr_in addr;
      size_t size = sizeof(struct sockaddr_in);
      getsockname(listensock, (struct sockaddr *) &addr, &size);
      listenname.sin_port = addr.sin_port;
      // fprintf(stderr,"port = %i\n", ntohs(listenname.sin_port));

      // Tell the OS to listen for connection requests on this socket.
      if (listen (listensock, 10) < 0)      // listen queue has 10 entries
	{
	  fprintf(stderr,"Server::init: can't listen on listen socket\n");
	  return -1;
	}
    }
  /**
    KOnline sends any KEEP_ALIVE_TIME seconds a UDP datagram to KOnserver. If KOnserver
    does not receive a packet after a time of 3*KEEP_ALIVE_TIME, it asumes that the client
    is offline and tries to send a 'are you alive' message to the client.
    The keepsocket is used to send the keepalive packets.
   */
  if( keepsocket < 0 )
    {
      keepsocket = socket(PF_INET, SOCK_DGRAM, 0);
      if( keepsocket < 0 )
	{
	  fprintf(stderr,"Server::init: unable to create udp keep alive socket");
	  return -1;
	}
      keepname.sin_family = AF_INET;
      //keepname.sin_port is not set
      keepname.sin_addr.s_addr = htonl (INADDR_ANY);
      /*
      if( bind(keepsocket, (struct sockaddr *) &keepname, sizeof(keepname)) < 0 )
	{
	  fprintf(stderr,"Server::init: can't bind keepsocket\n");
	  shutdown(keepsocket, 2);
	  close(keepsocket);
	  //keepsocket = -1;
	  return -1;
	}
	*/
    }
  return 0;
}

/**
  Take a look into init() for further information to this function.
  */
int Server::setServer( const char *h, unsigned short p )
{
  strncpy(host, h, 120);
  host[120] = '\n';
  port = p;
  if( !inet_aton( host, &servername.sin_addr ) )
    {
      struct hostent *hostinfo;
      hostinfo = gethostbyname(host);
      if( hostinfo == 0 )
	return -1;
      servername.sin_addr = *(struct in_addr *) hostinfo->h_addr;
    }
  servername.sin_family = AF_INET;
  servername.sin_port = htons(port);
  return 0;
}

/**
  Take a look into init() for further information to this function.
  */
void Server::setAccount( const char *pass, unsigned long id )
{
  my_id = id;
  if( pass != 0 )
    {
      strncpy( passwd, pass, 10 );
      passwd[10] = '\0';
    }
}

/**
  This is a protected function used internaly by the Server class.
  It creates a socket to communicate with KOnserver and establishes a connection.
  In case of a success the socket is bound to a new PipeCom object s_pipe.
 */
int Server::connect()
{
  sock = socket (PF_INET, SOCK_STREAM, 0);
  if (sock < 0)
    {
      fprintf(stderr,"Server::connect: can't create server socket\n");
      return -1;
    }
  //fprintf(stderr,"ip = %s\n", inet_ntoa(servername.sin_addr));
  //fprintf(stderr,"port = %i\n", ntohs(servername.sin_port));
  if ( 0 > ::connect(sock, (struct sockaddr *) &servername, sizeof(servername)) )
    {
      fprintf(stderr,"Server::connect: can't connect to server socket\n");
      return -1;
    }
  s_pipe = new PipeCom(sock);
  return 0;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::create( const char *name, const char *fname, const char *nname, 
		    const char *email, const char *pass, unsigned long *id)
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << (unsigned long) 0 << (short) NEW_CLIENT << (char *) name 
	  << (char *) fname
	  << (char *) nname
	  << (char *) email
	  << (char *) pass;
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"Server::create: read error from server\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  if( rcode == RE_OK )
    {
      *s_pipe >> *id;
      my_id = *id;
      strncpy( passwd, pass, 10 );
      passwd[10] = '\0';
    }
  disconnect();
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::change( const char *name, const char *fname, const char *nname, 
		    const char *email, const char *new_pass )
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << (unsigned long) my_id << (short) CHANGE_CLIENT << passwd
	  << (char *) name 
	  << (char *) fname
	  << (char *) nname
	  << (char *) email
	  << (char *) new_pass;
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"read error from client\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  disconnect();
  if( rcode == RE_OK )
    {
      strncpy( passwd, new_pass, 10 );
      passwd[10] = '\0';
    }
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::findUser(short key, const char *name, const char *fname, const char *nname,
		     const char *email, unsigned long id, struct users *rec)
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << my_id << (short) SEARCH_USER << key << (char *) name 
	  << (char *) fname
	  << (char *) nname
	  << (char *) email
	  << id;
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"Server::findUser: read error from server\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  char *temp;
  if( rcode == RE_OK )
    {
      *s_pipe >> temp;
      strcpy(rec->name, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->fname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->nname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->email, temp);
      delete temp;
      *s_pipe >> rec->user_id;
    }
  else
    disconnect();
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::findNext( struct users *rec )
{
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"Server::findNext: read error from server\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  char *temp;
  if( rcode == RE_OK )
    {
      *s_pipe >> temp;
      strcpy(rec->name, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->fname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->nname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->email, temp);
      delete temp;
      *s_pipe >> rec->user_id;
    }
  else
    disconnect();
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::addWatchList( unsigned long new_id )
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << (unsigned long) my_id << (short) ADD_WATCHLIST << passwd << (unsigned long) new_id; 
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"read error from client\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  disconnect();
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::serverConnect( struct users *rec, char *act )
{
  if( connect() < 0 )
    return RE_COM;
  // fprintf(stderr,"port = %i", ntohs(listenname.sin_port));
  *s_pipe << (unsigned long) my_id << (short) CONNECT 
	  << passwd
	  << (short) ntohs(listenname.sin_port);
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"read error from client\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  char *temp;
  if( rcode == RE_OK )
    {
      *s_pipe >> temp;
      strcpy(rec->name, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->fname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->nname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->email, temp);
      delete temp;
      *s_pipe >> rec->user_id;
      *s_pipe >> *act;
    }
  else
    disconnect();
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::serverDisconnect()
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << (unsigned long) my_id << (short) DISCONNECT << passwd;
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"read error from client\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  disconnect();
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::nextWUserId( struct users *rec, char *act)
{
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"read error from client\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  char *temp;
  if( rcode == RE_OK )
    {
      *s_pipe >> temp;
      strcpy(rec->name, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->fname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->nname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->email, temp);
      delete temp;
      *s_pipe >> rec->user_id;
      *s_pipe >> *act;
    }
  else
    disconnect();
  return rcode;
}

/**
  The client main program has to check for incoming data on 'listensock'.
  If any data arives the client calls this function.
  According to the message type it sets 'id', 'inbuf' and 'rec'.
  The function returns the message type.

  inbuf has a maximum length of MAX_MSG_LEN and has to be allocated by the client
  main program.
 */
int Server::incomingData( unsigned long *id, char *inbuf, struct users *rec )
{
  size_t size;
  int status;
  struct sockaddr_in clientname;
  /**
    accept() duplicates listensock and returns a new filedescriptor. The new fd is
    connected with konserver and receives the message.
   */
  if ( (status = accept (listensock, (struct sockaddr *) &clientname, (unsigned int *) &size)) < 0)
    {
      fprintf(stderr,"Server::incomingData: can't accept on listen socket\n");
      return -1;
    }
  fprintf(stderr, "client connect from host: %s", inet_ntoa (clientname.sin_addr) );
  fprintf(stderr, "   using port: %i\n", ntohs(clientname.sin_port) );
  // create a PipeCom object for the new filedescriptor
  l_pipe = new PipeCom(status);
  short length = 0;
  short command;
  if( (length = l_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"Server::incomingData: read error from client on listensocket\n");
      return -1;
    }
  *l_pipe >> command;
  switch(command) {
  case USER_ONLINE:
  case USER_OFFLINE:
    *l_pipe >> *id;
    break;
  case SERVER_SHUTDOWN:
    break;
  case STILL_ALIVE:
    /**
      If the server does not receive keep alive packets from the client, it sends
      a 'are you alive' message.
      The client answers with RE_OK and its id.
      */
    *l_pipe << (short) RE_OK << my_id;
    l_pipe->send();
    break;
  case INCOMING_MESSAGE:
    if( inbuf == 0 )
      { // receive buffer exceeded 
	/**
	  If the client is unable to process incoming messages, it sets inbuf=0.
	  Then incomingData() sends the message receive buffer exceeded to konserver
	  and konserver stops sending messages. You have to call getMsg() to get further
	  messages.
	 */
	*l_pipe << (short) RE_RECV_BUF_EXCEEDED;
	l_pipe->send();
      }
    else
      {
	char *temp;
	*l_pipe >> temp;
	strcpy(inbuf, temp);
	delete temp;
	*l_pipe >> temp;
	strcpy(rec->name, temp);
	delete temp;
	*l_pipe >> temp;
	strcpy(rec->fname, temp);
	delete temp;
	*l_pipe >> temp;
	strcpy(rec->nname, temp);
	delete temp;
	*l_pipe >> temp;
	strcpy(rec->email, temp);
	delete temp;
	*l_pipe >> temp;
	strcpy(rec->date, temp);
	delete temp;
	*l_pipe >> rec->user_id;
	*l_pipe << (short) RE_OK;
	l_pipe->send();
      }
    break;
  };
  // shuts the connection down and closes the filedescriptor.
  shutdown(status, 2);
  close(status);
  delete l_pipe;
  l_pipe = 0;
  return command;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::removeWatchList( unsigned long id )
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << (unsigned long) my_id << (short) REMOVE_WATCHLIST << passwd << id; 
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"read error from client\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  disconnect();
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::sendMessage( unsigned long rid, const char *msg, const char *date )
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << (unsigned long) my_id << (short) SEND_MSG << rid << (char *) msg << (char *) date; 
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"read error from client\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  disconnect();
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::getMsg()
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << (unsigned long) my_id << (short) GET_MSG;
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"read error from client\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  disconnect();
  return rcode;
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::remove()
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << (unsigned long) my_id << (short) DEL_CLIENT << passwd;
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"Server::read error from client\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  disconnect();
  return rcode;
}

/**
  Sends a UDP datagramm to the server to indicate that the client is still alive.
 */
void Server::sendKeepAlive()
{
  char buf[LL_SHORT+LL_LONG];
  short s = htons(KEEP_ALIVE);
  unsigned long l = htonl(my_id);
  buf[0] = *((char *) &s);          // ugly code, I know; next time I'll do it better
  buf[1] = *(((char *) &s)+1);
  buf[2] = *((char *) &l);
  buf[3] = *(((char *) &l)+1);
  buf[4] = *(((char *) &l)+2);
  buf[5] = *(((char *) &l)+3);
  /**
    Send UDP datagramm to konserver.
    The datagramm contains a signed short with value KEEP_ALIVE
    and an unsigned long containing the client id. 
    */
  sendto( keepsocket, buf, LL_SHORT+LL_LONG, 0, (struct sockaddr *) &servername,
	  sizeof(servername));
}

/**
  Look into server.h for a description of this function.
  You should not need to alter this function if you want to use class Server in
  your own program.
 */
int Server::getInfo( unsigned long id, const char *pass, struct users *rec )
{
  if( connect() < 0 )
    return RE_COM;
  *s_pipe << (unsigned long) 0 << (short) GET_USER_INFO << (char *) pass
	  << id;
  s_pipe->send();
  short length = 0;
  if( (length = s_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"Server::getInfo: read error from server\n");
      return -1;
    }
  short rcode;
  *s_pipe >> rcode;
  char *temp;
  if( rcode == RE_OK )
    {
      *s_pipe >> temp;
      strcpy(rec->name, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->fname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->nname, temp);
      delete temp;
      *s_pipe >> temp;
      strcpy(rec->email, temp);
      delete temp;
    }
  disconnect();
  return rcode;  
}

