/***************************************************************************
                          knd_pkt.cpp  -  description                              
                             -------------------                                         
    begin                : Thu Mar 25 14:26:46 GMT 1999
                                           
    copyright            : (C) 1999 by Mike Richardson                         
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include	"knd_pkt.h"

#include	<sys/time.h>
#include	<unistd.h>
#include	<time.h>

extern	int	g_pauseat	;	/* Packet load to pause at	*/
extern	int	g_pausefor	;	/* Pause time in seconds	*/

/*  ItoA	: Convert integer to string				*/
/*  val		: int		: Integer				*/
/*  (returns)	: char *	: String				*/

static	char	*ItoA
	(	int	val
	)
{
	static	char	b[32]  ;
	sprintf	(b, "%d", val) ;
	return	 b ;
}

/*  KNDPacket								*/
/*  KNDPacket	: Constructor for packet display object			*/
/*  parent	: KNDView *	: Parent				*/
/*  (returns)	: KNDPacket	:					*/

KNDPacket::KNDPacket
	(	KNDView	*_parent
	)
	:
	QFrame	   ((QWidget *)_parent),
	parent	   (_parent),
	l_accept   (this),
	l_dropped  (this),
	l_missed   (this),
	l_pktbytes (this),
	l_datbytes (this),
	v_accept   (this),
	v_dropped  (this),
	v_missed   (this),
	v_pktbytes (this),
	v_datbytes (this)
{
	plist	= NULL	;	/* No packet display object yet ...	*/
	lastp	= NULL	;	/* ... nor any items inserted		*/
	going	= false	;	/* Not at the moment ...		*/
	frozen	= false	;	/* ... nor this				*/
	pausefor= 0	;	/* ... not nor this neither		*/

	p_this	= 0	;
	p_last	= 0	;

	l_accept  .hide () ;
	l_dropped .hide () ;
	l_missed  .hide () ;
	l_pktbytes.hide () ;
	l_datbytes.hide () ;
	v_dropped .hide () ;
	v_missed  .hide () ;
	v_pktbytes.hide () ;
	v_datbytes.hide () ;

	memset	(packets, 0, sizeof(packets)) ;
}

/*  KNDPacket								*/
/*  ~KNDPacket	: Destructor for packet display object			*/
/*  (returns)	:		:					*/

KNDPacket::~KNDPacket ()
{
	/* Destroy the packet display object if any, and delete any	*/
	/* stored packets.						*/
	if (plist != NULL)
		delete plist ;

	for (int i = 0 ; i < MAXPACK ; i += 1)
		if (packets[i] != NULL)
			freePktInfo (packets[i]) ;
}

/*  KNDPacket								*/
/*  setOptions	: Set display options					*/
/*  _options	: int		: New options				*/
/*  (returns)	: void		:					*/

void	KNDPacket::setOptions
	(	int	_options
	)
{
	QRect	r = geometry () ;

	/* If there is a current packet display list then destroy it,	*/
	/* as we will be building a new one. Also drop any packets that	*/
	/* we have stored.						*/
	if (plist != NULL) delete plist ;
	plist	= NULL ;
	lastp	= NULL ;

	for (int i = 0 ; i < MAXPACK ; i += 1)
		if (packets[i] != NULL)
			freePktInfo (packets[i]) ;
	memset	(packets, 0, sizeof(packets)) ;
	offset	= 0 ;

	/* Note the specified options; if there are none then there is	*/
	/* no packet display at all.					*/
	if ((options = _options) == 0) return ;

	/* Create a new packet display object, and set it to just about	*/
	/* fill the window within which it is displayed.		*/
	plist	= new QListView (this) ;
	plist->setGeometry	(10, 10, r.width() - 20, r.height() - 40) ;

	/* Now add columns to the packet display object corresponding	*/
	/* to each of the selected options. The order is predefined,	*/
	/* the user has no control over this.				*/
	if (options & PO_TIME ) plist->addColumn ("Time",  100) ;
	if (options & PO_PROTO) plist->addColumn ("Proto",  80) ;
	if (options & PO_SRC  ) plist->addColumn ("Srce",  200) ;
	if (options & PO_DST  ) plist->addColumn ("Dest",  200) ;
	if (options & PO_INFO ) plist->addColumn ("Info",  110) ;
	if (options & PO_SIZE ) plist->addColumn ("Size",   70) ;

	l_accept  .setGeometry   ( 10, r.height() - 25, 70, 20) ;
	v_accept  .setGeometry   ( 90, r.height() - 25, 60, 20) ;
	l_dropped .setGeometry   (160, r.height() - 25, 70, 20) ;
	v_dropped .setGeometry   (240, r.height() - 25, 60, 20) ;
	l_missed  .setGeometry   (310, r.height() - 25, 70, 20) ;
	v_missed  .setGeometry   (390, r.height() - 25, 60, 20) ;
	l_pktbytes.setGeometry   (460, r.height() - 25, 70, 20) ;
	v_pktbytes.setGeometry   (540, r.height() - 25, 80, 20) ;
	l_datbytes.setGeometry   (630, r.height() - 25, 50, 20) ;
	v_datbytes.setGeometry   (690, r.height() - 25, 80, 20) ;

	l_accept  .setAlignment  (AlignRight) ;
	v_accept  .setAlignment  (AlignRight) ;
	l_dropped .setAlignment  (AlignRight) ;
	v_dropped .setAlignment  (AlignRight) ;
	l_missed  .setAlignment  (AlignRight) ;
	v_missed  .setAlignment  (AlignRight) ;
	l_pktbytes.setAlignment  (AlignRight) ;
	v_pktbytes.setAlignment  (AlignRight) ;
	l_datbytes.setAlignment  (AlignRight) ;
	v_datbytes.setAlignment  (AlignRight) ;

	v_accept  .setFrameStyle (QFrame::Panel|QFrame::Sunken) ;
	v_dropped .setFrameStyle (QFrame::Panel|QFrame::Sunken) ;
	v_missed  .setFrameStyle (QFrame::Panel|QFrame::Sunken) ;
	v_pktbytes.setFrameStyle (QFrame::Panel|QFrame::Sunken) ;
	v_datbytes.setFrameStyle (QFrame::Panel|QFrame::Sunken) ;

	l_accept  .setText   (i18n("Accepted")) ;
	l_dropped .setText   (i18n("Dropped" )) ;
	l_missed  .setText   (i18n("Missed"  )) ;
	l_pktbytes.setText   (i18n("Traffic" )) ;
	l_datbytes.setText   (i18n("Data"    )) ;

	QToolTip::add (&v_accept,   i18n("Packets accepted at interface" )) ;
	QToolTip::add (&v_dropped,  i18n("Packets dropped at interface"  )) ;
	QToolTip::add (&v_missed,   i18n("Packets missed at interface"   )) ;
	QToolTip::add (&v_pktbytes, i18n("Total accepted network traffic")) ;
	QToolTip::add (&v_datbytes, i18n("Total accepted data"		 )) ;

	v_accept  .setText ("0") ;
	v_dropped .setText ("0") ;
	v_missed  .setText ("0") ;
	v_pktbytes.setText ("0") ;
	v_datbytes.setText ("0") ;

	l_accept  .show () ;
	v_accept  .show () ;
#ifdef	_DROPPED
	l_dropped .show () ;
	v_dropped .show () ;
#endif	/* _DROPPED */
#ifdef	_MISSED
	l_missed  .show () ;
	v_missed  .show () ;
#endif	/* _MISSED */
	l_pktbytes.show () ;
	v_pktbytes.show () ;
	l_datbytes.show () ;
	v_datbytes.show () ;

	s_pktbytes = 0 ;
	s_datbytes = 0 ;

	p_last	   = 0 ;
	p_this	   = 0 ;

	going	   = false ;
	frozen	   = false ;
	pausefor   = 0	   ;
}

/*  KNDPacket								*/
/*  execute	: Set executing flag					*/
/*  _going	: bool		: Executing				*/
/*  (returns)	: void		:					*/

void	KNDPacket::execute
	(	bool	_going
	)
{
	going	= _going ;	/* Not the requsted execution state	*/
	frozen	= false	 ;	/* Changes always reset frozen state	*/
}

/*  KNDPacket								*/
/*  addPacket	: Add a new packet to the display			*/
/*  pktinfo	: PktInfo *	  : Packet information				*/
/*  (returns)	: void		  :					*/

void	KNDPacket::addPacket
	(	PktInfo	*pktinfo
	)
{
	/* Ignore the packet if there is no display ...			*/
	if (plist == NULL) return ;

	/* If we currently have the full complement of packets stored	*/
	/* then drop the oldest one; if on view, and not frozen or	*/
	/* paused then also drop if from the display.			*/
	if (packets[offset] != NULL)
	{
		freePktInfo (packets[offset]) ;
		packets[offset]  = NULL	      ;

		if (onview && !frozen && (pausefor == 0))
			delete	plist->firstChild() ;
	}

	/* Add a new packet to the list, then save the packet		*/
	/* information. Note that the "inuse" flag is set so that the	*/
	/* originator will not free the packet.				*/
	packets[offset] = pktinfo ;
	pktinfo->inuse	= true	  ;
	if ((offset += 1) >= MAXPACK) offset = 0 ;

	p_this		+= 1	  ;
	s_pktbytes	+= pktinfo->pktlen ;
	s_datbytes	+= pktinfo->datlen ;

	/* If the number of packets received so far this second exceeds	*/
	/* the pause threshhold, then set the pause counter, and insert	*/
	/* a pause marker ...						*/
	if ((p_this >= g_pauseat) && (pausefor == 0))
	{
		if (onview && !frozen) putDots () ;
		pausefor = g_pausefor	;
		return	 ;
	}

	/* Get the packet put into the list, again unless we are not	*/
	/* on view, or are frozen.					*/
	if (onview && !frozen && (pausefor == 0))
		putPacket (pktinfo, true) ;
}

/*  KNDPacket								*/
/*  putPacket	: Put packet into display				*/
/*  pktinfo	: PktInfo *	: New packet				*/
/*  makevis	: bool		: Make packet visible			*/
/*  (returns)	: void		:					*/

void	KNDPacket::putPacket
	(	PktInfo	*pktinfo,
		bool	makevis
	)
{
	char	*tptr	[6] ;
	char	tidx	;

	/* The scheme is to construct an array of up to size pointers,	*/
	/* corresponding to the selected options. We can then create a	*/
	/* new QListViewItem object with the requested values appearing	*/
	/* in the correct columns.					*/
	memset	(tptr, 0, sizeof(tptr)) ;
	tidx	= 0 ;

	if (options & PO_TIME ) tptr[tidx++] = pktinfo->time  ;
	if (options & PO_PROTO) tptr[tidx++] = pktinfo->proto ;
	if (options & PO_SRC  ) tptr[tidx++] = pktinfo->srce  ;
	if (options & PO_DST  ) tptr[tidx++] = pktinfo->dest  ;
	if (options & PO_INFO ) tptr[tidx++] = pktinfo->info  ;
	if (options & PO_SIZE ) tptr[tidx++] = pktinfo->size  ;

	lastp	= new QListViewItem (plist,
				     lastp,
				     tptr[0], tptr[1], tptr[2],
				     tptr[3], tptr[4], tptr[5]) ;

	/* If the mnake-visible flag is set then ensure that the just-	*/
	/* added list item is visible, and update the various counter	*/
	/* fields.							*/
	if (makevis)
	{
		plist->ensureItemVisible (lastp) ;

		v_accept  .setText (ItoA (pktinfo->accept )) ;
		v_dropped .setText (ItoA (pktinfo->dropped)) ;
		v_missed  .setText (ItoA (pktinfo->missed )) ;
		v_pktbytes.setText (ItoA (s_pktbytes	  )) ;
		v_datbytes.setText (ItoA (s_datbytes	  )) ;
	}
}

/*  KNDPacket								*/
/*  putDots	: Add a marked packet of dots				*/
/*  (returns)	: void		:					*/

void	KNDPacket::putDots ()
{
	static	PktInfo	dots	;
	static	bool	first	= true ;

	if (first)
	{	strcpy	(dots.time,	"......") ;
		strcpy	(dots.proto,	"......") ;
		strcpy	(dots.srce,	"......") ;
		strcpy	(dots.dest,	"......") ;
		strcpy	(dots.info,	"......") ;
		strcpy	(dots.size,	"......") ;
	}

	putPacket (&dots, true) ;
}

/*  KNDPacket								*/
/*  redisplay	: Rebuild display					*/
/*  (returns)	: void		:					*/

void	KNDPacket::redisplay ()
{
	int	idx	;
	int	slot	;

	plist->clear ()	;
	lastp	= NULL	;

	/* Put all recorded packets back into the display. We scan	*/
	/* forward from the current insertion offset, since will	*/
	/* contain the oldest packet.					*/
	for (idx = 0, slot = offset ; idx < MAXPACK ; idx += 1)
	{
		if (packets[slot] != NULL)
			putPacket (packets[slot], false) ;

		if ((slot += 1) >= MAXPACK)
			slot = 0 ;
	}

	/* If paused then add the line of dots that indicates this to	*/
	/* the viewer.							*/
	if (pausefor > 0) putDots () ;

	/* Make sure that the last entry is visible. If we put in some	*/
	/* dots then this will be the case already, but no neccessarily	*/
	/* otherwise.							*/
	plist->ensureItemVisible (lastp) ;
}

/*  KNDPacket								*/
/*  setOnview	: Set whether we are on view or not			*/
/*  _onview	: bool		: TRUE if on view			*/
/*  (returns)	: void		:					*/

void	KNDPacket::setOnview
	(	bool	_onview
	)
{
	if ((onview = _onview) && (plist != NULL) && !frozen)
		redisplay () ;
}

/*  KNDPacket								*/
/*  freeze	: Freeze or unfreeze the display			*/
/*  _frozen	: bool		: New state				*/

void	KNDPacket::freeze
	(	bool	_frozen
	)
{
	if (!(frozen = _frozen) && (plist != NULL) && onview && (pausefor == 0))
		redisplay () ;
}

/*  KNDPacket								*/
/*  timerTick	: Handle timer tick					*/
/*  second	: long		: Second number				*/
/*  (returns)	: void		:					*/

void	KNDPacket::timerTick
	(	long	second
	)
{
	p_last	= p_this ;
	p_this	= 0	 ;

	/* If the last second's packet count was below the pause	*/
	/* threshhold, and we are currently paused, then decrement the	*/
	/* pause. Once this falls to zero, redisplay everything,	*/
	/* provided that we are on view and are now frozen.		*/
	if ((p_last < g_pauseat) && (pausefor > 0))
		if (((pausefor -= 1) == 0) && onview && !frozen)
			redisplay () ;

	/* If we are still paused, then at least update the count	*/
	/* fields (again, provided that we are on view and not paused).	*/
	if ((pausefor > 0) && onview && !frozen)
	{
		int	slot	;
		PktInfo	*lastpkt;

		if ((slot = offset + MAXPACK - 1) >= MAXPACK) slot -= MAXPACK ;
		lastpkt	= packets[slot] ;

		v_accept  .setText (ItoA (lastpkt->accept )) ;
		v_dropped .setText (ItoA (lastpkt->dropped)) ;
		v_missed  .setText (ItoA (lastpkt->missed )) ;
		v_pktbytes.setText (ItoA (s_pktbytes	  )) ;
		v_datbytes.setText (ItoA (s_datbytes	  )) ;
	}
}
