// ----------------------------------------------------------------------------
// Filename:    KTimeclock.h
// $Date: 1999/06/10 15:45:59 $
// $Revision: 1.6 $
// ----------------------------------------------------------------------------

#ifndef KTimeclock_included
#define KTimeclock_included

#include <kprocess.h>
#include <qqueue.h>

#include "KTimeclockData.h"

/**
 * Tree-view display for managing time spent; implementation class.
 *
 * Implementation class for the tree-view display to allow for managing of time
 * spent on various tasks.  This class implements all of the functionality
 * provided by the tree-view as outlined in @ref KTimeclockData.  Please refer
 * to that reference for more information on the purpose of the member
 * functions implemented here.
 *
 * @author  Graham TerMarsch (gtermars@home.com)
 * @version $Revision: 1.6 $
 * @see     KTimeclockData
 */
class KTimeclock : public KTimeclockData
{
    Q_OBJECT

public:
    /**
     * Constructor
     *
     * @param   widget  Parent widget
     * @param   name    Name of this widget
     */
    KTimeclock( QWidget* parent = NULL, const char* name = NULL );

    /**
     * Destructor
     */
    virtual ~KTimeclock ();

public slots:
    virtual void AddProject ();
    virtual void AddSubProject ();
    virtual void AddTask ();

    virtual void EditItem ();
    virtual void DeleteItem ();
    virtual void ClearItem ();
    virtual void ClearAllItems ();

    virtual void Preferences ();

    virtual void StartTimer ();
    virtual void StopTimer ();

    virtual int save();
    virtual int load();

protected slots:
    virtual void SelectionChanged( QListViewItem* );
    virtual void ShowPopup( QListViewItem*, const QPoint&, int );

protected:
    virtual void timerEvent( QTimerEvent* );

private:
    /**
     * Internal function; tests for the existence of the given directory and
     * creates it if it doesn't exist.
     *
     * @param   name    Path of the directory to test/create, underneath $HOME.
     */
    void TestDir (const char* name);

    /**
     * Internal function; recursively saves the items in the tree-view out to a
     * given stream.  This function will recurse into itself whenever it
     * determines that it is at a deeper level of nesting of
     * projects/sub-projects.
     *
     * @param   str     Text stream to output to.
     * @param   item    Item to recursively save.
     * @param   depth   Current recursion depth.
     */
    void _recursiveSave( QTextStream& str, KTimeclockListItem* item, int depth);

    /**
     * Internal function; loads all of the items from the given queue into the
     * tree-view, making sure to preserve the nesting depth.  Note that this
     * function IS recursive and does call itself when it enters a new level of
     * nesting.
     *
     * @param   queue   Queue to load items from.
     * @param   parent  Nesting parent item.
     * @param   depth   Current nesting depth.
     */
    void _loadFromQueue( QQueue<QString>& queue, KTimeclockListItem* item, int depth);

    /**
     * Internal function; clears the time associated with all of the items in
     * the project list.  This function will recurse into itself in order to
     * process all child/sibling items from the given item.
     *
     * @param   item    Root item to clear time associated with
     */
    void _recursiveClear (KTimeclockListItem* item);
};
#endif // KTimeclock_included
