/***************************************************************************
                          kmud.cpp
                      -------------------
    description          : Main Application Class
    begin                : Sun Jun 20 14:17:54 MEST 1999
    copyright            : (C) 1999 by Kmud Developer Team
    email                : kmud-devel@kmud.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "kmud.h"
#include "kmudapi.h"

#include "dialogs/dlgconnect.h"
#include "dialogs/dlgoptionfonts.h"
#include "dialogs/dlgoptioncolors.h"
#include "dialogs/dlgoptionpreferences.h"
#include "dialogs/dlgconnectionwizard.h"
#include "dialogs/dlgreconnect.h"
#include "dialogs/dlgprofilealiases.h"
#include "dialogs/dlgprofiletriggers.h"
#include "dialogs/dlgprofilelog.h"
#include "dialogs/dlgprofilemacros.h"
#include "dialogs/dlgoptionmud.h"
#include "dialogs/dlgmultilineinput.h"
#include "dialogs/dlgpluginmanager.h"
#include "dialogs/dlgcustomtoolbar.h"

#include <kwm.h>

KmudApp::KmudApp()
{
	setCaption("Kmud " VERSION);

	// get the Application-instance
	app = (KmudApplication*)KApplication::getKApplication();
	
	// init member variables
	docker = NULL;
	accelerators = new KAccel(this);
	
	lastNumToolbarItems = -1;
	///////////////////////////////////////////////////////////////////
	// instatiate multiline input dialog, it's not modal so we just hide/show it
	multiLineInput = new DlgMultiLineInput(NULL,"MultiLineInput");
	multiLineInput->hide();
	connect( multiLineInput,SIGNAL(dialogClosed()),SLOT(slotViewMultiLineInputClose()) );

	// initialize the KmudDoc
	doc = new KmudDoc(this);
	doc->setCurrentCharacterID("");
	doc->setCurrentMudID("");

	///////////////////////////////////////////////////////////////////
	// read the config file options
	// read them _before_ the inits so we init the right
	readOptions();

	resize(mainWindowWidth,mainWindowHeight);

	autoReconnect = false;

	mapper = NULL;
	
	///////////////////////////////////////////////////////////////////
	// call inits to invoke all other construction parts
	initMenuBar();
	initToolBar();
	initStatusBar();
	refreshEnablesDisables();	
	initView();
	initTimer();
	initAccelerators();	
	
	///////////////////////////////////////////////////////////////////
	// create new Telnet
	mud = new CTelnet;

	///////////////////////////////////////////////////////////////////
	// create new Mud log
	mudlog = new CMudLog;

	///////////////////////////////////////////////////////////////////
	// create new KmudAPI (needed by plugin manager)
	kmudapi=new KmudAPI(this);

	///////////////////////////////////////////////////////////////////
	// create new plugin manager (needed by mudstream)
	plugin_manager = new CPluginManager(kmudapi);
	plugin_manager->searchPlugins();

	autoloadPlugins.first();
	while (autoloadPlugins.current()!=NULL)
	{
		plugin_manager->loadPlugin(autoloadPlugins.current());
		plugin_manager->startPlugin(autoloadPlugins.current());
		autoloadPlugins.next();
	}

	///////////////////////////////////////////////////////////////////
	// create new Mud stream
	mudstream = new CMudStream(this);


	mudstream->setMapper(mapper);
	mudstream->setCharDBEntry(NULL);

	connect (mudstream, SIGNAL(processedOutput()),this, SLOT(slotMudOutputReceived()) );
	QObject::connect( view, SIGNAL(enteredInput(QString)), mudstream, SLOT(slotProcessInput(QString)) );


	///////////////////////////////////////////////////////////////////
	// enable bars dependend on config file setups
	if (!bViewToolbar)
		enableToolBar(KToolBar::Hide,0);
	if (!bViewCustomToolbar)
		enableToolBar(KToolBar::Hide,1);
			
	if (!bViewStatusbar)
		enableStatusBar(KStatusBar::Hide);


	menuBar()->setMenuBarPos(menu_bar_pos);
	mainToolBar->setBarPos(tool_bar_pos);
	customToolBar->setBarPos(custom_tool_bar_pos);

}

KmudApp::~KmudApp()
{
	// Make sure that is disable as the applications is about to quit
	autoReconnect = false;
	delete mainToolBar;
	delete customToolBar;
	delete mud;
	delete mudstream;
	delete mudlog;
	delete mud_menu;
	delete view_menu;
	delete option_menu;
	delete profile_menu;
	delete connectTimer;
	delete connectIcon;

	delete plugin_manager;
}

KmudView* KmudApp::getView()
{
	return view;
}

KmudDoc* KmudApp::getDoc()
{
	return doc;
}

CTelnet* KmudApp::getConnection()
{
	return mud;
}

CMudLog* KmudApp::getMudLog()
{
	return mudlog;
}


CPluginManager* KmudApp::getPluginManager()
{
	return plugin_manager;
}

CMapWindow* KmudApp::getMapper()
{
	return mapper;
}

void KmudApp::enableCommand(int id_)
{
	///////////////////////////////////////////////////////////////////
	// enable menu and toolbar functions by their ID's
	menuBar()->setItemEnabled(id_,true);
	mainToolBar->setItemEnabled(id_,true);
}

void KmudApp::disableCommand(int id_)
{
	///////////////////////////////////////////////////////////////////
	// disable menu and toolbar functions by their ID's
	menuBar()->setItemEnabled(id_,false);
	mainToolBar->setItemEnabled(id_,false);
}

void KmudApp::initTimer()
{
	//////////////////////////////////////////////////////////////////
	// setup the connection timer
	connectTimer = new QTimer( this );
	connect( connectTimer, SIGNAL(timeout()),this, SLOT(slotConnectionTime()) );
	connectHour = 0;
	connectMin = 0;
	connectSec = 0;

	mudOutputTimer = new QTimer(this);
	connect (mudOutputTimer, SIGNAL(timeout()),this, SLOT(slotMudOutputTimeout()) );
	mudOutputBlinkTimer = new QTimer(this);
	connect (mudOutputBlinkTimer, SIGNAL(timeout()),this, SLOT(slotMudOutputBlinker()) );

// now initialized by readOptions()
//	mudOutputNotifyOn = true;
//	mudOutputNotifyWaitSeconds = 10;
	mudOutputDoNotify = false;
	mudOutputIconBlinkOn = false;

}

void KmudApp::initMenuBar()
{

	QString connectStr;
	QString title,host,login,port;
	int loop,loop2;

	///////////////////////////////////////////////////////////////////
	// MENUBAR  

	///////////////////////////////////////////////////////////////////
	// menuBar entry mud_menu
	mud_menu = new QPopupMenu();
	mud_menu->insertItem(Icon("mini/kmud.xpm"), i18n("&New Window"), ID_MUD_NEW_WINDOW );
	mud_menu->insertSeparator();
	mud_menu->insertItem(Icon("kmud_connectionwizard.xpm"), i18n("&Connection Wizard"), ID_MUD_CONNECTION_WIZARD );
	mud_menu->insertItem(Icon("kmud_quickconnect.xpm"), i18n("&Quick Connect"), ID_MUD_CONNECT );
	mud_menu->insertItem(Icon("kmud_disconnect.xpm"), i18n("Close"), ID_MUD_CLOSE );
	mud_menu->setItemEnabled(ID_MUD_CLOSE, false);
	mud_menu->insertSeparator();
	mud_menu->insertItem(i18n("C&lose Window"), ID_MUD_CLOSE_WINDOW);
	mud_menu->insertSeparator();
	mud_menu->insertItem(i18n("E&xit"), ID_MUD_QUIT );
	mud_menu->insertSeparator();

	loop2=1;
	for (loop=(int)lastConnectHost.count()-1; loop>=0; loop--)
	{
		connectStr.sprintf("&%d: ",loop2++);
		title = lastConnectTitle.at(loop);
		login = lastConnectLogin.at(loop);
		host = lastConnectHost.at(loop);
		port = lastConnectPort.at(loop);
		if (title=="" && login == "")		
		{
			if (host!="" && port!="")
			  mud_menu->insertItem(connectStr + host + ":" + port,ID_MUD_QUICKCONNECTBASE + loop);
//			else
//			  mud_menu->insertItem(connectStr);
		}
		else
			mud_menu->insertItem(connectStr + login + "@" + title ,ID_MUD_QUICKCONNECTBASE + loop);					
	}	

	// mud_menu key accelerators
	mud_menu->setAccel(CTRL+Key_N, ID_MUD_NEW_WINDOW);
	mud_menu->setAccel(CTRL+Key_O, ID_MUD_CONNECTION_WIZARD);
	mud_menu->setAccel(CTRL+Key_E, ID_MUD_CONNECT);
	mud_menu->setAccel(CTRL+Key_Q, ID_MUD_QUIT);
	mud_menu->setAccel(ALT+Key_X, ID_MUD_QUIT);

	///////////////////////////////////////////////////////////////////
	// menuBar entry view_menu
	view_menu = new QPopupMenu();
	view_menu->setCheckable(true);
	view_menu->insertItem(i18n("Main Tool&bar"), ID_VIEW_TOOLBAR);
	view_menu->insertItem(i18n("&Programmble Toolbar"), ID_VIEW_CUSTOM_TOOLBAR);	
	view_menu->insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR );
	view_menu->insertItem(i18n("&Alternative Input "), ID_VIEW_ALTINPUT );
	view_menu->insertItem(i18n("Multiline &Input "), ID_VIEW_MULTILINEINPUT );
	view_menu->insertItem(i18n("Auto &Mapper"),ID_VIEW_MAPPER);

	view_menu->setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
	view_menu->setItemChecked(ID_VIEW_CUSTOM_TOOLBAR, bViewCustomToolbar);	
	view_menu->setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);
	view_menu->setItemChecked(ID_VIEW_ALTINPUT, bViewAltInput);
	view_menu->setItemChecked(ID_VIEW_MULTILINEINPUT, multiLineInput->isVisible());

	///////////////////////////////////////////////////////////////////
	// menuBar entry option_menu
	option_menu = new QPopupMenu();
	option_menu->setCheckable(true);
	option_menu->insertItem(i18n("&Fonts..."), ID_OPTION_FONT);
	option_menu->insertItem(i18n("&Colors..."), ID_OPTION_COLORS);
	option_menu->insertItem(i18n("&Keys..."), ID_OPTION_KEYS);
	option_menu->insertItem(i18n("&Preferences..."), ID_OPTION_PREFERENCES);
	option_menu->insertSeparator();
	mudOptionsID = option_menu->insertItem(i18n("Current &MUD Settings..."), ID_OPTION_MUD);

	///////////////////////////////////////////////////////////////////
	// menuBar entry profile_menu
	profile_menu = new QPopupMenu();
	profile_menu->setCheckable(true);
	aliasID = profile_menu->insertItem(Icon("kmud_aliases.xpm"), i18n("&Aliases..."), ID_PROFILE_ALIASES);
	triggerID = profile_menu->insertItem(Icon("kmud_triggers.xpm"), i18n("&Triggers..."), ID_PROFILE_TRIGGERS);
	loggingID = profile_menu->insertItem(Icon("kmud_log.xpm"), i18n("&Logging..."), ID_PROFILE_LOGGING);
	toolbarID = profile_menu->insertItem(i18n("&Toolbar..."),ID_PROFILE_TOOLBAR);
	profile_menu->insertSeparator();
	saveID = profile_menu->insertItem(i18n("&Save Current Profile"), ID_PROFILE_SAVE);

	///////////////////////////////////////////////////////////////////
	// menuBar entry plugin_menu
	plugin_menu = new QPopupMenu();
	plugin_menu->setCheckable(true);
	pluginManagerID = plugin_menu->insertItem(i18n("Plugin &Manager"), ID_PLUGIN_MANAGER);
	plugin_menu->insertSeparator();

	///////////////////////////////////////////////////////////////////
	// menuBar entry help_menu
	help_menu = new QPopupMenu();
	QString aboutString="";
	aboutString+=i18n("Kmud Version ");
	aboutString+=VERSION;
	aboutString+=i18n("\nCopyright (c) 1999,2000,2001 The Kmud Team <kmud-devel@kmud.de>\n    Stephan Uhlmann <stephan@kmud.de>\n    John-Paul Stanford <jp@stanwood.org.uk>\n    Roalt Zijlstra <roalt@dnd.utwente.nl>\n    Tomer Brisker <brisker@actcom.co.il>\n    Jrgen Kosche <mnementh@gmx.de>\n    Kevin Krammer <kevin.krammer@gmx.at>\n    Andre Alexander Bell <andre.bell@gmx.de>\nPlease see the AUTHORS file for all contributors.\n\nKmud Homepage: http://www.kmud.de\n\nKmud is distributed under the GNU General Public License.\nSee the file COPYING for further detail.");
	help_menu = kapp->getHelpMenu(true, aboutString);
	// create boss key entry
	help_menu->insertSeparator();
	help_menu->insertItem(i18n("Help! The Boss!"), ID_HELP_BOSS);

	///////////////////////////////////////////////////////////////////
	// MENUBAR CONFIGURATION
	// set menuBar() the current menuBar and the position due to config file
	KMenuBar *menu = new KMenuBar(this);
	menu->insertItem(i18n("&Mud"), mud_menu);
	menu->insertItem(i18n("&View"), view_menu);
	menu->insertItem(i18n("&Options"), option_menu);
	menu->insertItem(i18n("&Profile"), profile_menu);
	menu->insertItem(i18n("P&lugins"), plugin_menu);
 	menu->insertSeparator();
	menu->insertItem(i18n("&Help"), help_menu);
	setMenu(menu);

	///////////////////////////////////////////////////////////////////
	// CONNECT THE SUBMENU SLOTS WITH SIGNALS

	CONNECT_CMD(mud_menu);
	CONNECT_CMD(view_menu);
	CONNECT_CMD(option_menu);
	CONNECT_CMD(profile_menu);
	CONNECT_CMD(plugin_menu);
	CONNECT_CMD(help_menu);
}

void KmudApp::initToolBar()
{

	///////////////////////////////////////////////////////////////////
	// TOOLBAR
	// set toolBar() the current toolBar and the position due to config file
	mainToolBar = new KToolBar(this);
	addToolBar(mainToolBar);
	
	toolBarConnectButtonPosition=mainToolBar->insertButton(Icon("kmud_connectionwizard.xpm"), ID_MUD_CONNECTION_WIZARD, true, i18n("Connection Wizard"));
	mainToolBar->insertSeparator();
	aliasToolID = mainToolBar->insertButton(Icon("kmud_aliases.xpm"), ID_PROFILE_ALIASES, true, i18n("Aliases"));
	triggerToolID = mainToolBar->insertButton(Icon("kmud_triggers.xpm"), ID_PROFILE_TRIGGERS, true, i18n("Triggers"));
	loggingToolID = mainToolBar->insertButton(Icon("kmud_log.xpm"), ID_TOOLBAR_LOGGING_TOGGLE, true, i18n("Toggle Logging"));
	mapperToolID = mainToolBar->insertButton(Icon("kmud_mapper.xpm"), ID_VIEW_MAPPER, false, i18n("Show/Hide Mapper"));

	mainToolBar->insertSeparator();
	mainToolBar->insertButton(Icon("help.xpm"), ID_HELP, SIGNAL(pressed()), kapp, SLOT(appHelpActivated()), true, i18n("Help"));

	customToolBar = new KToolBar(this);
	addToolBar(customToolBar);
	updateCustomToolbar();
	
	///////////////////////////////////////////////////////////////////
	// CONNECT THE TOOLBAR SLOTS WITH SIGNALS - add new created toolbars
	CONNECT_TOOLBAR(mainToolBar);
	connect(customToolBar,SIGNAL(clicked(int)),this,SLOT(slotCustomToolbarClicked(int)));

	refreshEnablesDisables();

}

/** Used to update the custom tool bar */
void KmudApp::updateCustomToolbar(void)
{
	if (lastNumToolbarItems>-1)
	{
		for (int i=0; i<=lastNumToolbarItems;i++)
		{
			customToolBar->hideItem(i);
			customToolBar->removeItem(i);
		}		
	}

	CCharacterProfile* profile = doc->getCharacterProfile(doc->getCurrentCharacterID());

	if (profile)
	{
		ToolbarEntry* entry = profile->firstToolbarEntry();
		int count = 0;

	 	while (entry !=NULL)
	 	{
	 		if (entry->icon == "")
	 		{
	 			QLabel *tmpIcon = new QLabel(this);
	 			tmpIcon->setText(entry->name);
	 			
	 			QFont font("Helvetica",10);
	 			font.setBold(false);
			    font.setItalic(false);

				QFontMetrics fm(font);
				int w = fm.width(entry->name);
//				int h = fm.height();
	 			
		 		QPixmap *pm = new QPixmap();
		 		CCustomBarButton *button = new CCustomBarButton(*pm,count,customToolBar,"",w+5,(const char *)entry->name);
		 		
	 			customToolBar->insertWidget(count,button->width(),button);	 			
	 			
 			    connect(button, SIGNAL(clicked(int)), this, SLOT(slotCustomToolbarClicked(int)));
		 	}
		 	else
		 	{
		 		customToolBar->insertButton(Icon("mini/"+entry->icon),count,true,entry->name);		 		
		 	}
		 	
		 	
		 	entry = profile->nextToolbarEntry();
		 	count ++;
	 	}
	
	
		lastNumToolbarItems = count-1;	
	}
	else
		lastNumToolbarItems = -1;	
	
}

void KmudApp::initStatusBar()
{
  	///////////////////////////////////////////////////////////////////
  	//STATUSBAR
	connectIcon = new QLabel(statusBar());
	connectIcon->setPixmap(Icon("mini/mini-connect.xpm"));
	connectIcon->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  	statusBar()->setInsertOrder(KStatusBar::RightToLeft);
        statusBar()->insertItem(i18n("00:00:00"), ID_STATUS_TIME );
        statusBar()->insertWidget(connectIcon,25,ID_STATUS_ICON); 
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG );
}

void KmudApp::initView()
{ 
	////////////////////////////////////////////////////////////////////
	// set the main widget here
	KApplication *app=KApplication::getKApplication();
	view = new KmudView(app,doc,this);
	setView(view);

	///////////////////////////////////////////////////////////////////
	// set view options as defined in readOptions()
	view->setOutputDefaultForegroundColor(mudForegroundColor);
	view->setOutputDefaultBackgroundColor(mudBackgroundColor);
	view->setOutputDefaultFont(mudFont);
	view->setInputFont(inputFont);
	view->setInputBackgroundColor(mudBackgroundColor);
	view->setLocalEchoColor(mudLocalEchoColor);

	view->setDebugHighlightColor(mudDebugHighlightColor);

	for (int col=BLACK;col<ANSICOLOR_COUNT;col++)
	{
		view->setOutputANSIColor(col,mudANSIColor[col]);
	}
	
	for (int col=BG_BLACK;col<BGCOLOR_COUNT;col++)
	{
	  view->setOutputBgColor(col,mudBgColor[col]);
	}
	
	view->setAltInput(bViewAltInput);
	view->setMaxHistoryLines(maxInputLines);
	view->setMaxMudViewLines(maxOutputLines);
	view->setSharedInputHistory(sharedInputHistory);
	view->setEnableSpecialChars(enableSpecialChars);
	view->setSeparator(separator);
	view->setSpeedwalkingChar (speedwalkingChar);
	view->setExternChar (externChar);
	view->setUseBeep(useBeep);

	view->setLocalEchoMode(localEchoMode);
	view->setInputEchoMode(inputEchoMode);

	view->setKeepLastInput(keepLastInput);
	view->setSelectAllOnFocus(selectAllOnFocus);

	//connect( multiLineInput,SIGNAL(sendText(QString)),view, SLOT(slotAppendToOutput(QString)) );
}


void KmudApp::saveOptions()
{
	KConfig *config = kapp->getConfig();
	QStrList autoLoadPluginList;

	config->setGroup("APPEARANCE");
	config->writeEntry("WindowWidth",width());
	config->writeEntry("WindowHeight",height());
	config->writeEntry("ShowToolbar",mainToolBar->isVisible());
	config->writeEntry("ShowCustomToolbar",customToolBar->isVisible());	
	config->writeEntry("ShowStatusbar",statusBar()->isVisible());
	config->writeEntry("ShowAltInput",bViewAltInput);
	config->writeEntry("MenuBarPos", (int)menuBar()->menuBarPos());
	config->writeEntry("ToolBar_Pos", (int)mainToolBar->barPos());
	config->writeEntry("CustomToolBar_Pos",(int)customToolBar->barPos());

	config->setGroup("OPTIONS");

	config->writeEntry("MudTextColor",view->getOutputDefaultForegroundColor());

	config->writeEntry("MudBackgroundColor",view->getOutputDefaultBackgroundColor());

	config->writeEntry("MudFont",view->getOutputDefaultFont());

	config->writeEntry("InputFont",view->getInputFont());

	config->writeEntry("MudLocalEchoColor",view->getLocalEchoColor());

	config->writeEntry("MudDebugHighlightColor", view->getDebugHighlightColor());

	config->writeEntry("MaxOutputLines",maxOutputLines);
	config->writeEntry("MaxInputLines",maxInputLines);
	config->writeEntry("SharedInputHistory",sharedInputHistory);
	config->writeEntry("Local_Echo",localEchoMode);
	config->writeEntry("Local_Echo_SuppressNewline",localEchoSuppressNewline);
	config->writeEntry("Input_Echo",inputEchoMode);
	config->writeEntry("Keep_Last_input",keepLastInput);
	config->writeEntry("Select_All_On_Focus",selectAllOnFocus);
	config->writeEntry("AutoReconnect",autoReconnectOption);
	config->writeEntry("AutoReconnectTime",autoTime);
	config->writeEntry("EnableSpecialChars",enableSpecialChars);
	config->writeEntry("CharSeparator",separator);
	config->writeEntry("CharSpeedwalking", speedwalkingChar);
	config->writeEntry("CharExternCommand", externChar);
	config->writeEntry("IntelligentHistory", doc->getIntelligentHistory());
	config->writeEntry("NumpadMacros", app->getNumpadMacros());
	config->writeEntry("UseBeep",useBeep);
	config->writeEntry("OutputNotify",mudOutputNotifyOn);
	config->writeEntry("OutputNotifyWaitSeconds",mudOutputNotifyWaitSeconds);
	config->writeEntry("ConnectTimeout",connecttimeout);

	config->setGroup("HISTORY");
	if (lastConnectHost.at(0)!=NULL) 
		config->writeEntry("0Host",lastConnectHost.at(0)); 
	else 
		config->writeEntry("0Host","");
	if (lastConnectPort.at(0)!=NULL) 
		config->writeEntry("0Port",lastConnectPort.at(0)); 
	else 
		config->writeEntry("0Port","");
	if (lastConnectTitle.at(0)!=NULL) 
		config->writeEntry("0Title",lastConnectTitle.at(0)); 
	else 
		config->writeEntry("0Title","");
	if (lastConnectLogin.at(0)!=NULL) 
		config->writeEntry("0Login",lastConnectLogin.at(0)); 
	else 
		config->writeEntry("0Login","");
	if (lastConnectHost.at(1)!=NULL) 
		config->writeEntry("1Host",lastConnectHost.at(1));
	else 
		config->writeEntry("1Host","");
	if (lastConnectPort.at(1)!=NULL) 
		config->writeEntry("1Port",lastConnectPort.at(1)); 
	else 
		config->writeEntry("1Port","");
	if (lastConnectTitle.at(1)!=NULL) 
		config->writeEntry("1Title",lastConnectTitle.at(1)); 
	else 
		config->writeEntry("1Title","");
	if (lastConnectLogin.at(1)!=NULL) 
		config->writeEntry("1Login",lastConnectLogin.at(1)); 
	else 
		config->writeEntry("1Login","");
	if (lastConnectHost.at(2)!=NULL) 
		config->writeEntry("2Host",lastConnectHost.at(2));
	else 
		config->writeEntry("2Host","");
	if (lastConnectPort.at(2)!=NULL) 
		config->writeEntry("2Port",lastConnectPort.at(2)); 
	else config->writeEntry("2Port","");
	if (lastConnectTitle.at(2)!=NULL) 
		config->writeEntry("2Title",lastConnectTitle.at(2)); 
	else 
		config->writeEntry("2Title","");
	if (lastConnectLogin.at(2)!=NULL) 
		config->writeEntry("2Login",lastConnectLogin.at(2)); 
	else 
		config->writeEntry("2Login","");
	if (lastConnectHost.at(3)!=NULL) 
		config->writeEntry("3Host",lastConnectHost.at(3)); 
	else 
		config->writeEntry("3Host","");
	if (lastConnectPort.at(3)!=NULL) 
		config->writeEntry("3Port",lastConnectPort.at(3)); 
	else 
		config->writeEntry("3Port","");
	if (lastConnectTitle.at(3)!=NULL) 
		config->writeEntry("3Title",lastConnectTitle.at(3)); 
	else 
		config->writeEntry("3Title","");
	if (lastConnectLogin.at(3)!=NULL) 
		config->writeEntry("3Login",lastConnectLogin.at(3)); 
	else 
		config->writeEntry("3Login","");
	if (lastConnectHost.at(4)!=NULL) 
		config->writeEntry("4Host",lastConnectHost.at(4)); 
	else 
		config->writeEntry("4Host","");
	if (lastConnectPort.at(4)!=NULL) 
		config->writeEntry("4Port",lastConnectPort.at(4)); 
	else 
		config->writeEntry("4Port","");
	if (lastConnectTitle.at(4)!=NULL) 
		config->writeEntry("4Title",lastConnectTitle.at(4)); 
	else 
		config->writeEntry("4Title","");
	if (lastConnectLogin.at(4)!=NULL) 
		config->writeEntry("4Login",lastConnectLogin.at(4)); 
	else 
		config->writeEntry("4Login","");

	config->setGroup("ANSI COLORS");
	
	for (int col=BLACK;col<ANSICOLOR_COUNT;col++)
	{
		config->writeEntry(ANSICOLOR_NAME[col], view->getOutputANSIColor(col));
	}

	for (int col=BG_BLACK;col<BGCOLOR_COUNT;col++)
	{
		config->writeEntry(BGCOLOR_NAME[col],view->getOutputBgColor(col));
	}
	
	config->setGroup("PLUGINS");
	config->writeEntry("autoload",autoloadPlugins);
	
	// save accelerators
	accelerators->writeSettings(config);

 	config->setGroup("PING");
	config->writeEntry("executable", doc->getPingExecutable());
  config->writeEntry("enabled", doc->getPingEnabled());
}

void KmudApp::readOptions()
{
	///////////////////////////////////////////////////////////////////
	// read the config file entries
	KConfig *config = kapp->getConfig();

	config->setGroup("APPEARANCE");
	mainWindowWidth = config->readNumEntry("WindowWidth",600);
	mainWindowHeight = config->readNumEntry("WindowHeight",400);
	bViewToolbar = config->readBoolEntry("ShowToolbar", true);
	bViewCustomToolbar = config->readBoolEntry("ShowCustomToolbar", false);
	bViewStatusbar = config->readBoolEntry("ShowStatusbar", true);
	bViewAltInput = config->readBoolEntry("ShowAltInput", false);
	menu_bar_pos = (KMenuBar::menuPosition)config->readNumEntry("MenuBarPos", KMenuBar::Top); 
	tool_bar_pos = (KToolBar::BarPosition)config->readNumEntry("ToolBar_Pos", KToolBar::Top);
	custom_tool_bar_pos = (KToolBar::BarPosition)config->readNumEntry("CustomToolBar_Pos", KToolBar::Top);


	config->setGroup("OPTIONS");
	mudForegroundColor=QColor(0,255,0);
	mudForegroundColor=config->readColorEntry("MudTextColor", &mudForegroundColor);
	mudBackgroundColor=QColor(0,0,0);
	mudBackgroundColor=config->readColorEntry("MudBackgroundColor", &mudBackgroundColor);
	mudFont=QFont("fixed");
	mudFont=config->readFontEntry("MudFont", &mudFont);
	inputFont=QFont("helvetica");
	inputFont=config->readFontEntry("InputFont", &inputFont);
	mudLocalEchoColor=QColor(255,0,0);
	mudLocalEchoColor=config->readColorEntry("MudLocalEchoColor", &mudLocalEchoColor);
	mudDebugHighlightColor=QColor(0, 255, 0);
	mudDebugHighlightColor=config->readColorEntry("MudDebugHighlightColor", &mudDebugHighlightColor);
	
	maxOutputLines = config->readNumEntry("MaxOutputLines",5000);
	maxInputLines = config->readNumEntry("MaxInputLines",1000);
	sharedInputHistory = config->readBoolEntry("SharedInputHistory",false);
	localEchoMode = config->readNumEntry("Local_Echo",2);
	localEchoSuppressNewline = config->readBoolEntry("Local_Echo_SuppressNewline",false);
	inputEchoMode = config->readNumEntry("Input_Echo",2);
	enableSpecialChars = config->readBoolEntry("EnableSpecialChars",true);	
	separator = config->readEntry("CharSeparator", ";");
	speedwalkingChar = config->readEntry ("CharSpeedwalking", ".");
	externChar = config->readEntry ("CharExternCommand", "#");
	doc->setIntelligentHistory(config->readBoolEntry("IntelligentHistory", false));
	app->setNumpadMacros(config->readBoolEntry("NumpadMacros", true));
	keepLastInput = config->readBoolEntry("Keep_Last_Input",false);
	selectAllOnFocus = config->readBoolEntry("Select_All_On_Focus",false);
	autoReconnectOption = config->readBoolEntry("AutoReconnect",true);
	autoTime = config->readNumEntry("AutoReconnectTime",10);
	useBeep = config->readBoolEntry("UseBeep",true);
	mudOutputNotifyOn = config->readBoolEntry("OutputNotify",true);
	mudOutputNotifyWaitSeconds = config->readNumEntry("OutputNotifyWaitSeconds",10);
	connecttimeout = config->readNumEntry("ConnectTimeout",10);


	config->setGroup("HISTORY");
	lastConnectHost.clear();
	lastConnectPort.clear();
	lastConnectHost.append(config->readEntry("0Host",""));
	lastConnectPort.append(config->readEntry("0Port",""));
	lastConnectTitle.append(config->readEntry("0Title",""));
	lastConnectLogin.append(config->readEntry("0Login",""));
	lastConnectHost.append(config->readEntry("1Host",""));
	lastConnectPort.append(config->readEntry("1Port",""));
	lastConnectTitle.append(config->readEntry("1Title",""));
	lastConnectLogin.append(config->readEntry("1Login",""));
	lastConnectHost.append(config->readEntry("2Host",""));
	lastConnectPort.append(config->readEntry("2Port",""));
	lastConnectTitle.append(config->readEntry("2Title",""));
	lastConnectLogin.append(config->readEntry("2Login",""));
	lastConnectHost.append(config->readEntry("3Host",""));
	lastConnectPort.append(config->readEntry("3Port",""));
	lastConnectTitle.append(config->readEntry("3Title",""));
	lastConnectLogin.append(config->readEntry("3Login",""));
	lastConnectHost.append(config->readEntry("4Host",""));
	lastConnectPort.append(config->readEntry("4Port",""));
	lastConnectTitle.append(config->readEntry("4Title",""));
	lastConnectLogin.append(config->readEntry("4Login",""));


	config->setGroup("ANSI COLORS");
	
	mudANSIColor[BLACK]=QColor(0, 0, 0);
	mudANSIColor[DARK_RED]=QColor(128, 0, 0);
	mudANSIColor[DARK_GREEN]=QColor(0, 128, 0);
	mudANSIColor[DARK_YELLOW]=QColor(128, 128, 0);
	mudANSIColor[DARK_BLUE]=QColor(0, 0, 128);
	mudANSIColor[DARK_MAGENTA]=QColor(128, 0, 128);
	mudANSIColor[DARK_CYAN]=QColor(0, 128, 128);
	mudANSIColor[DARK_WHITE]=QColor(192, 192, 192);
	mudANSIColor[GRAY]=QColor(128, 128, 128);
	mudANSIColor[LIGHT_RED]=QColor(255, 0, 0);
	mudANSIColor[LIGHT_GREEN]=QColor(0, 255, 0);
	mudANSIColor[LIGHT_YELLOW]=QColor(255, 255, 0);
	mudANSIColor[LIGHT_BLUE]=QColor(0, 0, 255);
	mudANSIColor[LIGHT_MAGENTA]=QColor(255, 0, 255);
	mudANSIColor[LIGHT_CYAN]=QColor(0, 255, 255);
	mudANSIColor[LIGHT_WHITE]=QColor(255, 255, 255);
	
	mudBgColor[BG_BLACK]=QColor(0, 0, 0);
	mudBgColor[BG_RED]=QColor(128, 0, 0);
	mudBgColor[BG_GREEN]=QColor(0, 128, 0);
	mudBgColor[BG_YELLOW]=QColor(128, 128, 0);
	mudBgColor[BG_BLUE]=QColor(0, 0, 128);
	mudBgColor[BG_MAGENTA]=QColor(128, 0, 128);
	mudBgColor[BG_CYAN]=QColor(0, 128, 128);
	mudBgColor[BG_WHITE]=QColor(192, 192, 192);
	
	for (int col=BLACK;col<ANSICOLOR_COUNT;col++)
	{
	  mudANSIColor[col]=
	  	config->readColorEntry(ANSICOLOR_NAME[col],&mudANSIColor[col]);
	}
	
	for (int col=BG_BLACK;col<BGCOLOR_COUNT;col++)
	{
	  mudBgColor[col]=
	  	config->readColorEntry(BGCOLOR_NAME[col], &mudBgColor[col]);
	}

	config->setGroup("PLUGINS");
	config->readListEntry("autoload",autoloadPlugins);

	config->setGroup("PING");
	doc->setPingExecutable(config->readEntry("executable", "/bin/ping"));
  doc->setPingEnabled(config->readBoolEntry("enabled", true));
}

/** Connect to a mud that as been setup in with the wizard
  * eg. ID -> "Sunbright@MidnightSun"
  */
int KmudApp::connectWizardId(QString ID)
{
	QString character;
	QString mud;
	QString port;
	CharDBEntry* entry;
	CMudProfile* mudPro;
	int pos;

	// Find the mud Title and Charcter from given ID
	pos = ID.find('@',0,false);
	character = ID.left(pos);
	mud = ID.right(ID.length()-pos-1);

	entry=doc->getEntry(character,mud);	
	if (entry!=NULL)
	{
		mudPro = doc->getMudProfile(entry->mudfile);
		
		if (mudPro!=NULL)
		{
			// Attempt to connect to the mud
			port.sprintf("%d",mudPro->getPort());		
			connectToMud(mudPro->getTitle(),entry->login,mudPro->getHostName(),port);
		}
		else
		{
			// Unable to find the entry so tell the user
			QMessageBox NotFound("Kmud",ID + i18n(" can't be found"),QMessageBox::Information,
				QMessageBox::Ok | QMessageBox::Default,0,0,0,"NoConnectMessageBox1",TRUE,0);
			NotFound.exec();			
		}

	}
	else
	{
		// Unable to find the entry so tell the user
		QMessageBox NotFound("Kmud",ID + i18n(" can't be found"),QMessageBox::Information,
				QMessageBox::Ok | QMessageBox::Default,0,0,0,"NoConnectMessageBox1",TRUE,0);
		NotFound.exec();			
	}
}


void KmudApp::connectToMud(QString mudTitle, QString login,QString mudHost,QString mudPort)
{
	QString connectStr,mHost,mPort,mTitle,mLogin;
	CharDBEntry* entry;
	CCharacterProfile* prof;
	int connectReturn=0;
	int loop,loop2;
	unsigned int port;


	if (mudPort=="") port=23;
	else port=mudPort.toUInt();

	connectReturn = mud->connect(mudHost,port,connecttimeout);
	mPort.setNum(mud->getHostPort());

	switch ( connectReturn )
	{
		case -1 : {
			QMessageBox noconnect1("Kmud",i18n("Unknown Host")+QString(" (")+mudHost+")",QMessageBox::Warning,
					QMessageBox::Ok | QMessageBox::Default,0,0,0,"NoConnectMessageBox1",TRUE,0);
			noconnect1.exec();
			break;
			}
		case -2 : {
			QMessageBox noconnect2("Kmud",i18n("Cant create socket"),QMessageBox::Warning,
					QMessageBox::Ok | QMessageBox::Default,0,0,0,"NoConnectMessageBox2",TRUE,0);
			noconnect2.exec();
			break;
			}
		case -3 : {
			QMessageBox noconnect3("Kmud",i18n("Connection to ") + mud->getHostIP() + ":" + mPort + i18n(" refused"),QMessageBox::Warning,
					QMessageBox::Ok | QMessageBox::Default,0,0,0,"NoConnectMessageBox3",TRUE,0);
			noconnect3.exec();
			break;
			}

		case -4 : {
			QMessageBox noconnect4("Kmud",i18n("Connect to ") + mud->getHostIP() + i18n(" timed out"),QMessageBox::Warning,
					QMessageBox::Ok | QMessageBox::Default,0,0,0,"NoConnectMessageBox4",TRUE,0);
			noconnect4.exec();
			break;
			}

		case -5 : {
			QMessageBox noconnect5("Kmud",i18n("Network unreachable ") + QString("(") + mud->getHostIP() + ":" + mPort + ")",QMessageBox::Warning,
					QMessageBox::Ok | QMessageBox::Default,0,0,0,"NoConnectMessageBox5",TRUE,0);
			noconnect5.exec();
			break;
			}

		case -6 : {
			QMessageBox noconnect6("Kmud",i18n("No route to host ") + mud->getHostIP(),QMessageBox::Warning,
					QMessageBox::Ok | QMessageBox::Default,0,0,0,"NoConnectMessageBox6",TRUE,0);
			noconnect6.exec();
			break;
			}

		case -10 : {
			QMessageBox noconnect10("Kmud",i18n("Failed to connect to ") + mud->getHostIP() + ":" + mPort,QMessageBox::Warning,
					QMessageBox::Ok | QMessageBox::Default,0,0,0,"NoConnectMessageBox10",TRUE,0);
			noconnect10.exec();
			}

	} // switch ( connectReturn )

	if (connectReturn>=0)
	{
		QObject::connect( mud, SIGNAL(filledBuffer(QString)), mudstream, SLOT(slotProcessOutput(QString)) );
		//QObject::connect( view, SIGNAL(enteredInput(QString)), mudstream, SLOT(slotProcessInput(QString)) );
		QObject::connect( multiLineInput,SIGNAL(sendText(QString)),mudstream, SLOT(slotProcessInput(QString)) );
		QObject::connect( mud, SIGNAL(changedEcho(bool)), view, SLOT(setLocalEcho(bool)) );
		QObject::connect( mud, SIGNAL(changedEcho(bool)), view, SLOT(setInputEcho(bool)) );
		QObject::connect( mud, SIGNAL(closedConnection()), view, SLOT(slotMudClosed()) );
		QObject::connect( mud, SIGNAL(closedConnection()), this, SLOT(slotMudClose()) );


		entry = doc->getEntry(login,mudTitle);
		if (entry!=NULL)
		{
			mudstream->setCharDBEntry(entry);
			
			doc->setCurrentCharacterID(entry->profile);
			doc->setCurrentMudID(entry->mudfile);

			prof = doc->getCharacterProfile(doc->getCurrentCharacterID());
			if (prof->getLoggingEnabled())
			{
				prof->setLoggingEnabled(mudlog->open(prof->getLogFilename()));
			}
		}
		else
		{
			doc->setCurrentCharacterID("");
			doc->setCurrentMudID("");
		}


		if ( (login!="") && (mudTitle!="") )
			setCaption(login+"@"+mudTitle+" - Kmud " VERSION);
		else
			setCaption(mudHost+" - Kmud " VERSION);


		// TODO: we have four different lists hist,port,title and login
		//       better: make one structure and one list of this struct

		for (loop=0;loop<=(int)lastConnectHost.count()-1; loop++)
		{
		 	// remove list, we will regenerate new afterwards
	  		if (mud_menu->indexOf(ID_MUD_QUICKCONNECTBASE + loop)!=-1)
				{
					mud_menu->removeItem(ID_MUD_QUICKCONNECTBASE + loop);
				}
		}	// for loop ...

		for (loop=0;loop<=(int)lastConnectHost.count()-1; loop++)
		{
		 	// remove older connects to the same mud
			if ((lastConnectHost.at(loop)==mudHost)
			&& (lastConnectPort.at(loop)==mudPort)
			&& (lastConnectTitle.at(loop)==mudTitle)
			&& (lastConnectLogin.at(loop)==login))
			{
				lastConnectHost.remove(loop);
				lastConnectPort.remove(loop);
				lastConnectTitle.remove(loop);
				lastConnectLogin.remove(loop);
			}
		}	// for loop ...



	 	// Add mud to the history list;
		lastConnectHost.append(mudHost);
		lastConnectPort.append(mudPort);
		lastConnectTitle.append(mudTitle);
		lastConnectLogin.append(login);

		// too much now? -> remove fist entry
		while (	(lastConnectHost.count()>5) )
		{
			lastConnectHost.removeFirst();
			lastConnectPort.removeFirst();
			lastConnectTitle.removeFirst();
			lastConnectLogin.removeFirst();
		} // while >5

	 	// Regenerate mud menu with the new list
		loop2=1;
		for (loop=(int)lastConnectHost.count()-1; loop>=0; loop--)
		{
			connectStr.sprintf("&%d: ",loop2++);
			mTitle = lastConnectTitle.at(loop);
			mLogin = lastConnectLogin.at(loop);
			mHost = lastConnectHost.at(loop);
			mPort = lastConnectPort.at(loop);
			if (mTitle=="" && mLogin == "")		
			{
				if (mHost!="" && mPort!="")
			  		mud_menu->insertItem(connectStr + mHost + ":" + mPort,ID_MUD_QUICKCONNECTBASE + loop);
//				else
//			  		mud_menu->insertItem(connectStr,ID_MUD_QUICKCONNECTBASE + loop);
			}
			else
				mud_menu->insertItem(connectStr + mLogin + "@" + mTitle ,ID_MUD_QUICKCONNECTBASE + loop);					
		}	// for loop ...


		// update icons on toolbar
		mainToolBar->removeItem(ID_MUD_CONNECTION_WIZARD);
		mainToolBar->insertButton(Icon("kmud_disconnect.xpm"), ID_MUD_CLOSE, true, i18n("Close"),toolBarConnectButtonPosition);

		refreshLoggingToolbarIcon();


		// Update the connection pixmap on the status bar
		connectIcon->setPixmap(Icon("mini/mini-connected.xpm"));
		// Setup and start the connection timer
		statusBar()->changeItem(i18n("00:00:00"), ID_STATUS_TIME );
		connectHour = 0;
		connectMin = 0;
		connectSec = 0;
		connectTimer->start(1000,FALSE);

		mudOutputTimer->start(10000,true); // WARNING: hard coded to 10 sec. make me configurable

		// Make a copy of the prameters to this function so that the autoreconnect
		// is able to reconncect.
		lastTitle = mudTitle;
		lastLogin = login;
		lastHost = mudHost;
		lastPort = mudPort;

		autoReconnect = true;	

		mud_menu->setItemEnabled(ID_MUD_CONNECTION_WIZARD, false);
		mud_menu->setItemEnabled(ID_MUD_CONNECT, false);
		mud_menu->setItemEnabled(ID_MUD_CLOSE, true);
		mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT0, false);
		mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT1, false);
		mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT2, false);
		mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT3, false);
		mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT4, false);
		updateCustomToolbar();
		
	} // if (connectReturn>=0)

	// enable/disable icons and menu slots on the current state of connection
	refreshEnablesDisables();
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

/**
  * This method hides the main window, after it assured that the docker
  * widget exists to get it back.
  */
void KmudApp::slotHideMainWindow()
{
  if (docker == NULL) // lazy creation
  {
    docker = new KmudDocker(this, Icon("mini/kmud.xpm"));
  }

  docker->show();
  hide();
}

void KmudApp::slotMudNewWindow()
{
	slotStatusMsg(i18n("Opening a new Application window..."));
	(new KmudApp)->show();
	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotMudConnectionWizard()
{
	int ret=0;
	CharDBEntry* entry;

	slotStatusMsg(i18n("Connection Wizard..."));

	DlgConnectionWizard d(doc,this,"ConnectionWizardDialog");


	ret=d.exec(); // Connect=1, OK=2, Cancel=Reject

	if (ret)
	{
		/* Read in MUD options */
		if (doc->getEntry(d.getLogin(),d.getMudTitle()))
		{
			doc->getMudProfile(doc->getEntry(d.getLogin(),d.getMudTitle())->mudfile)->readData();
			interpretType = doc->getMudProfile(doc->current()->mudfile)->getInterpretationType();
			view->setInterpretType(interpretType);
		}
		/* Finished reading in options*/

		switch (ret)
		{
			case 1:
				if ( !mud->isConnected() )
				{
					connectToMud(d.getMudTitle(),d.getLogin(),d.getMudHost(),d.getMudPort());

/*

- this is all done in connectToMud
- doc->setCurrentMudID(d.getSelectedMudID()) is WRONG because it always sets the
  selected id, even when the connect to the mud failed


					if ((d.getLogin()!="") && (d.getMudTitle()!=""))
					{
						currentLogin=d.getLogin();
						currentMudTitle=d.getMudTitle();
						doc->setCurrentMudID(d.getSelectedMudID());
						doc->setCurrentCharacterID(d.getSelectedCharacterID());
					}
					else
					{
						currentLogin="";
						currentMudTitle="";
						doc->setCurrentCharacterID("");
						doc->setCurrentMudID("");
					}
					refreshEnablesDisables();
*/
				} // end if mud->isConnected()
				break;
			case 2:
				if ( !mud->isConnected() )
				{
					if ((d.getLogin()!="") && (d.getMudTitle()!=""))
					{
						entry = doc->getEntry(d.getLogin(),d.getMudTitle());
						mudstream->setCharDBEntry(entry);
						doc->setCurrentMudID(d.getSelectedMudID());
						doc->setCurrentCharacterID(d.getSelectedCharacterID());
					}
					else
					{
						doc->setCurrentCharacterID("");
						doc->setCurrentMudID("");
					}
					refreshEnablesDisables();
					refreshLoggingToolbarIcon();

					if ( (d.getLogin()!="") && (d.getMudTitle()!="") )
					{
						setCaption(d.getLogin()+"@"+d.getMudTitle()+" - Kmud " VERSION);
						lastTitle = d.getMudTitle();
						lastLogin = d.getLogin();
					}
				} // end if mud->isConnected()
				break;
		} //switch (ret)

	} // if (ret)

	slotStatusMsg(i18n("Ready."));
}

/** Used to connect to a previous mud for the history list */
void KmudApp::slotMudQuickConnect(int histNum)
{
	connectToMud(lastConnectTitle.at(histNum),lastConnectLogin.at(histNum),lastConnectHost.at(histNum),lastConnectPort.at(histNum));
}


void KmudApp::slotMudConnect()
{
	slotStatusMsg(i18n("Connecting to MUD..."));

	DlgConnect d(this,"ConnectDialog");

	d.setHost(lastConnectHost.last());
	d.setPort(lastConnectPort.last());

	if ( !mud->isConnected() && d.exec())
	{
		connectToMud("","",d.getHost(),d.getPort());
		interpretType = 0; // auto detect by default
	} // end if mud->isConnected()

	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotMudClose()
{
	bool quit=false;
	slotStatusMsg(i18n("Closing connection..."));
	
	QObject::disconnect( mud, SIGNAL(filledBuffer(QString)), mudstream, SLOT(slotProcessOutput(QString)) );
	//QObject::disconnect( view, SIGNAL(enteredInput(QString)), mudstream, SLOT(slotProcessInput(QString)) );
 	QObject::disconnect( multiLineInput,SIGNAL(sendText(QString)),mudstream, SLOT(slotProcessInput(QString)) );
	QObject::disconnect( mud, SIGNAL(changedEcho(bool)), view, SLOT(setLocalEcho(bool)) );
	QObject::disconnect( mud, SIGNAL(changedEcho(bool)), view, SLOT(setInputEcho(bool)) );
	QObject::disconnect( mud, SIGNAL(closedConnection()), this, SLOT(slotMudClose()) );
	QObject::disconnect( mud, SIGNAL(closedConnection()), view, SLOT(slotMudClosed()) );
	view->setLocalEcho(true);
	view->setInputEcho(true);

	if (mud->isConnected())
		mud->close();

	mudlog->close();

	if (mapper)
	{
		slotCloseMapper();
	}

	mudstream->reset ();

	if (mainToolBar->getButton(ID_MUD_CLOSE)!=NULL)
	{
		mainToolBar->removeItem(ID_MUD_CLOSE);
		mainToolBar->insertButton(Icon("kmud_connectionwizard.xpm"), ID_MUD_CONNECTION_WIZARD, true, i18n("Connect"),toolBarConnectButtonPosition);
	}

	doc->setCurrentCharacterID("");
	doc->setCurrentMudID("");
	refreshEnablesDisables();

	setCaption("Kmud " VERSION);

	slotStatusMsg(i18n("Ready."));
	connectIcon->setPixmap(Icon("mini/mini-connect.xpm"));
	statusBar()->changeItem(i18n("00:00:00"), ID_STATUS_TIME );
	connectTimer->stop();

	mud_menu->setItemEnabled(ID_MUD_CONNECTION_WIZARD, true);
	mud_menu->setItemEnabled(ID_MUD_CONNECT, true);
	mud_menu->setItemEnabled(ID_MUD_CLOSE, false);
	mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT0, true);
	mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT1, true);
	mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT2, true);
	mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT3, true);
	mud_menu->setItemEnabled(ID_MUD_QUICKCONNECT4, true);

	updateCustomToolbar();
	
	if (autoReconnect && autoReconnectOption)
	{
		DlgReconnect d(this,"ReconnectDialog");
		d.setAutoTime(autoTime);
		switch (d.exec())
		{
			case 1 : 
				connectToMud(lastTitle,lastLogin,lastHost,lastPort);
				break;
			case 2 : 
				quit = true; 
				break;
			case 3 : 
				slotMudConnect(); 
				break;
			case 4 : 
				slotMudConnectionWizard(); 
				break;
		}
	}
	if (quit) close();
}


void KmudApp::slotMudCloseWindow()
{
	close();
}

void KmudApp::slotMudQuit()
{ 

	///////////////////////////////////////////////////////////////////
	// exits the Application
	if(queryExit())
	{
		saveOptions();
		slotCloseMapper();		
		KTMainWindow::deleteAll();
		kapp->quit();
	}
	else
		slotStatusMsg(i18n("Ready."));
}


void KmudApp::slotViewToolBar()
{
	///////////////////////////////////////////////////////////////////
	// turn Toolbar on or off
	bViewToolbar=!bViewToolbar;
	menuBar()->setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
	enableToolBar(KToolBar::Toggle,0);
	slotStatusMsg(i18n("Ready."));

}

void KmudApp::slotViewCustomToolBar()
{
	bViewCustomToolbar=!bViewCustomToolbar;
	menuBar()->setItemChecked(ID_VIEW_CUSTOM_TOOLBAR, bViewCustomToolbar);
	enableToolBar(KToolBar::Toggle,1);
	
	updateCustomToolbar();
	slotStatusMsg(i18n("Ready."));	
}

void KmudApp::slotViewStatusBar()
{
	///////////////////////////////////////////////////////////////////
	//turn Statusbar on or off
	bViewStatusbar=!bViewStatusbar;
	menuBar()->setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);
	statusBar()->clear();
	enableStatusBar();
	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotViewAltInput()
{
	///////////////////////////////////////////////////////////////////
	//turn alternative Iinput on or off
	bViewAltInput=!bViewAltInput;
	menuBar()->setItemChecked(ID_VIEW_ALTINPUT, bViewAltInput);

	view->setAltInput(bViewAltInput);

	slotStatusMsg(i18n("Ready."));
}


void KmudApp::slotViewMultiLineInput()
{
	///////////////////////////////////////////////////////////////////
	//turn multiline Iinput on or off

	if (multiLineInput->isVisible())
		multiLineInput->hide();
	else 
		multiLineInput->show();

	view_menu->setItemChecked(ID_VIEW_MULTILINEINPUT, multiLineInput->isVisible());

	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotViewMultiLineInputClose()
{
	///////////////////////////////////////////////////////////////////
	//called when multiline inout is closed (not a modal dialog!)

	multiLineInput->hide();
	view_menu->setItemChecked(ID_VIEW_MULTILINEINPUT, false);

	slotStatusMsg(i18n("Ready."));
}


void KmudApp::slotViewMapper()
{
	if (doc->getCurrentMudID() == "")
	{
		QMessageBox::information(this, i18n("Error"), i18n("There is no active MUD.  Please use\nthe Connection Wizard to select one."));
		return;
	}

	////////////////////////////////////////////////////////////////////
	//show the auto mapper
	if (!mapper)
	{	
		mapper = new CMapWindow(lastTitle,lastLogin,doc,view);
		mudstream->setMapper(mapper);
		view_menu->setItemChecked(ID_VIEW_MAPPER,true);

		connect( mapper,SIGNAL(slotMapperClosed()),SLOT(slotCloseMapper()) );
		connect( mapper,SIGNAL(movePlayer(QString)),SLOT(slotMovePlayer(QString)));
  	accelerators->connectItem("map_undo", mapper, SLOT(slotUndoAutoCreation()));
	}
	else
	{
		slotCloseMapper();
	}

	slotStatusMsg(i18n("Ready."));
}


void KmudApp::slotMovePlayer(QString dirCmd)
{
	QString moveDir = dirCmd;
	view->slotInputReturnPressed(moveDir);

}


void KmudApp::slotCloseMapper(void)
{
	if (mapper)
	{
  	accelerators->disconnectItem("map_undo", mapper, SLOT(slotUndoAutoCreation()));
		mapper->saveOptions();
		mapper->disconnect();
		mapper->close(true);
		mapper = NULL;
		mudstream->setMapper(NULL);
		view_menu->setItemChecked(ID_VIEW_MAPPER,false);
	}
}


void KmudApp::slotOptionFont()
{
	slotStatusMsg(i18n("Changing font settings..."));

	DlgOptionFonts d(this,"FontSetupDialog");

	readOptions();

	d.setOutputFont(view->getOutputDefaultFont());
	d.setInputFont(view->getInputFont());

	if (d.exec())
	{
		view->setOutputDefaultFont(d.getOutputFont());
		mudFont=d.getOutputFont();
		view->setInputFont(d.getInputFont());
		inputFont=d.getInputFont();

		saveOptions();
	}

	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotOptionColors()
{
	slotStatusMsg(i18n("Changing color settings..."));

	DlgOptionColors d(this,"ColorSetupDialog", interpretType);

	readOptions();
	d.setForegroundColor(view->getOutputDefaultForegroundColor());
	d.setBackgroundColor(view->getOutputDefaultBackgroundColor());
	d.setLocalEchoColor(view->getLocalEchoColor());

	d.setDebugHighlightColor(view->getDebugHighlightColor());

	for (int col=BLACK;col<ANSICOLOR_COUNT;col++)
	{
	  d.setANSIColor(col,view->getOutputANSIColor(col));
	}
	
  for (int col=BG_BLACK;col<BGCOLOR_COUNT;col++)
  {
    d.setBgColor(col,view->getOutputBgColor(col));
  }

	if (d.exec())
	{
		mudForegroundColor = d.getForegroundColor();
		mudBackgroundColor = d.getBackgroundColor();
		mudLocalEchoColor = d.getLocalEchoColor();
		mudDebugHighlightColor = d.getDebugHighlightColor();
		view->setOutputDefaultForegroundColor(mudForegroundColor);
		view->setOutputDefaultBackgroundColor(mudBackgroundColor);
		view->setInputBackgroundColor(mudBackgroundColor);
		view->setLocalEchoColor(mudLocalEchoColor);
		view->setDebugHighlightColor(mudDebugHighlightColor);

		for (int col=BLACK;col<ANSICOLOR_COUNT;col++)
		{
		  mudANSIColor[col]=d.getANSIColor(col);
			view->setOutputANSIColor(col, mudANSIColor[col]);
		}

		for (int col=BG_BLACK;col<BGCOLOR_COUNT;col++)
		{
		  mudBgColor[col]=d.getBgColor(col);
			view->setOutputBgColor(col, mudBgColor[col]);
		}
		
		saveOptions();
	}

	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotOptionMud()
{
	if (doc->getCurrentMudID() == "") {
		QMessageBox::information(this, i18n("Error"), i18n("There is no active MUD.  Please use\nthe Connection Wizard to select one."));
		return;
	}

	slotStatusMsg(i18n("Changing MUD Settings"));
	DlgOptionMud d(doc, this, "Mud Settings");

//	d.setInterpretType(interpretType);
	if (d.exec())
	{
		interpretType = d.getInterpretType();
		view->setInterpretType(interpretType);
	}

	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotOptionPreferences()
{
	slotStatusMsg(i18n("Changing preferences..."));

	DlgOptionPreferences d(this, "Preferences");

	d.setAutoReconnect(autoReconnectOption);
	d.setKeepLastInput(keepLastInput);
	d.setLocalEchoMode(localEchoMode);
	d.setLocalEchoSuppressNewline(localEchoSuppressNewline);
	d.setInputEchoMode(inputEchoMode);
	d.setOutputSize(maxOutputLines);
	d.setInputSize(maxInputLines);
	d.setSharedInputHistory(sharedInputHistory);
	d.setEnableSpecialChars(enableSpecialChars);
	d.setSeparator(separator);
	d.setSpeedwalkingChar (speedwalkingChar);
	d.setExternChar (externChar);
	d.setIntelligentHistory(doc->getIntelligentHistory());
	d.setNumpadMacros(app->getNumpadMacros());
	d.setSelectAllOnFocus(selectAllOnFocus);
	d.setUseBeep(useBeep);
	d.setAutoTime(autoTime);
	d.setOutputNotifyOn(mudOutputNotifyOn);
	d.setOutputNotifySeconds(mudOutputNotifyWaitSeconds);
	d.setConnectTimeoutSeconds(connecttimeout);
  d.setPingData(doc->getPingExecutable(), doc->getPingEnabled());
	
	if (d.exec())
	{
		localEchoMode=d.getLocalEchoMode();
		localEchoSuppressNewline=d.getLocalEchoSuppressNewline();
		inputEchoMode=d.getInputEchoMode();
		maxOutputLines=d.getOutputSize();
		maxInputLines=d.getInputSize();
		sharedInputHistory=d.getSharedInputHistory();
		keepLastInput=d.getKeepLastInput();
		selectAllOnFocus=d.getSelectAllOnFocus();
		autoReconnectOption=d.getAutoReconnect();
		enableSpecialChars=d.getEnableSpecialChars();
		separator=d.getSeparator();
		speedwalkingChar = d.getSpeedwalkingChar ();
		externChar = d.getExternChar ();
		doc->setIntelligentHistory(d.getIntelligentHistory());
		app->setNumpadMacros(d.getNumpadMacros());
		useBeep=d.getUseBeep();
		autoTime=d.getAutoTime();
		mudOutputNotifyOn=d.getOutputNotifyOn();
		mudOutputNotifyWaitSeconds=d.getOutputNotifySeconds();
		connecttimeout=d.getConnectTimeoutSeconds();	

		view->setLocalEchoMode(localEchoMode);
		view->setLocalEchoSupressNewline(localEchoSuppressNewline);
		view->setInputEchoMode(inputEchoMode);
		view->setMaxHistoryLines(maxInputLines);
		view->setMaxMudViewLines(maxOutputLines);
		view->setSharedInputHistory(sharedInputHistory);
		view->setKeepLastInput(keepLastInput);
		view->setSelectAllOnFocus(selectAllOnFocus);
		view->setEnableSpecialChars(enableSpecialChars);
		view->setSeparator(separator);
		view->setSpeedwalkingChar (speedwalkingChar);
		view->setExternChar (externChar);
		view->setUseBeep(useBeep);
		view->setAutoTime(autoTime);
		view->setAutoReconnect(autoReconnectOption);
		
		doc->setPingExecutable(d.getPingPath());
		doc->setPingEnabled(d.getPingEnabled());
		
		mudOutputDoNotify = false;
		mudOutputTimer->start(mudOutputNotifyWaitSeconds*1000,true);
			
		saveOptions();
	}
	
	slotStatusMsg(i18n("Ready."));
}
	
void KmudApp::slotProfileAliases()
{
	if (doc->getCurrentCharacterID() == "") {
		QMessageBox::information(this, i18n("Error"), i18n("There is no active profile.\nPlease connect through Connection Wizard."));
		return;
	}

	slotStatusMsg(i18n("Editing aliases..."));


	DlgProfileAliases d(doc, this, "Aliases");

	if (d.exec())
	{
	}

	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotProfileToobar()
{
	if (doc->getCurrentCharacterID() == "") {
		QMessageBox::information(this, i18n("Error"), i18n("There is no active profile.\nPlease connect through Connection Wizard."));
		return;
	}
	slotStatusMsg(i18n("Editing custom toolbar..."));
	
	DlgCustomToolbar d(doc,this,"Programmable Toolbar");
	
	if (d.exec())
	{
		updateCustomToolbar();
	}
	
	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotProfileTriggers()
{
	if (doc->getCurrentCharacterID() == "") {
		QMessageBox::information(this, i18n("Error"), i18n("There is no active profile.\nPlease connect through Connection Wizard."));
		return;
	}

	slotStatusMsg(i18n("Editing triggers..."));

	DlgProfileTriggers d(doc, this, "Triggers");

	if (d.exec())
	{
	}

	slotStatusMsg(i18n("Ready."));
}

void KmudApp::slotProfileLogging()
{
	if (doc->getCurrentCharacterID() == "") {
		QMessageBox::information(this, i18n("Error"), i18n("There is no active profile.\nPlease connect through Connection Wizard."));
		return;
	}

	slotStatusMsg(i18n("Customizing logging..."));

	DlgProfileLog d(this, "Logging");

	CCharacterProfile* prof;

	prof = doc->getCharacterProfile(doc->getCurrentCharacterID());
	if (prof)
	{
		d.setLoggingEnabled(prof->getLoggingEnabled());
		d.setLogANSIEnabled(prof->getLogANSIEnabled());
		d.setLogFilename(prof->getLogFilename());
	}

	if (d.exec())
	{
 		if (prof)
		{
			prof->setLoggingEnabled(d.getLoggingEnabled());
			prof->setLogANSIEnabled(d.getLogANSIEnabled());
			prof->setLogFilename(d.getLogFilename());
			prof->writeData();
			mudlog->close();
			if (prof->getLoggingEnabled())
			{
				prof->setLoggingEnabled(mudlog->open(prof->getLogFilename()));
			}
		}
		refreshLoggingToolbarIcon();
	}

	slotStatusMsg(i18n("Ready."));
}



void KmudApp::slotProfileSave()
{
	slotStatusMsg(i18n("Saving profile..."));
	slotStatusMsg(i18n("Ready."));
}



void KmudApp::slotPluginManager()
{
	slotStatusMsg(i18n("Running Plugin Manager..."));

	DlgPluginManager d(plugin_manager, this, "PluginManager");
	QStrList checkedList, uncheckedList;
	QString pluginfilename;

	if (d.exec())
	{
		checkedList=d.getCheckedPlugins();
		uncheckedList=d.getUncheckedPlugins();

		// if checked and not running -> run
		pluginfilename=checkedList.first();
		while (pluginfilename!=NULL)
		{
			if (plugin_manager->isPluginRunning(pluginfilename)==false)
			{
				plugin_manager->loadPlugin(pluginfilename);
				plugin_manager->startPlugin(pluginfilename);
			}
			pluginfilename=checkedList.next();
		}


		// if unchecked and not unloaded -> unload
		pluginfilename=uncheckedList.first();
		while (pluginfilename!=NULL)
		{
			if (plugin_manager->isPluginUnloaded(pluginfilename)==false)
			{
				plugin_manager->stopPlugin(pluginfilename);
				plugin_manager->unloadPlugin(pluginfilename);
			}
			pluginfilename=uncheckedList.next();
		}

		autoloadPlugins=checkedList;
	} // if (d.exec())

	slotStatusMsg(i18n("Ready."));
}



void KmudApp::slotStatusMsg(const char *text)
{
	///////////////////////////////////////////////////////////////////
	// change status message permanently
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG );
}


void KmudApp::slotStatusHelpMsg(const char *text)
{
	///////////////////////////////////////////////////////////////////
	// change status message of whole statusbar temporary (text, msec)
	statusBar()->message(text, 2000);
}

	
void KmudApp::slotConnectionTime(void)
{	
	///////////////////////////////////////////////////////////////////
	// Update the staus bar to show the new amount of time a user
	// has been connected to the mud
	char time[10];	

	connectSec++;
	if (connectSec==60)
	{
		connectMin++;
		connectSec=0;
	}
	if (connectMin==60)
	{
		connectHour++;
		connectMin=0;
	}
	
	// construct the connection status string a
	// and add leading zeros to the hour,mins and seconds.	
	if (connectHour<10)
		sprintf(time,"0%d:",connectHour);
	else
		sprintf(time,"%d:",connectHour);
	if (connectMin<10)
		sprintf(time,"%s0%d:",time,connectMin);
	else
		sprintf(time,"%s%d:",time,connectMin);
	if (connectSec<10)
		sprintf(time,"%s0%d",time,connectSec);
	else
		sprintf(time,"%s%d",time,connectSec);

	statusBar()->changeItem(i18n(time), ID_STATUS_TIME );
}


void KmudApp::slotMudOutputReceived()
{
	if (mudOutputDoNotify)
		// notify user by doing some action (blink mini-icon, play wav,...
		mudOutputBlinkTimer->start(1000,false);
	mudOutputDoNotify = false;
	mudOutputTimer->start(mudOutputNotifyWaitSeconds*1000,true);
}

void KmudApp::slotMudOutputTimeout()
{
	mudOutputDoNotify = true;
}

void KmudApp::slotMudOutputBlinker()
{
	if (mudOutputNotifyOn)
	{
		if (mudOutputIconBlinkOn)
			KWM::setMiniIcon(winId(),Icon("kmud_null.xpm"));
		else
			KWM::setMiniIcon(winId(),Icon("mini/kmud.xpm"));

		mudOutputIconBlinkOn=!mudOutputIconBlinkOn;

		if (KWM::isActive(winId()))
		{
			mudOutputBlinkTimer->stop();
			mudOutputIconBlinkOn=false;
			KWM::setMiniIcon(winId(),Icon("mini/kmud.xpm"));
		}
	}
}


void KmudApp::slotToolbarLoggingToggleClicked()
{
	CCharacterProfile* prof;

	prof=doc->getCharacterProfile(doc->getCurrentCharacterID());

	if (prof!=NULL)
	{
		if (prof->getLoggingEnabled())
		{
		  prof->setLoggingEnabled(false);
		}
		else
		{
			prof->setLoggingEnabled(mudlog->open(prof->getLogFilename()));
		}
	}

	refreshLoggingToolbarIcon();
}


void KmudApp::closeEvent(QCloseEvent* ce)
{
	saveOptions(); // save options before we quit
	KTMainWindow::closeEvent(ce);
}

void KmudApp::refreshEnablesDisables()
{
	bool flag=doc->getCurrentCharacterID() != "";
	if (!flag)
	{
	  // no profile set, reset alias expansion and triggers
	  doc->setAliasesDisabled(false);
	  doc->setTriggersDisabled(false);
	}
	profile_menu->setItemEnabled(aliasID, flag);
	profile_menu->setItemEnabled(triggerID, flag);
	profile_menu->setItemEnabled(loggingID, flag);
	profile_menu->setItemEnabled(saveID, flag);
	profile_menu->setItemEnabled(toolbarID, flag);	
	mainToolBar->setItemEnabled(aliasID, flag && !doc->getAliasesDisabled());
	mainToolBar->setItemEnabled(triggerID, flag && !doc->getTriggersDisabled());
	mainToolBar->setItemEnabled(ID_TOOLBAR_LOGGING_TOGGLE, flag);
	mainToolBar->setItemEnabled(ID_VIEW_MAPPER, flag);
	
	flag=doc->getCurrentMudID() != "";
	option_menu->setItemEnabled(mudOptionsID, flag);
}

void KmudApp::refreshLoggingToolbarIcon()
{

	CCharacterProfile* prof;

	prof=doc->getCharacterProfile(doc->getCurrentCharacterID());

	if (prof!=NULL)
	{
		if (prof->getLoggingEnabled())
			mainToolBar->setButton(ID_TOOLBAR_LOGGING_TOGGLE,true);
		else
			mainToolBar->setButton(ID_TOOLBAR_LOGGING_TOGGLE,false);
	}


}


QPopupMenu* KmudApp::getPluginMenu()
{
	return plugin_menu;
}

void KmudApp::slotCustomToolbarClicked(int handle)
{
	CCharacterProfile* profile = doc->getCharacterProfile(doc->getCurrentCharacterID());

	if (profile)
	{
		ToolbarEntry* entry = profile->getToolbarEntryAt(handle);
		
		QString command = entry->command;
		view->slotInputReturnPressed(command);
	}
}

void KmudApp::commandCallback(int id_){
	switch (id_){
		ON_CMD(ID_MUD_NEW_WINDOW,	slotMudNewWindow())
		ON_CMD(ID_MUD_CONNECTION_WIZARD,slotMudConnectionWizard())
		ON_CMD(ID_MUD_CONNECT,		slotMudConnect())
		ON_CMD(ID_MUD_CLOSE,		autoReconnect = false;slotMudClose())
		ON_CMD(ID_MUD_CLOSE_WINDOW,	slotMudCloseWindow())
		ON_CMD(ID_MUD_QUIT,		slotMudQuit())
		ON_CMD(ID_MUD_QUICKCONNECT0,	slotMudQuickConnect(0))
		ON_CMD(ID_MUD_QUICKCONNECT1,	slotMudQuickConnect(1))
		ON_CMD(ID_MUD_QUICKCONNECT2,	slotMudQuickConnect(2))
		ON_CMD(ID_MUD_QUICKCONNECT3,	slotMudQuickConnect(3))
		ON_CMD(ID_MUD_QUICKCONNECT4,	slotMudQuickConnect(4))

		ON_CMD(ID_VIEW_TOOLBAR,		slotViewToolBar())
		ON_CMD(ID_VIEW_CUSTOM_TOOLBAR, slotViewCustomToolBar())		
		ON_CMD(ID_VIEW_STATUSBAR,	slotViewStatusBar())
		ON_CMD(ID_VIEW_ALTINPUT,	slotViewAltInput())
		ON_CMD(ID_VIEW_MULTILINEINPUT,	slotViewMultiLineInput())
		ON_CMD(ID_VIEW_MAPPER,		slotViewMapper())

		ON_CMD(ID_OPTION_FONT,		slotOptionFont())
		ON_CMD(ID_OPTION_COLORS,	slotOptionColors())
		ON_CMD(ID_OPTION_KEYS,    slotOptionKeys())
		ON_CMD(ID_OPTION_PREFERENCES,	slotOptionPreferences())
		ON_CMD(ID_OPTION_MUD,		slotOptionMud())

		ON_CMD(ID_PROFILE_ALIASES,	slotProfileAliases())
		ON_CMD(ID_PROFILE_TRIGGERS,	slotProfileTriggers())
		ON_CMD(ID_PROFILE_LOGGING,	slotProfileLogging())
		ON_CMD(ID_PROFILE_TOOLBAR,	slotProfileToobar())
		ON_CMD(ID_PROFILE_SAVE,		slotProfileSave())

		ON_CMD(ID_PLUGIN_MANAGER,		slotPluginManager())

		ON_CMD(ID_TOOLBAR_LOGGING_TOGGLE,		slotToolbarLoggingToggleClicked())
	
	  	ON_CMD(ID_HELP_BOSS, slotHideMainWindow())
	}
}

void KmudApp::statusCallback(int id_){
	switch (id_){
		ON_STATUS_MSG(ID_MUD_NEW_WINDOW,   i18n("Opens a new application window"))
		ON_STATUS_MSG(ID_MUD_CONNECTION_WIZARD,      i18n("Invokes the Connection Wizard"))
		ON_STATUS_MSG(ID_MUD_CONNECT,      i18n("Connect to a MUD"))
		ON_STATUS_MSG(ID_MUD_CLOSE,        i18n("Closes a connection"))
		ON_STATUS_MSG(ID_MUD_CLOSE_WINDOW, i18n("Closes the current window"))
		ON_STATUS_MSG(ID_MUD_QUIT,         i18n("Exits the program"))
		ON_CMD(ID_MUD_QUICKCONNECT0,			 i18n("Connects to a previous Mud"))
		ON_CMD(ID_MUD_QUICKCONNECT1,			 i18n("Connects to a previous Mud"))
		ON_CMD(ID_MUD_QUICKCONNECT2,			 i18n("Connects to a previous Mud"))
		ON_CMD(ID_MUD_QUICKCONNECT3,			 i18n("Connects to a previous Mud"))
		ON_CMD(ID_MUD_QUICKCONNECT4,			 i18n("Connects to a previous Mud"))

		ON_STATUS_MSG(ID_VIEW_TOOLBAR,      i18n("Enables / disables the main Toolbar"))
		ON_STATUS_MSG(ID_VIEW_CUSTOM_TOOLBAR,      i18n("Enables / disables the programmable Toolbar"))		
		ON_STATUS_MSG(ID_VIEW_STATUSBAR,    i18n("Enables / disables the Statusbar"))
		ON_STATUS_MSG(ID_VIEW_ALTINPUT,     i18n("Enables / disables the alternative Input"))
		ON_STATUS_MSG(ID_VIEW_MULTILINEINPUT,     i18n("Enables / disables the  multiline Input"))
		ON_STATUS_MSG(ID_VIEW_MAPPER,       i18n("Invokes the auto mapper"))

		ON_STATUS_MSG(ID_OPTION_FONT,       i18n("Change font settings"))
		ON_STATUS_MSG(ID_OPTION_COLORS,     i18n("Change color settings"))
		ON_STATUS_MSG(ID_OPTION_KEYS,       i18n("Change accelerator keys"))
		ON_STATUS_MSG(ID_OPTION_PREFERENCES,     i18n("Change preferences"))
		ON_STATUS_MSG(ID_OPTION_MUD,        i18n("Change MUD Settings"))

		ON_STATUS_MSG(ID_PROFILE_ALIASES,     i18n("Edit aliases profile"))
		ON_STATUS_MSG(ID_PROFILE_TRIGGERS,    i18n("Edit triggers profile"))
		ON_STATUS_MSG(ID_PROFILE_TOOLBAR,     i18n("Edit programmable toolbar profile"))
		ON_STATUS_MSG(ID_PROFILE_LOGGING,     i18n("Customize logging"))
		ON_STATUS_MSG(ID_PROFILE_SAVE,        i18n("Save profile"))

		ON_STATUS_MSG(ID_PLUGIN_MANAGER,        i18n("Plugin Manager"))

		ON_STATUS_MSG(ID_TOOLBAR_LOGGING_TOGGLE,    i18n("Toggle logging"))
	}
}

/** refreshes key accelerator bindings */
void KmudApp::refreshAccelerators()
{
  // Boss Key
	accelerators->disconnectItem("boss_key", this, SLOT(slotHideMainWindow()));
	accelerators->connectItem("boss_key", this, SLOT(slotHideMainWindow()));
  accelerators->changeMenuAccel(help_menu, ID_HELP_BOSS, "boss_key");

  // Toggle Alias resolving
	accelerators->disconnectItem("disable_aliases", this, SLOT(slotToggleAliasResolving()));
	accelerators->connectItem("disable_aliases", this, SLOT(slotToggleAliasResolving()));

	// Toggle Triggers
  accelerators->disconnectItem("disable_triggers", this, SLOT(slotToggleTriggers()));
	accelerators->connectItem("disable_triggers", this, SLOT(slotToggleTriggers()));

}

/** initialises key accelerators */
void KmudApp::initAccelerators()
{
	//define defaults, then load current values form config file
	
	// Boss Key, default is ESCAPE
	if (accelerators->defaultKey("boss_key") == 0)
	{
	  accelerators->insertItem(i18n("Boss key"), "boss_key", Key_Escape);
	}
	
	// Toggle alias resolving, default is CTRL+A
	if (accelerators->defaultKey("disable_aliases") == 0)
	{
	  accelerators->insertItem(i18n("Enable/disable Aliases"), "disable_aliases", "CTRL+A");
	}
	
	// Toggle triggers on/off, default is CTRL+T
	if (accelerators->defaultKey("disable_triggers") == 0)
	{
	  accelerators->insertItem(i18n("Enable/disable Triggers"), "disable_triggers", "CTRL+T");
	}
	
	// Undo map change, default is CTRL-BACKSPACE
	if (accelerators->defaultKey("map_undo") == 0)
	{
	  accelerators->insertItem(i18n("Mapper Undo"), "map_undo", "CTRL+Backspace");
	}
	// read values from kmudrc
	accelerators->readSettings(kapp->getConfig());	
	
	// refresh acclerator-menu bindings
	refreshAccelerators();
}

/** opens the key configuration dialog */
void KmudApp::slotOptionKeys()
{
  KKeyDialog::configureKeys(accelerators, false);
  refreshAccelerators();
}

/** overrides KTMainWindow::hide() */
void KmudApp::hide()
{
  // Hide associated top level windows

  if (mapper != NULL)
  {
    mapper->hide();
  }

  // finally use overridden method
  KTMainWindow::hide();
}

/** override KTMainWindow::show() */
void KmudApp::show()
{
  KTMainWindow::show();

  if (mapper != NULL)
  {
    mapper->show();
  }
}

/** toggle the state of alias resolving (on/off) */
void KmudApp::slotToggleAliasResolving()
{
  doc->setAliasesDisabled(!doc->getAliasesDisabled());
  refreshEnablesDisables();
}

/** enable/disable triggers */
void KmudApp::slotToggleTriggers()
{
  doc->setTriggersDisabled(!doc->getTriggersDisabled());
  refreshEnablesDisables();
}
