/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <termios.h>
#include <stdlib.h>

#include <qglobal.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qbuttongroup.h>
#include <qtooltip.h>
#include <qstring.h>
#include <qsize.h>

#include <kapplication.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kseparator.h>

#include "configure.h"
#include "global.h"
#include "modem.h"



Configure::Configure(QWidget *parent, const char *name): KDialogBase (IconList, i18n("Configure"), Help|Default|Ok|Apply|Cancel, Ok, parent, name, true, true)
{
	QGridLayout *grid;
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *label1, *label2, *label3;
	QGroupBox *gb;
	QButtonGroup *bg;
	QFrame *page;
	KSeparator *sep;
	QSize size;
	int i;

	config = kapp->config();

	setHelp("configure-help");
	setIconListAllVisible(true);
	
	/* Page 1: General */
	page = addPage(i18n("General"), i18n("General Settings"), UserIcon("general"));
	
	vbox = new QVBoxLayout(page, 12);

	gb = new QGroupBox(i18n("User Interface"), page);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	gen_tooltips = new QCheckBox(i18n("Show tooltips"), gb);
	gen_tooltips->setMinimumSize(gen_tooltips->sizeHint());
	svbox->addWidget(gen_tooltips);
	svbox->activate();
	
	vbox->addWidget(gb);

	gen_viewmode = 0;

	bg = new QButtonGroup(page);
	bg->setTitle(i18n("Preview"));
	svbox = new QVBoxLayout(bg, 10);
	svbox->addSpacing(8);

	gen_intern_single = new QRadioButton(i18n("Use internal viewer with one window"), bg);
	gen_intern_single->setMinimumSize(gen_intern_single->sizeHint());
	gen_intern_single->setChecked(true);
	svbox->addWidget(gen_intern_single);

	gen_intern_multi = new QRadioButton(i18n("Use internal viewer with multiple windows"), bg);
	gen_intern_multi->setMinimumSize(gen_intern_multi->sizeHint());
	svbox->addWidget(gen_intern_multi);

	gen_external = new QRadioButton(i18n("Use default viewer for TIFF images"), bg);
	gen_external->setMinimumSize(gen_external->sizeHint());
	svbox->addWidget(gen_external);

	gen_custom = new QRadioButton(i18n("Use custom external viewer"), bg);
	gen_custom->setMinimumSize(gen_custom->sizeHint());
	svbox->addWidget(gen_custom);

	shbox = new QHBoxLayout();
	svbox->addLayout(shbox);
	shbox->addSpacing(19);
	
	gen_viewer = new QLineEdit(bg);
	gen_viewer->setMinimumSize(gen_viewer->sizeHint());
	shbox->addWidget(gen_viewer);
	
	svbox->activate();

	vbox->addWidget(bg);
	
	vbox->addStretch(1);
	vbox->activate();

	connect(bg, SIGNAL(clicked(int)), SLOT(previewSelected(int)));

	/* Page 2: Modem */
	page = addPage(i18n("Modem"), i18n("Settings for the Modem"), UserIcon("modem"));
	
	vbox = new QVBoxLayout(page, 12);

	bg = new QButtonGroup(page);
	bg->setTitle(i18n("Modem Device"));
	svbox = new QVBoxLayout(bg, 10);
	svbox->addSpacing(8);
	
	mod_serial = new QRadioButton(i18n("Serial Port"), bg);
	mod_serial->setChecked(true);
	svbox->addWidget(mod_serial);

	shbox = new QHBoxLayout();
	svbox->addLayout(shbox);
	shbox->addSpacing(19);
	
	label1 = new QLabel(i18n("Device:"), bg);
	shbox->addWidget(label1);
	
	mod_ser_device = new QComboBox(false, bg);
	for (i=0; i < MODEM_SER_DEVNUM; i++)
		mod_ser_device->insertItem(modem_ser_devices[i], i);
	shbox->addWidget(mod_ser_device);
	shbox->addStretch();
	
	shbox = new QHBoxLayout();
	svbox->addLayout(shbox);
	shbox->addSpacing(19);
	
	label2 = new QLabel(i18n("Speed:"), bg);
	shbox->addWidget(label2);

	mod_ser_speed = new QComboBox(false, bg);
	mod_ser_speed->insertItem("19200 Bps", 0);
	mod_ser_speed->insertItem("38400 Bps", 1);
#ifdef B57600
	mod_ser_speed->insertItem("57600 Bps", 2);
#endif
#ifdef B115200
	mod_ser_speed->insertItem("115200 Bps", 3);
#endif
#ifdef B230400
	mod_ser_speed->insertItem("230400 Bps", 4);
#endif
	shbox->addWidget(mod_ser_speed);
	shbox->addStretch();

	mod_usb = new QRadioButton(i18n("USB Port"), bg);
	svbox->addWidget(mod_usb);

	shbox = new QHBoxLayout();
	svbox->addLayout(shbox);
	shbox->addSpacing(19);
	
	label3 = new QLabel(i18n("Device:"), bg);
	shbox->addWidget(label3);
	
	mod_usb_device = new QComboBox(false, bg);
	for (i=0; i < MODEM_USB_DEVNUM; i++)
		mod_usb_device->insertItem(modem_usb_devices[i], i);
	shbox->addWidget(mod_usb_device);
	shbox->addStretch();

	size = label1->sizeHint().expandedTo(label2->sizeHint()).expandedTo(label3->sizeHint());
	label1->setFixedSize(size);
	label2->setFixedSize(size);
	label3->setFixedSize(size);
	
	size = mod_ser_device->sizeHint().expandedTo(mod_ser_speed->sizeHint()).expandedTo(mod_usb_device->sizeHint());
	mod_ser_device->setFixedSize(size);
	mod_ser_speed->setFixedSize(size);
	mod_usb_device->setFixedSize(size);

	svbox->activate();

	vbox->addWidget(bg);
	
	gb = new QGroupBox(i18n("Download"), page);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	mod_loadnew = new QCheckBox(i18n("Download new facsimiles only"), gb);
	mod_loadnew->setMinimumSize(mod_loadnew->sizeHint());
	svbox->addWidget(mod_loadnew);
	mod_deletefax = new QCheckBox(i18n("Delete facsimiles from modem"), gb);
	mod_deletefax->setMinimumSize(mod_deletefax->sizeHint());
	svbox->addWidget(mod_deletefax);
	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();

	connect(bg, SIGNAL(clicked(int)), SLOT(deviceSelected(int)));

	/* Page 3: Print */
	page = addPage(i18n("Print"), i18n("Settings for Printing"), UserIcon("print"));
	
	vbox = new QVBoxLayout(page, 12);

	gb = new QGroupBox(i18n("Paper Margins"), page);
	grid = new QGridLayout(gb, 3, 5, 10);

	grid->addRowSpacing(0, 8);
	label1 = new QLabel(i18n("Left (mm):"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 1, 0);
	prt_lmargin = new QSpinBox(gb);
	prt_lmargin->setMinimumSize(prt_lmargin->sizeHint());
	grid->addWidget(prt_lmargin, 1, 1);

	label1 = new QLabel(i18n("Right (mm):"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 2, 0);
	prt_rmargin = new QSpinBox(gb);
	prt_rmargin->setMinimumSize(prt_rmargin->sizeHint());
	grid->addWidget(prt_rmargin, 2, 1);

	label1 = new QLabel(i18n("Top (mm):"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 1, 3);
	prt_tmargin = new QSpinBox(gb);
	prt_tmargin->setMinimumSize(prt_tmargin->sizeHint());
	grid->addWidget(prt_tmargin, 1, 4);

	label1 = new QLabel(i18n("Bottom (mm):"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 2, 3);
	prt_bmargin = new QSpinBox(gb);
	prt_bmargin->setMinimumSize(prt_bmargin->sizeHint());
	grid->addWidget(prt_bmargin, 2, 4);

	sep = new KSeparator(KSeparator::VLine, gb);
	grid->addMultiCellWidget(sep, 1, 2, 2, 2);
	
	grid->setColStretch(1, 1);
	grid->setColStretch(4, 1);

	grid->activate();

	vbox->addWidget(gb);

	vbox->addStretch();

	vbox->activate();

	/* Page 4: Mail */
	page = addPage(i18n("Mail"), i18n("Settings for Sending Mails"), UserIcon("mail"));
	
	vbox = new QVBoxLayout(page, 12);

	gb = new QGroupBox(i18n("SMTP Server"), page);

	grid = new QGridLayout(gb, 3, 3, 12);

	grid->addRowSpacing(0, fontMetrics().lineSpacing());

	label1 = new QLabel(i18n("Name:"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 1, 0);

	mail_server = new QLineEdit(gb);
	mail_server->setMinimumSize(mail_server->sizeHint());
	grid->addMultiCellWidget(mail_server, 1, 1, 1, 2);

	label1 = new QLabel(i18n("Port:"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 2, 0);

	mail_port = new QLineEdit(gb);
	mail_port->setFixedWidth(50);
	mail_port->setMinimumHeight(mail_port->sizeHint().height());
	grid->addWidget(mail_port, 2, 1);

	grid->setColStretch(2, 1);

	grid->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Identity"), page);

	grid = new QGridLayout(gb, 5, 2, 12);

	grid->addRowSpacing(0, fontMetrics().lineSpacing());

	label1 = new QLabel(i18n("Name:"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 1, 0);

	mail_name = new QLineEdit(gb);
	mail_name->setMinimumSize(mail_name->sizeHint());
	grid->addWidget(mail_name, 1, 1);

	label1 = new QLabel(i18n("Organization:"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 2, 0);

	mail_org = new QLineEdit(gb);
	mail_org->setMinimumSize(mail_org->sizeHint());
	grid->addWidget(mail_org, 2, 1);

	label1 = new QLabel(i18n("Email address:"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 3, 0);

	mail_email = new QLineEdit(gb);
	mail_email->setMinimumSize(mail_email->sizeHint());
	grid->addWidget(mail_email, 3, 1);

	label1 = new QLabel(i18n("Reply-to address:"), gb);
	label1->setMinimumSize(label1->sizeHint());
	grid->addWidget(label1, 4, 0);

	mail_reply = new QLineEdit(gb);
	mail_reply->setMinimumSize(mail_reply->sizeHint());
	grid->addWidget(mail_reply, 4, 1);

	grid->setColStretch(1, 1);

	grid->activate();

	vbox->addWidget(gb);
	vbox->addStretch(1);

	vbox->activate();

	config->setGroup(ID_PREF_GROUP_GENERAL);
	if (config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS)) {
		QToolTip::add(gen_tooltips, i18n("Show tooltips in dialogs"));
		QToolTip::add(gen_intern_single, i18n("Use build-in preview dialog with one window"));
		QToolTip::add(gen_intern_multi, i18n("Use build-in preview dialog with multiple windows"));
		QToolTip::add(gen_external, i18n("Use default external viewer for TIFF images"));
		QToolTip::add(gen_custom, i18n("Use a user specified viewer for preview"));
		QToolTip::add(gen_viewer, i18n("Command to invoke for preview"));
		QToolTip::add(mod_serial, i18n("Use serial device with the modem"));
		QToolTip::add(mod_ser_device, i18n("Serial device of the modem"));
		QToolTip::add(mod_ser_speed, i18n("Speed for serial communication"));
		QToolTip::add(mod_usb, i18n("Use USB device with the modem"));
		QToolTip::add(mod_usb_device, i18n("USB device of the modem"));
		QToolTip::add(mod_loadnew, i18n("Download only new facsimiles from modem"));
		QToolTip::add(mod_deletefax, i18n("Delete facsimiles from modem after download"));
		QToolTip::add(prt_lmargin, i18n("Left margin on paper"));
		QToolTip::add(prt_rmargin, i18n("Right margin on paper"));
		QToolTip::add(prt_tmargin, i18n("Top margin on paper"));
		QToolTip::add(prt_bmargin, i18n("Bottom margin on paper"));
		QToolTip::add(mail_server, i18n("Name of the SMTP server"));
		QToolTip::add(mail_port, i18n("Port of the SMTP server"));
		QToolTip::add(mail_name, i18n("Your full name"));
		QToolTip::add(mail_org, i18n("Your organization"));
		QToolTip::add(mail_email, i18n("Your email address"));
		QToolTip::add(mail_reply, i18n("The address replies are sent to"));
	}

	reset();
}


Configure::~Configure()
{
}


void Configure::slotDefault()
{
	QString s;
	KConfig mailconf("emaildefaults", true);

	gen_tooltips->setChecked(PREF_GEN_TOOLTIPS);
	previewSelected(PREF_GEN_VIEWMODE);
#if PREF_GEN_VIEWMODE == 0
	gen_intern_single->setChecked(true);
#endif
#if PREF_GEN_VIEWMODE == 1
	gen_intern_multi->setChecked(true);
#endif
#if PREF_GEN_VIEWMODE == 2
	gen_external->setChecked(true);
#endif
#if PREF_GEN_VIEWMODE == 3
	gen_custom->setChecked(true);
#endif
	gen_viewer->setText(PREF_GEN_VIEWER);

#if PREF_MOD_USB == 0
	deviceSelected(0);
	mod_serial->setChecked(true);
#endif
	mod_ser_device->setCurrentItem(PREF_MOD_SER_DEVICE);
	mod_ser_speed->setCurrentItem(PREF_MOD_SER_SPEED);
#if PREF_MOD_USB == 1
	deviceSelected(1);
	mod_usb->setChecked(true);
#endif
	mod_usb_device->setCurrentItem(PREF_MOD_USB_DEVICE);
	mod_loadnew->setChecked(PREF_MOD_LOADNEW);
	mod_deletefax->setChecked(PREF_MOD_DELETEFAX);

	prt_lmargin->setValue(PREF_PRT_LMARGIN);
	prt_rmargin->setValue(PREF_PRT_RMARGIN);
	prt_tmargin->setValue(PREF_PRT_TMARGIN);
	prt_bmargin->setValue(PREF_PRT_BMARGIN);

	mailconf.setGroup("Defaults");
	mailconf.setGroup(QString("PROFILE_") + mailconf.readEntry("Profile"));
	if (mailconf.readEntry("OutgoingServerType") == "smtp")
		mail_server->setText(mailconf.readEntry("OutgoingServer", PREF_MAIL_SERVER));
	else
		mail_server->setText(PREF_MAIL_SERVER);
	s.setNum(PREF_MAIL_PORT);
	mail_port->setText(s);
	mail_name->setText(mailconf.readEntry("FullName", PREF_MAIL_NAME));
	mail_org->setText(mailconf.readEntry("Organization", PREF_MAIL_ORG));
	mail_email->setText(mailconf.readEntry("EmailAddress", PREF_MAIL_EMAIL));
	mail_reply->setText(mailconf.readEntry("ReplyAddr", PREF_MAIL_REPLY));
}


void Configure::slotOk()
{
	slotApply();
	accept();
}


void Configure::slotApply()
{
	config->setGroup(ID_PREF_GROUP_GENERAL);
	config->writeEntry(ID_PREF_GEN_TOOLTIPS, gen_tooltips->isChecked());
	config->writeEntry(ID_PREF_GEN_VIEWMODE, gen_viewmode);
	config->writeEntry(ID_PREF_GEN_VIEWER, gen_viewer->text());
	
	config->setGroup(ID_PREF_GROUP_MODEM);
	config->writeEntry(ID_PREF_MOD_USB, mod_usb->isChecked());
	config->writeEntry(ID_PREF_MOD_SER_DEVICE, mod_ser_device->currentItem());
	config->writeEntry(ID_PREF_MOD_SER_SPEED, mod_ser_speed->currentItem());
	config->writeEntry(ID_PREF_MOD_USB_DEVICE, mod_usb_device->currentItem());
	config->writeEntry(ID_PREF_MOD_LOADNEW, mod_loadnew->isChecked());
	config->writeEntry(ID_PREF_MOD_DELETEFAX, mod_deletefax->isChecked());

	config->setGroup(ID_PREF_GROUP_PRINT);
	config->writeEntry(ID_PREF_PRT_LMARGIN, prt_lmargin->value());
	config->writeEntry(ID_PREF_PRT_RMARGIN, prt_rmargin->value());
	config->writeEntry(ID_PREF_PRT_TMARGIN, prt_tmargin->value());
	config->writeEntry(ID_PREF_PRT_BMARGIN, prt_bmargin->value());

	config->setGroup(ID_PREF_GROUP_MAIL);
	config->writeEntry(ID_PREF_MAIL_SERVER, mail_server->text());
	config->writeEntry(ID_PREF_MAIL_PORT, mail_port->text().toInt());
	config->writeEntry(ID_PREF_MAIL_NAME, mail_name->text());
	config->writeEntry(ID_PREF_MAIL_ORG, mail_org->text());
	config->writeEntry(ID_PREF_MAIL_EMAIL, mail_email->text());
	config->writeEntry(ID_PREF_MAIL_REPLY, mail_reply->text());

	config->sync();
}


void Configure::reset()
{
	QString s;
	KConfig mailconf("emaildefaults", true);

	config->setGroup(ID_PREF_GROUP_GENERAL);
	gen_tooltips->setChecked(config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS));
	
	previewSelected(config->readNumEntry(ID_PREF_GEN_VIEWMODE, PREF_GEN_VIEWMODE));
	switch (gen_viewmode) {
		case 0:
			gen_intern_single->setChecked(true);
			break;
		case 1:
			gen_intern_multi->setChecked(true);
			break;
		case 2:
			gen_external->setChecked(true);
			break;
		case 3:
			gen_custom->setChecked(true);
			break;
		default:
			break;
	}
	
	gen_viewer->setText(config->readEntry(ID_PREF_GEN_VIEWER, PREF_GEN_VIEWER));
	
	config->setGroup(ID_PREF_GROUP_MODEM);
	if (config->readBoolEntry(ID_PREF_MOD_USB, PREF_MOD_USB)) {
		deviceSelected(1);
		mod_usb->setChecked(true);
	} else {
		deviceSelected(0);
		mod_serial->setChecked(true);
	}
	mod_ser_device->setCurrentItem(config->readNumEntry(ID_PREF_MOD_SER_DEVICE, PREF_MOD_SER_DEVICE));
	mod_ser_speed->setCurrentItem(config->readNumEntry(ID_PREF_MOD_SER_SPEED, PREF_MOD_SER_SPEED));
	mod_usb_device->setCurrentItem(config->readNumEntry(ID_PREF_MOD_USB_DEVICE, PREF_MOD_USB_DEVICE));
	mod_loadnew->setChecked(config->readBoolEntry(ID_PREF_MOD_LOADNEW, PREF_MOD_LOADNEW));
	mod_deletefax->setChecked(config->readBoolEntry(ID_PREF_MOD_DELETEFAX, PREF_MOD_DELETEFAX));

	config->setGroup(ID_PREF_GROUP_PRINT);
	prt_lmargin->setValue(config->readNumEntry(ID_PREF_PRT_LMARGIN, PREF_PRT_LMARGIN));
	prt_rmargin->setValue(config->readNumEntry(ID_PREF_PRT_RMARGIN, PREF_PRT_RMARGIN));
	prt_tmargin->setValue(config->readNumEntry(ID_PREF_PRT_TMARGIN, PREF_PRT_TMARGIN));
	prt_bmargin->setValue(config->readNumEntry(ID_PREF_PRT_BMARGIN, PREF_PRT_BMARGIN));

	config->setGroup(ID_PREF_GROUP_MAIL);

	mailconf.setGroup("Defaults");
	mailconf.setGroup(QString("PROFILE_") + mailconf.readEntry("Profile"));
	if (mailconf.readEntry("OutgoingServerType") == "smtp")
		mail_server->setText(config->readEntry(ID_PREF_MAIL_SERVER, mailconf.readEntry("OutgoingServer", PREF_MAIL_SERVER)));
	else
		mail_server->setText(config->readEntry(ID_PREF_MAIL_SERVER));
	s.setNum(config->readNumEntry(ID_PREF_MAIL_PORT, PREF_MAIL_PORT));
	mail_port->setText(s);
	mail_name->setText(config->readEntry(ID_PREF_MAIL_NAME, mailconf.readEntry("FullName", PREF_MAIL_NAME)));
	mail_org->setText(config->readEntry(ID_PREF_MAIL_ORG, mailconf.readEntry("Organization", PREF_MAIL_ORG)));
	mail_email->setText(config->readEntry(ID_PREF_MAIL_EMAIL, mailconf.readEntry("EmailAddress", PREF_MAIL_EMAIL)));
	mail_reply->setText(config->readEntry(ID_PREF_MAIL_REPLY, mailconf.readEntry("ReplyAddr", PREF_MAIL_REPLY)));
}


void Configure::previewSelected(int id_)
{
	gen_viewmode = id_;
	gen_viewer->setEnabled(gen_viewmode == 3);
}


void Configure::deviceSelected(int id_)
{
	mod_ser_device->setEnabled(id_ == 0);
	mod_ser_speed->setEnabled(id_ == 0);
	mod_usb_device->setEnabled(id_ == 1);
}
