/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <qfile.h>
#include <qglobal.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qprogressdialog.h>

#include <kapplication.h>
#include <ksimpleconfig.h>
#include <kmessagebox.h>
#include <klocale.h>

#include "faxfiler.h"
#include "global.h"
#include "mlofile.h"
#include "filters.h"


FaxFiler::FaxFiler(QWidget *parent, const char *name): QSplitter(parent, name)
{
	config = kapp->config();
	current_folder = 0;

	folderlist = new FolderView(this);
	faxlist = new FaxView(this);

	connect(folderlist, SIGNAL(hasCurrent(FolderViewItem *)), SLOT(folderHasCurrentSlot(FolderViewItem *)));
	connect(folderlist, SIGNAL(move(FolderViewItem *)), SLOT(moveCallback(FolderViewItem *)));
	connect(folderlist, SIGNAL(copy(FolderViewItem *)), SLOT(copyCallback(FolderViewItem *)));
	connect(folderlist, SIGNAL(empty()), SLOT(emptyFolder()));
	
	connect(faxlist, SIGNAL(hasCurrent(bool)), SLOT(faxHasCurrentSlot(bool)));
	connect(faxlist, SIGNAL(hasSelection(bool)), SLOT(faxHasSelectionSlot(bool)));
	connect(faxlist, SIGNAL(command(int)), SLOT(commandSlot(int)));
	
	readOptions();
}


FaxFiler::~FaxFiler()
{
}


void FaxFiler::setInboxFolder()
{
	folderlist->setCurrentItem((QListViewItem *)folderlist->inboxFolder());
}


void FaxFiler::readOptions()
{
	folderlist->readOptions();
	faxlist->readOptions();

	config->setGroup("FilerWindow Options");

	if (config->hasKey("Sizes"))
		setSizes(config->readIntListEntry("Sizes"));
	else {
		QValueList<int> def_sizes;
		def_sizes.append(154);
		def_sizes.append(436);
		setSizes(def_sizes);
	}
}


void FaxFiler::saveOptions()
{
	folderlist->saveOptions();
	faxlist->saveOptions();
	
	config->setGroup("FilerWindow Options");

	config->writeEntry("Sizes", sizes());
}


void FaxFiler::saveCurrentFolder()
{
	KSimpleConfig *sconfig;
	QString s;
	FaxViewItem *item;
	int i;

	if (!current_folder)
		return;

	QFile::remove(expandPath(currentPath() + "/" + KMLOFAX_FAXDIR));
	sconfig = new KSimpleConfig(expandPath(currentPath() + "/" + KMLOFAX_FAXDIR), false);
	Q_CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	sconfig->writeEntry("Entries", faxlist->childCount());
	item = (FaxViewItem *)faxlist->firstChild();
	for (i = 0; (item); i++) {
		s.sprintf("Entry %d", i);
		sconfig->setGroup(s);
		sconfig->writeEntry("FileName", item->name());
		sconfig->writeEntry("New", item->isNew());
		sconfig->writeEntry("Time", item->time());
		sconfig->writeEntry("Sender", item->sender());
		sconfig->writeEntry("Pages", item->pages());
		sconfig->writeEntry("Size", item->size());
		item = (FaxViewItem *)item->nextSibling();
	}
	sconfig->sync();
	delete sconfig;
}


void FaxFiler::takeFaxItem(FolderViewItem *folder, const QString& name, bool isnew)
{
	TiffFile tiffile;
	KSimpleConfig *sconfig;
	int i, number;
	QString group;

	sconfig = new KSimpleConfig(expandPath(folder->path() + "/" + KMLOFAX_FAXDIR));
	Q_CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries", 0);
	for (i = 0; i < number; i++) {
		group = QString("Entry %1").arg(i);
		if (sconfig->hasGroup(group)) {
			sconfig->setGroup(group);
			if (sconfig->readEntry("FileName", "") == name)
				break;
		}
	}
	if (i == number) {
		number++;
		sconfig->setGroup("General");
		sconfig->writeEntry("Entries", number);
		sconfig->setGroup(QString("Entry %1").arg(number-1));
		folder->setFiles(number);
	}
	sconfig->writeEntry("New", isnew);
	sconfig->writeEntry("FileName", name);
	tiffile.setName(expandPath(folder->path() + "/" + name));
	if (tiffile.open(IO_ReadOnly)) {
		sconfig->writeEntry("Time", tiffile.time());
		sconfig->writeEntry("Sender", tiffile.sender());
		sconfig->writeEntry("Pages", tiffile.pages());
		sconfig->writeEntry("Size", tiffile.size());
		if (current_folder == folder)
			faxlist->takeFaxItem(name, tiffile.time(), tiffile.sender(), tiffile.pages(), tiffile.size(), isnew);
		tiffile.close();
	}
	delete sconfig;
}


void FaxFiler::newFaxItem(const QString& name)
{
	takeFaxItem(folderlist->inboxFolder(), name, true);
}


void FaxFiler::moveFax(const QString& name, FolderViewItem *to, bool copy)
{
	bool is_new = faxlist->faxNew(name);

	if (copy) {
		QFile source(expandPath(currentPath() + "/" + name));
		QFile target(expandPath(to->path() + "/" + name));
		char buffer[4096];
		int result;
		if (!source.open(IO_ReadOnly))
			return;
		if (!target.open(IO_WriteOnly)) {
			source.close();
			return;
		}
		while ((result = source.readBlock(buffer, 4096)) > 0)
			target.writeBlock(buffer, result);
		target.close();
		source.close();
	}
	else {
		QDir d;
		d.remove(expandPath(to->path() + "/" + name));
		if (!d.rename(expandPath(currentPath() + "/" + name), expandPath(to->path() + "/" + name)))
			return;
	}

	takeFaxItem(to, name, is_new);
}


void FaxFiler::moveFax(FolderViewItem *to, bool copy, bool all)
{
	FaxViewItem *item;
	bool remove = false;

	if ((!current_folder) || (!to) || (current_folder == to))
		return;

	item = (FaxViewItem *)faxlist->firstChild();
	while (item) {
		if (item->isSelected() || all) {
			moveFax(item->name(), to, copy);
			remove = !copy;
		}
		if (remove) {
			FaxViewItem *item2 = item;
			item = (FaxViewItem *)item->nextSibling();
			delete item2;
			remove = false;
		}
		else
			item = (FaxViewItem *)item->nextSibling();
	}
	
	if (!copy)
		saveCurrentFolder();

	folderlist->setFiles(current_folder);
}


void FaxFiler::setFolder(FolderViewItem *folder)
{
	KSimpleConfig *sconfig;
	int i, number = 0;
	QString group;

	current_folder = folder;

	faxlist->clear();
	emit hasCurrentFax(false);

	if (current_folder->type() == FolderViewItem::Root)
		return;

	sconfig = new KSimpleConfig(expandPath(current_folder->path() + "/" + KMLOFAX_FAXDIR), true);
	Q_CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");
	for (i = 0; i < number; i++) {
		group = QString("Entry %1").arg(i);
		if (sconfig->hasGroup(group)) {
			sconfig->setGroup(group);
			faxlist->insertFaxItem(sconfig->readEntry("FileName"), sconfig->readDateTimeEntry("Time"), sconfig->readEntry("Sender"), sconfig->readNumEntry("Pages"), sconfig->readNumEntry("Size"), sconfig->readBoolEntry("New"));
		}
	}
	delete sconfig;

	emit hasCurrentFax(faxlist->currentItem() != 0);
}


QString FaxFiler::currentPath()
{
	if (current_folder)
		return current_folder->path();

	return QString::null;
}


QString FaxFiler::currentFaxName()
{
	return faxlist->currentFaxName();
}


void FaxFiler::setCurrentFaxNew(bool isnew)
{
	faxlist->setCurrentFaxNew(isnew);
	setFaxNew(faxlist->currentFaxName(), isnew);
}


bool FaxFiler::currentFaxNew()
{
	return faxlist->currentFaxNew();
}


QString FaxFiler::currentFaxSender()
{
	return faxlist->currentFaxSender();
}


void FaxFiler::deleteFax(bool all)
{
//	if (current_folder == folderlist->trashFolder()) {
	if (current_folder->type() == FolderViewItem::Trash) {
		FaxViewItem *item;
		bool new_fax = false;

		item = (FaxViewItem *)faxlist->firstChild();
		while (item) {
			if (item->isSelected() || all) {
				if (item->isNew()) {
					new_fax = true;
					item = (FaxViewItem *)item->nextSibling();
				}
				else {
					FaxViewItem *item_buff;
					QDir d;
					d.remove(expandPath(current_folder->path() + "/" + item->name()));
					item_buff = item;
					item = (FaxViewItem *)item->nextSibling();
					delete item_buff;
				}
			}
			else
				item = (FaxViewItem *)item->nextSibling();
		}

		saveCurrentFolder();
		folderlist->setFiles(current_folder);

		if (new_fax)
			KMessageBox::sorry(this, i18n("One or more facsimiles are marked as new\nand have not been deleted.\nPlease view or print them first."), i18n("Warning"));
	}
	else
		moveFax(folderlist->trashFolder(), false, all);
}


void FaxFiler::selectAllFax()
{
	faxlist->selectAll(true);
}

	
void FaxFiler::setFaxNew(const QString& name, bool isnew)
{
	KSimpleConfig *sconfig;
	int i, number = 0;
	QString group;

	if (name.isEmpty())
		return;

	sconfig = new KSimpleConfig(expandPath(current_folder->path() + "/" + KMLOFAX_FAXDIR), false);
	Q_CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");
	for (i = 0; i < number; i++) {
		group = QString("Entry %1").arg(i);
		if (sconfig->hasGroup(group)) {
			sconfig->setGroup(group);
			if (sconfig->readEntry("FileName") == name) {
				sconfig->writeEntry("New", isnew);
				break;
			}
		}
	}
	delete sconfig;
}


int FaxFiler::rebuildFolder()
{
	int res;
	
	res = rebuildFolder(current_folder);
	
//	setFolder(folderlist->name());

	return res;
}


void FaxFiler::updateFromPre21()
{
	KSimpleConfig *sconfig;
	int i, number = 0;
	QString group, fname;
	MLO2TiffFilter filter;
	
	config->setGroup("General");
	if (config->readBoolEntry("Update21Done", false))
		return;
	
	sconfig = new KSimpleConfig(expandPath(KMLOFAX_FAXDIR), true);
	Q_CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");

	if (!number) {
		config->writeEntry("Update21Done", true);
		return;
	}

	i = KMessageBox::warningYesNoCancel(this,
		i18n("There are %1 old facsimile files.\n" \
		     "Should they be converted into the new file\n" \
		     "format to be used with KMLOFax %2?").arg(number).arg(VERSION),
		i18n("Question"));

	if (i == KMessageBox::Cancel)
		return;

	config->writeEntry("Update21Done", true);

	if (i == KMessageBox::No)
		return;

	QProgressDialog progress(i18n("Converting files..."), i18n("Abort"), number, this, "progress", true);

	for (i = 0; i < number; i++) {
		progress.setProgress(i);
		kapp->processEvents();
		if (progress.wasCancelled())
			break;
		group = QString("Entry %1").arg(i);
		if (sconfig->hasGroup(group)) {
			sconfig->setGroup(group);
			fname = sconfig->readEntry("FileName");
			filter.setFile(expandPath(fname));
			filter.setTime(mloDateSplit(fname));
			if (fname.right(4) == ".FAX")
				fname.replace(fname.length()-3, 3, "tif");
			if (filter.convertFile(expandPath(QString(INBOX_DIR) + "/" + fname)))
				takeFaxItem(folderlist->inboxFolder(), fname, sconfig->readBoolEntry("New"));
		}
	}

	progress.setProgress(number);

	delete sconfig;
}


void FaxFiler::newFolder()
{
	folderlist->createFolder();
}


void FaxFiler::renameFolder()
{
	folderlist->renameFolder();
}


void FaxFiler::removeFolder()
{
	folderlist->removeFolder();
}


int FaxFiler::rebuildFolder(FolderViewItem *folder)
{
	TiffFile tifffile;
	int number = 0;
	KSimpleConfig *sconfig;

	QFile::remove(expandPath(folder->path() + "/" + KMLOFAX_FAXDIR));
	
	sconfig = new KSimpleConfig(expandPath(folder->path() + "/" + KMLOFAX_FAXDIR), false);
	Q_CHECK_PTR(sconfig);
	
	QDir d(expandPath(folder->path()));
	const QFileInfoList *files;
	
	d.setNameFilter("*.tif");
	files = d.entryInfoList();

	if (files) {
		kapp->setOverrideCursor(waitCursor);
		QFileInfoListIterator it(*files);
		QFileInfo *f;
		while ((f=it.current()) != 0) {
			++it;
			tifffile.setName(expandPath(folder->path() + "/" + f->fileName()));
			if (!(tifffile.open(IO_ReadOnly)))
				continue;
			sconfig->setGroup(QString("Entry %1").arg(number++));
			sconfig->writeEntry("FileName", f->fileName());
			sconfig->writeEntry("Time", tifffile.time());
			sconfig->writeEntry("New", false);
			sconfig->writeEntry("Sender", tifffile.sender());
			sconfig->writeEntry("Pages", tifffile.pages());
			sconfig->writeEntry("Size", tifffile.size());
			tifffile.close();
		}
		kapp->restoreOverrideCursor();
	}

	sconfig->setGroup("General");
	sconfig->writeEntry("Entries", number);
	delete sconfig;

	folder->setFiles(number);

	return number;
}


void FaxFiler::moveCallback(FolderViewItem *to)
{
	moveFax(to, false, false);
}


void FaxFiler::copyCallback(FolderViewItem *to)
{
	moveFax(to, true, false);
}


void FaxFiler::emptyFolder()
{
	if (!current_folder)
		return;

	if (KMessageBox::warningYesNo(this, i18n("Do you really want to empty the folder \"%1\"?").arg(current_folder->name()), i18n("Question"), i18n("&Empty"), i18n("&Cancel")) == KMessageBox::Yes)
		deleteFax(true);
}


void FaxFiler::folderHasCurrentSlot(FolderViewItem *folder)
{
	setFolder(folder);
	emit hasCurrentFolder(folder->type() == FolderViewItem::Root, folder->type() == FolderViewItem::User);
}


void FaxFiler::faxHasCurrentSlot(bool yesno)
{
	emit hasCurrentFax(yesno);
}


void FaxFiler::faxHasSelectionSlot(bool yesno)
{
	moveMenu->setEnabled(yesno);
	copyMenu->setEnabled(yesno);
}


void FaxFiler::commandSlot(int id_)
{
	emit command(id_);
}
