/***************************************************************************
                          utils.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <dlfcn.h>
#include <qlineedit.h>
#include <qmultilinedit.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qtimer.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kio/netaccess.h>
#include <kglobal.h>
#include <kurl.h>
#include "kpldoc.h"
#include "utils.h"
#include "sourcewidget.h"

ChooseFuncDlg::ChooseFuncDlg(QWidget* _parent, QFile* f, QLineEdit* func0) :
 KDialogBase(Plain, QString::null, Ok | Cancel, Ok, _parent), func(func0)
{
  setPlainCaption(i18n("Functions"));
  QFrame* frame = plainPage();
  QVBoxLayout* vbox = new QVBoxLayout(frame, 0, spacingHint());
  vbox->addWidget(funcList = new QListBox(frame));
  QTextStream t(f);
  while (!t.eof()) {
    QString line = t.readLine();
    if (line[9] == 'T')
      funcList->insertItem(line.mid(11));
  }
  f->close();
  connect(funcList, SIGNAL(selected(int)), SLOT(slotList(int)));
  resize(200, 200);
}

ChooseFuncDlg::~ChooseFuncDlg()
{
}

void ChooseFuncDlg::slotList(int i)
{
  func->setText(funcList->text(i));
  accept();
}

void ChooseFuncDlg::slotOk()
{
  int i = funcList->currentItem();
  if (i != -1) {
    func->setText(funcList->text(i));
    accept();
  }
}

EditParameterDlg::EditParameterDlg(QWidget* _parent, KplDoc* m, double* p0,
                                   const QString& file, const QString& func,
                                   bool applyButton) :
 KDialogBase(Plain, i18n("Parameters"),
             applyButton ? Ok | Cancel | Apply : Ok | Cancel,
             Ok, _parent, 0, true, true), p(p0)
{
  QHBoxLayout* hbox = 0;
  QVBoxLayout* vbox;
  QFrame* frame = plainPage();
  if (m->options()->showSource) {
    hbox = new QHBoxLayout(frame, 0, spacingHint());
    vbox = new QVBoxLayout(hbox);
  } else
    vbox = new QVBoxLayout(frame, 0, spacingHint());
  QGroupBox* g = new QGroupBox(0, Qt::Vertical, frame);
  QGridLayout* grid = new QGridLayout(g->layout(), QMIN(KPL_NPMAX, 10),
                                      2 + 3 * ((KPL_NPMAX - 1) / 10),
                                      spacingHint());
  vbox->addWidget(g);
  QString s;
  for (int i = 0; i < KPL_NPMAX; i++) {
    int ix = 3 * (i / 10);
    int iy = i % 10;
    grid->addWidget(new QLabel(s.sprintf("p%i", i), g), iy, ix);
    grid->addWidget(par[i] = new QLineEdit(m->number(p[i]), g), iy, ix + 1);
    par[i]->setMaximumWidth(80);
  }
  for (int i = 0; i < ((KPL_NPMAX - 1) / 10); i++)
    grid->addItem(new QSpacerItem(20, 10), 0, 3 * i + 2);
  if (m->options()->showSource) {
    QFileInfo fi(file);
    int pp = func.find("__");
    s = fi.dirPath(true) + "/" + fi.baseName() + ((pp != -1) ? ".cpp" : ".c");
    QFile f(s);
    if (f.open(IO_ReadOnly)) {
      SourceWidget* sw = new SourceWidget(frame, &f);
      hbox->addWidget(sw, 1);
      sw->setMinimumWidth(400);
      if (pp != -1) {
        s = func;
        sw->highlight(s.left(pp));
      } else
        sw->highlight(func);
      QTimer::singleShot(0, sw, SLOT(slotHighlight()));
    }
  }
}

EditParameterDlg::~EditParameterDlg()
{
}

void EditParameterDlg::getValues(bool ok)
{
  bool valid = true;
  QString s;
  double pt[KPL_NPMAX];
  for (int i = 0; i < KPL_NPMAX; i++) {
    s = par[i]->text();
    pt[i] = s.toDouble(&valid);
    if (!valid) {
      KMessageBox::sorry(this, s.sprintf(i18n("invalid parameter %i"), i));
      return;
    }
  }
  memcpy(p, pt, KPL_NPMAX * sizeof(double));
  if (ok)
    accept();
  else
    emit applyClicked();
}

void EditParameterDlg::slotOk()
{
  getValues(true);
}

void EditParameterDlg::slotApply()
{
  getValues(false);
}

bool Utils::getWriteURL(QWidget* parent, KURL& url, const QString& filter,
                        KplDoc* m)
{
  url = KFileDialog::getSaveURL(m->currentDir(), filter);
  if (url.isEmpty())
    return false;
  if (url.isLocalFile()) {
    QFileInfo fi(url.path());
    if (!fi.exists())
      return true;
    if (!fi.isWritable()) {
      KMessageBox::sorry(parent, i18n("Writing to this file not possible."));
      return false;
    }
  } else
    if (!KIO::NetAccess::exists(url))
      return true;
  return (KMessageBox::warningYesNo(parent, url.path() + "\n" +
    i18n("A file with this name exists already.\n"
         "Do you want to overwrite it?")) == KMessageBox::Yes);
}

void Utils::getFile(QLineEdit* e, KplDoc* m)
{
  QString s = KFileDialog::getOpenFileName(m->currentDir(), "*.so\n*");
  if (!s.isEmpty()) {
    e->setText(s);
    m->setCurrentDir(s);
  }
}

bool Utils::loadPar(const KURL& url, QWidget* parent, double* p, KplDoc* m)
{
  bool success = false;
  QString fn;
  if (KIO::NetAccess::download(url, fn)) {
    QFile f(fn);
    if (f.open(IO_ReadOnly)) {
      memset(p, 0, KPL_NPMAX * sizeof(double));
      QTextStream t(&f);
      int i = -1;
      while (!t.eof() && (++i < KPL_NPMAX))
        p[i] = t.readLine().toDouble();
      f.close();
      m->setCurrentDir(url);
      success = true;
    } else
      KMessageBox::error(parent,
                         i18n("while trying to open parameter file"));
    KIO::NetAccess::removeTempFile(fn);
  }
  return success;
}

void Utils::getPar(QWidget* parent, double* p, KplDoc* m)
{
  KURL url = KFileDialog::getOpenURL(m->currentDir(), "*.par\n*");
  if (!url.isEmpty())
    loadPar(url, parent, p, m);
}

bool Utils::saveParameter(QWidget* parent, const KURL& url, double* p,
                          double* err, KplDoc* m)
{
  QFile f((url.isLocalFile()) ? url.path() : m->tmpFile());
  bool success = f.open(IO_WriteOnly);
  if (success) {
    QTextStream t(&f);
    for (int i = 0; i < KPL_NPMAX; i++) {
      t << m->number(p[i]);
      if (err)
        t << m->separator() << m->number(err[i]);
      t << "\n";
    }
    f.close();
    m->setCurrentDir(url);
    if (!url.isLocalFile())
      m->copyTmp(url);
  } else
    KMessageBox::error(parent, i18n("while trying to open parameter file"));
  return success;
}

void Utils::saveFunPar(QWidget* parent, double* p, KplDoc* m)
{
  KURL url;
  if (getWriteURL(parent, url, "*.par\n*", m))
    saveParameter(parent, url, p, 0, m);
}

bool Utils::minMaxFile(double* xmin, double* xmax, const double* x, int n)
{
  *xmin = 1.0e300;
  *xmax = -*xmin;
  for (int i = 0; i < n; i++) {
    if ((x[i] > 1.0e300) || (x[i] < -1.0e300))
      return true;
    *xmin = QMIN(*xmin, x[i]);
    *xmax = QMAX(*xmax, x[i]);
  }
  return (*xmax == *xmin);
}

bool Utils::autoSc(double* xmi, double* xma, double* tic, int* mtic, int* ndig,
                   double xmin, double xmax, double fx, double c, bool _log)
{
  if (_log) {
    if (xmin < 1.0e-300)
      return true;
    if (xmin == xmax) {
      xmax = 1.001 * xmin;
      xmin = QMAX(0.999 * xmin, 1.0e-300);
    }
    *ndig = -1;
    double d = c * (log10(xmax) - log10(xmin));
    if (d > 1.0) {
      *mtic = 1;
      double f = pow(10.0, (-double(Utils::int9(log10(d + 1.0e-301)))));
      d *= f;
      if (d > 7.071)
        *tic = 10.0 / f;
      else
        if (d > 3.162)
          *tic = 5.0 / f;
        else
          *tic = ((d > 1.414) ? 2.0 : 1.0) / f;
      *tic = QMIN(*tic, 100.0);
      *xmi = pow(10.0, *tic * Utils::int9(log10(xmin) / *tic + 1.0e-4));
      *xma = pow(10.0, *tic * Utils::int9(log10(xmax) / *tic + 0.9999));
      *tic = pow(10.0, *tic);
      return false;
    } else {
      *mtic = 10;
      if (d < 0.7)
        *tic = (d < 0.5) ? 2.0 : 3.0;
      else
        *tic = 10.0;
    }
    double xl = log10(xmin);
    double mant = xl - int(xl + 1.0e-6 * fabs(xl));
    if (fabs(mant) < 1.0e-9)
      mant = 0.0;
    if (mant < 0.0)
      mant += 1.0;
    int e = qRound(xl - mant);
    if (*tic == 2.0)
      if (mant < log10(1.999999))
        mant = 0.0;
      else
        if (mant < log10(4.999999))
          mant = log10(2.0);
        else
          mant = log10(5.0);
    else
      if (*tic == 3.0)
        if (mant < log10(2.999999))
          mant = 0.0;
        else
          mant = log10(3.0);
      else
        mant = 0.0;
    *xmi = pow(10.0, e + mant);
    xl = log10(xmax);
    mant = xl - int(xl + 1.0e-6 * fabs(xl));
    if (fabs(mant) < 1.0e-9)
      mant = 0.0;
    if (mant < 0.0)
      mant += 1.0;
    e = qRound(xl - mant);
    if (*tic == 2.0)
      if (mant > log10(5.000001))
        mant = 1.0;
      else
        if (mant > log10(2.000001))
          mant = log10(5.0);
        else {
          if (mant)
            mant = log10(2.0);
        }
    else
      if (*tic == 3.0) {
        if (mant > log10(3.000001))
          mant = 1.0;
        else
          if (mant)
            mant = log10(3.0);
      } else
        if (mant)
          mant = 1.0;
    *xma = pow(10.0, e + mant);
  } else {
    if (xmin == xmax) {
      double dx;
      if (xmin)
        dx = 1.0e-3 * fabs(xmin);
      else
        dx = 1.0e-3 / fx;
      xmin -= dx;
      xmax += dx;
    }
    double d = c * fx * (xmax - xmin);
    double f = pow(10.0, (-double(Utils::int9(log10(d + 1.0e-301)))));
    d *= f;
    if (d > 7.071)
      *mtic = 10;
    else
      if (d > 3.162)
        *mtic = 5;
      else
        *mtic = (d > 1.414) ? 2 : 1;
    *tic = *mtic / f;
    if ((*mtic == 1) || (*mtic == 10))
      *mtic = 2;
    if (!(*ndig = -Utils::int9(log10(*tic + 1.0e-301) + 1.0e-6)))
      *ndig = -1;
    *xmi = *tic * Utils::int9(fx * xmin / *tic + 1.0e-4);
    *xma = *tic * Utils::int9(fx * xmax / *tic + 0.9999);
  }
  return false;
}

void Utils::expo(double a, int* ie, double* f)
{
  if ((*ie = int9(log10(1.01 * a + 1.0e-301))) < 0)
    *ie = *ie - 1;
  *ie = -3 * (*ie / 3);
  *f = pow(10.0, *ie);
}

int Utils::int9(double a)
{
  int i = int(a);
  if (a < 0.0)
    i--;
  return i;
}

QString Utils::relPath(const KURL& uPlo, const KURL& path, bool abs)
{
  if (!abs) {
    KURL u = path;
    if (!(uPlo.isLocalFile() ^ u.isLocalFile())) {
      if (!uPlo.isLocalFile()) {
        KURL t(uPlo);
        t.setPath("/");
        if (!t.isParentOf(u))
          return path.url();
      }
      QString dPlo = uPlo.directory(false);
      QString dFile = u.directory(false);
      int nMax = QMIN(dPlo.length(), dFile.length());
      int n;
      for (n = 0; n < nMax; n++)
        if (dPlo[n] != dFile[n])
          break;
      if (n > 0)
        n = dFile.findRev('/', n) + 1;
      QString s = u.path().mid(n);
      int nUp = dPlo.contains('/') - dPlo.left(n).contains('/');
      for (int i = 0; i < nUp; i++)
        s.prepend("../");
      if (!u.isLocalFile()) {
        u.setPath(s);
        s = u.url();
      }
      return s;
    }
  }
  return path.isLocalFile() ? path.path() : path.url();
}

void Utils::saveSize(QWidget* w, const QString& groupName)
{
  KConfig* config = KGlobal::config();
  config->setGroup(groupName);
  config->writeEntry("Width", w->width());
  config->writeEntry("Height", w->height());
}

void Utils::saveGeometry(QWidget* w, const QString& groupName)
{
  saveSize(w, groupName);
  KConfig* config = KGlobal::config();
  config->writeEntry("x", w->x());
  config->writeEntry("y", w->y());
}

void Utils::setSize(QWidget* w, const QString& groupName)
{
  KConfig* config = KGlobal::config();
  config->setGroup(groupName);
  w->resize(config->readNumEntry("Width", w->minimumWidth()),
            config->readNumEntry("Height", w->minimumHeight()));
}

void Utils::setGeometry(QWidget* w, const QString& groupName)
{
  KConfig* config = KGlobal::config();
  config->setGroup(groupName);
  w->setGeometry(config->readNumEntry("x"), config->readNumEntry("y", 20),
                 config->readNumEntry("Width", w->minimumWidth()),
                 config->readNumEntry("Height", w->minimumHeight()));
}
