/***************************************************************************
                          funitem.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Aug 29 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <qfileinfo.h>
#include <qlibrary.h>
#include <qtextstream.h>
#include <qurl.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ksimpleconfig.h>
#include "arrayitem.h"
#include "funitem.h"
#include "kgraph.h"
#include "lmfit.h"
#include "utils.h"
#ifndef KPL_CLASSES_ONLY
#include <qlistview.h>
#include "funcdlg.h"
#include "kpldoc.h"
#endif

FunItem::FunItem() : logxo(false), tmin(0.0), tmax(0.0), dt(0.0),
 tmino(0.0), tmaxo(0.0), dto(0.0), fkty(0), fktyo(0), liby(0)
{
  init();
}

FunItem::FunItem(const FunItem& f) :
 ScaledItem(f), logxo(f.logxo), tmin(f.tmin), tmax(f.tmax), dt(f.dt),
 tmino(f.tmino), tmaxo(f.tmaxo), dto(f.dto), fkty(f.fkty), fktyo(f.fktyo),
 tv(f.tv), yv(f.yv), namey(f.namey), pathy(f.pathy), liby(0)
{
  memcpy(py, f.py, sizeof(py));
  memcpy(pyo, f.pyo, sizeof(pyo));
  tv.detach();
  yv.detach();
  if (f.liby)
    getFuncAddr(pathy.path(), namey, &liby, &fkty);
}

FunItem::FunItem(KplNamespace::AutoStruct* aut) : ScaledItem(aut),
 logxo(false), tmin(0.0), tmax(0.0), dt(0.0), tmino(0.0), tmaxo(0.0), dto(0.0),
 fkty(0), fktyo(0), liby(0)
{
  init();
}

FunItem::FunItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut,
                 const KURL& uPlo) :
 ScaledItem(plo, aut), logxo(false), tmino(0.0), tmaxo(0.0), dto(0.0),
 fktyo(0)
{
  tmin = plo->readDoubleNumEntry("xfmin");
  tmax = plo->readDoubleNumEntry("xfmax", tmin + 1.0);
  dt = plo->readDoubleNumEntry("dx");
  memset(pyo, 0, sizeof(pyo));
  QStringList list = plo->readListEntry("p", ' ');
  int cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    py[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  namey = plo->readEntry("name", "");
  QString s = plo->readEntry("path", "");
  pathy = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  pathy.cleanPath();
  getFuncAddr(pathy.path(), namey, &liby, &fkty);
}

FunItem::FunItem(bool act, int fill, int sym, const QString& col,
                 double xn, double yn, double xmin, double xmax, double dx,
                 const QString& name, const KURL& path, double relSize)
 : ScaledItem(act, fill, sym, col, xn, yn, relSize), logxo(false), tmin(xmin),
 tmax(xmax), dt(dx), tmino(0.0), tmaxo(0.0), dto(0.0), fkty(0), fktyo(0),
 namey(name), pathy(path), liby(0)
{
  init();
  getFuncAddr(pathy.path(), namey, &liby, &fkty);
}

FunItem::~FunItem()
{
  delete liby;
}

FunItem& FunItem::operator=(const FunItem& f)
{
  if (this != &f) {
    *(ScaledItem*)this = f;
    logxo = f.logxo;
    tmin = f.tmin;
    tmax = f.tmax;
    dt = f.dt;
    tmino = f.tmino;
    tmaxo = f.tmaxo;
    dto = f.dto;
    fkty = f.fkty;
    fktyo = f.fktyo;
    memcpy(py, f.py, sizeof(py));
    memcpy(pyo, f.pyo, sizeof(pyo));
    tv = f.tv;
    tv.detach();
    yv = f.yv;
    yv.detach();
    namey = f.namey;
    pathy = f.pathy;
    delete liby;
    liby = 0;
    if (f.liby)
      getFuncAddr(pathy.path(), namey, &liby, &fkty);
  }
  return *this;
}

KplItem::ItemTypes FunItem::iType() const
{
  return Function;
}

void FunItem::draw(KplGraph* g)
{
  if (fkty && active) {
    setProperties(g);
    double sav = g->relSize();
    g->setRelSize(relsiz * sav);
    int n = calcTable(g->logx);
    g->plArray(tv.data(), yv.data(), fx, fy, n, fillStyle, true, x0, y0);
    g->setRelSize(sav);
  }
}

#ifndef KPL_CLASSES_ONLY
void FunItem::writePlo(KSimpleConfig* plo, const KURL& url, bool _abs,
                       KplDoc* m) const
{
  plo->writeEntry("Type", "FUNITEM");
  ScaledItem::writePlo(plo, url, _abs, m);
  char frm = m->options()->format;
  int prec = m->options()->prec;
  plo->writeEntry("xfmin", tmin, true, false, frm, prec);
  plo->writeEntry("xfmax", tmax, true, false, frm, prec);
  plo->writeEntry("dx", dt, true, false, frm, prec);
  QStringList list;
  for (int i = 0; i < KPL_NPMAX; i++)
    list.append(m->number(py[i]));
  plo->writeEntry("p", list, ' ');
  plo->writeEntry("name", namey);
  plo->writeEntry("path", Utils::relPath(url, pathy, _abs));
}

void FunItem::setText(QListViewItem* it, bool*, bool* funcs) const
{
  *funcs = true;
  it->setText(1, i18n("Function"));
  QFileInfo fi(pathy.path());
  KGraph g;
  it->setPixmap(2, g.pixmap(symb, color));
  it->setText(2, QString("y = ") + fi.baseName() + "." + namey + "(x)");
}

int FunItem::editItem(QWidget* parent, KplDoc* m, int)
{
  FuncDlg dlg(parent, m, this);
  return dlg.exec();
}

void FunItem::exportTable(QTextStream& ts, KplDoc* m) const
{
  int n = calcTable(logxo);
  for (int i = 0; i < n; i++)
    ts << m->number(fx * tv[i]) << m->separator()
       << m->number(fy * yv[i]) << "\n";
}
#endif

KplItem* FunItem::copy() const
{
  return new FunItem(*this);
}

void FunItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (fkty) {
    Utils::expo(QMAX(fabs(tmin), fabs(tmax)), iext, fxt);
    double xmin, xmax, ymin, ymax;
    minMax(&xmin, &xmax, &ymin, &ymax);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), ieyt, fyt);
  }
}

void FunItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (fkty) {
    *xmi = tmin + x0;
    *xma = tmax + x0;
    int n = calcTable(logxo);
    Utils::minMaxFile(ymi, yma, yv.data(), n);
    *ymi += y0;
    *yma += y0;
  }
}

void FunItem::setPar(int i, double value, bool yFun)
{
  if (yFun)
    py[i] = value;
}

int FunItem::fit(ArrayItem* ad, bool* bFit, KplNamespace::DataErrorStruct* err,
                 bool nonLin, double tol, int itMax, double* chisq,
                 double* corr, double* pErr, double* avgErr)
{
  QPtrList<ArrayItem> al;
  al.append(ad);
  QPtrList<FunItem> fl;
  fl.append(this);
  return fit(&al, &fl, bFit, py, 0, 0, err, nonLin, tol, itMax, 0, chisq, corr,
             pErr, avgErr);
}

int FunItem::fit(QPtrList<ArrayItem>* ad, QPtrList<FunItem>* fd, bool* bFit,
                 double* p, QMemArray<double>* _fvec, QMemArray<double>* _sig,
                 KplNamespace::DataErrorStruct* err, bool nonLin, double tol,
                 int itMax, LMFit* _lm, double* chisq, double* corr,
                 double* pErr, double* avgErr, QObject* parent)
{
  for (FunItem* f = fd->first(); f; f = fd->next())
    if (!f->fkty)
      return -101;
  int i, k;
  int np = 0;
  for (i = 0; i < KPL_NPMAX; i++)
    if (bFit[i])
      np++;
  if (!np)
    return -102;
  ArrayItem* a;
  int n = 0;
  for (a = ad->first(); a; a = ad->next())
    n += a->n;
  QMemArray<double> *fvec, *sig;
  if (_fvec) {
    fvec = _fvec;
    fvec->resize(n);
  } else
    fvec = new QMemArray<double>(n);
  if (_sig) {
    sig = _sig;
    sig->resize(n);
  } else
    sig = new QMemArray<double>(n);
  int j = 0;
  for (k = 0; k < (int) ad->count(); k++) {
    a = ad->at(k);
    if ((!err) || err[k].fitErrCol) {
      for (i = 0; i < a->n; i++)
        (*sig)[j++] = a->x[a->ie][a->istart + i];
    } else {
      double (*fktErrMod)(double, const double*);
      QLibrary* lib;
      if (!getFuncAddr(err[k].errModPath, err[k].errModName, &lib, &fktErrMod))
        return -103;
      for (i = 0; i < a->n; i++)
        (*sig)[j++] = fktErrMod(a->x[err[k].errModArg ? a->iy : a->ix]
                                    [a->istart + i], err[k].pErrMod);
      delete lib;
    }
  }
  LMFit* lm = _lm ? _lm : new LMFit(ad, fd, bFit, p, parent);
  lm->sig = sig->data();
  int wadim = (6 + np + n) * np + n;
  double* wa = new double[wadim];
  memset(wa, 0, wadim * sizeof(double));
  double* guess = wa + 3 * np + n;
  double* diag = guess + np;
  double* fjac = diag + np;
  double* qtf = fjac + n * np;
  double* alpha = qtf + np;
  j = 0;
  for (i = 0; i < KPL_NPMAX; i++)
    if (bFit[i])
      guess[j++] = p[i];
  bool aborted = lm->userBreak = false;
  QMemArray<int> ip(KPL_NPMAX);
  ip[0] = -1;
  int info = 0;
  if (nonLin) {
    int nfev;
    lm->lmdif(n, np, guess, fvec->data(), tol, tol, 0.0,
              (np + 1) * itMax, 0.0, diag, 1, 100.0, 1, &info, &nfev,
              fjac, n, ip.data(), qtf, wa, wa + np, wa + 2 * np, wa + 3 * np);
    aborted = (ip[0] < 0);
    if (info >= 8)
      info = 4;
  } else
    aborted = lm->linFit(n, np, guess, fvec->data(), fjac, ip.data(), qtf, wa,
                         wa + np, wa + 2 * np, wa + 3 * np);
  if (!aborted) {
    j = 0;
    for (i = 0; i < KPL_NPMAX; i++)
      if (bFit[i])
        p[i] = guess[j++];
    if (chisq) {
      double chi = LMFit::enorm(n, fvec->data());
      *chisq = chi * chi;
    }
    if (corr) {
      for (i = 0; i < np; i++) {
        for (j = 0; j < np; j++) {
          alpha[ip[i] + ip[j] * np] = 0.0;
          for (int k = 0; k <= j; k++)
            if (k <= i)
              alpha[ip[i] + ip[j] * np] += fjac[i * n + k] * fjac[j * n + k];
        }
      }
      LMFit::minv(alpha, corr, np, wa, ip.data());
      int ij = 0;
      for (i = 0; i < np; i++) {
        diag[i] = sqrt(corr[ij]);
        ij += np + 1;
      }
      k = 0;
      for (i = 0; i < np; i++)
        for (j = 0; j < np; j++) {
          if (double temp = diag[i] * diag[j])
            corr[k] /= temp;
          k++;
        }
      int ny = n - np;
      if (chisq && pErr && ny) {
        double temp = sqrt(*chisq / ny);
        for (i = 0; i < np; i++)
          diag[i] *= temp;
        k = 0;
        for (i = 0; i < KPL_NPMAX; i++)
          pErr[i] = bFit[i] ? diag[k++] : 0.0;
      }
    }
    if (avgErr) {
      *avgErr = 0.0;
      for (i = 0; i < n; i++) {
        double temp = (*fvec)[i] * (*sig)[i];
        *avgErr += temp * temp;
      }
      *avgErr = sqrt(*avgErr / n);
    }
  }
  delete [] wa;
  if (!_lm)
    delete lm;
  if (!_sig)
    delete sig;
  if (!_fvec)
    delete fvec;
  if (aborted)
    info = -100;
  return info;
}

bool FunItem::getFuncAddr(const QString& path, const QString& name,
                          QLibrary** lib,
                          double (**fkt)(double, const double*))
{
  *lib = 0;
  *fkt = 0;
  if (path.isEmpty() || name.isEmpty())
    KMessageBox::sorry(0, i18n("no library or no function specified!"));
  else {
    *lib = new QLibrary(path);
    if (!(*lib)->load())
      KMessageBox::error(0, path + "\n" + i18n("library cannot be opened!"));
    else
      if ((*fkt = (double (*)(double, const double *)) (*lib)->resolve(name)))
        return true;
      else
        KMessageBox::error(0, path + ": " + name + "\n" +
                           i18n("function name cannot be resolved!"));
    delete *lib;
    *lib = 0;
  }
  return false;
}

void FunItem::init()
{
  memset(py, 0, sizeof(py));
  memset(pyo, 0, sizeof(pyo));
}

int FunItem::calcTable(bool logx) const
{
  bool newv = tv.isEmpty();
  double d;
  int n;
  if (dt) {
    n = int(fabs((tmax - tmin) / dt)) + 1;
    d = logx ? (log10(tmax / tmin) / (n - 1)) : dt;
  } else {
    n = 101;
    d = 0.01 * (logx ? log10(tmax / tmin) : (tmax - tmin));
  }
  int i;
  if (newv || (tmin != tmino) || (tmax != tmaxo) || (d != dto) ||
      (logx != logxo)) {
    tv.detach();
    tv.resize(n);
    int n1 = n - 1;
    for (i = 0; i < n; i++)
      tv[i] = (i == n1) ? tmax : (logx ? pow(10.0, log10(tmin) + i * d) :
                                         (tmin + i * d));
  }
  bool pchanged = false;
  for (i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (py[i] != pyo[i])))
      break;
  if (pchanged || newv || (tmin != tmino) || (tmax != tmaxo) ||
      (d != dto) || (fkty != fktyo) || (logx != logxo)) {
    yv.detach();
    yv.resize(n);
  }
  if (pchanged || newv || (tmin != tmino) || (tmax > tmaxo) ||
      (d != dto) || (fkty != fktyo) || (logx != logxo))
    for (i = 0; i < n; i++)
      yv[i] = ((*fkty)(tv[i], py));
  if (pchanged)
    for (i = 0; i < KPL_NPMAX; i++)
      pyo[i] = py[i];
  if (fkty != fktyo)
    fktyo = fkty;
  if (tmin != tmino)
    tmino = tmin;
  if (tmax != tmaxo)
    tmaxo = tmax;
  if (d != dto)
    dto = d;
  if (logx != logxo)
    logxo = logx;
  return n;
}

ParFunItem::ParFunItem(): fktx(0), fktxo(0), libx(0)
{
  init();
}

ParFunItem::ParFunItem(const ParFunItem& f) :
 FunItem(f), fktx(f.fktx), fktxo(f.fktxo), xv(f.xv),
 namex(f.namex), pathx(f.pathx), libx(0)
{
  memcpy(px, f.px, sizeof(px));
  memcpy(pxo, f.pxo, sizeof(pxo));
  xv.detach();
  if (f.libx)
    getFuncAddr(pathx.path(), namex, &libx, &fktx);
}

ParFunItem::ParFunItem(KplNamespace::AutoStruct* aut) :
 FunItem(aut), fktx(0), fktxo(0), libx(0)
{
  init();
}

ParFunItem::ParFunItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut,
                       const KURL& uPlo) : fktx(0), fktxo(0), libx(0)
{
  active = plo->readBoolEntry("active", true);
  relsiz = plo->readDoubleNumEntry("relsiz", 1.0);
  ScaledItem::readPlo(plo, aut);
  tmin = plo->readDoubleNumEntry("tmin");
  tmax = plo->readDoubleNumEntry("tmax", tmin + 1.0);
  dt = plo->readDoubleNumEntry("dt");
  QStringList list = plo->readListEntry("px", ' ');
  int cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    px[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  list = plo->readListEntry("py", ' ');
  cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    py[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  namex = plo->readEntry("namex", "");
  namey = plo->readEntry("namey", "");
  QString s = plo->readEntry("pathx", "");
  pathx = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  pathx.cleanPath();
  s = plo->readEntry("pathy", "");
  pathy = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  pathy.cleanPath();
  getFuncAddr(pathx.path(), namex, &libx, &fktx);
  getFuncAddr(pathy.path(), namey, &liby, &fkty);
}

ParFunItem::ParFunItem(bool act, int fill,int sym, const QString& col,
                       double xn, double yn, double tmi, double tma, double d,
                       const QString& name_x, const KURL& path_x,
                       const QString& name_y, const KURL& path_y,
                       double relSize) :
 FunItem(act, fill, sym, col, xn, yn, tmi, tma, d, name_y, path_y, relSize),
 fktxo(0), namex(name_x), pathx(path_x)
{
  init();
  getFuncAddr(pathx.path(), namex, &libx, &fktx);
}

ParFunItem::~ParFunItem()
{
  delete libx;
}

ParFunItem& ParFunItem::operator=(const ParFunItem& f)
{
  if (this != &f) {
    *(FunItem*)this = f;
    fktx = f.fktx;
    fktxo = f.fktxo;
    memcpy(px, f.px, sizeof(px));
    memcpy(pxo, f.pxo, sizeof(pxo));
    xv = f.xv;
    xv.detach();
    namex = f.namex;
    pathx = f.pathx;
    delete libx;
    libx = 0;
    if (f.libx)
      getFuncAddr(pathx.path(), namex, &libx, &fktx);
  }
  return *this;
}

KplItem::ItemTypes ParFunItem::iType() const
{
  return ParFunction;
}

void ParFunItem::draw(KplGraph* g)
{
  if (fkty && active) {
    setProperties(g);
    double sav = g->relSize();
    g->setRelSize(relsiz * sav);
    int n = calcTable();
    g->plArray(xv.data(), yv.data(), fx, fy, n, fillStyle, true, x0, y0);
    g->setRelSize(sav);
  }
}

#ifndef KPL_CLASSES_ONLY
void ParFunItem::writePlo(KSimpleConfig* plo, const KURL& url, bool _abs,
                          KplDoc* m) const
{
  plo->writeEntry("Type", "PARFUNITEM");
  ScaledItem::writePlo(plo, url, _abs, m);
  char frm = m->options()->format;
  int prec = m->options()->prec;
  plo->writeEntry("tmin", tmin, true, false, frm, prec);
  plo->writeEntry("tmax", tmax, true, false, frm, prec);
  plo->writeEntry("dt", dt, true, false, frm, prec);
  QStringList lx, ly;
  for (int i = 0; i < KPL_NPMAX; i++) {
    lx.append(m->number(px[i]));
    ly.append(m->number(py[i]));
  }
  plo->writeEntry("px", lx, ' ');
  plo->writeEntry("py", ly, ' ');
  plo->writeEntry("namex", namex);
  plo->writeEntry("namey", namey);
  plo->writeEntry("pathx", Utils::relPath(url, pathx, _abs));
  plo->writeEntry("pathy", Utils::relPath(url, pathy, _abs));
}

void ParFunItem::setText(QListViewItem* it, bool*, bool*) const
{
  it->setText(1, i18n("Par. function"));
  QFileInfo fi1(pathx.path());
  QFileInfo fi2(pathy.path());
  KGraph g;
  it->setPixmap(2, g.pixmap(symb, color));
  it->setText(2, QString("x = ") + fi1.baseName() + "." + namex
              + "(t), y = " + fi2.baseName() + "." + namey + "(t)");
}

int ParFunItem::editItem(QWidget* parent, KplDoc* m, int)
{
  ParFuncDlg dlg(parent, m, this);
  return dlg.exec();
}

void ParFunItem::exportTable(QTextStream& ts, KplDoc* m) const
{
  int n = calcTable();
  for (int i = 0; i < n; i++)
    ts << m->number(fx * xv[i]) << m->separator()
       << m->number(fy * yv[i]) << "\n";
}
#endif

KplItem* ParFunItem::copy() const
{
  return new ParFunItem(*this);
}

void ParFunItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (fktx && fkty) {
    double xmin, xmax, ymin, ymax;
    minMax(&xmin, &xmax, &ymin, &ymax);
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), iext, fxt);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), ieyt, fyt);
  }
}

void ParFunItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (fktx && fkty) {
    int n = calcTable();
    Utils::minMaxFile(xmi, xma, xv.data(), n);
    Utils::minMaxFile(ymi, yma, yv.data(), n);
    *xmi += x0;
    *xma += x0;
    *ymi += y0;
    *yma += y0;
  }
}

void ParFunItem::setPar(int i, double value, bool yFun)
{
  if (yFun)
    py[i] = value;
  else
    px[i] = value;
}

void ParFunItem::init()
{
  memset(px, 0, sizeof(px));
  memset(pxo, 0, sizeof(pxo));
}

int ParFunItem::calcTable() const
{
  bool newv = tv.isEmpty();
  double d;
  int n;
  if (dt) {
    n = int(fabs((tmax - tmin) / dt)) + 1;
    d = dt;
  } else {
    n = 101;
    d = 0.01 * (tmax - tmin);
  }
  int i;
  if (newv || (tmin != tmino) || (tmax != tmaxo) || (d != dto)) {
    tv.detach();
    tv.resize(n);
    int n1 = n - 1;
    for (i = 0; i < n; i++)
      tv[i] = (i == n1) ? tmax : (tmin + i * d);
  }
  bool pchanged = false;
  for (i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (py[i] != pyo[i])))
      break;
  if (pchanged || newv || (tmin != tmino) || (tmax != tmaxo) ||
      (d != dto) || (fkty != fktyo)) {
    yv.detach();
    yv.resize(n);
  }
  if (pchanged || newv || (tmin != tmino) || (tmax > tmaxo) ||
      (d != dto) || (fkty != fktyo))
    for (i = 0; i < n; i++)
      yv[i] = ((*fkty)(tv[i], py));
  if (pchanged) {
    for (i = 0; i < KPL_NPMAX; i++)
      pyo[i] = py[i];
    pchanged = false;
  }
  for (i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (px[i] != pxo[i])))
      break;
  if (pchanged || newv || (tmin != tmino) || (tmax != tmaxo) ||
      (d != dto) || (fktx != fktxo)) {
    xv.detach();
    xv.resize(n);
  }
  if (pchanged || newv || (tmin != tmino) || (tmax > tmaxo) ||
      (d != dto) || (fktx != fktxo))
    for (i = 0; i < n; i++)
      xv[i] = ((*fktx)(tv[i], px));
  if (pchanged)
    for (i = 0; i < KPL_NPMAX; i++)
      pxo[i] = px[i];
  if (fktx != fktxo)
    fktxo = fktx;
  if (fkty != fktyo)
    fktyo = fkty;
  if (tmin != tmino)
    tmino = tmin;
  if (tmax != tmaxo)
    tmaxo = tmax;
  if (d != dto)
    dto = d;
  return n;
}
