/***************************************************************************
                          splineitem.h  -  description
                             -------------------
    begin                : Wed Feb 27 2002
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SPLINEITEM_H
#define SPLINEITEM_H

#include <qmemarray.h>
#include "kplitem.h"

class ArrayItem;

/**
  * Spline item class. Class for smoothing spline items for plotting spline
  * interpolations, derivatives or integrals by lines or markers.
  * @author Werner Stille
  */
class SplineItem : public ScaledItem {
public:
  /**
    * Constructor.
    * @param k degree of spline.
    */
  SplineItem(int k = 3);
  /** Copy constructor */
  SplineItem(const SplineItem& f);
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  SplineItem(KplNamespace::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    * @param url URL of plot file.
    */
  SplineItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut,
             const KURL& url);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param active true for visible item.
    * @param fillStyle > 0 for filled polygon representation.
    * @param symb 0 for circles, < 0 for different markers,
    * >0 for different line types.
    * @param color color for spline plot.
    * @param fx normalization factor for x values.
    * @param fy normalization factor for y values.
    * @param xmin minimum argument value.
    * @param xmax maximum argument value.
    * @param dx argument stepsize.
    * @param k degree of spline.
    * @param deriv order of derivative.
    * @param low lower x limit for spline integral.
    * @param relSize relative size of lines or symbols.
    */
  SplineItem(bool active, int fillStyle, int symb, const QString& color,
             double fx, double fy, double xmin, double xmax, double dx = 0.0,
             int k = 3, int deriv = 0, double low = 0.0, double relSize = 1.0);
  /** Destructor */
  ~SplineItem();
  /** Assignment operator */
  SplineItem& operator=(const SplineItem& f);
  virtual ItemTypes iType() const;
  virtual void draw(KplGraph* g);
#ifndef KPL_CLASSES_ONLY
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;
  virtual void setText(QListViewItem* it, bool* arrays, bool* funcs) const;
  virtual int editItem(QWidget* parent, KplDoc* m, int i);
  virtual void exportTable(QTextStream& ts, KplDoc* m) const;
#endif
  virtual KplItem* copy() const;
  virtual void expoItem(int* iex, int* iey, double* fx, double* fy) const;
  virtual void minMax(double* xmi, double* xma, double* ymi, double* yma) const;
  /**
   * Finds roots of a cubic spline or its derivatives.
   * @param zero array for roots.
   * @param mest dimension of zero array.
   * @param nz number of roots.
   * @param deriv order of derivative.
   * @param offset value of the spline or derivative.
   * @return error code.
   */
  int roots(double* zero, int mest, int* nz, int deriv = 0,
            double offset = 0.0) const;
  /**
   * Performs interpolating (non-smoothing) spline fit.
   * @param x array with x values.
   * @param y array with y values.
   * @param np number of points.
   * @param xmin lower boundary of approximation interval.
   * @param xmax upper boundary of approximation interval.
   * @return error code.
   */
  int interpolation(const double* x, const double *y, int np, double xmin,
                    double xmax);
  /**
   * Performs smoothing spline fit.
   * @param ad pointer to array item.
   * @param xmin lower boundary of approximation interval.
   * @param xmax upper boundary of approximation interval.
   * @param chisq chi-square.
   * @param nk number of knots.
   * @param smf smoothing factor.
   * @param err pointer to structure containing the data error settings.
   * @param avgErr average error.
   * @return error code.
   */
  int fit(const ArrayItem* ad, double xmin, double xmax, double* chisq,
          int* nk, double smf = 0.0, KplNamespace::DataErrorStruct* err = 0,
          double* avgErr = 0);
  /**
   * Performs smoothing spline fit.
   * @param x array with x values.
   * @param y array with y values.
   * @param e array with error values.
   * @param np number of points.
   * @param xmin lower boundary of approximation interval.
   * @param xmax upper boundary of approximation interval.
   * @param chisq chi-square.
   * @param nk number of knots.
   * @param smf smoothing factor.
   * @param err pointer to structure containing the data error settings.
   * @param avgErr average error.
   * @return error code.
   */
  int fit(const double* x, const double* y, const double* e, int np,
          double xmin, double xmax, double* chisq, int* nk,
          double smf = 0.0, KplNamespace::DataErrorStruct* err = 0,
          double* avgErr = 0);
  /**
   * Calculates interpolated spline values.
   * @param x array with x values.
   * @param y array with y values.
   * @param np number of points.
   * @param deriv order of derivative.
   * @param x0 lower x limit in case of spline integral.
   * @return number of rows.
   */
  int calcValues(double* x, double* y, int np, int deriv = 0,
                 double x0 = 0.0) const;
  /**
   * Sorts arrays and calculates error column.
   * @param x0 array with x values.
   * @param y0 array with y values.
   * @param e0 array with error values.
   * @param np number of points.
   * @param err pointer to structure containing the data error settings.
   * @param x array for sorted x values.
   * @param y array for y values.
   * @param e array for error values.
   * @return error code.
   */
  static int sortedArrays(const double* x0, const double* y0, const double* e0,
                          int np, KplNamespace::DataErrorStruct* err,
                          double* x, double* y, double* e);
  /**
   * Compares table rows for qsort.
   * @param e1 pointer to row 1.
   * @param e2 pointer to row 2.
   * @return -1 if x1 < x2, 0 if x1 = x2, 1 if x1 > x2.
   */
  static int cmpasc(const void* e1, const void* e2);

protected:
  /**
   * Calculates spline value table.
   * @param logx true for logarithmic steps
   * @return number of rows.
   */
  int calcTable(bool logx) const;
  mutable bool logxo;
  int kdeg, nk, nderiv;
  mutable int kdego, nko, nderivo;
  double xmin, xmax, dx, xlow;
  mutable double xmino, xmaxo, dxo, xlowo;
  mutable QMemArray<double> xv, yv, t, c;

  friend class ItemDlg;
  friend class KplView;
  friend class RootDlg;
  friend class SplFitDlg;
  friend class SplineDlg;
};

#endif
