/***************************************************************************
                              kst2dplot.cpp
                             ---------------
    begin                : Mar 28, 2004
    copyright            : (C) 2004 The University of Toronto
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <stdio.h>
#include <math.h>
#include <limits.h>

// include files for Qt
#include <qapplication.h>
#include <qbitmap.h>
#include <qclipboard.h>
#include <qcolor.h>
#include <qcursor.h>
#include <qmessagebox.h>
#include <qnamespace.h>
#include <qpainter.h>
#include <qrect.h>
#include <qstring.h>
#include <qstringlist.h>

// include files for KDE
#include <kdebug.h>
#include <klocale.h>
#include <kmdimainfrm.h>
#include <kpopupmenu.h>

// application specific includes
#include "kst.h"
#include "kst2dplot.h"
#include "kstcurvedialog_i.h"
#include "kstdatacollection.h"
#include "kstfitcurve.h"
#include "kstfitdialog_i.h"
#include "kstlabeldialog_i.h"
#include "kstlegend.h"
#include "kstlinestyle.h"
#include "kstplotdialog_i.h"
#include "kstsettings.h"
#include "ksttoplevelview.h"
#include "kstviewwidget.h"
#include "kstviewwindow.h"

Kst2DPlot::Kst2DPlot(const QString& in_tag,
                 KstScaleModeType yscale_in,
                 KstScaleModeType xscale_in,
                 double xmin_in, double ymin_in,
                 double xmax_in, double ymax_in) : KstPlotBase("Kst2DPlot") {
  // Must stay here for plot loading correctness
  _pos_x = 0.0;
  _pos_y = 0.0;
  _width = 0.0;
  _height = 0.0;

  commonConstructor(in_tag, yscale_in, xscale_in, xmin_in, ymin_in,
                    xmax_in, ymax_in);

  // Use default foreground and background colours in KstSettings
  _backgroundColor = KstSettings::globalSettings()->backgroundColor;
  _foregroundColor = KstSettings::globalSettings()->foregroundColor;

  draw();
}


Kst2DPlot::Kst2DPlot(QDomElement& e) : KstPlotBase(e) {
  QString in_tag = "unknown";
  KstScaleModeType yscale_in = AUTO, xscale_in = AUTO;
  double xmin_in = 0, ymin_in = 0, xmax_in = 1, ymax_in = 1;
  QStringList ctaglist;
  KstLabel *in_toplabel = 0L, *in_xlabel = 0L, *in_ylabel = 0L, *in_ticklabel = 0L;
  KstLabel *in_alabel = 0L;
  KstLegend *in_legend = 0L;
  bool x_log = false, y_log = false;

  //temporary colors
  QColor in_foreColor = KstSettings::globalSettings()->foregroundColor;
  QColor in_backColor = KstSettings::globalSettings()->backgroundColor;


  // Must stay here for plot loading correctness
  _pos_x = 0.0;
  _pos_y = 0.0;
  _width = 0.0;
  _height = 0.0;

  /* parse the DOM tree */
  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement el = n.toElement(); // try to convert the node to an element.
    if (!el.isNull()) { // the node was really an element.
      if (el.tagName() == "width") {
        _width = el.text().toDouble();
      } else if (el.tagName() == "height") {
        _height = el.text().toDouble();
      } else if (el.tagName() == "pos_x") {
        _pos_x = el.text().toDouble();
      } else if (el.tagName() == "pos_y") {
        _pos_y = el.text().toDouble();
      } else if (el.tagName() == "xscalemode") {
        xscale_in = (KstScaleModeType) el.text().toInt();
      } else if (el.tagName() == "yscalemode") {
        yscale_in = (KstScaleModeType) el.text().toInt();
      } else if (el.tagName() == "xmin") {
        xmin_in = el.text().toDouble();
      } else if (el.tagName() == "xmax") {
        xmax_in = el.text().toDouble();
      } else if (el.tagName() == "ymin") {
        ymin_in = el.text().toDouble();
      } else if (el.tagName() == "ymax") {
        ymax_in = el.text().toDouble();
      } else if (el.tagName() == "toplabel") {
        delete in_toplabel;
        in_toplabel = new KstLabel(" ");
        in_toplabel->read(el);
      } else if (el.tagName() == "xlabel") {
        delete in_xlabel;
        in_xlabel = new KstLabel(" ");
        in_xlabel->read(el);
      } else if (el.tagName() == "ylabel") {
        delete in_ylabel;
        in_ylabel = new KstLabel(" ");
        in_ylabel->read(el);
      } else if (el.tagName() == "ticklabel") {
        delete in_ticklabel;
        in_ticklabel = new KstLabel(" ");
        in_ticklabel->read(el);
      } else if (el.tagName() == "legend") {
        delete in_legend;
        in_legend = new KstLegend();
        in_legend->read(el);
      } else if (el.tagName() == "label") {
        in_alabel = new KstLabel(" ");
        in_alabel->read(el);
        labelList.append(in_alabel);
      } else if (el.tagName() == "curvetag") {
        ctaglist.append(el.text());
      } else if (el.tagName() == "xlog") {
        x_log = true;
      } else if (el.tagName() == "ylog") {
        y_log = true;
      } else if (el.tagName() == "plotforecolor") {
        in_foreColor.setNamedColor(el.text());
      } else if (el.tagName() == "plotbackcolor") {
        in_backColor.setNamedColor(el.text());
      }
    }
    n = n.nextSibling();
  }

  commonConstructor(tagName(), yscale_in, xscale_in, xmin_in, ymin_in,
                    xmax_in, ymax_in, x_log, y_log);

  KstBaseCurveList l =
    kstObjectSubList<KstDataObject,KstBaseCurve>(KST::dataObjectList);
  for (unsigned i = 0; i < ctaglist.count(); i++) {
    KstBaseCurveList::Iterator it = l.findTag(ctaglist[i]);
    if (it != l.end()) {
      addCurve(*it);
    }
  }

  if (in_toplabel) {
    delete TopLabel;
    TopLabel = in_toplabel;
  }

  if (in_xlabel) {
    delete XLabel;
    XLabel = in_xlabel;
  }

  if (in_ylabel) {
    delete YLabel;
    YLabel = in_ylabel;
  }

  if (in_ticklabel) {
    delete TickLabel;
    TickLabel = in_ticklabel;
  }

  if (in_legend) {
    delete Legend;
    Legend = in_legend;
  }

  TickLabel->setDoScalarReplacement(false);

  //If no colours, use default ones
  _foregroundColor = in_foreColor;
  _backgroundColor = in_backColor;

  draw();
}

void Kst2DPlot::commonConstructor(const QString &in_tag,
                                KstScaleModeType yscale_in,
                                KstScaleModeType xscale_in,
                                double xmin_in,
                                double ymin_in,
                                double xmax_in,
                                double ymax_in,
                                bool x_log,
                                bool y_log) {
  _zoomPaused = false;
  _dirty = true;
  _oldSize.setWidth(0);
  _oldSize.setHeight(0);
  _oldXAlignment = 0;
  _hasFocus = false;
  _copy_x = _copy_y = KST::NOPOINT;
  _standardActions |= Delete | Edit | Zoom | Pause;
  _draggableLabel = -1;
  _type = "plot";

  _xLog = x_log;
  _yLog = y_log;

  setTagName(in_tag);
  _isTied = false;

  XMin = xmin_in;
  XMax = xmax_in;
  YMin = ymin_in;
  YMax = ymax_in;

  _xScaleMode = xscale_in;
  _yScaleMode = yscale_in;

  // Verify that scale limits make sense.  If not, go to auto.
  if (XMax <= XMin) { // not OK: ignore request
    XMin = 0;
    XMax = 1;
    if (_xScaleMode != AUTOUP) {
      _xScaleMode = AUTO;
    }
  }
  if (YMax <= YMin) {
    YMin = 0;
    YMax = 1;
    if (_yScaleMode != AUTOUP) {
      _yScaleMode = AUTO;
    }
  }

  // Turn on AutoDeletion
  _plotScaleList.setAutoDelete(true);
  //Push Scale onto PlotScaleList
  pushScale();

  XLabel = new KstLabel;
  XLabel->setJustification(CxBy);
  XLabel->setRotation(0);

  YLabel = new KstLabel;
  YLabel->setJustification(CxTy);
  YLabel->setRotation(270);

  TopLabel = new KstLabel;
  TopLabel->setJustification(LxBy);
  TopLabel->setRotation(0);

  TickLabel = new KstLabel;
  TickLabel->setRotation(0);
  TickLabel->setDoScalarReplacement(false);

  Legend = new KstLegend;
  Legend->setJustification(CxBy);
  Legend->setRotation(0);

  //setForegroundColor("black");
  //setBackgroundColor("white");

  labelList.setAutoDelete(true);

  setpixrect();
}


Kst2DPlot::~Kst2DPlot() {
  delete XLabel;
  XLabel = 0L;
  delete YLabel;
  YLabel = 0L;
  delete TopLabel;
  TopLabel = 0L;
  delete TickLabel;
  TickLabel = 0L;
  delete Legend;
  Legend = 0L;
}

/*** initialize the fonts in a plot: boost size by font_size ***/
void Kst2DPlot::initFonts(const QFont& in_font, int font_size) {
  // Still have to set symbol font info as well //
  // We also may want to change different label fonts separately. //
  int point_size;

  point_size = in_font.pointSize() + font_size;

  TopLabel->setFontName(in_font.family());
  TopLabel->setSize(point_size - 12);

  XLabel->setFontName(in_font.family());
  XLabel->setSize(point_size - 12);

  YLabel->setFontName(in_font.family());
  YLabel->setSize(point_size - 12);

  TickLabel->setFontName(in_font.family());
  TickLabel->setSize(point_size - 12);

  Legend->setFontName(in_font.family());
  Legend->setSize(point_size - 12);

}
  /* set font sizes */


void Kst2DPlot::setTopLabel(const QString& in_label) {
  TopLabel->setText(in_label);
}

void Kst2DPlot::setXLabel(const QString& in_label) {
  XLabel->setText(in_label);
}

void Kst2DPlot::setYLabel(const QString& in_label) {
  YLabel->setText(in_label);
}

void Kst2DPlot::setXScale(double xmin_in, double xmax_in) {
  if (xmax_in > xmin_in) { // only accept new range if valid
    XMax = xmax_in;
    XMin = xmin_in;
  }
}

void Kst2DPlot::setYScale(double ymin_in, double ymax_in) {
  if (ymax_in > ymin_in) { // only accept new range if valid
    YMax = ymax_in;
    YMin = ymin_in;
  }
}

void Kst2DPlot::setLXScale(double xmin_in, double xmax_in) {
  if (xmax_in > xmin_in) { // only accept new range if valid
    if (_xLog) {
      XMax = pow(10.0, xmax_in);
      XMin = pow(10.0, xmin_in);
    } else {
      XMax = xmax_in;
      XMin = xmin_in;
    }
  }
}

void Kst2DPlot::setLYScale(double ymin_in, double ymax_in) {
  if (ymax_in > ymin_in) { // only accept new range if valid
    if (_yLog) {
      YMax = pow(10.0, ymax_in);
      YMin = pow(10.0, ymin_in);
    } else {
      YMax = ymax_in;
      YMin = ymin_in;
    }
  }
}

void Kst2DPlot::setScale(double xmin_in, double ymin_in,
                  double xmax_in, double ymax_in) {
  setXScale(xmin_in, xmax_in);
  setYScale(ymin_in, ymax_in);
}

void Kst2DPlot::setLScale(double xmin_in, double ymin_in,
                  double xmax_in, double ymax_in) {
  setLXScale(xmin_in, xmax_in);
  setLYScale(ymin_in, ymax_in);
}

void Kst2DPlot::getScale(double &xmin, double &ymin,
                       double &xmax, double &ymax) const {
  xmin = XMin;
  xmax = XMax;
  ymin = YMin;
  ymax = YMax;
}

void Kst2DPlot::getLScale(double& x_min, double& y_min,
                       double& x_max, double& y_max) const {
  if (_xLog) {
    x_min = XMin > 0 ? log10(XMin) : -350;
    x_max = XMax > 0 ? log10(XMax) : -340;
  } else {
    x_max = XMax;
    x_min = XMin;
  }

  if (_yLog) {
    y_min = YMin > 0 ? log10(YMin) : -350;
    y_max = YMax > 0 ? log10(YMax) : -340;
  } else {
    y_max = YMax;
    y_min = YMin;
  }
}


KstScaleModeType Kst2DPlot::getXScaleMode() const {
  return _xScaleMode;
}

KstScaleModeType Kst2DPlot::getYScaleMode() const {
  return _yScaleMode;
}

void Kst2DPlot::setXScaleMode(KstScaleModeType scalemode_in) {
  _xScaleMode = scalemode_in;
}

void Kst2DPlot::setYScaleMode(KstScaleModeType scalemode_in) {
  _yScaleMode = scalemode_in;
}

void Kst2DPlot::addCurve(KstBaseCurvePtr incurve, bool set_dirty) {
  Curves.append(incurve);
  if (set_dirty) setDirty();
}

void Kst2DPlot::removeCurve(KstBaseCurvePtr incurve) {
  Curves.remove(incurve);
  setDirty();
}

void Kst2DPlot::updateScale() {
  double mid, delta;

  switch (_xScaleMode) {
      case AUTO:  // set scale so all of all curves fits
        if (Curves.isEmpty()) {
          XMin = 0;
          XMax = 1.0;
        } else {
          if (_xLog) {
            XMin = Curves[0]->minPosX();
          } else {
            XMin = Curves[0]->minX();
          }
          XMax = Curves[0]->maxX();
          for (unsigned i = 1; i < Curves.count(); i++) {
            if (_xLog) {
              if (XMin > Curves[i]->minPosX()) {
                XMin = Curves[i]->minPosX();
              }
            } else {
              if (XMin > Curves[i]->minX()) {
                XMin = Curves[i]->minX();
              }
            }
            if (XMax < Curves[i]->maxX()) {
              XMax = Curves[i]->maxX();
            }
          }
          if (XMax <= XMin) {  // if curves had no variation in them
            XMin -= 0.1;
            XMax = XMin + 0.2;
          }
        }
        break;
      case NOSPIKE:  // set scale so all of all curves fits
        if (Curves.isEmpty()) {
          XMin = 0;
          XMax = 1.0;
        } else {
          if (_xLog) {
            XMin = Curves[0]->minPosX();
          } else {
            XMin = Curves[0]->ns_minX();
          }
          XMax = Curves[0]->ns_maxX();
          for (unsigned i = 1; i < Curves.count(); i++) {
            if (_xLog) {
              if (XMin > Curves[i]->minPosX()) {
                XMin = Curves[i]->minPosX();
              }
            } else {
              if (XMin > Curves[i]->ns_minX()) {
                XMin = Curves[i]->ns_minX();
              }
            }
            if (XMax < Curves[i]->ns_maxX()) {
              XMax = Curves[i]->ns_maxX();
            }
          }
          if (XMax <= XMin) {  // if curves had no variation in them
            XMin -= 0.1;
            XMax = XMin + 0.2;
          }
        }
        break;
      case AC: // keep range const, but set mid to mid of all curves
        if (XMax <= XMin) { // make sure that range is legal
          XMin = 0; XMax = 1;
        }
        if (Curves.isEmpty()) {
          XMin = -0.5;
          XMax = 0.5;
        } else {
          mid = Curves[0]->midX();
          for (unsigned i = 1; i < Curves.count(); i++) {
            mid += Curves[i]->midX();
          }
          mid /= Curves.count();
          delta = XMax - XMin;
          XMin = mid - delta / 2.0;
          XMax = mid + delta / 2.0;
        }

        break;
      case FIXED:  // don't change the range
        if (XMin >= XMax) {  // has to be legal, even for fixed scale...
          if (XMax == 0) {
            XMax =  0.5;
            XMin = -0.5;
          } else {
            XMax += XMax*0.01;
            XMin -= XMin*0.01;
          }
        }
        break;
      case AUTOUP:  // only change up
        for (unsigned i = 0; i < Curves.count(); i++) {
          if (_xLog) {
            if (XMin > Curves[i]->minPosX()) {
              XMin = Curves[i]->minPosX();
            }
          } else {
            if (XMin > Curves[i]->minX()) {
              XMin = Curves[i]->minX();
            }
          }
          if (XMax < Curves[i]->maxX()) {
            XMax = Curves[i]->maxX();
          }
        }

        if (XMin >= XMax) {  // has to be legal, even for autoup...
          if (XMax == 0) {
            XMax =  0.5;
            XMin = -0.5;
          } else {
            XMax += XMax * 0.01;
            XMax = XMin * 0.02;
          }
        }
        break;
      default:
        fprintf(stderr, "Bug in Kst2DPlot::updateScale: bad scale mode\n");
        break;
  }

  switch (_yScaleMode) {
      case AUTO:  // set scale so all of all curves fits
        if (Curves.isEmpty()) {
          YMin = 0;
          YMax = 1.0;
        } else {
          bool isPsd = false;

          if (_yLog) {
            YMin = Curves[0]->minPosY();
          } else {
            YMin = Curves[0]->minY();
          }
          YMax = Curves[0]->maxY();
          if (Curves[0]->type() == KST_PSDCURVE)
            isPsd = true;
          for (unsigned i = 1; i < Curves.count(); i++) {
            if (_yLog) {
              if (YMin > Curves[i]->minPosY()) {
                YMin = Curves[i]->minPosY();
              }
            } else {
              if (YMin > Curves[i]->minY()) {
                YMin = Curves[i]->minY();
              }
            }
            if (YMax < Curves[i]->maxY()) {
              YMax = Curves[i]->maxY();
            }
            if (Curves[0]->type() == KST_PSDCURVE) {
              isPsd = true;
            }
          }
          if (YMax <= YMin) {  // if curves had no variation in them
            YMin -= 0.1;
            YMax = YMin + 0.2;
          }
          if (isPsd && !_yLog) { /* psd plots should default to 0min */
            YMin = 0;
          }
        }
        break;
      case NOSPIKE:  // set scale so all of all curves fits
        if (Curves.isEmpty()) {
          YMin = 0;
          YMax = 1.0;
        } else {
          bool isPsd = false;

          YMin = _yLog ? Curves[0]->minPosY() : Curves[0]->ns_minY();
          YMax = Curves[0]->ns_maxY();

          if (Curves[0]->type() == KST_PSDCURVE) {
            isPsd = true;
          }

          for (unsigned i = 1; i < Curves.count(); i++) {
            if (_yLog) {
              if (YMin > Curves[i]->minPosY()) {
                YMin = Curves[i]->minPosY();
              }
            } else {
              if (YMin > Curves[i]->ns_minY()) {
                YMin = Curves[i]->ns_minY();
              }
            }
            if (YMax < Curves[i]->ns_maxY()) {
              YMax = Curves[i]->ns_maxY();
            }
            if (Curves[0]->type() == KST_PSDCURVE) {
              isPsd = true;
            }
          }
          if (YMax <= YMin) {  // if curves had no variation in them
            YMin -= 0.1;
            YMax = YMin + 0.2;
          }
          if (isPsd && !_yLog) { /* psd plots should default to 0min */
            YMin = 0;
          }
        }
        break;
      case AC: // keep range const, but set mid to mean of all curves
        if (YMax <= YMin) { // make sure that range is legal
          YMin = 0;
          YMax = 1;
        }
        if (Curves.isEmpty()) {
          YMin = -0.5;
          YMax = 0.5;
        } else {
          mid = Curves[0]->midY();
          for (unsigned i = 1; i < Curves.count(); i++) {
            mid += Curves[i]->midY();
          }
          mid /= Curves.count();
          delta = YMax - YMin;
          YMin = mid - delta / 2.0;
          YMax = mid + delta / 2.0;
        }

        break;
      case FIXED:  // don't change the range
        if (YMin >= YMax) {  // has to be legal, even for fixed scale...
          if (YMax == 0) {
            YMax =  0.5;
            YMin = -0.5;
          } else {
            YMax += YMax*0.01;
            YMin -= YMin*0.01;
          }
        }
        break;
      case AUTOUP:  // only change up
        for (unsigned i = 0; i < Curves.count(); i++) {
          if (_yLog) {
            if (YMin > Curves[i]->minPosY()) {
              YMin = Curves[i]->minPosY();
            }
          } else {
            if (YMin > Curves[i]->minY()) {
              YMin = Curves[i]->minY();
            }
          }
          if (YMax < Curves[i]->maxY()) {
            YMax = Curves[i]->maxY();
          }
        }

        if (YMin >= YMax) {  // has to be legal, even for autoup...
          if (YMax == 0) {
            YMax =  0.5;
            YMin = -0.5;
          } else {
            YMax += YMax*0.01;
            YMin -= YMin*0.01;
          }
        }
        break;
      default:
        fprintf(stderr, "Bug in Kst2DPlot::updateScale: bad scale mode\n");
        break;
  }

}


inline int d2i(double x) {
  return int(floor(x+0.5));
}

inline void SafeD2I(double& dx, int& x) {

  /* make sure that we can work in integers, which is faster */
  /* we have to covert for p.drawline, so we may as well do it now */
  if (dx > double(INT_MAX)) {
    x = INT_MAX;
  } else if (dx < double(INT_MIN)) {
    x = INT_MIN;
  } else {
    x = int(dx);
  }
}

static void genAxisTikLabel(char *label, double z, bool isLog) {
  if (isLog) {
    if (z > -4 && z < 4) {
      sprintf(label, "%.9g", pow(10,z));
    } else {
      sprintf(label, "10^{%.0f}", z);
    }
  } else {
    sprintf(label, "%.9g", z);
  }
}

double Kst2DPlot::xInternalAlignment() {
  _buffer.buffer().resize(size());
  QPainter p(&_buffer.buffer());

  double YTick, Yorg; // Tick interval and position
  double x_min, x_max, y_min, y_max;

  double xleft_bdr_px, xright_bdr_px, ytop_bdr_px, ybot_bdr_px;
  int x_px, y_px;

  updateScale();

  getLScale(x_min, y_min, x_max, y_max);

  QRect v = p.window();
  x_px = v.width();
  y_px = v.height();

  setTicks(YTick, Yorg, y_max, y_min, _yLog);

  setBorders(xleft_bdr_px, xright_bdr_px, ytop_bdr_px, ybot_bdr_px,
             YTick, Yorg, p);

  return xleft_bdr_px;
}


static void set2dPlotTickPix(double& xtickpix, double& ytickpix, int x_pix, int y_pix) {
  /* set tick size: 4 points on a full letter size plot */
  if (x_pix < y_pix) {
    xtickpix = 4.0 * x_pix / 540.0;
    ytickpix = 4.0 * y_pix / 748.0;
  } else {
    ytickpix = 4.0 * y_pix / 540.0;
    xtickpix = 4.0 * x_pix / 748.0;
  }
  xtickpix = (xtickpix + ytickpix) / 2.0; // average of x and y scaling
  if (xtickpix < 2.0) {
    xtickpix = 2.0; // but at least 2 pixels
  }
  ytickpix = xtickpix;
}

void Kst2DPlot::setBorders(double& xleft_bdr_px, double& xright_bdr_px,
                         double& ytop_bdr_px, double& ybot_bdr_px,
                         double YTick, double Yorg,
                         QPainter& p) {
  int x_px, y_px,i;
  char TmpStr[120];

  QRect v = p.window();

  x_px = v.width();
  y_px = v.height();

  /*********************************************/
  /* Set Borders                               */
  ytop_bdr_px = 1.3 * TopLabel->lineSpacing(p);
  ybot_bdr_px = 1.3 * XLabel->lineSpacing(p) + TickLabel->ascent(p);

  double ifloat = (YMax - Yorg)/YTick;
  i = int(floor(ifloat));

  // necessary?  -- Broken!  Unreliable.
  if (double(i) == ifloat) {
    --i;
  }

  --i;

  genAxisTikLabel(TmpStr, i * YTick + Yorg, _yLog);
  TickLabel->setJustification(RxCy);
  TickLabel->setText(TmpStr);
  xleft_bdr_px = TickLabel->width(p);

  /* now check min label */
  ifloat = (YMin - Yorg)/YTick;
  i = int(floor(ifloat));

  // necessary?  -- Broken!  Unreliable.
  if (double(i) == ifloat) {
    ++i;
  }

  ++i;

  genAxisTikLabel(TmpStr, i * YTick + Yorg, _yLog);
  TickLabel->setText(TmpStr);
  i = TickLabel->width(p);
  if (i > xleft_bdr_px) {
    xleft_bdr_px = i;
  }
  xleft_bdr_px += 1.5 * YLabel->lineSpacing(p) + 5;

  xright_bdr_px = x_px / 30;

  xleft_bdr_px = ceil(xleft_bdr_px);
  xright_bdr_px = ceil(xright_bdr_px);
  ytop_bdr_px = ceil(ytop_bdr_px);
  ybot_bdr_px = ceil(ybot_bdr_px);

}


void Kst2DPlot::drawDotAt(QPainter& p, double x, double y) {
  int X1, Y1;

  if (_xLog) {
    x = x > 0 ? log10(x) : -350;
  }
  if (_yLog) {
    y = y > 0 ? log10(y) : -350;
  }

  X1 = (int)(_m_X * x + _b_X);
  Y1 = (int)(_m_Y * y + _b_Y);

  if (PlotRegion.contains(X1, Y1)) {
    p.setPen(QPen(QColor(255,0,0), 2));
    p.drawArc((int)X1 - 2, (int)Y1 - 2, 4, 4, 0, 5760);
    p.setPen(QPen(QColor(0,0,0), 0));
    p.drawArc((int)X1 - 3, (int)Y1 - 3, 6, 6, 0, 5760);
  }
}

void Kst2DPlot::edit() {
  KstApp *app = KstApp::inst();
  KMdiChildView *c = app->activeWindow();

  if (c) {
    app->showPlotDialog(c->caption(), tagName());
  } else {
    app->showPlotDialog();
  }
}


void Kst2DPlot::parentResized() {
  KstPlotBase::parentResized();
  setDirty();
}


void Kst2DPlot::resize(const QSize& size) {
  KstPlotBase::resize(size);
  setDirty();
}


void Kst2DPlot::updateTieBox(QPainter& p) {
//kdDebug() << "Update tie box for " << tagName() << endl;
  QRect tr = GetTieBoxRegion();
  if (isTied()) {
    p.fillRect(tr, QColor("slate gray"));
  } else {
    p.fillRect(tr, backgroundColor());
  }
  p.drawRect(tr);
  if (_hasFocus) {
    QSize sz = tr.size();
    sz /= 2;
    tr.setSize(sz);
    tr.moveTopLeft(tr.topLeft() + QPoint(3*tr.width()/4, 3*tr.height()/4));
    p.fillRect(tr, Qt::black);
  }
}

void Kst2DPlot::paint(KstPaintType type, QPainter& p) {
  if ((type == P_EXPORT) || (type == P_PRINT)) {
    QRect window_hold = p.window();
    QRect viewport_hold = p.viewport();
    p.setViewport(geometry().left(), geometry().top(),
                  geometry().width(), geometry().height());
    if (type == P_PRINT) {
      draw(p, 5.0);
    } else {
      draw(p, 1.);
    }
    p.setWindow(window_hold);
    p.setViewport(viewport_hold);
  } else {
    if (_zoomPaused) {
      return;
    }

    /* check for optimizations */
    bool doDraw = true;
    if (type == P_PAINT) {
      if (_oldSize == size()) {
        doDraw = false;
      }
    } else if (type == P_ZOOM) {
      if (KST::alignment.x(size()) == _oldXAlignment) {
        doDraw = false;
      }
    }
    if (_dirty) {
      doDraw = true;
    }

    _oldXAlignment = KST::alignment.x(size());
    _oldSize = size();

    setpixrect();

    if (doDraw) {
      draw();
      _dirty = false;
    }
    _buffer.paintInto(p, geometry());
    updateTieBox(p);
  }

  KstPlotBase::paint(type, p);
}

void Kst2DPlot::draw() {
  if (_zoomPaused) {
    return;
  }

  _buffer.buffer().resize(size());
  _buffer.buffer().fill(backgroundColor());
  QPainter p(&_buffer.buffer());
  p.setWindow(0,0,geometry().width(), geometry().height());

  draw(p);

  setpixrect();
}

void Kst2DPlot::draw(QPainter &p, double resolutionEnhancement) {
/* Draws to the buffer */

  if (_zoomPaused) {
    return;
  }

  double in_xleft_bdr_px = resolutionEnhancement * KST::alignment.x(p.window().size());

  double XTick, YTick, Xorg, Yorg; // Tick interval and position
  double x_min, x_max, y_min, y_max;

  // Plot dimensions in pixels
  double xleft_bdr_px, xright_bdr_px, ytop_bdr_px, ybot_bdr_px;
  double x_orig_px, y_orig_px, xtick_len_px, ytick_len_px;
  double xtick_px, ytick_px;
  int x_px, y_px;

  // Used for drawing curves
  double Lx, Hx, Ly, Hy;
  double rX, rY, rEX, rEY;
  double X1, Y1, EX, EY;
  double last_x1, last_y1;
  double X2, Y2;
  int X2i, last_x1i = 0;
  double maxY = 0, minY = 0;
  int penWidth;

  int i_pt;
  KstBaseCurve *c;

  char TmpStr[120];

  int i, j;
  bool overlap, nopoint;

  QRect old_window = p.window();

  p.setWindow(0, 0, (int)(p.viewport().width() * resolutionEnhancement),
                    (int)(p.viewport().height() * resolutionEnhancement));

  setpixrect();
  updateScale();

  getLScale(x_min, y_min, x_max, y_max);

  QRect v(0, 0, p.window().width(), p.window().height());
  //p.fillRect(v, printMode ? backgroundColor() : KstSettings::globalSettings()->backgroundColor);
  p.fillRect(v, _backgroundColor);

  x_px = v.width();
  y_px = v.height();

  penWidth = x_px / 999;
  if (penWidth == 1) {
    penWidth = 0;
  }

  p.setPen(QPen(_foregroundColor, penWidth));

  setTicks(XTick, Xorg, x_max, x_min, _xLog);
  setTicks(YTick, Yorg, y_max, y_min, _yLog);

  set2dPlotTickPix(xtick_len_px, ytick_len_px, x_px, y_px);

  setBorders(xleft_bdr_px, xright_bdr_px, ytop_bdr_px, ybot_bdr_px, YTick, Yorg, p);
  if (in_xleft_bdr_px > 0.001) { // x border overridden
    xleft_bdr_px = in_xleft_bdr_px;
  }

  x_orig_px = (Xorg - x_min) / (x_max - x_min) *
         (double)(x_px - (xleft_bdr_px + xright_bdr_px)) + xleft_bdr_px;
  y_orig_px = (y_max - Yorg) / (y_max-y_min) *
         (double)(y_px - (ytop_bdr_px + ybot_bdr_px)) + ytop_bdr_px;
  xtick_px = (XTick / (x_max - x_min)) * ((double)x_px - (xleft_bdr_px + xright_bdr_px));
  ytick_px = (YTick / (y_max - y_min)) * ((double)y_px - (ytop_bdr_px + ybot_bdr_px));

  /* return if the plot is too small to draw */
  if (x_px - xright_bdr_px - xleft_bdr_px < 10) {
    p.setWindow(old_window);
    return;
  }

  if (y_px - ybot_bdr_px - ytop_bdr_px + 1.0 - ytop_bdr_px < 10) {
    p.setWindow(old_window);
    return;
  }

  RelPlotRegion.setRect(d2i(xleft_bdr_px),
                        d2i(ytop_bdr_px),
                        d2i(x_px - xright_bdr_px - xleft_bdr_px + 1.0),
                        d2i(y_px - ybot_bdr_px - ytop_bdr_px + 1.0));

  setpixrect();

  RelWinRegion.setRect(0, 0, (int)x_px, (int)y_px);

  RelPlotAndAxisRegion.setRect(
    d2i(YLabel->lineSpacing(p) + 1),
    d2i(ytop_bdr_px),
    d2i(x_px - YLabel->lineSpacing(p) - xright_bdr_px),
    d2i(y_px - XLabel->lineSpacing(p) - ytop_bdr_px));

  /******************************************************************/
  /* Draw the axis and labels */
  /* Draw Labels */

  YLabel->draw(p, (YLabel->lineSpacing(p) - YLabel->ascent(p))/2, y_px/2);
  XLabel->draw(p, x_px/2, y_px-(XLabel->lineSpacing(p) - XLabel->ascent(p))/2);

  TopLabel->draw(p, d2i(xleft_bdr_px), d2i(0.85*(ytop_bdr_px)));

  /* Draw Axis */
  p.drawRect(RelPlotRegion);

  if (_xLog) {
    /* Draw X Ticks */
    for (i = -1; xtick_px * i + x_orig_px > xleft_bdr_px - 1; i--); // find starting i;
    //i++;
    for (;xtick_px * i + x_orig_px < x_px - xright_bdr_px + 1; i++) {
      // draw major ticks
      X1 = (x_orig_px + (double)i * xtick_px);
      if (X1 > xleft_bdr_px && X1 < x_px - xright_bdr_px) {
        p.drawLine(d2i(X1),
                   d2i(ytop_bdr_px),
                   d2i(X1),
                   d2i(ytop_bdr_px + 2.0 * xtick_len_px));

        p.drawLine(d2i(X1),
                   d2i(y_px - ybot_bdr_px),
                   d2i(X1),
                   d2i(y_px - ybot_bdr_px - 2.0 * xtick_len_px));
      }
      // draw minor ticks
      if (XTick == 1.0) {
        for (j = 2; j < 10; j++) {
          X2 = log10((double)j) * (double)xtick_px + X1;
          if (X2 > xleft_bdr_px && X2 < x_px - xright_bdr_px) {
            p.drawLine(d2i(X2),
                       d2i(ytop_bdr_px),
                       d2i(X2),
                       d2i(ytop_bdr_px + xtick_len_px));

            p.drawLine(d2i(X2),
                       d2i(y_px - ybot_bdr_px),
                       d2i(X2),
                       d2i(y_px - ybot_bdr_px - xtick_len_px));
          }
        }
      }
    }
  } else {
    /* Draw X Ticks */
    for (i = -1; xtick_px * i / 5.0 + x_orig_px > xleft_bdr_px - 1; i--); // find starting i
    i++;
    for (; xtick_px * i / 5 + x_orig_px < x_px - xright_bdr_px ; i++) {
      X1 = x_orig_px + (double)i * xtick_px / 5.0;
      if (i % 5 == 0) {
        p.drawLine(d2i(X1),
                   d2i(ytop_bdr_px),
                   d2i(X1),
                   d2i(ytop_bdr_px + 2.0 * xtick_len_px));

        p.drawLine(d2i(X1),
                   d2i(y_px - ybot_bdr_px),
                   d2i(X1),
                   d2i(y_px - ybot_bdr_px - 2 * xtick_len_px));
      } else {
        p.drawLine(d2i(X1),
                   d2i(ytop_bdr_px),
                   d2i(X1),
                   d2i(ytop_bdr_px + xtick_len_px));
        p.drawLine(d2i(X1),
                   d2i(y_px - ybot_bdr_px),
                   d2i(X1),
                   d2i(y_px - ybot_bdr_px - xtick_len_px));
      }
    }
  }

  /* Draw Y Ticks */
  if (_yLog) {
    for (i = -1; ytick_px * i + y_orig_px > ytop_bdr_px - 1; i--);
    //i++;
    for (; ytick_px * i + y_orig_px < y_px - ybot_bdr_px + 1; i++) {
      // draw major ticks
      Y1 = y_orig_px + (double)i * ytick_px;
      if (Y1 > ytop_bdr_px) {
        p.drawLine(d2i(xleft_bdr_px),
                   d2i(Y1),
                   d2i(xleft_bdr_px + 2.0 * ytick_len_px - 1.0),
                   d2i(Y1));
        p.drawLine(d2i(x_px - xright_bdr_px),
                   d2i(Y1),
                   d2i(x_px - xright_bdr_px - 2.0 * ytick_len_px - 1.0),
                   d2i(Y1));
      }
      if (YTick == 1.0) {
        for (j = 2; j < 10; j++) {
          Y2 = (-log10((double)j) + 1.0) * (double)ytick_px + Y1;
          if (Y2 > ytop_bdr_px && Y2 < y_px - ybot_bdr_px) {
            p.drawLine(d2i(xleft_bdr_px),
                       d2i(Y2),
                       d2i(xleft_bdr_px + ytick_len_px),
                       d2i(Y2));
            p.drawLine(d2i(x_px - xright_bdr_px),
                       d2i(Y2),
                       d2i(x_px - xright_bdr_px - ytick_len_px),
                       d2i(Y2));
          }
        }
      }
    }
  } else {
    for (i = -1; ytick_px * i / 5 + y_orig_px > ytop_bdr_px - 1; i--);
    i++;
    for (; ytick_px * i / 5 + y_orig_px < y_px - ybot_bdr_px + 1; i++) {
      Y1 = y_orig_px + (double)i * ytick_px / 5.0;
      if (i % 5 == 0) {
        p.drawLine(d2i(xleft_bdr_px),
                   d2i(Y1),
                   d2i(xleft_bdr_px + 2.0 * ytick_len_px),
                   d2i(Y1));
        p.drawLine(d2i(x_px - xright_bdr_px),
                   d2i(Y1),
                   d2i(x_px - xright_bdr_px - 2.0 * ytick_len_px),
                   d2i(Y1));
      } else {
        p.drawLine(d2i(xleft_bdr_px),
                   d2i(Y1),
                   d2i(xleft_bdr_px + ytick_len_px),
                   d2i(Y1));
        p.drawLine(d2i(x_px - xright_bdr_px),
                   d2i(Y1),
                   d2i(x_px - xright_bdr_px - ytick_len_px),
                   d2i(Y1));
      }
    }
  }

  /* Print Numbers */
  /* x axis numbers */
  TickLabel->setJustification(CxTy);
  for (i = -1; xtick_px * i + x_orig_px > xleft_bdr_px - 1; i--);
  i++;
  for (; xtick_px * i + x_orig_px < x_px - xright_bdr_px + 1; i++) {
    genAxisTikLabel(TmpStr, i * XTick + Xorg, _xLog);
    TickLabel->setText(TmpStr);
    TickLabel->draw(p, d2i(x_orig_px + i * xtick_px),
                   d2i(y_px - (0.85 * ybot_bdr_px)));
  }

  /* y axis numbers */
  TickLabel->setJustification(RxCy);
  for (i = -1; ytick_px * i + y_orig_px > ytop_bdr_px - 1; i--);
  i++;
  for (; ytick_px * i + y_orig_px < y_px - ybot_bdr_px + 1; i++) {
    genAxisTikLabel(TmpStr, -(i*YTick) + Yorg, _yLog);
    TickLabel->setText(TmpStr);
    TickLabel->draw(p, d2i(xleft_bdr_px - TickLabel->lineSpacing(p) / 4),
                   d2i(y_orig_px + i * ytick_px));
  }

  /*** plot the arbitrary labels **/
  for (KstLabel *label = labelList.first(); label; label = labelList.next()) {
    label->draw(p, d2i(xleft_bdr_px + RelPlotRegion.width() * label->x()),
                d2i(ytop_bdr_px + RelPlotRegion.height() * label->y()));
  }

  /*** plot the legend **/
  if (!Legend->getFront()) {
    Legend->draw( &Curves, p, d2i(xleft_bdr_px + RelPlotRegion.width() * Legend->x()),
                d2i(ytop_bdr_px + RelPlotRegion.height() * Legend->y()));
  }

  /*******************************************************************/
  /* Plot the Curves */
  Lx = (double)xleft_bdr_px + 1;
  Hx = (double)(x_px - xright_bdr_px - 1);
  Ly = (double)ytop_bdr_px + 1;
  Hy = (double)(y_px - ybot_bdr_px - 1);
  _m_X = (Hx - Lx)/(x_max - x_min);
  _m_Y = (Ly - Hy)/(y_max - y_min);
  _b_X = Lx - _m_X * x_min;
  _b_Y = Hy - _m_Y * y_min;

  for (int i_curve = 0; i_curve < (int)Curves.count(); i_curve++) {
    c = Curves[i_curve];
    c->readLock();
    overlap = false;

    if (c->sampleCount() > 0) {
      int i0, iN;
      if (c->xIsRising()) {
        i0 = c->getIndexNearX(x_min);
        if (i0>0) i0--;
        iN = c->getIndexNearX(x_max);
        if (iN<c->sampleCount() - 1) iN++;
      } else {
        i0 = 0;
        iN = c->sampleCount() - 1;
      }

      p.setPen(QPen(c->getColor(), c->lineWidth(), KstLineStyle[c->lineStyle()]));

      if (c->hasLines()) {

        /* Read i_pt = 0 */
        nopoint = false;
        c->getPoint(i0, rX, rY);

// Optimize - isnan seems expensive, at least in gcc debug mode
//            cachegrind backs this up.
        if (_xLog) {
          if (rX > 0) {
            rX = log10(rX);
          } else {
            rX = -350;
          }
        }

        if (_yLog) {
          if (rY > 0) {
            rY = log10(rY);
          } else {
            rY = -350;
          }
        }

        X1 = _m_X*rX + _b_X;
        Y1 = _m_Y*rY + _b_Y;

        last_x1 = X1;
        last_x1i = (int)X1;
        last_y1 = Y1;

        for (i_pt = i0 + 1; i_pt <= iN; i_pt++) {
          X2 = last_x1;
          X2i = last_x1i;
          Y2 = last_y1;

          /* read next i_pt */
          nopoint = false;
          c->getPoint(i_pt, rX, rY);
#undef isnan
#define isnan(x) (x != x)
          while ((isnan(rX) || isnan(rY)) && i_pt < iN) {
#undef isnan
            nopoint = true;
            i_pt++;
            c->getPoint(i_pt, rX, rY);
          }

          if (_xLog) {
            rX = rX > 0 ? log10(rX) : -350;
          }
          if (_yLog) {
            rY = rY > 0 ? log10(rY) : -350;
          }

          X1 = _m_X*rX + _b_X;
          Y1 = _m_Y*rY + _b_Y;

          last_x1 = X1;
          last_x1i = (int)X1;
          last_y1 = Y1;
          if (nopoint) {
            if (overlap) {
              if (X2 >= Lx && X2 <= Hx) {
                if (maxY > Hy && minY <= Hy) {
                  maxY = Hy;
                }
                if (minY < Ly && maxY >= Ly) {
                  minY = Ly;
                }
                if (minY >= Ly && minY <= Hy && maxY >= Ly && maxY <= Hy) {
                  p.drawLine((int)X2, (int)minY, (int)X2, (int)maxY);
                }
              }
              overlap = false;
            }
          } else if (last_x1i == X2i) {
            if (overlap) {
              if (Y1 > maxY) {
                maxY = Y1;
              }
              if (Y1 < minY) {
                minY = Y1;
              }
            } else {
              if (Y1 < Y2) {
                minY = Y1;
                maxY = Y2;
              } else {
                maxY = Y1;
                minY = Y2;
              }
              overlap = true;
            }
          } else {
            if (!((X1 < Lx && X2 < Lx) || (X1 > Hx && X2 > Hx))) {
              /* trim all lines to be within plot */
              if (X1 < Lx && X2 > Lx) {
                Y1 = (Y2 - Y1) / (X2 - X1) * (Lx - X1) + Y1;
                X1 = Lx;
              } else if (X2 < Lx && X1 > Lx) {
                Y2 = (Y1 - Y2) / (X1 - X2) * (Lx - X2) + Y2;
                X2 = Lx;
              }

              if (X1 < Hx && X2 > Hx) {
                Y2 = (Y2 - Y1) / (X2 - X1) * (Hx - X1) + Y1;
                X2 = Hx;
              } else if (X2 < Hx && X1 > Hx) {
                Y1 = (Y1 - Y2) / (X1 - X2) * (Hx - X2) + Y2;
                X1 = Hx;
              }

              if (Y1 < Ly && Y2 > Ly) {
                X1 = (X2 - X1) / (Y2 - Y1) * (Ly - Y1) + X1;
                Y1 = Ly;
              } else if (Y2 < Ly && Y1 > Ly) {
                X2 = (X1 - X2) / (Y1 - Y2) * (Ly - Y2) + X2;
                Y2 = Ly;
              }

              if (Y1 < Hy && Y2 > Hy) {
                X2 = (X2 - X1) / (Y2 - Y1) * (Hy - Y1) + X1;
                Y2 = Hy;
              } else if (Y2 < Hy && Y1 > Hy) {
                X1 = (X1 - X2) / (Y1 - Y2) * (Hy - Y2) + X2;
                Y1 = Hy;
              }
            }

            if (overlap) {
              if (X2 >= Lx && X2 <= Hx) {
                if (maxY > Hy && minY <= Hy)
                  maxY = Hy;
                if (minY < Ly && maxY >= Ly)
                  minY = Ly;
                if (minY >= Ly && minY <= Hy && maxY >= Ly && maxY <= Hy) {
                  p.drawLine((int)X2, (int)minY, (int)X2, (int)maxY);
                }
              }
              overlap = false;
            }

            /* make sure both ends are in range: */
            if (X1 >= Lx && X1 <= Hx && X2 >= Lx && X2 <= Hx) {
              if (Y1 >= Ly && Y1 <= Hy && Y2 >= Ly && Y2 <= Hy) {
                p.drawLine((int)X1, (int)Y1, (int)X2, (int)Y2);
              }
            }
          }  /* end if (X1 == X2) */
        } // end for
      } // end if c->hasLines()

      if (c->hasPoints()) {
        for (i_pt = i0; i_pt < iN; i_pt++) {
          c->getPoint(i_pt, rX, rY);
          if (_xLog) {
            rX = rX > 0 ? log10(rX) : -350;
          }
          if (_yLog) {
            rY = rY > 0 ? log10(rY) : -350;
          }

          X1 = _m_X * rX + _b_X;
          Y1 = _m_Y * rY + _b_Y;
          if (X1 >= Lx && X1 <= Hx && Y1 >= Ly && Y1 <= Hy) {
            c->Point.draw(&p, (int)X1, (int)Y1);
          }
        }
      }
      if (c->hasXError()) {
        bool do_low_flag = true;
        bool do_high_flag = true;

        for (i_pt = i0; i_pt < iN; i_pt++) {
          do_low_flag = true;
          do_high_flag = true;
          c->getEXPoint(i_pt, rX, rY, rEX);
          EX = _m_X * rEX;
          X1 = _m_X * rX + _b_X - EX;
          X2 = _m_X * rX + _b_X + EX;
          Y1 = _m_Y * rY + _b_Y;
          if (X1 < Lx && X2 > Lx) {
            X1 = Lx;
            do_low_flag = false;
          }
          if (X1 < Hx && X2 > Hx) {
            X2 = Hx;
            do_high_flag = false;
          }

          if (X1 >= Lx && X2 <= Hx && Y1 >= Ly && Y1 <= Hy) {
            p.drawLine((int)X1, (int) Y1, (int) X2, (int) Y1);
            if (do_low_flag) {
              p.drawLine((int)X1, (int)Y1 + c->Point.getDim(&p),
                  (int)X1, (int)Y1 - c->Point.getDim(&p));
            }
            if (do_high_flag) {
              p.drawLine((int)X2, (int)Y1 + c->Point.getDim(&p),
                  (int)X2, (int)Y1 - c->Point.getDim(&p));
            }
          }
        }
      }

      if (c->hasYError()) {
        bool do_low_flag = true;
        bool do_high_flag = true;

        for (i_pt = i0; i_pt<iN; i_pt++) {
          do_low_flag = true;
          do_high_flag = true;
          c->getEYPoint(i_pt, rX, rY, rEY);
          EY = _m_Y * rEY;
          X1 = _m_X * rX + _b_X;
          Y1 = _m_Y * rY + _b_Y - EY;
          Y2 = _m_Y * rY + _b_Y + EY;
          if (Y1 < Ly && Y2 > Ly) {
            Y1 = Ly;
            do_low_flag = false;
          }
          if (Y1 < Hy && Y2 > Hy) {
            Y2 = Hy;
            do_high_flag = false;
          }

          if (X1 >= Lx && X1 <= Hx && Y1 >= Ly && Y2 <= Hy) {
            p.drawLine((int)X1, (int) Y1, (int) X1, (int) Y2);
            if (do_low_flag) {
              p.drawLine((int)X1 + c->Point.getDim(&p), (int)Y1,
                  (int)X1 - c->Point.getDim(&p), (int)Y1);
            }
            if (do_high_flag) {
              p.drawLine((int)X1 + c->Point.getDim(&p), (int)Y2,
                  (int)X1 - c->Point.getDim(&p), (int)Y2);
            }
          }
        }
      }
    }
    c->readUnlock();
  }

  /*** plot the legend **/
  if (Legend->getFront()) {
    Legend->draw(&Curves, p, d2i(xleft_bdr_px + RelPlotRegion.width() * Legend->x()),
                d2i(ytop_bdr_px + RelPlotRegion.height() * Legend->y()));
  }

  p.flush();

  p.setWindow(old_window);

}


QRect Kst2DPlot::GetPlotRegion() const {
  return PlotRegion;
}

QRect Kst2DPlot::GetWinRegion() const {
  return WinRegion;
}

QRect Kst2DPlot::GetPlotAndAxisRegion() const {
  return PlotAndAxisRegion;
}

QRect Kst2DPlot::GetTieBoxRegion() const {
  int left, top;
  const int dim = 11;

  if (WinRegion.right() - PlotRegion.right() > dim + 3) {
    left = PlotRegion.right() + 2;
  } else {
    left = WinRegion.right() - dim - 1;
  }
  if (PlotRegion.top() - WinRegion.top() > dim + 3) {
    top = PlotRegion.top() - 2 - dim;
  } else {
    top = WinRegion.top()+1;
  }

  return QRect(left, top, dim, dim);
}

void Kst2DPlot::setpixrect() {
  PlotRegion = RelPlotRegion;
  PlotRegion.moveBy(geometry().x(), geometry().y());
  WinRegion = RelWinRegion;
  WinRegion.moveBy(geometry().x(), geometry().y());
  PlotAndAxisRegion = RelPlotAndAxisRegion;
  PlotAndAxisRegion.moveBy(geometry().x(), geometry().y());
}


KstObject::UpdateType Kst2DPlot::update() {
  setDirty(); // FIXME
  return NO_CHANGE;
}


void Kst2DPlot::save(QTextStream& ts) {
  unsigned i;

  KstPlotBase::save(ts);

  ts << "  <xscalemode>" << _xScaleMode << "</xscalemode>" << endl;
  ts << "  <yscalemode>" << _yScaleMode << "</yscalemode>" << endl;

  ts << "  <xmin>" << XMin << "</xmin>" << endl;
  ts << "  <xmax>" << XMax << "</xmax>" << endl;
  ts << "  <ymin>" << YMin << "</ymin>" << endl;
  ts << "  <ymax>" << YMax << "</ymax>" << endl;

  ts << "  <toplabel>" << endl;
  TopLabel->save(ts);
  ts << "  </toplabel>" << endl;

  ts << "  <xlabel>" << endl;
  XLabel->save(ts);
  ts << "  </xlabel>" << endl;

  ts << "  <ylabel>" << endl;
  YLabel->save(ts);
  ts << "  </ylabel>" << endl;

  ts << "  <ticklabel>" << endl;
  TickLabel->save(ts);
  ts << "  </ticklabel>" << endl;

  ts << "  <legend>" << endl;
  Legend->save(ts);
  ts << "  </legend>" << endl;

  if (isXLog()) ts << "  <xlog/>" << endl;
  if (isYLog()) ts << "  <ylog/>" << endl;

  for (i = 0; i < labelList.count(); i++) {
    ts << "  <label>" << endl;
    labelList.at(i)->save(ts);
    ts << "  </label>" << endl;
  }

  for (i = 0; i < Curves.count(); i++) {
    ts << "  <curvetag>" << Curves[i]->tagName() << "</curvetag>" << endl;
  }

  //save the plot colors, but only if they are different from default
  if (_foregroundColor != KstSettings::globalSettings()->foregroundColor) {
    ts << "  <plotforecolor>" << _foregroundColor.name() << "</plotforecolor>" << endl;
  }
  if (_backgroundColor != KstSettings::globalSettings()->backgroundColor) {
    ts << "  <plotbackcolor>" << _backgroundColor.name() << "</plotbackcolor>" << endl;
  }

}


void Kst2DPlot::saveTag(QTextStream& ts) {
  ts << "<" << type() << ">" << endl;
  ts << "  <tag>" << tagName() << "</tag>" << endl;
  for (KstViewObjectList::Iterator i = _children.begin(); i != _children.end(); ++i) {
    (*i)->saveTag(ts);
  }
  ts << "</" << type() << ">" << endl;
}


void Kst2DPlot::pushScale() {
  struct KstPlotScale *ps;
  ps = new struct KstPlotScale;

  ps->xmin = XMin;
  ps->ymin = YMin;
  ps->xmax = XMax;
  ps->ymax = YMax;
  ps->xscalemode = _xScaleMode;
  ps->yscalemode = _yScaleMode;
  ps->xlog = _xLog;
  ps->ylog = _yLog;

  _plotScaleList.append(ps);
}


bool Kst2DPlot::popScale() {
  struct KstPlotScale *ps;

  if (_plotScaleList.count() > 1) {
    _plotScaleList.removeLast();
    ps = _plotScaleList.last();
    XMin = ps->xmin;
    XMax = ps->xmax;
    YMin = ps->ymin;
    YMax = ps->ymax;
    _xScaleMode = ps->xscalemode;
    _yScaleMode = ps->yscalemode;
    _xLog = ps->xlog;
    _yLog = ps->ylog;
    return true;
  }
  return false;
}


/****************************************************************/
/*                                                              */
/*        Place a '\' in front of special characters (ie, '_')  */
/*                                                              */
/****************************************************************/
static void EscapeSpecialChars(QString& label) {
  unsigned int i_char;

  for (i_char = 0; i_char < label.length(); i_char++) {
    if (label.at(i_char) == '_') {
      label.insert(i_char, '\\');
      i_char++;
    }
  }
}


void Kst2DPlot::GenerateDefaultLabels() {
  int n_curves, i_curve;
  QString xlabel, ylabel, toplabel;

  n_curves = Curves.count();

  if (n_curves < 1) return;

  if (n_curves == 1) {
    xlabel = Curves[0]->getXLabel();
    ylabel = Curves[0]->getYLabel();
    toplabel = Curves[0]->getTopLabel();
  } else {
    xlabel = Curves[0]->getXLabel();
    ylabel = QString::null;
    toplabel = QString::null;

    ylabel = Curves[0]->getYLabel();
    toplabel = Curves[0]->getTopLabel();
    for (i_curve = 1; i_curve < n_curves - 1; i_curve++) {
      ylabel += QString(", ") + Curves[i_curve]->getYLabel();
      if (toplabel != Curves[i_curve]->getTopLabel()) {
        toplabel += QString(", ") + Curves[i_curve]->getTopLabel();
      }
    }

    ylabel = i18n("%1 and %2").arg(ylabel).arg(Curves[n_curves - 1]->getYLabel());
    if (toplabel != Curves[i_curve]->getTopLabel() &&
        !Curves[i_curve]->getTopLabel().isEmpty()) {
      toplabel = i18n("%1 and %2").arg(toplabel).arg(Curves[n_curves - 1]->getTopLabel());
    }
  }

  EscapeSpecialChars(xlabel);
  EscapeSpecialChars(ylabel);
  EscapeSpecialChars(toplabel);

  setXLabel(xlabel);
  setYLabel(ylabel);
  setTopLabel(toplabel);
}


void Kst2DPlot::setTicks(double& tick, double& org,
                       double max, double min, bool is_log) {
  double St,Mant1,Mant2;  // used to generate tick interval
  double Exp,Log,FLog;    // used to generate tick interval

  if (is_log && max - min < 11 && max - min > 1) {
    tick = 1.0;
  } else {
    /* Determine tik Interval */
    St = (max - min) / 5.0;       /* tiks */
    Log = log10(St);
    FLog = (double)floor(Log);
    Exp = pow(10., FLog);
    Mant1 = fabs(Exp - St) < fabs(2.0 * Exp - St) ? Exp : 2.0 * Exp;
    Mant2 = fabs(5.0 * Exp - St) < fabs(10.0 * Exp - St) ? 5.0 * Exp : 10.0 * Exp;
    tick = fabs(Mant1 - St) < fabs(Mant2 - St) ? Mant1 : Mant2;
  }

  /* Determine Location of Origin */
  if (min > 0) {
    org = (double)ceil(min / tick) * tick;
  } else if (max < 0) {
    org = (double)floor(max / tick) * tick;
  } else {
    org = 0;
  }
}


void Kst2DPlot::setLog(bool x_log, bool y_log) {
  _xLog = x_log;
  _yLog = y_log;
}


bool Kst2DPlot::isXLog() const {
  return _xLog;
}


bool Kst2DPlot::isYLog() const {
  return _yLog;
}


bool Kst2DPlot::isTied() const {
  return _isTied;
}


void Kst2DPlot::toggleTied() {
  _isTied = !_isTied;
}


void Kst2DPlot::setTied(bool in_tied) {
  _isTied = in_tied;
}


void Kst2DPlot::editCurve(int id) {
  KstBaseCurvePtr curve = *(Curves.findTag(_curveEditMap[id]));
  if (curve) {
    curve->_showDialog();  // Hmm, this isn't supposed to be called.
  }
}


void Kst2DPlot::matchAxis(int id) {
  Kst2DPlotPtr p = (Kst2DPlot*)_plotMap[id];
  if (p) {
    double x0, x1, y0, y1;
    p->getScale(x0, y0, x1, y1);
    setLog(p->isXLog(), p->isYLog());
    setXScaleMode(FIXED); //p->getXScaleMode());
    setYScaleMode(FIXED); //p->getYScaleMode());
    setXScale(x0, x1);
    setYScale(y0, y1);
    pushScale();
    setDirty();
    if (_menuView) {
      _menuView->paint();
    }
  }
}


void Kst2DPlot::fitCurve(int id) {
  KstApp *app = KstApp::inst();
  KMdiChildView *c = app->activeWindow();
  KstBaseCurvePtr curve = *(Curves.findTag(_curveEditMap[id]));

  KstFitDialogI::globalInstance()->setCurve(curve->tagName(), tagName(), c->caption());
  KstFitDialogI::globalInstance()->show_I();
}


void Kst2DPlot::removeCurve(int id) {
  KstBaseCurvePtr curve = *(Curves.findTag(_curveRemoveMap[id]));
  if (curve) {
    Curves.remove(curve);
    setDirty();
    if (_menuView) {
      _menuView->paint();
    }
  }
}


bool Kst2DPlot::popupMenu(KPopupMenu *menu, const QPoint& pos, KstViewObjectPtr topLevelParent) {
  bool bHasEntry = false;
  int n_curves = Curves.count();
  int id;
  int i;

  KstTopLevelViewPtr tlv = dynamic_cast<KstTopLevelView*>(topLevelParent.data());
  _menuView = tlv ? tlv->widget() : 0L;
  KstViewObject::popupMenu(menu, pos, topLevelParent);

  KPopupMenu *submenu = new KPopupMenu(menu);
  id = menu->insertItem(i18n("&Match Axis"), submenu);
  Kst2DPlotList pl = globalPlotList();
  i = 0;
  _plotMap.clear();
  for (Kst2DPlotList::Iterator j = pl.begin(); j != pl.end(); ++j) {
    if ((*j).data() != this) {
      _plotMap[i] = *j; // don't think there is any way around this.
                        // We have to hope that it's safe until the menu is
                        // done.
      submenu->insertItem((*j)->tagName(), i);
      submenu->connectItem(i++, this, SLOT(matchAxis(int)));
      bHasEntry = true;
    }
  }
  menu->setItemEnabled(id, bHasEntry);
  bHasEntry = false;

  submenu = new KPopupMenu(menu);
  menu->insertItem(i18n("Z&oom"), submenu);
  submenu->insertItem(i18n("Zoom &Maximum"), this, SLOT(menuZoomMax()), Key_M);
  submenu->insertItem(i18n("Zoom Max &Spike Insensitive"),
                      this, SLOT(menuZoomSpikeInsensitiveMax()), Key_S);
  submenu->insertItem(i18n("Zoom P&revious"), this, SLOT(menuZoomPrev()), Key_R);
  submenu->insertItem(i18n("Y-Zoom Mean-centered"), this, SLOT(menuYZoomAc()), Key_A);
  submenu->insertSeparator();
  submenu->insertItem(i18n("X-Zoom Maximum"),
                        this, SLOT(menuXZoomMax()), CTRL + Key_M);
  submenu->insertItem(i18n("X-Zoom Out"),
                        this, SLOT(menuXZoomOut()), SHIFT + Key_Right);
  submenu->insertItem(i18n("X-Zoom In"),
                        this, SLOT(menuXZoomIn()), SHIFT + Key_Left);
  submenu->insertItem(i18n("Toggle Log X Axis"),
                        this, SLOT(menuXLogSlot()), Key_G);
  submenu->insertSeparator();
  submenu->insertItem(i18n("Y-Zoom Maximum"),
                        this, SLOT(menuYZoomMax()), SHIFT + Key_M);
  submenu->insertItem(i18n("Y-Zoom Out"),
                        this, SLOT(menuYZoomOut()), SHIFT + Key_Up);
  submenu->insertItem(i18n("Y-Zoom In"),
                        this, SLOT(menuYZoomIn()), SHIFT + Key_Down);
  submenu->insertItem(i18n("Toggle Log Y Axis"),
                        this, SLOT(menuYLogSlot()), Key_L);
  submenu = new KPopupMenu(menu);
  menu->insertItem(i18n("&Scroll"), submenu);
  submenu->insertItem(i18n("Left"), this, SLOT(menuMoveLeft()), Key_Left);
  submenu->insertItem(i18n("Right"), this, SLOT(menuMoveRight()), Key_Right);
  submenu->insertItem(i18n("Up"), this, SLOT(menuMoveUp()), Key_Up);
  submenu->insertItem(i18n("Down"), this, SLOT(menuMoveDown()), Key_Down);

  n_curves = Curves.count();
  if (n_curves > 0) {
    menu->insertSeparator();

    _curveEditMap.clear();
    _curveFitMap.clear();
    _curveRemoveMap.clear();

    submenu = new KPopupMenu(menu);
    bHasEntry = false;
    for (i = 0; i < n_curves; i++) {
      QString tag = Curves[i]->tagName();
      _curveEditMap[i] = tag;
      submenu->insertItem(tag, i);
      submenu->connectItem(i, this, SLOT(editCurve(int)));
      bHasEntry = true;
    }

    id = menu->insertItem(i18n("Edit..."), submenu);
    menu->setItemEnabled(id, bHasEntry);

    submenu = new KPopupMenu(menu);

    bHasEntry = false;
    for (i = 0; i < n_curves; i++) {
      QString tag = Curves[i]->tagName();
      _curveFitMap[i] = tag;
      submenu->insertItem(tag, i);
      submenu->connectItem(i, this, SLOT(fitCurve(int)));
      bHasEntry = true;
    }

    id = menu->insertItem(i18n("Fit..."), submenu);
    submenu->setItemEnabled(id, bHasEntry);

    submenu = new KPopupMenu(menu);
    bHasEntry = false;
    for (i = 0; i < n_curves; i++) {
      QString tag = Curves[i]->tagName();
      _curveRemoveMap[i] = tag;
      submenu->insertItem(tag, i);
      submenu->connectItem(i, this, SLOT(removeCurve(int)));
      bHasEntry = true;
    }

    id = menu->insertItem(i18n("Remove"), submenu);
    menu->setItemEnabled(id, bHasEntry);
  }

  return true;
}

bool Kst2DPlot::layoutPopupMenu(KPopupMenu *menu, const QPoint& pos, KstViewObjectPtr topLevelParent) {
  _layoutActions |= Delete | Raise | Lower | RaiseToTop | LowerToBottom | Rename | MoveTo;

  KstViewObject::layoutPopupMenu(menu, pos, topLevelParent);
  return true;
}


KstViewObjectPtr create_Kst2DPlot() {
  return KstViewObjectPtr(new Kst2DPlot);
}


KstViewObjectFactoryMethod Kst2DPlot::factory() const {
  return &create_Kst2DPlot;
}


bool Kst2DPlot::mouseHandler() const {
  return true;
}


void Kst2DPlot::removeFocus(QPainter& p) {
  p.setClipRegion(_lastClipRegion);
  setHasFocus(false);
  updateTieBox(p);
}


void Kst2DPlot::setHasFocus(bool has) {
  _hasFocus = has;
}


void Kst2DPlot::mouseMoveEvent(QWidget *view, QMouseEvent *e) {
  int x, y;
  int iWidth, iHeight;
  int i_label = -1;

  if (!_hasFocus) {
    KstViewWidget *w = dynamic_cast<KstViewWidget*>(view);
    QPainter p(view);
    if (w) {
      w->viewObject()->forEachChild<QPainter&>(&Kst2DPlot::removeFocus, p);
    }
    setHasFocus(true);
    p.setClipRegion(_lastClipRegion);
    updateTieBox(p);
  }

  if (globalZoomType() == LABEL_TOOL) { // HACK
    _mouse.mode = LABEL_TOOL;
  } else if (_mouse.mode == LABEL_TOOL) {
    _mouse.mode = INACTIVE;
  }

  if (e->state() & Qt::LeftButton && _mouse.mode == LABEL_TOOL &&
        (i_label = labelNumber(e)) >= 0 && _draggableLabel == i_label) {
    // Start a drag
    KstLabel *label = labelList.at(i_label);
    if (label) {
#define LABEL_TRANSPARENT
      QRegion oldExtents = label->extents;
      QPixmap pm(GetWinRegion().width(), GetWinRegion().height());
      QRect rectBounding = oldExtents.boundingRect();

#ifdef LABEL_TRANSPARENT
      QBitmap bm(GetWinRegion().width(), GetWinRegion().height(), true);
      { // Scope is needed to kill off painter before we resize
        QPainter p(&bm);
        label->draw(p, 0, label->v_offset, false);
      }
#endif

      pm.fill(_backgroundColor);

      { // Scope is needed to kill off painter before we resize
        QPainter p(&pm);
        label->draw(p, 0, label->v_offset, false);
      }

      //
      // do not allow the pixmap to be increased in size, else we will be
      //  drawing a partially uninitialised pixmap during the drag operation...
      //
      iWidth = rectBounding.width();
      if (GetWinRegion().width() < iWidth) {
        iWidth = GetWinRegion().width();
      }
      iHeight = rectBounding.height();
      if (GetWinRegion().height() < iHeight) {
        iHeight = GetWinRegion().height();
      }

      pm.resize(iWidth, iHeight);
#ifdef LABEL_TRANSPARENT
      bm.resize(iWidth, iHeight);
      pm.setMask(bm);
#endif

      label->extents = oldExtents; // restore them in case the drag is canceled
      QDragObject *d = new KstLabelDrag(view, static_cast<KstViewWindow*>(view->parent())->caption(), this, i_label, _draggablePoint - GetWinRegion().topLeft() - rectBounding.topLeft(), pm);
      d->dragMove();
      _draggableLabel = -2;
      static_cast<KstViewWidget*>(view)->viewObject()->releaseMouse(this);
    }
    return;
  } else if (e->state() & Qt::LeftButton && _mouse.mode == LABEL_TOOL && legendUnder(e)) {
    // Start a drag
    KstLegend *legend = Legend;
    QRegion oldExtents = legend->extents;
    QPixmap pm(GetWinRegion().width(), GetWinRegion().height());
    pm.fill(_backgroundColor);
    QRect rectBounding = oldExtents.boundingRect();

    { // Scope is needed to kill off painter before we resize
      QPainter p(&pm);
      p.setBackgroundColor(_backgroundColor);
      legend->draw(&Curves, p, 0, 0);
    }

    //
    // do not allow the pixmap to be increased in size, else we will be
    //  drawing a partially uninitialised pixmap during the drag operation...
    //
    iWidth = rectBounding.width();
    if (GetWinRegion().width() < iWidth) {
      iWidth = GetWinRegion().width();
    }
    iHeight = rectBounding.height();
    if (GetWinRegion().height() < iHeight) {
      iHeight = GetWinRegion().height();
    }

    pm.resize(iWidth, iHeight);
    legend->extents = oldExtents; // restore them in case the drag is canceled
    QDragObject *d = new KstLegendDrag(view, static_cast<KstViewWindow*>(view->parent())->caption(), this, _draggablePoint - GetWinRegion().topLeft() - rectBounding.topLeft(), pm);
    d->dragMove();
    _draggableLabel = -2;
    static_cast<KstViewWidget*>(view)->viewObject()->releaseMouse(this);
    return;
  }

  KstMouseModeType newType = _mouse.mode;
  QRect pr = GetPlotRegion();
  if (e->state() & Qt::LeftButton && _mouse.zooming()) {
    // LEAVE BLANK
  } else if (KstApp::inst()->dataMode() && pr.contains(e->pos())) {
    double near_x, near_y;
    double newxpos = 0, newypos = 0;
    double d, best_d = 1.0E300;
    int i_near_x;
    KstBaseCurvePtr curve;
    QPoint pos = e->pos();

    double xmin, ymin, xmax, ymax, xpos, ypos, dx_per_pix;
    getLScale(xmin, ymin, xmax, ymax);

    // find mouse location in plot units
    xpos = (double)(pos.x() - pr.left())/(double)pr.width() *
           (xmax - xmin) + xmin;
    if (isXLog()) {
      xpos = pow(10.0, xpos);
    }

    // convert 1 pixel to plot units.
    dx_per_pix = (double)(pos.x()+2 - pr.left()) / (double)pr.width() *
                 (xmax - xmin) + xmin;
    if (isXLog()) {
      dx_per_pix = pow(10.0, dx_per_pix);
    }
    dx_per_pix -= xpos;

    ypos = (double)(pos.y() - pr.top())/(double)pr.height();
    ypos = ypos * (ymin - ymax) + ymax;

    if (isYLog()) {
      ypos = pow(10.0, ypos);
    }

    for (KstBaseCurveList::Iterator i = Curves.begin(); i != Curves.end(); ++i) {
      i_near_x = (*i)->getIndexNearXY(xpos, dx_per_pix, ypos);
      (*i)->getPoint(i_near_x, near_x, near_y);
      d = fabs(ypos - near_y);
      if (d < best_d) {
        newypos = near_y;
        newxpos = near_x;
        best_d = d;
        curve = *i;
      }
    }

    if (curve.data()) {
      QString msg = i18n("%3 (%1, %2)").arg(newxpos).arg(newypos,0,'G').arg(curve->tagName());
      if (_copy_x != newxpos || _copy_y != newypos) {
        _copy_x = newxpos;
        _copy_y = newypos;

        static_cast<KstViewWidget*>(view)->paint();
        QPainter p(view);
        p.setClipRegion(_lastClipRegion);
        drawDotAt(p, newxpos, newypos);
        KstApp::inst()->slotUpdateDataMsg(msg);
      }
    } else {
      _copy_x = KST::NOPOINT;
      _copy_y = KST::NOPOINT;
      static_cast<KstViewWidget*>(view)->paint();
      KstApp::inst()->slotUpdateDataMsg(QString::null);
    }
  } else if (pr.contains(e->pos())) {
    QPoint pos = e->pos();

    double xmin, ymin, xmax, ymax, xpos, ypos;
    getLScale(xmin, ymin, xmax, ymax);

    // find mouse location in plot units
    xpos = (double)(pos.x() - pr.left())/(double)pr.width() *
           (xmax - xmin) + xmin;
    if (isXLog()) {
      xpos = pow(10.0, xpos);
    }

    ypos = (double)(pos.y() - pr.top())/(double)pr.height();
    ypos = ypos * (ymin - ymax) + ymax;

    if (isYLog()) {
      ypos = pow(10.0, ypos);
    }
    _copy_x = xpos;
    _copy_y = ypos;
    KstApp::inst()->slotUpdateDataMsg(i18n("(%1, %2)").arg(xpos,0,'G').arg(ypos,0,'G'));
  } else {
    KstApp::inst()->slotUpdateDataMsg(QString::null);
  }

  if (_mouse.mode == XY_ZOOMBOX) {
    if (e->x() > pr.right()) {
      x = pr.right() + 1;
    } else if (e->x() < pr.left()) {
      x = pr.left();
    } else {
      x = e->x();
    }

    if (e->y() > pr.bottom()) {
      y = pr.bottom() + 1;
    } else if (e->y() < pr.top()) {
      y = pr.top();
    } else {
      y = e->y();
    }

    zoomRectUpdate(view, newType, x, y);
    setCursorForMode(view);
  } else if (_mouse.mode == Y_ZOOMBOX) {
    x = pr.right();

    if (e->y() > pr.bottom()) {
      y = pr.bottom() + 1;
    } else if (e->y() < pr.top()) {
      y = pr.top();
    } else {
      y = e->y();
    }

    zoomRectUpdate(view, newType, x, y);
    setCursorForMode(view);
  } else if (_mouse.mode == X_ZOOMBOX) {
    if (e->x() > pr.right()) {
      x = pr.right() + 1;
    } else if (e->x() < pr.left()) {
      x = pr.left();
    } else {
      x = e->x();
    }

    y = pr.bottom();
    zoomRectUpdate(view, newType, x, y);
    setCursorForMode(view);
  } else if (_mouse.mode == LABEL_TOOL) {
    if (labelNumber(e) < 0) {
      setCursorForMode(view, LABEL_TOOL);
    } else {
      view->setCursor(QCursor(Qt::ArrowCursor));
    }
  } else {
    ButtonState s = e->stateAfter();
    if (pr.contains(e->pos())) {
      if (s & Qt::ShiftButton) {
        setCursorForMode(view, Y_ZOOMBOX);
      } else if (s & Qt::ControlButton) {
        setCursorForMode(view, X_ZOOMBOX);
      } else {
        setCursorForMode(view, globalZoomType());
      }
    } else {
      view->setCursor(QCursor(Qt::ArrowCursor));
    }
  }
}


bool Kst2DPlot::legendUnder(QMouseEvent *e) {
  QPoint pt = GetWinRegion().topLeft();
  QPoint ptCheck(e->x() - pt.x(), e->y() - pt.y());
  return Legend && Legend->extents.contains(ptCheck);
}


int Kst2DPlot::labelNumber(QMouseEvent *e) {
  int i_label = -1;
  uint cnt = labelList.count();
  QPoint pt = GetWinRegion().topLeft();
  QPoint ptCheck(e->x() - pt.x(), e->y() - pt.y());
  for (uint i = 0; i < cnt; ++i) {
    if (labelList.at(i)->extents.contains(ptCheck)) {
      i_label = i;
      break;
    }
  }
  return i_label;
}


void Kst2DPlot::mousePressEvent(QWidget *view, QMouseEvent *e) {
  QRect win_rect, plot_rect, tie_rect, plot_and_axis_rect;
  KstApp *ParentApp = KstApp::inst();

  static_cast<KstViewWidget*>(view)->viewObject()->grabMouse(this);

  /* Find where the mouse was to determine which mode to be in */
  /* which button */
  if (e->button() == Qt::LeftButton) {
    _draggableLabel = labelNumber(e);
    _draggablePoint = e->pos();

    win_rect = GetWinRegion();
    plot_rect = GetPlotRegion();
    tie_rect = GetTieBoxRegion();
    plot_and_axis_rect = GetPlotAndAxisRegion();
    //kdDebug() << e->pos() << " " << win_rect << " " << plot_rect << endl;
    if (tie_rect.contains(e->pos())) {
      toggleTied();
      // So inefficient, but I have some sort of weird bug making it necessary
      static_cast<KstViewWidget*>(view)->paint();
      return;
    } else if (plot_rect.contains(e->pos())) {
      if (_mouse.mode == LABEL_TOOL) {
      } else {
        if (e->state() & Qt::ShiftButton) {
          _mouse.mode = Y_ZOOMBOX;
        } else if (e->state() & Qt::ControlButton) {
          _mouse.mode = X_ZOOMBOX;
        } else {
          _mouse.mode = globalZoomType();
          assert(_mouse.mode != INACTIVE);
        }
        if (_mouse.mode != LABEL_TOOL) {
          _mouse.plotGeometry = GetPlotRegion();
          _mouse.zoomStart(_mouse.mode, e->pos());
          _zoomPaused = true;
        }
      }
      return;
    } else if (plot_and_axis_rect.contains(e->pos())) {
      ParentApp->plotDialog()->show_I(static_cast<KstViewWidget*>(view)->viewObject()->tagName(), tagName());
      ParentApp->plotDialog()->TabWidget->setCurrentPage(LIMITS_TAB);
      return;
    } else if (win_rect.contains(e->pos())) {
      ParentApp->plotDialog()->show_I(static_cast<KstViewWidget*>(view)->viewObject()->tagName(), tagName());
      ParentApp->plotDialog()->TabWidget->setCurrentPage(LABELS_TAB);
      return;
    }
  } else if (e->button() == Qt::RightButton) {
    win_rect = GetPlotRegion();
    if (win_rect.contains(e->pos())) {
      _mouse.mode = INACTIVE;
      _mouse.pressLocation = e->pos();
      return;
    }
  } else if (e->button() == Qt::MidButton) {
    win_rect = GetPlotRegion();
    if (win_rect.contains(e->pos())) {
      _mouse.mode = INACTIVE;
      _mouse.pressLocation = e->pos();
      zoomPrev(static_cast<KstViewWidget *>(view));
      return;
    }
    return;
  } else {
    // cout << "unknown button: " << e->button() << "\n";
  }
}


void Kst2DPlot::mouseReleaseEvent(QWidget *view, QMouseEvent *e) {
  double xmin, xmax, ymin, ymax;
  double new_xmin, new_xmax, new_ymin, new_ymax;
  QRect plotregion;
  bool doUpdate = false;

  _zoomPaused = false;
  static_cast<KstViewWidget*>(view)->viewObject()->releaseMouse(this);

  QRect newg = _mouse.mouseRect();
  if (_mouse.mode == XY_ZOOMBOX) {
    if (_mouse.rectBigEnough()) {
      QPainter p(view);
      p.setClipRegion(_lastClipRegion);
      p.setRasterOp(Qt::NotROP);
      p.drawWinFocusRect(newg);

      getLScale(xmin, ymin, xmax, ymax);
      plotregion = GetPlotRegion();
      new_xmin = (double)(newg.left() - plotregion.left())/
                 (double)plotregion.width() * (xmax - xmin) + xmin;
      new_xmax = (double)(newg.right() - plotregion.left() + 1) /
                 (double)plotregion.width() * (xmax - xmin) + xmin;
      new_ymin = (double)(newg.bottom() - plotregion.top() + 1) /
                 (double)plotregion.height() * (ymin - ymax) + ymax;
      new_ymax = (double)(newg.top() - plotregion.top())/
                 (double)plotregion.height() * (ymin - ymax) + ymax;

      setXScaleMode(FIXED);
      setYScaleMode(FIXED);
      setLScale(new_xmin, new_ymin, new_xmax, new_ymax);
      pushScale();
      doUpdate = true;
      _mouse.lastLocation = _mouse.pressLocation;
      if (isTied()) {
        Kst2DPlotList pl = static_cast<KstViewWidget*>(view)->viewObject()->findChildrenType<Kst2DPlot>(true);
        for (Kst2DPlotList::Iterator i = pl.begin(); i != pl.end(); ++i) {
          Kst2DPlotPtr p = *i;
          if (p->isTied() && p.data() != this) {
            p->setXScaleMode(FIXED);
            p->setYScaleMode(FIXED);
            p->setLScale(new_xmin, new_ymin, new_xmax, new_ymax);
            p->pushScale();
            p->update();
          }
        }
      }
    }
  } else if (_mouse.mode == Y_ZOOMBOX) {
    if (newg.height() >= _mouse.minMove) {
      QPainter p(view);
      p.setClipRegion(_lastClipRegion);
      p.setRasterOp(Qt::NotROP);
      p.drawWinFocusRect(newg);

      getLScale(xmin, ymin, xmax, ymax);
      plotregion = GetPlotRegion();
      new_ymin = (double)(newg.bottom() - plotregion.top() + 1) /
                 (double)plotregion.height() * (ymin - ymax) + ymax;
      new_ymax = (double)(newg.top() - plotregion.top()) /
                 (double)plotregion.height() * (ymin - ymax) + ymax;

      setYScaleMode(FIXED);
      setLYScale(new_ymin, new_ymax);
      pushScale();
      doUpdate = true;
      _mouse.lastLocation = _mouse.pressLocation;
      if (isTied()) {
        Kst2DPlotList pl = static_cast<KstViewWidget*>(view)->viewObject()->findChildrenType<Kst2DPlot>(true);
        for (Kst2DPlotList::Iterator i = pl.begin(); i != pl.end(); ++i) {
          Kst2DPlotPtr p = *i;
          if (p->isTied() && p.data() != this) {
            p->setYScaleMode(FIXED);
            p->setLYScale(new_ymin, new_ymax);
            p->pushScale();
            p->update();
          }
        }
      }
    }
  } else if (_mouse.mode == X_ZOOMBOX) {
    if (newg.width() >= _mouse.minMove) {
      QPainter p(view);
      p.setClipRegion(_lastClipRegion);
      p.setRasterOp(Qt::NotROP);
      p.drawWinFocusRect(newg);

      getLScale(xmin, ymin, xmax, ymax);
      plotregion = GetPlotRegion();
      new_xmin = (double)(newg.left() - plotregion.left()) /
                 (double)plotregion.width() * (xmax - xmin) + xmin;
      new_xmax = (double)(newg.right() -
                          plotregion.left() + 1) /
                 (double)plotregion.width() * (xmax - xmin) + xmin;

      setXScaleMode(FIXED);
      setLXScale(new_xmin, new_xmax);
      pushScale();
      doUpdate = true;
      _mouse.lastLocation = _mouse.pressLocation;
      if (isTied()) {
        Kst2DPlotList pl = static_cast<KstViewWidget*>(view)->viewObject()->findChildrenType<Kst2DPlot>(true);
        for (Kst2DPlotList::Iterator i = pl.begin(); i != pl.end(); ++i) {
          Kst2DPlotPtr p = *i;
          if (p->isTied() && p.data() != this) {
            p->setXScaleMode(FIXED);
            p->setLXScale(new_xmin, new_xmax);
            p->pushScale();
            p->update();
          }
        }
      }
    }

  } else if (_mouse.mode == LABEL_TOOL) {
    plotregion = GetPlotRegion();

    double x = double(e->x() - plotregion.left()) / double(plotregion.width());
    double y = double(e->y() - plotregion.top()) / double(plotregion.height());
    int i_label;

    if ((i_label = labelNumber(e)) >= 0 && _draggableLabel == i_label)  {
      KstApp::inst()->labelDialog()->showI(this, i_label, x, y);
    } else if ( plotregion.contains(e->pos()) && _draggableLabel == -1) {
      KstApp::inst()->labelDialog()->showI(this, -1, x, y);
    }

    return; // no need to update, and we don't want to set INACTIVE
  }

  _mouse.mode = INACTIVE;
  //needrecreate = true;
  if (doUpdate) {
    setDirty();
    static_cast<KstViewWidget*>(view)->paint();
  }
}


void KstMouse::zoomUpdate(KstMouseModeType t, const QPoint& location) {
  mode = t;
  lastLocation = location;
}


void KstMouse::zoomStart(KstMouseModeType t, const QPoint& location) {
  mode = t;
  pressLocation = lastLocation = location;
}


bool KstMouse::rectBigEnough() const {
  QRect r = mouseRect();
  return r.width() >= minMove && r.height() >= minMove;
}


QRect KstMouse::mouseRect() const {
  QRect rc;
  rc = QRect(QMIN(pressLocation.x(), lastLocation.x()), QMIN(pressLocation.y(), lastLocation.y()), QABS(pressLocation.x() - lastLocation.x()), QABS(pressLocation.y() - lastLocation.y()));
  switch (mode) {
    case X_ZOOMBOX:
        rc.setTop(plotGeometry.top());
        rc.setBottom(plotGeometry.bottom());
      break;
    case Y_ZOOMBOX:
        rc.setLeft(plotGeometry.left());
        rc.setRight(plotGeometry.right());
      break;
    default:
      break;
  }
  return rc;
}


bool KstMouse::zooming() const {
  return mode == XY_ZOOMBOX || mode == X_ZOOMBOX || mode == Y_ZOOMBOX;
}


void Kst2DPlot::zoomRectUpdate(QWidget *view, KstMouseModeType t, int x, int y) {
    QPoint newp(x, y);

    if (_mouse.lastLocation != newp) {
      QPainter p(view);
      p.setClipRegion(_lastClipRegion);
      p.setRasterOp(Qt::NotROP);
      if (_mouse.rectBigEnough()) {
        p.drawWinFocusRect(_mouse.mouseRect());
      }
      _mouse.zoomUpdate(t, newp);
      if (_mouse.rectBigEnough()) {
        p.drawWinFocusRect(_mouse.mouseRect());
      }
    }
}


void Kst2DPlot::setCursorForMode(QWidget *view) {
  setCursorForMode(view, _mouse.mode);
}


void Kst2DPlot::setCursorForMode(QWidget *view, KstMouseModeType mode) {
  switch (mode) {
    case Y_ZOOMBOX:
      view->setCursor(QCursor(Qt::SizeVerCursor));
      break;
    case X_ZOOMBOX:
      view->setCursor(QCursor(Qt::SizeHorCursor));
      break;
    case XY_ZOOMBOX:
      view->setCursor(QCursor(Qt::CrossCursor));
      break;
    case LABEL_TOOL:
      view->setCursor(QCursor(Qt::IbeamCursor));
      break;
    default:
      if (GetPlotRegion().contains(view->mapFromGlobal(QCursor::pos()))) {
        view->setCursor(QCursor(Qt::CrossCursor));
      } else {
        view->setCursor(QCursor(Qt::ArrowCursor));
      }
      break;
  }
}


void Kst2DPlot::keyReleaseEvent(QWidget *view, QKeyEvent *e) {
  if (_mouse.mode != INACTIVE) {
    e->ignore();
    return;
  }

  KstMouseModeType newType = globalZoomType();
  QPoint c = view->mapFromGlobal(QCursor::pos());
  QRect pr = GetPlotRegion();
  int x = _mouse.pressLocation.x(), y = _mouse.pressLocation.y();

  if (newType == Y_ZOOMBOX) {
    if (c.y() > pr.bottom()) {
      y = pr.bottom() + 1;
    } else if (c.y() < pr.top()) {
      y = pr.top();
    } else {
      y = c.y();
    }
  } else if (newType == X_ZOOMBOX) {
    if (c.x() > pr.right()) {
      x = pr.right() + 1;
    } else if (c.x() < pr.left()) {
      x = pr.left();
    } else {
      x = c.x();
    }
  } else {
    if (c.x() > pr.right()) {
      x = pr.right() + 1;
    } else if (c.x() < pr.left()) {
      x = pr.left();
    } else {
      x = c.x();
    }

    if (c.y() > pr.bottom()) {
      y = pr.bottom() + 1;
    } else if (c.y() < pr.top()) {
      y = pr.top();
    } else {
      y = c.y();
    }
  }

  if (_mouse.zooming() || _mouse.mode == INACTIVE) {
    if (_mouse.mode != INACTIVE) {
      QPoint newp(x, y);
      QPainter p(view);
      p.setClipRegion(_lastClipRegion);
      p.setRasterOp(Qt::NotROP);
      if (_mouse.rectBigEnough()) {
        p.drawWinFocusRect(_mouse.mouseRect());
      }

      _mouse.zoomUpdate(newType, newp);
      if (_mouse.rectBigEnough()) {
        p.drawWinFocusRect(_mouse.mouseRect());
      }
    }
  }
  setCursorForMode(view, newType);
  e->accept();
}


void Kst2DPlot::cancelZoom(QWidget *view) {
  if (_mouse.rectBigEnough()) {
    QPainter p(view);
    p.setClipRegion(_lastClipRegion);
    p.setRasterOp(Qt::NotROP);
    p.drawWinFocusRect(_mouse.mouseRect());
  }

  _mouse.lastLocation = _mouse.pressLocation; // make rectBigEnough() false
  _mouse.mode = INACTIVE;
}


void Kst2DPlot::menuXZoomMax() {
  if (_menuView) {
    xZoomMax(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuYZoomMax() {
  if (_menuView) {
    yZoomMax(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuZoomMax() {
  if (_menuView) {
    zoomMax(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuXLogSlot() {
  if (_menuView) {
    xLogSlot(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuYLogSlot() {
  if (_menuView) {
    yLogSlot(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuZoomPrev() {
  if (_menuView) {
    zoomPrev(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuYZoomAc() {
  if (_menuView) {
    yZoomAc(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuZoomSpikeInsensitiveMax() {
  if (_menuView) {
    zoomSpikeInsensitiveMax(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuXZoomIn() {
  if (_menuView) {
    xZoomIn(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuXZoomOut() {
  if (_menuView) {
    xZoomOut(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuYZoomIn() {
  if (_menuView) {
    yZoomIn(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuYZoomOut() {
  if (_menuView) {
    yZoomOut(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuMoveUp() {
  if (_menuView) {
    moveUp(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuMoveDown() {
  if (_menuView) {
    moveDown(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuMoveLeft() {
  if (_menuView) {
    moveLeft(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::menuMoveRight() {
  if (_menuView) {
    moveRight(_menuView);
    _menuView->paint();
  }
}


void Kst2DPlot::moveLeft(KstViewWidget *view) {
  Q_UNUSED(view)
  moveSelfHorizontal(true);
  updateTiedPlots(&Kst2DPlot::moveSelfHorizontal, true);
  pushScale();
  setDirty();
}


void Kst2DPlot::moveRight(KstViewWidget *view) {
  Q_UNUSED(view)
  moveSelfHorizontal(false);
  updateTiedPlots(&Kst2DPlot::moveSelfHorizontal, false);
  pushScale();
  setDirty();
}


void Kst2DPlot::moveUp(KstViewWidget *view) {
  Q_UNUSED(view)
  moveSelfVertical(true);
  updateTiedPlots(&Kst2DPlot::moveSelfVertical, true);
  pushScale();
  setDirty();
  view->paint();
}


void Kst2DPlot::moveDown(KstViewWidget *view) {
  Q_UNUSED(view)
  moveSelfVertical(false);
  updateTiedPlots(&Kst2DPlot::moveSelfVertical, false);
  pushScale();
  setDirty();
  view->paint();
}


void Kst2DPlot::moveSelfHorizontal(bool left) {
  double xmin, xmax, ymin, ymax;
  double new_xmin, new_xmax;

  getLScale(xmin, ymin, xmax, ymax);

  if (left) {
    new_xmin = xmin - (xmax - xmin)*0.1;
    new_xmax = xmax - (xmax - xmin)*0.1;
  } else {
    new_xmin = xmin + (xmax - xmin)*0.1;
    new_xmax = xmax + (xmax - xmin)*0.1;
  }

  setXScaleMode(FIXED);
  setLXScale(new_xmin, new_xmax);
}


void Kst2DPlot::moveSelfVertical(bool up) {
  double xmin, xmax, ymin, ymax;
  double new_ymin, new_ymax;

  getLScale(xmin, ymin, xmax, ymax);

  if (up) {
    new_ymin = ymin + (ymax - ymin)*0.25;
    new_ymax = ymax + (ymax - ymin)*0.25;
  } else {
    new_ymin = ymin - (ymax - ymin)*0.25;
    new_ymax = ymax - (ymax - ymin)*0.25;
  }

  if (globalZoomType() == X_ZOOMBOX) {
    getLScale(xmin, ymin, xmax, ymax);
    if (up) {
      new_ymin = ymin + (ymax - ymin)*0.25;
      new_ymax = ymax + (ymax - ymin)*0.25;
    } else {
      new_ymin = ymin - (ymax - ymin)*0.25;
      new_ymax = ymax - (ymax - ymin)*0.25;
    }
  }

  setYScaleMode(FIXED);
  setLYScale(new_ymin, new_ymax);
}


void Kst2DPlot::zoomSelfVertical(bool in) {
  double xmin, xmax, ymin, ymax;
  double new_ymin, new_ymax;

  getLScale(xmin, ymin, xmax, ymax);

  if (in) {
    new_ymin = ymin + (ymax - ymin)*0.1666666;
    new_ymax = ymax - (ymax - ymin)*0.1666666;
  } else {
    new_ymin = ymin - (ymax - ymin)*0.25;
    new_ymax = ymax + (ymax - ymin)*0.25;
  }

  setYScaleMode(FIXED);
  setLYScale(new_ymin, new_ymax);
}


void Kst2DPlot::yZoomIn(KstViewWidget *view) {
  Q_UNUSED(view)
  zoomSelfVertical(true);
  updateTiedPlots(&Kst2DPlot::zoomSelfVertical, true);
  pushScale();
  setDirty();
}


void Kst2DPlot::yZoomOut(KstViewWidget *view) {
  Q_UNUSED(view)
  zoomSelfVertical(false);
  updateTiedPlots(&Kst2DPlot::zoomSelfVertical, false);
  pushScale();
  setDirty();
}


void Kst2DPlot::zoomSelfHorizontal(bool in) {
  double xmin, xmax, ymin, ymax;
  double new_xmin, new_xmax;

  getLScale(xmin, ymin, xmax, ymax);

  if (in) {
    new_xmin = xmin + (xmax - xmin)*0.1666666;
    new_xmax = xmax - (xmax - xmin)*0.1666666;
  } else {
    new_xmin = xmin - (xmax - xmin)*0.25;
    new_xmax = xmax + (xmax - xmin)*0.25;
  }

  setXScaleMode(FIXED);
  setLXScale(new_xmin, new_xmax);
}


void Kst2DPlot::xZoomIn(KstViewWidget *view) {
  Q_UNUSED(view)
  zoomSelfHorizontal(true);
  updateTiedPlots(&Kst2DPlot::zoomSelfHorizontal, true);
  pushScale();
  setDirty();
}


void Kst2DPlot::xZoomOut(KstViewWidget *view) {
  Q_UNUSED(view)
  zoomSelfHorizontal(false);
  updateTiedPlots(&Kst2DPlot::zoomSelfHorizontal, false);
  pushScale();
  setDirty();
}


void Kst2DPlot::xZoomMax(KstViewWidget *view) {
  Q_UNUSED(view)
  setXScaleMode(AUTO);
  updateTiedPlots(&Kst2DPlot::setXScaleMode, AUTO);
  pushScale();
  setDirty();
}


void Kst2DPlot::yZoomMax(KstViewWidget *view) {
  Q_UNUSED(view)
  setYScaleMode(AUTO);
  updateTiedPlots(&Kst2DPlot::setYScaleMode, AUTO);
  pushScale();
  setDirty();
}


void Kst2DPlot::zoomMax(KstViewWidget *view) {
  Q_UNUSED(view)
  setXScaleMode(AUTO);
  setYScaleMode(AUTO);
  // Hmm could make a method that does both at once.
  updateTiedPlots(&Kst2DPlot::setXScaleMode, AUTO);
  updateTiedPlots(&Kst2DPlot::setYScaleMode, AUTO);
  pushScale();
  setDirty();
}


void Kst2DPlot::xLogSlot(KstViewWidget *view) {
  Q_UNUSED(view)
  setLog(!isXLog(), isYLog());
  setDirty();
}


void Kst2DPlot::yLogSlot(KstViewWidget *view) {
  Q_UNUSED(view)
  setLog(isXLog(), !isYLog());
  setDirty();
}


void Kst2DPlot::zoomPrev(KstViewWidget *view) {
  if (popScale()) {
    if (isTied()) {
      Kst2DPlotList pl = view->viewObject()->findChildrenType<Kst2DPlot>(true);
      for (Kst2DPlotList::Iterator i = pl.begin(); i != pl.end(); ++i) {
        Kst2DPlotPtr p = *i;
        if (p->isTied() && p.data() != this) {
          p->popScale();
          p->cancelZoom(view);
          p->update();
        }
      }
    }
    setDirty();
  }
}


void Kst2DPlot::yZoomAc(KstViewWidget *view) {
  Q_UNUSED(view)
  setYScaleMode(AC);
  pushScale();
  updateTiedPlots(&Kst2DPlot::setYScaleMode, AC);
  setDirty();
}


void Kst2DPlot::zoomSpikeInsensitiveMax(KstViewWidget *view) {
  Q_UNUSED(view)
  setXScaleMode(NOSPIKE);
  setYScaleMode(NOSPIKE);
  updateTiedPlots(&Kst2DPlot::setXScaleMode, NOSPIKE);
  updateTiedPlots(&Kst2DPlot::setYScaleMode, NOSPIKE);
  pushScale();
  setDirty();
}


void Kst2DPlot::keyPressEvent(QWidget *vw, QKeyEvent *e) {
  if (_mouse.mode == LABEL_TOOL) {
    e->ignore();
    return;
  }

  bool handled = true;
  ButtonState s = e->stateAfter();
  KstViewWidget *view = static_cast<KstViewWidget*>(vw);
  switch (e->key()) {
    case Key_A:
      yZoomAc(view);
      break;
    case Key_G:
      xLogSlot(view);
      break;
    case Key_L:
      yLogSlot(view);
      break;
    case Key_M:
      if (s & Qt::ShiftButton) {
        yZoomMax(view);
      } else if (s & Qt::ControlButton) {
        xZoomMax(view);
      } else {
        zoomMax(view);
      }
      break;
    case Key_P:
      pauseToggle();
      setDirty();
      break;
    case Key_R:
      zoomPrev(view);
      break;
    case Key_S:
      zoomSpikeInsensitiveMax(view);
      break;
    case Key_Z:
      zoomToggle();
      cancelZoom(view);
      setDirty();
      break;
    case Key_Left:
      if (s & Qt::ShiftButton) {
        xZoomIn(view);
      } else {
        moveLeft(view);
      }
      break;
    case Key_Right:
      if (s & Qt::ShiftButton) {
        xZoomOut(view);
      } else {
        moveRight(view);
      }
      break;
    case Key_Up:
      if (s & Qt::ShiftButton) {
        yZoomOut(view);
      } else {
        moveUp(view);
      }
      break;
    case Key_Down:
      if (s & Qt::ShiftButton) {
        yZoomIn(view);
      } else {
        moveDown(view);
      }
      break;
    default:
      handled = false;
      break;
  }

  if (handled) {
    view->paint();
    e->accept();
    return;
  }

  if (_mouse.zooming()) {
    KstMouseModeType newType = _mouse.mode;

    if (e->key() == Qt::Key_Escape) {
      cancelZoom(view);
    } else {
      QPoint newp = _mouse.lastLocation;

      QPainter p(view);
      p.setClipRegion(_lastClipRegion);
      p.setRasterOp(Qt::NotROP);
      if (_mouse.rectBigEnough()) {
        p.drawWinFocusRect(_mouse.mouseRect());
      }

      _mouse.zoomUpdate(newType, newp);
      if (_mouse.rectBigEnough()) {
        p.drawWinFocusRect(_mouse.mouseRect());
      }
    }
    setCursorForMode(view);
  } else {
    if (_mouse.mode == INACTIVE && GetPlotRegion().contains(view->mapFromGlobal(QCursor::pos()))) {
      if (s & Qt::ShiftButton) {
        setCursorForMode(view, Y_ZOOMBOX);
      } else if (s & Qt::ControlButton) {
        setCursorForMode(view, X_ZOOMBOX);
      } else {
        setCursorForMode(view, globalZoomType());
      }
    } else {
      e->ignore();
      return;
    }
  }
  e->accept();
}


KstMouseModeType Kst2DPlot::globalZoomType() const {
  switch (KstApp::inst()->getZoomRadio()) {
  case KstApp::XZOOM:
    return X_ZOOMBOX;
  case KstApp::YZOOM:
    return Y_ZOOMBOX;
  case KstApp::XYZOOM:
    return XY_ZOOMBOX;
  case KstApp::TEXT:
    return LABEL_TOOL;
  case KstApp::LAYOUT:
    return LAYOUT_TOOL;
  default:
    break;
  }

  return INACTIVE;
}


void Kst2DPlot::dragEnterEvent(QWidget *view, QDragEnterEvent *e) {
  _draggableLabel = -2;

  e->accept( e->provides("application/x-kst-label-number") ||
            (e->source() == view && e->provides("application/x-kst-legend")));
}


void Kst2DPlot::dragMoveEvent(QWidget *view, QDragMoveEvent *e) {
  _draggableLabel = -2;

  e->accept( e->provides("application/x-kst-label-number") ||
            (e->source() == view && e->provides("application/x-kst-legend")));
}


void Kst2DPlot::dropEvent(QWidget *view, QDropEvent *e) {
  QString windowname, plotname;
  QPoint hs;
  bool bAccept = false;

  if (e->provides("application/x-kst-label-number")) {
    QDataStream ds(e->encodedData("application/x-kst-label-number"), IO_ReadOnly);
    int label;

    ds >> windowname >> plotname >> label >> hs;

    if (GetPlotRegion().contains(e->pos())) {
      KstApp *app = KstApp::inst();
      KstViewWindow *w = dynamic_cast<KstViewWindow*>(app->findWindow(windowname));
      if (w) {
        Kst2DPlotPtr p = dynamic_cast<Kst2DPlot*>(w->view()->findChild(plotname).data());

        if (p) {
          KstLabel *l = p->labelList.take(label);
          if (l) {
            p->update();
            labelList.append(l);

            QPoint pos = e->pos() - GetWinRegion().topLeft() - hs;
            QRect rectBounding = l->extents.boundingRect();
            pos.setX(pos.x() - rectBounding.left());
            pos.setY(pos.y() - rectBounding.top());
            QSize divisor = GetPlotRegion().size();
            l->offsetRelPosition(float(pos.x())/divisor.width(), float(pos.y())/divisor.height());
            e->accept(true);
            bAccept = true;

            if (!_hasFocus) {
              QPainter p(view);
              w->view()->forEachChild<QPainter&>(&Kst2DPlot::removeFocus, p);
              setHasFocus(true);
            }

            if (e->source() != view) {
              static_cast<KstViewWidget*>(e->source())->paint();
            }

            setDirty();
          }
          static_cast<KstViewWidget*>(view)->paint();
        }
      }
    }
  } else if (e->provides("application/x-kst-legend")) {
    if (e->source() == view) {
      QDataStream ds(e->encodedData("application/x-kst-legend"), IO_ReadOnly);

      ds >> windowname >> plotname >> hs;

      if (GetPlotRegion().contains(e->pos())) {
        KstApp *app = KstApp::inst();
        KstViewWindow *w = dynamic_cast<KstViewWindow*>(app->findWindow(windowname));
        if (w) {
          Kst2DPlotPtr p = dynamic_cast<Kst2DPlot*>(w->view()->findChild(plotname).data());

          if (p && p.data() == this) {
            KstLegend *l = Legend;

            QPoint pos = e->pos() - GetWinRegion().topLeft() - hs;
            QRect rectBounding = l->extents.boundingRect();
            pos.setX(pos.x() - rectBounding.left());
            pos.setY(pos.y() - rectBounding.top());
            QSize divisor = GetPlotRegion().size();
            l->offsetRelPosition(float(pos.x())/divisor.width(), float(pos.y())/divisor.height());
            e->accept(true);
            bAccept = true;

            if (!_hasFocus) {
              QPainter p(view);
              w->view()->forEachChild<QPainter&>(&Kst2DPlot::removeFocus, p);
              setHasFocus(true);
            }
            setDirty();
            static_cast<KstViewWidget*>(view)->paint();
          }
        }
      }
    }
  }

  if (!bAccept) {
    e->accept(false);
  }
}


void Kst2DPlot::copy() {
  // Don't set the selection because while it does make sense, it
  // is far too easy to swipe over Kst while trying to paste a selection
  // from one window to another.

  // FIXME: we should also provide a custom mime source so that we can
  //        actually manipulate points of data within Kst.
  QString msg = i18n("(%1, %2)").arg(_copy_x, 0, 'G').arg(_copy_y, 0, 'G');
  QApplication::clipboard()->setText(msg);
}


// FIXME: this does not strip out dupes.  Why?  Because we can't have them
//        by definition at present.  this might change one day.  Unfortunately
//        removal of dupes makes O(n) code become O(n^2).
Kst2DPlotList Kst2DPlot::globalPlotList() {
  Kst2DPlotList rc;
  KstApp *app = KstApp::inst();
  KMdiIterator<KMdiChildView*> *pIterator = app->createIterator();
  if (pIterator) {
    while (pIterator->currentItem()) {
      KstViewWindow *pView = dynamic_cast<KstViewWindow*>(pIterator->currentItem());
      if (pView) {
        KstTopLevelViewPtr tlv = pView->view();
        Kst2DPlotList sub = tlv->findChildrenType<Kst2DPlot>(true);
        rc += sub;
      }
      pIterator->next();
    }
    app->deleteIterator(pIterator);
  }
  return rc;
}


void Kst2DPlot::setDirty() {
  _dirty = true;
}


#ifndef WHEEL_DELTA
#define WHEEL_DELTA 120
#endif

void Kst2DPlot::wheelEvent(QWidget *view, QWheelEvent *e) {
  KstViewWidget *vw = dynamic_cast<KstViewWidget*>(view);
  if (!vw || !GetPlotRegion().contains(e->pos())) {
    return;
  }

  bool forward = e->delta() >= 0;
  int absDelta = forward ? e->delta() : -e->delta();
  bool alt = e->state() & Qt::AltButton;
  if (e->state() & Qt::ControlButton) {
    // Advance?  Something else?
  } else if (e->state() & Qt::ShiftButton) {
    for (int i = 0; i < absDelta/WHEEL_DELTA; ++i) {
      if (forward) {
        if (alt) {
          yZoomIn(vw);
        } else {
          xZoomIn(vw);
        }
      } else {
        if (alt) {
          yZoomOut(vw);
        } else {
          xZoomOut(vw);
        }
      }
    }
    vw->paint();
  } else {
    for (int i = 0; i < absDelta/WHEEL_DELTA; ++i) {
      if (forward) {
        if (alt) {
          moveUp(vw);
        } else {
          moveRight(vw);
        }
      } else {
        if (alt) {
          moveDown(vw);
        } else {
          moveLeft(vw);
        }
      }
    }
    vw->paint();
  }

  e->accept();
}

#include "kst2dplot.moc"
// vim: ts=2 sw=2 et
