#include <klistview.h>
#include <kiconview.h>
#include <kstatusbar.h>
#include <kprogress.h>
#include <kcombobox.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kurl.h>
#include <kapp.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kstandarddirs.h>
#include <ktrader.h>
#include <kservice.h>
#include <krun.h>
#include <kurl.h>

#include <qwidgetstack.h>
#include <qstringlist.h>
#include <qstring.h>
#include <qsplitter.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qdir.h>
#include <qfile.h>
#include <qevent.h>
#include <qpopmenu.h>
#include <qvaluevector.h>

#include "camera_browser.h"
#include "camera_folder.h"
#include "camera_iconview.h"
#include "camera_iconviewitem.h"
#include "gphoto_controller.h"
#include "savefile_dialog.h"
#include "gphoto_events.h"
#include "camera_fileinfo_dialog.h"


Camera_Browser::Camera_Browser(QWidget *parent): QWidget(parent) {


  m_cancelOperation = false;
  m_busyProcessing = false;

  m_camGlobFolder = "/";
  m_gpController = NULL;


  m_mainLayout = new QGridLayout(this);
  m_mainLayout->setSpacing( 6 );
  m_mainLayout->setMargin( 11 );

  m_checkboxThumbs = new QCheckBox( "ThumbNails", this);
  m_camLabel = new QLabel("Camera Not Initialized", this);
  m_mainLayout->addWidget(m_checkboxThumbs, 0, 0);
  m_mainLayout->addWidget(m_camLabel, 0, 1);

  m_splitter = new QSplitter(this);
  m_camFolderTree = new KListView(m_splitter);
  m_camIconViewStack = new QWidgetStack(m_splitter);

  m_splitter->setOpaqueResize (true);
  m_splitter->setResizeMode( m_camFolderTree, QSplitter::Stretch );
  m_splitter->setResizeMode( m_camIconViewStack, QSplitter::Stretch );

  m_mainLayout->addMultiCellWidget(m_splitter, 1, 1, 0, 1);
  m_mainLayout->setRowStretch (1, 1);

  m_camFolderTree->addColumn( tr( "Camera Folders" ) );
  m_camFolderTree->setFullWidth();

  m_camIconViewRoot = new KIconView(m_camIconViewStack);
  m_camIconViewRoot->setResizeMode( KIconView::Adjust );
  m_camIconViewRoot->setItemsMovable( FALSE );

  m_numIconViewStacks = 0;
  m_camIconViewStack->addWidget(m_camIconViewRoot, 0);
  m_camIconViewStack->raiseWidget(0);


  m_GroupBox = new QGroupBox(this);
  m_GroupBox->setFrameShape( QGroupBox::NoFrame );
  m_GroupBox->setTitle( QString::null );
  m_GroupBox->setColumnLayout(0, Qt::Vertical );
  m_GroupBox->layout()->setSpacing( 0 );
  m_GroupBox->setInsideMargin( 0 );
  m_GroupBoxLayout = new QGridLayout( m_GroupBox->layout() );
  m_GroupBoxLayout->setAlignment( Qt::AlignTop );

  QLabel* camDirBoxLabel = new QLabel("Download Directory : ", m_GroupBox);
  m_GroupBoxLayout->addWidget( camDirBoxLabel, 0, 0);

  m_camDirBox = new KComboBox( true, m_GroupBox, "camDirBox" );
  m_camDirBox->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)7, (QSizePolicy::SizeType)0 ) );
  m_camDirBox->setDuplicatesEnabled( false );
  m_camDirBox->insertItem(QDir::currentDirPath());
  m_camDirBox->insertItem(QDir::homeDirPath());
  m_GroupBoxLayout->addWidget( m_camDirBox, 0, 1);

  m_camDirButton = new QPushButton(m_GroupBox);
  m_camDirButton->setText( i18n( "Browse" ) );
  m_GroupBoxLayout->addWidget( m_camDirButton, 0, 2);

  m_mainLayout->addMultiCellWidget(m_GroupBox, 2, 2, 0, 1);



  m_statusBar = new KStatusBar(this);
  m_statusBar->insertItem ("Ready", 0, 7, false);
  m_statusBar->setItemAlignment (0, AlignLeft | AlignVCenter);
  QLabel *spacer = new  QLabel(m_statusBar);
  m_statusBar->addWidget(spacer, 1, false);
  m_progressBar = new KProgress(m_statusBar);
  m_progressBar->setTotalSteps(100);
  m_statusBar->addWidget( m_progressBar, 5, false);


  m_mainLayout->addMultiCellWidget(m_statusBar, 3, 3, 0, 1);



  camera_new();



  connect( m_camFolderTree, SIGNAL( executed(QListViewItem *) ),
	   this, SLOT( slot_camFolderTreeClicked(QListViewItem *) ) );

  connect( m_checkboxThumbs, SIGNAL( toggled(bool) ),
	   this, SLOT( slot_showThumbnails(bool) ) );
  connect( m_camDirButton, SIGNAL( clicked() ),
	   this, SLOT( slot_camDirBrowse() ) );


}


Camera_Browser::~Camera_Browser()
{

  if (m_gpController) {
    delete m_gpController;
  }

}


void Camera_Browser::camera_new() {

  if (m_gpController)
    delete m_gpController;

  totalItemCount = 0;
  totalImageCount = 0;
  totalThumbsRequested = 0;
  totalThumbsDownloaded = 0;
  totalImagesRequested = 0;
  totalImagesDownloaded = 0;


  m_gpController = new Gphoto_Controller(this);
  m_gpController->start();

  if (m_gpController->cameraSupportsThumbNails())
    m_camAbilities_Preview = true;
  else
    m_camAbilities_Preview = false;

  if (m_gpController->cameraSupportsDelete())
    m_camAbilities_Delete = true;
  else
    m_camAbilities_Delete = false;

  if (m_gpController->cameraSupportsUpload())
    m_camAbilities_Upload = true;
  else
    m_camAbilities_Upload = false;

  if (m_gpController->cameraSupportsMkDir())
    m_camAbilities_MkDir = true;
  else
    m_camAbilities_MkDir = false;

  if (m_gpController->cameraSupportsDelDir())
    m_camAbilities_DelDir = true;
  else
    m_camAbilities_DelDir = false;


  if (m_camAbilities_Preview) {
    m_checkboxThumbs->setEnabled(true);
    m_checkboxThumbs->setChecked(true);
  }



}


void Camera_Browser::camera_setSelection(int select)
{

  Camera_IconView *iconView = (Camera_IconView*) m_camIconViewStack->visibleWidget();
  if (!iconView) return;

  switch(select) {
  case (All):    iconView->selectAll(true); break;
  case (None):   iconView->clearSelection(); break;
  case (Invert): iconView->invertSelection(); break;
  default: break;
  }

}

void Camera_Browser::camera_hideFolders()
{
  if (m_camFolderTree->isHidden())
    m_camFolderTree->show();
  else
    m_camFolderTree->hide();
}

void Camera_Browser::camera_hideFolders(bool hide)
{
  if (hide)
    m_camFolderTree->hide();
  else
    m_camFolderTree->show();

}

void Camera_Browser::camera_hideStatusBar()
{
  if (m_statusBar->isHidden())
    m_statusBar->show();
  else
    m_statusBar->hide();
}




void Camera_Browser::camera_connect() {

  if (!(m_gpController->isCameraInitialised()))
    m_gpController->initializeCamera();

}

void Camera_Browser::camera_disconnect() {

  delete m_gpController;
  m_gpController = NULL;

  m_camFolderTree->clear();
  m_numIconViewStacks = 0;
  m_camIconViewStack->raiseWidget(0);

  resetStatusBar();
  m_camLabel->setText("Camera Not Initialized");

  camera_new();

}

void Camera_Browser::camera_download()
{

  m_overwriteAll = false;

  if (!(m_gpController->isCameraInitialised())) return;
  if (!(m_camFolderTree->childCount())) return;

  Camera_IconView *iconView = (Camera_IconView*) m_camIconViewStack->visibleWidget();
  if (!iconView) return;

  int count = 0;
  for ( QIconViewItem *i = iconView->firstItem(); i; i=i->nextItem() ) {
    if (i->isSelected()) ++count;
  }

  if (count == 0) return;

  bool proceed = true;

  for (QIconViewItem *i = iconView->firstItem(); i; i=i->nextItem()) {
    Camera_IconViewItem *item = static_cast<Camera_IconViewItem*>(i);
    if (item->isSelected()) {
      downloadOneImage(item, proceed);
      if (!proceed) return;
    }
  }


}


void Camera_Browser::camera_downloadAll()
{


  m_overwriteAll = false;

  if (!(m_gpController->isCameraInitialised())) return;
  if (!(m_camFolderTree->childCount())) return;

  Camera_Folder *camFolder;
  Camera_IconView* iconView;

  QListViewItemIterator it(m_camFolderTree);
  for ( ; it.current(); ++it ) {

    camFolder = static_cast<Camera_Folder*>(it.current());
    if (!camFolder) return;
    if (camFolder == m_camVirtualFolder) continue;

    iconView = camFolder->m_iconView;

    bool proceed = true;

    for (QIconViewItem *i = iconView->firstItem(); i; i=i->nextItem()) {
      Camera_IconViewItem *item = static_cast<Camera_IconViewItem*>(i);
      downloadOneImage(item, proceed);
      if (!proceed) return;
    }

  }

}


void Camera_Browser :: camera_delete() {

  if (!(m_gpController->isCameraInitialised())) return;
  if (!(m_camFolderTree->childCount())) return;

  Camera_IconView *iconView =
    (Camera_IconView*) m_camIconViewStack->visibleWidget();
  if (!iconView) return;

  int count = 0;
  QStringList imageList;

  for ( QIconViewItem *i = iconView->firstItem(); i; i=i->nextItem() ) {
    if (i->isSelected()) {
      ++count;
      imageList.append(i->text());
    }
  }

  if (count == 0) return;

  QString warnMsg("About to delete these Image(s)\nAre you sure?");
  if (KMessageBox::warningContinueCancelList(this, warnMsg,imageList,
					     QString("Warning"),
					     QString("Delete"))
      ==  KMessageBox::Continue) {

    Camera_IconViewItem *item =
      static_cast<Camera_IconViewItem*>(iconView->firstItem());

    while(item) {

      Camera_IconViewItem *nextItem = (Camera_IconViewItem*) item->nextItem();

      if (item->isSelected())
	m_gpController->deleteImage(item->fileInfo().getFolder(),
                                    item->fileInfo().getName());


      item = nextItem;

    }

  }

}


void Camera_Browser :: camera_deleteAll() {

  if (!(m_gpController->isCameraInitialised())) return;
  if (!(m_camFolderTree->childCount())) return;

  Camera_Folder *camFolder;
  Camera_IconView* iconView;

  QString warnMsg("About to delete All Image(s)\nAre you sure?");

  if (KMessageBox::warningContinueCancel (this, warnMsg, QString("Warning"),
					  QString("Delete"))
      == KMessageBox::Continue) {

    QListViewItemIterator it(m_camFolderTree);
    for ( ; it.current(); ++it ) {

      camFolder = static_cast<Camera_Folder*>(it.current());
      if (!camFolder) return;
      if (camFolder == m_camVirtualFolder) continue;

      iconView = camFolder->m_iconView;

      Camera_IconViewItem *item =
	static_cast<Camera_IconViewItem*>(iconView->firstItem());

      while(item) {

	Camera_IconViewItem *nextItem = (Camera_IconViewItem*) item->nextItem();

	m_gpController->deleteImage(item->fileInfo().getFolder(),
                                    item->fileInfo().getName());

	item = nextItem;


      }

    }

  }

}




void Camera_Browser::camera_cancel()
{
  m_gpController->stopOperations();
}


QString Camera_Browser::cameraModel() const {

  QString camModel(m_gpController->cameraModel());
  return camModel;

}


QString Camera_Browser::cameraPort() const {

  QString camPort(m_gpController->cameraPort());
  return camPort;

}


void Camera_Browser::setDefaultDownloadDir(const QString& downloadDir) {

  if (downloadDir.isNull()) return;

  if (downloadDir.isEmpty()) return;

  if (m_camDirBox->contains(downloadDir)) {
    for (int i = 0; i < m_camDirBox->count(); i++)
      if (downloadDir == m_camDirBox->text(i)) {
	m_camDirBox->setCurrentItem(i);
      }
  }
  else {
      m_camDirBox->insertItem(downloadDir,0);
  }

}

void Camera_Browser::checkThumbnailsButton(bool on) {

  if (m_camAbilities_Preview) {
    m_checkboxThumbs->setChecked(on);
  }

}

void Camera_Browser::startWithGlobalFolder(bool val) {

  m_showGlobalFolder = val;

}

void Camera_Browser::setThumbnailSize(ThumbnailSize::Size size) {

  if (m_thumbnailSize.mySize() == size) return;

  m_thumbnailSize.setSize(size);

  if (!m_gpController) return;
  if ( !(m_gpController->isCameraInitialised()) ) return;

  Camera_Folder *camFolder;
  Camera_IconView* iconView;

  QListViewItemIterator it(m_camFolderTree);
  for ( ; it.current(); ++it ) {

    camFolder = static_cast<Camera_Folder*>(it.current());
    if (!camFolder) continue;

    iconView = camFolder->m_iconView;

    if (camFolder == m_camVirtualFolder) continue;


    Camera_IconViewItem* item = (Camera_IconViewItem*) iconView->firstItem();

    while (item) {

      if (item->stateThumbNailed())
	item->setThumbNail(m_thumbnailSize);
      else
	item->setPixmapForMime(m_thumbnailSize);

      item = (Camera_IconViewItem*) item->nextItem();

    }

    iconView->arrangeItemsInGrid();
    m_camVirtualFolder->m_iconView->arrangeItemsInGrid();
  }


  QListViewItemIterator it2(m_camFolderTree);
  for ( ; it2.current(); ++it2 ) {

    camFolder = static_cast<Camera_Folder*>(it2.current());
    if (!camFolder) continue;

    camFolder->m_iconView->arrangeItemsInGrid();

  }


}



void Camera_Browser::downloadOneImage(Camera_IconViewItem *item,
				      bool& proceedFurther)
{

  proceedFurther = true;

  QString downloadDir=m_camDirBox->currentText();

  QDir dir(downloadDir);
  if (!dir.exists()) {
    camera_cancel();
    KMessageBox::error(0, downloadDir + " Directory does not exist");
    return;
  }

  QString saveFile;
  saveFile=downloadDir + "/" + item->fileInfo().getName();
  while (QFile::exists(saveFile) && !(m_overwriteAll)) {

    bool m_overwrite=false;

    SaveFile_Dialog *dlg = new SaveFile_Dialog(saveFile);
    if (dlg->exec()== QDialog::Rejected) {
      delete dlg;
      camera_cancel();
      proceedFurther = false;
      return;
    }

    switch(dlg->saveFileOperation()) {
    case (Skip): {
      delete dlg;
      return;
    }
    case (SkipAll): {
      delete dlg;
      proceedFurther = false;
      return;
    }
    case (Overwrite): {
      m_overwrite = true;
      delete dlg;
      break;
    }
    case (OverwriteAll): {
      m_overwriteAll = true;
      delete dlg;
      break;
    }
    case (Rename): {
      saveFile = downloadDir+"/"+dlg->renameFile();
      delete dlg;
      break;
    }
    default:  {
      delete dlg;
      proceedFurther = false;
      camera_cancel();
      return;
    }

    }

    if (m_overwrite) break;
  }

  m_gpController->downloadImage(item->fileInfo().getFolder(),
                                item->fileInfo().getName(),
                                saveFile);

}



void Camera_Browser::slot_showThumbnails(bool checked) {


  if (!m_gpController) return;

  Camera_Folder *camFolder;
  Camera_IconView* iconView;

  QListViewItemIterator it(m_camFolderTree);
  for ( ; it.current(); ++it ) {

    camFolder = static_cast<Camera_Folder*>(it.current());
    if (!camFolder) return;

    iconView = camFolder->m_iconView;

    if (camFolder == m_camVirtualFolder)
        continue;


    if (checked) {

      Camera_IconViewItem* item = (Camera_IconViewItem*) iconView->firstItem();

      while (item) {

	QString mime;
	mime = item->fileInfo().getMime();

	if (mime.contains("image")) {

            if ( !(item->isThumbNailed()) ) {

                m_gpController->getThumbNail(item->fileInfo().getFolder(),
                                             item->fileInfo().getName());
                m_progressBar->setTotalSteps(totalImageCount);

            }
            else {
                item->setThumbNail(m_thumbnailSize);
            }
	}
	item = (Camera_IconViewItem*) item->nextItem();
      }


    }

    else {

      Camera_IconViewItem* item = (Camera_IconViewItem*) iconView->firstItem();

      while (item) {
	item->setPixmapForMime(m_thumbnailSize);
	item = (Camera_IconViewItem*) item->nextItem();
      }


    }


    iconView->arrangeItemsInGrid();
    m_camVirtualFolder->m_iconView->arrangeItemsInGrid();

  }

}

void Camera_Browser::slot_setStatusMsg(const QString& msg)
{
  m_statusBar->changeItem(msg, 0);
}

void Camera_Browser::slot_setProgressVal(int val)
{

  if (val >= 0 && val <= 100) {
    m_progressBar->setProgress(val);
  }

}

void Camera_Browser::resetStatusBar()
{
  m_statusBar->changeItem("Ready", 0);
  m_progressBar->setTotalSteps(100);
  m_progressBar->setProgress(0);
}



void Camera_Browser::slot_camFolderTreeClicked(QListViewItem *item) {

  if ( !item ) return;

  Camera_Folder* camFolder = static_cast<Camera_Folder*>( item );
  m_camIconViewStack->raiseWidget(camFolder->m_iconView);

  if ((camFolder->isOpen()))
    camFolder->setOpen(false);
  else
    camFolder->setOpen(true);

  QString numImagesString;
  numImagesString.setNum(camFolder->getNumImages());
  slot_setStatusMsg(camFolder->getPath() + " Folder With " +
		    numImagesString + " Items");

}


void Camera_Browser::slot_camIconViewClicked(QIconViewItem *item) {


    if (m_camIconViewStack->visibleWidget() == m_camIconViewRoot) return;

    if (!item) {

        Camera_IconView *iconView =
            (Camera_IconView*) m_camIconViewStack->visibleWidget();

        QString numImagesString;
        numImagesString.setNum(iconView->cameraFolder()->getNumImages());
        slot_setStatusMsg(iconView->cameraFolder()->getPath() + " Folder With " +
                          numImagesString + " Items");

    }

    else {

        Camera_IconViewItem *it = static_cast<Camera_IconViewItem*>(item);
        if (!it) return;

        QString msg(it->fileInfo().getName());
        if (!(it->fileInfo().getMime().isEmpty())) {
            msg += " (";
            msg += it->fileInfo().getMime();
            msg += ")";
        }
        if (it->fileInfo().getSize() >= 0) {
            QString numString;
            numString.setNum(it->fileInfo().getSize()/1000000.0,'f',3);
            msg += " ";
            msg += numString;
            msg += " MB";
        }


        slot_setStatusMsg(msg);

    }

}

void Camera_Browser::slot_camIconViewRightClicked(QIconViewItem* item,
                                                  const QPoint& pos)
{
    if (m_camIconViewStack->visibleWidget() == m_camIconViewRoot) return;
    if (!item) return;

    Camera_IconViewItem *it = static_cast<Camera_IconViewItem*>(item);
    if (!it) return;

    QPopupMenu *popMenu = new QPopupMenu();
    QPopupMenu *openWithMenu = new QPopupMenu();

    //---------------------------------------------------------------------------

    QValueVector<KService::Ptr> serviceVector;

    KTrader::OfferList offers = KTrader::self()->query(it->fileInfo().getMime(),
                                                       "Type == 'Application'");
    KTrader::OfferList::Iterator iter;
    KService::Ptr ptr;
    int index = 100;
    for( iter = offers.begin(); iter != offers.end(); ++iter ) {
        ptr = *iter;
        openWithMenu->insertItem( ptr->pixmap(KIcon::Small), ptr->name(), index++);
        serviceVector.push_back(ptr);
    }

    //---------------------------------------------------------------------------

    popMenu->insertItem( i18n("Open With ..."), openWithMenu, 10 );
    popMenu->insertItem( i18n("Properties ..."), 11 );
    popMenu->insertSeparator();
    popMenu->insertItem( BarIcon("filesave", KIcon::SizeSmall),
                         i18n("Save"), 12 );
    popMenu->insertItem( BarIcon("edittrash", KIcon::SizeSmall),
                         i18n("Delete"), 13 );

    int id = popMenu->exec(pos);
    switch (id) {

    case 11: {
        Camera_FileInfo_Dialog* fileInfoDlg =
            new Camera_FileInfo_Dialog(it->fileInfo(), this);
        fileInfoDlg->show();
        break;
    }

    case 12: {
        camera_download();
        break;
    }

    case 13: {
        camera_delete();
        break;
    }

    default:
        break;
    }

    //---------------------------------------------------------------------------

    if (id >= 100) {

        imageServicePtr = serviceVector[id-100];

        m_gpController->openImage(it->fileInfo().getFolder(),
                                  it->fileInfo().getName());

    }

    //---------------------------------------------------------------------------

    serviceVector.clear();


    delete openWithMenu;
    delete popMenu;

}


void Camera_Browser::slot_camDirBrowse()
{

  KURL result =
    KURL(KFileDialog::getExistingDirectory( m_camDirBox->text(m_camDirBox->currentItem()),
					    this ));
  if (!result.isEmpty()) {
    for (int i = 0; i < m_camDirBox->count(); i++) {
      if (result.cmp(KURL(m_camDirBox->text(i)), true)) {
	m_camDirBox->setCurrentItem(i);
	return;
      }
    }
    m_camDirBox->insertItem(result.path(), 0);
    m_camDirBox->setCurrentItem(0);
  }

}


void Camera_Browser::customEvent(QCustomEvent *event) {

  if (event->type() >= QCustomEvent::User) {

    switch (event->type()) {

    case Gphoto_InitializeT:
      eventCameraInitialized(event);
      break;

    case Gphoto_GetSubFoldersT:
      eventSubFolders(event);
      break;

    case Gphoto_GetImagesInfoT:
      eventGetImagesInfo(event);
      break;

    case Gphoto_GetThumbNailT:
      eventGetThumbNail(event);
      break;

    case Gphoto_DeleteImageT:
      eventDeleteImage(event);
      break;

    case Gphoto_OpenImageT:
      eventOpenImage(event);
      break;

    case Gphoto_StatusMsgT:
      eventCameraStatusMsg(event);
      break;

    case Gphoto_ProgressT:
      eventCameraProgress(event);
      break;

    case Gphoto_ErrorMsgT:
      eventCameraErrorMsg(event);
      break;

    case Gphoto_BusyT:
      eventCameraBusy(event);
      break;

    default:
      break;

    }

  }

}


void Camera_Browser::eventCameraInitialized(QCustomEvent *event) {

  Gphoto_Event_InitializeCamera*
    ev(static_cast<Gphoto_Event_InitializeCamera *>(event));

  QString cameraModel, cameraPort;

  cameraModel = QString(ev->cameraModel());
  cameraPort  = QString(ev->cameraPort());


  m_camLabel->setText(cameraModel + " (" + cameraPort + ")");
  m_statusBar->message(cameraModel + " (" + cameraPort + ") " +
		       " Initialized", 1000);


  // Clear the IconView Stack
  m_numIconViewStacks = 0;
  m_camIconViewStack->raiseWidget(0);
  m_camFolderTree->clear();

  // Create the root Folder
  Camera_Folder *camFolder = new Camera_Folder(m_camFolderTree,
					       m_camGlobFolder,
					       m_camGlobFolder);
  camFolder->m_iconView = new Camera_IconView(m_camIconViewStack,
					      camFolder);
  m_camIconViewStack->addWidget(camFolder->m_iconView,
				++(m_numIconViewStacks));

  connect(camFolder->m_iconView, SIGNAL(clicked(QIconViewItem *)),
	  this, SLOT(slot_camIconViewClicked(QIconViewItem *)));
  connect(camFolder->m_iconView, SIGNAL( rightButtonPressed(QIconViewItem*, const QPoint&) ),
          this, SLOT( slot_camIconViewRightClicked(QIconViewItem*, const QPoint&) ) );


  // Global Virtual Folder
  m_camVirtualFolder = new Camera_Folder(m_camFolderTree,
					 "Global", "All Pictures");
  m_camVirtualFolder->m_iconView = new Camera_IconView(m_camIconViewStack,
						       m_camVirtualFolder);
  m_camIconViewStack->addWidget(m_camVirtualFolder->m_iconView,
				++(m_numIconViewStacks));

  connect(m_camVirtualFolder->m_iconView, SIGNAL(clicked(QIconViewItem *)),
	  this, SLOT(slot_camIconViewClicked(QIconViewItem *)));
  connect(m_camVirtualFolder->m_iconView, SIGNAL( rightButtonPressed(QIconViewItem*, const QPoint&) ),
          this, SLOT( slot_camIconViewRightClicked(QIconViewItem*, const QPoint&) ) );


  if (m_showGlobalFolder) {
    slot_camFolderTreeClicked(m_camVirtualFolder);
    m_camFolderTree->setSelected(m_camVirtualFolder, true);
  }

  // Request subfolder Names
  m_gpController->getSubFolders(m_camGlobFolder);

  emit signal_cameraConnected();

}


void Camera_Browser::eventSubFolders(QCustomEvent *event) {

  Gphoto_Event_GetSubFolders*
    ev(static_cast<Gphoto_Event_GetSubFolders *>(event));

  if (ev->finished()) {

    QListViewItemIterator it(m_camFolderTree);
    for ( ; it.current(); ++it ) {

      Camera_Folder* camFolder = static_cast<Camera_Folder*>(it.current());
      if (!camFolder) return;
      if (camFolder == m_camVirtualFolder) continue;

      m_gpController->getImagesInfo(camFolder->getPath());

    }
    return;

  }


  if (ev->success()) {

    QString parentFolder(ev->parentFolder());
    QStringList subFolderList(ev->subFolderList());
    QStringList subFolderNameList(ev->subFolderNameList());

    QListViewItemIterator it(m_camFolderTree);
    for ( ; it.current(); ++it ) {

      Camera_Folder* camFolder = static_cast<Camera_Folder*>(it.current());
      if (!camFolder) return;
      if (camFolder == m_camVirtualFolder) continue;

      if (parentFolder == camFolder->getPath()) {


	QStringList::Iterator subFolderNameIter = subFolderNameList.begin();


	for (QStringList::Iterator subFolderIter = subFolderList.begin();
             subFolderIter != subFolderList.end();
             ++subFolderIter ) {
	  Camera_Folder *camFolderChild =
	    new Camera_Folder(camFolder, *subFolderIter, *subFolderNameIter);
	  camFolderChild->m_iconView = new Camera_IconView(m_camIconViewStack,
							   camFolderChild);
	  m_camIconViewStack->addWidget(camFolderChild->m_iconView,
					++(m_numIconViewStacks));
	  connect(camFolderChild->m_iconView, SIGNAL(clicked(QIconViewItem *)),
		  this, SLOT(slot_camIconViewClicked(QIconViewItem *)));
          connect(camFolderChild->m_iconView, SIGNAL( rightButtonPressed(QIconViewItem*, const QPoint&) ),
                  this, SLOT( slot_camIconViewRightClicked(QIconViewItem*, const QPoint&) ) );
	  subFolderNameIter++;

	}


	break;
      }

    }

  }
  else {

    QString errorMsg("Failed to get SubFolders from Folder ");
    errorMsg += ev->parentFolder();

    KMessageBox::error (this, errorMsg);
  }

}


void Camera_Browser::eventGetImagesInfo(QCustomEvent *event) {

    Gphoto_Event_GetImagesInfo*
        ev(static_cast<Gphoto_Event_GetImagesInfo *>(event));


    QString parentFolder(ev->parentFolder());
    List_MT<Camera_FileInfo> infoList(ev->imagesInfoList());


    QListViewItemIterator it(m_camFolderTree);
    for ( ; it.current(); ++it ) {

        Camera_Folder* camFolder = static_cast<Camera_Folder*> (it.current());
        if (!camFolder) return;
        if (camFolder == m_camVirtualFolder) continue;

        if (parentFolder == camFolder->getPath()) {

            camFolder->setNumImages(infoList.count());

            while( !(infoList.isEmpty()) ) {

                Camera_FileInfo info( infoList.first() );
                infoList.pop_front();

                Camera_IconViewItem* item =
                    new Camera_IconViewItem(camFolder->m_iconView,
                                            info,
                                            m_camVirtualFolder->m_iconView);
                item->setPixmapForMime(m_thumbnailSize);

                totalItemCount++;

                if (info.getMime().contains("image")) {
                    totalImageCount++;
                    if (m_checkboxThumbs->isChecked())
                        m_gpController->getThumbNail(parentFolder,
                                                     info.getName());
                    totalThumbsRequested++;

                }

            }

            m_camVirtualFolder->setNumImages(totalItemCount);

            m_progressBar->setTotalSteps(totalThumbsRequested);

            return;

        }

    }


}


void Camera_Browser :: eventGetThumbNail(QCustomEvent *event) {

  Gphoto_Event_GetThumbNail*
    ev(static_cast<Gphoto_Event_GetThumbNail *>(event));

  Camera_Folder *camFolder;
  Camera_IconView* iconView;

  QListViewItemIterator it(m_camFolderTree);
  for ( ; it.current(); ++it ) {

    camFolder = static_cast<Camera_Folder*>(it.current());
    if (!camFolder) return;
    if (camFolder == m_camVirtualFolder) continue;

    iconView = camFolder->m_iconView;


    for (QIconViewItem *i = iconView->firstItem(); i; i=i->nextItem()) {

      Camera_IconViewItem *item = static_cast<Camera_IconViewItem*>(i);
      if (item->text() == ev->imageName()) {

	QPixmap thumb;
	ev->getThumbNail(thumb);
	item->setThumbNail(thumb, m_thumbnailSize);

	m_progressBar->advance(1);
	totalThumbsDownloaded++;

	if (totalThumbsDownloaded == 1) {
	  QListViewItemIterator it2(m_camFolderTree);
	  for ( ; it2.current(); ++it2 ) {
	    Camera_Folder* camFolder2 =
	      static_cast<Camera_Folder*>(it2.current());
	    Camera_IconView* iconView2 = camFolder2->m_iconView;

	    iconView2->arrangeItemsInGrid();
	  }
	}


	if (totalThumbsDownloaded >= totalImageCount) {

	  m_progressBar->reset();
	  QListViewItemIterator it2(m_camFolderTree);

	  for ( ; it2.current(); ++it2 ) {
	    Camera_Folder* camFolder2 =
	      static_cast<Camera_Folder*>(it2.current());
	    Camera_IconView* iconView2 = camFolder2->m_iconView;

	    iconView2->arrangeItemsInGrid();
	  }


	}


	return;

      }

    }


  }

}

void Camera_Browser :: eventDeleteImage(QCustomEvent *event) {

  Gphoto_Event_DeleteImage*
    ev(static_cast<Gphoto_Event_DeleteImage *>(event));

  Camera_Folder *camFolder;
  Camera_IconView* iconView;

  QListViewItemIterator it(m_camFolderTree);
  for ( ; it.current(); ++it ) {

    camFolder = static_cast<Camera_Folder*>(it.current());
    if (!camFolder) continue;
    if (camFolder == m_camVirtualFolder) continue;

    if (ev->parentFolder() == camFolder->getPath()) {

      iconView = camFolder->m_iconView;

      for (QIconViewItem *i = iconView->firstItem(); i; i=i->nextItem()) {

	Camera_IconViewItem *item = static_cast<Camera_IconViewItem*>(i);
	if (item->fileInfo().getName() == ev->imageName()) {
            delete item;

            iconView->arrangeItemsInGrid();
            m_camVirtualFolder->m_iconView->arrangeItemsInGrid();
            camFolder->setNumImages(camFolder->getNumImages()-1);
            m_camVirtualFolder->setNumImages(m_camVirtualFolder->getNumImages()-1);
            return;
	}


      }

    }

  }

}

void Camera_Browser::eventOpenImage(QCustomEvent *event)
{

    Gphoto_Event_OpenImage*
        ev(static_cast<Gphoto_Event_OpenImage *>(event));

    QString savedFile = locateLocal("tmp", ev->imageName());

    if (imageServicePtr) KRun::run(*imageServicePtr, KURL(savedFile));

}

void Camera_Browser :: eventCameraStatusMsg(QCustomEvent *event) {

  Gphoto_Event_CameraStatusMsg*
    ev(static_cast<Gphoto_Event_CameraStatusMsg *>(event));

  slot_setStatusMsg(ev->statusMsg());

}

void Camera_Browser :: eventCameraProgress(QCustomEvent *event) {

  Gphoto_Event_CameraProgress*
    ev(static_cast<Gphoto_Event_CameraProgress *>(event));

   m_progressBar->setTotalSteps(100);
   slot_setProgressVal(ev->val());

}

void Camera_Browser :: eventCameraErrorMsg(QCustomEvent *event) {

  Gphoto_Event_CameraErrorMsg*
    ev(static_cast<Gphoto_Event_CameraErrorMsg *>(event));

  QString errorMsg("Camera Error : ");

  KMessageBox::error(0, errorMsg + ev->errorMsg());

}


void Camera_Browser :: eventCameraBusy(QCustomEvent *event) {

  Gphoto_Event_Busy*
    ev(static_cast<Gphoto_Event_Busy *>(event));

  if (ev->busy()) {
    if (!m_busyProcessing) {
      emit signal_startedProcessing();
      m_busyProcessing = true;
    }
  }
  else {
    if (m_busyProcessing) {
      emit signal_stoppedProcessing();
      m_busyProcessing = false;
      resetStatusBar();
    }
  }


}
