/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGREPORTBANK_H
#define SKGREPORTBANK_H
/** @file
 * A report class for bank document
*
* @author Stephane MANKOWSKI
*/
#include <QVariant>

#include "skgerror.h"
#include "skgreport.h"
#include "skgbankmodeler_export.h"
#include "skgunitobject.h"

class SKGDocument;
/**
 * A report class for bank document
 */
class SKGBANKMODELER_EXPORT SKGReportBank : public SKGReport
{
    Q_OBJECT
    /**
     * The budget table
     */
    Q_PROPERTY(QVariantList budget_table READ getBudgetTable)

    /**
     * The unit table
     */
    Q_PROPERTY(QVariantList unit_table READ getUnitTable)

    /**
     * The portfolio
     */
    Q_PROPERTY(QVariantList portfolio READ getPortfolio)

    /**
     * The account table
     */
    Q_PROPERTY(QVariantList account_table READ getAccountTable)

    /**
     * The bank table
     */
    Q_PROPERTY(QVariantList bank_table READ getBankTable)

    /**
     * The scheduled operations
     */
    Q_PROPERTY(QVariantList scheduled_operations READ getScheduledOperations)

    /**
     * The main categories of the period
     */
    Q_PROPERTY(QVariantList categories_period READ getMainCategoriesForPeriod)

    /**
     * The main categories of the previous period
     */
    Q_PROPERTY(QVariantList categories_previous_period READ getMainCategoriesForPreviousPeriod)

    /**
     * The main categories of the period (for compatibility)
     */
    Q_PROPERTY(QVariantList categories_month READ getMainCategoriesForPeriod)

    /**
     * The main categories of the previous period (for compatibility)
     */
    Q_PROPERTY(QVariantList categories_previous_month READ getMainCategoriesForPreviousPeriod)
    /**
     * The income versus expenditure
     */
    Q_PROPERTY(QVariantList income_vs_expenditure READ getIncomeVsExpenditure)

    /**
     * The main categories variations
     */
    Q_PROPERTY(QStringList categories_variations READ get5MainCategoriesVariation)

public:
    /**
     * Default Constructor
     */
    explicit SKGReportBank(SKGDocument* iDocument);

    /**
     * Default Destructor
     */
    virtual ~SKGReportBank();

    /**
     * Get the budget table
     * @return the budget table
     */
    virtual QVariantList getBudgetTable();

    /**
     * Get the unit table
     * @return the unit table
     */
    virtual QVariantList getUnitTable();

    /**
     * Get the portfolio
     * @return the portfolio
     */
    virtual QVariantList getPortfolio();

    /**
     * Get the account table
     * @return the account table
     */
    virtual QVariantList getAccountTable();

    /**
     * Get the bank table
     * @return the bank table
     */
    virtual QVariantList getBankTable();

    /**
     * Get the scheduled operations
     * @return the scheduled operations
     */
    virtual QVariantList getScheduledOperations();

    /**
     * Get the main categories of the period
     * @return the main categories of the period
     */
    virtual QVariantList getMainCategoriesForPeriod();

    /**
     * Get the main categories of the previous period
     * @return the main categories of the previous period
     */
    virtual QVariantList getMainCategoriesForPreviousPeriod();

    /**
     * Get the main categories variations
     * @return the main categories variations
     */
    virtual QStringList get5MainCategoriesVariation();

    /**
     * Get the income versus expenditure
     * @param iOnSubOperation the computation is done on sub operation
     * @param iTransfert the computation includes transferts
     * @param iTracker the computation includes tracked operations
     * @param iWhereClause1 the period where clause (if empty then use the period of the report)
     * @param iWhereClause2 the period where clause (if empty then use the previous period of the report)
     * @param iTracker the computation includes tracked operations
     * @return the income versus expenditure
     */
    virtual QVariantList getIncomeVsExpenditure(bool iOnSubOperation = false, bool iTransfert = true, bool iTracker = true, const QString& iWhereClause1 = QString(), const QString& iWhereClause2 = QString());

protected:
    /**
     * Enrich the grantlee mapping
     * @param iMapping the mapping
     */
    virtual void addItemsInMapping(QVariantHash& iMapping);

private:
    Q_DISABLE_COPY(SKGReportBank)

    typedef struct {
        SKGUnitObject unit;
        double initalAmount;
        double purchaseAmount;
        double currentAmount;
        double quantity;
    } unitValues;
};

#endif  // SKGREPORTBANK_H
