/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "klchildmeta.h"
#include "klchild.h"
#include "qtstream.h"

/**
* KLChildMeta class
*/

bool KLChildMeta::dumpObject(KLChild *, KLDumpDevice &) const
{
  return true; // Nothing to be done.
}


void KLChildMeta::restoreObject(KLChild *, KLDumpDevice &) const
{
  // Nothing to be done. 
}


KLChild *KLChildMeta::createObjectEdit(KLChild *object, bool extended=false) const
{
  if( parentMeta ) 
    return parentMeta->createObjectEdit(object,extended);
  else {
    HASSERT(false); // There's something wrong !
    return NULL;
  }
}


bool KLChildMeta::dumpCompleteObject(KLChild *object, KLDumpDevice &stream)
{
  const KLChildMeta *metaClass = object->getRealMetaClass();
  HASSERT( metaClass );
  stream.writeLevelStart(metaClass->className);
  while( metaClass ) {
    HASSERT( object->metaInherits(metaClass->className) );
    stream.writeComment( "Data of " + metaClass->className + "(" + metaClass->userClassName + ")" );
    if( !metaClass->dumpObject(object,stream) ) 
      return false;
    metaClass = metaClass->parentMeta;
  }
  stream.writeLevelEnd();
  return true;
}


KLChild *KLChildMeta::restoreCompleteObject(KLDumpDevice &stream,KLChild *object)
{
  QString linestr;
  ulong line = stream.linestart, level = 0;
  if( !stream.readLine(linestr,line,level) )
    return NULL;

  HASSERT( stream.reglist );
  if( linestr.left(2) == "{ " ) 
    linestr = linestr.mid(2,linestr.length());
  KLChildMeta *metaClass = stream.reglist->findMetaClass(linestr);
  if( !metaClass ) {
    WARN("Unknown metaclass:" + linestr);
    return 0;
  }
  if( !object ) 
    object = metaClass->createObject(stream.appenv);
  else
    HASSERT( object->metaIsA(metaClass->className) );
  if( !object )
    return 0;
  
  while( metaClass ) {
    metaClass->restoreObject(object,stream);
    metaClass = metaClass->parentMeta;
  }
  return object;
}

/** 
* Dervided classes from KLChildMeta
*/


KLChildMetaStandAlone::ClassType KLChildMetaStandAlone::classType() const
{
  return StandAlone;
}

KLChildMetaFixUnique::ClassType KLChildMetaFixUnique::classType() const
{
  return FixUnique;
}

KLChild *KLChildMetaFixUnique::createObject(void *) const
{
  return a_object;
}

KLChildMetaMultiple::ClassType KLChildMetaMultiple::classType() const
{
  return Multiple;
}

//////////////////////////////////////////////////////////////////////////////////////////
// KLMetaRegList
//////////////////////////////////////////////////////////////////////////////////////////
void KLMetaRegList::regMetaClass(KLChildMeta *metaClass)
{
  HASSERT( metaClass );
  index.insert(metaClass->className, metaClass);
  append(metaClass);
}


void KLMetaRegList::unRegMetaClass(KLChildMeta *metaClass)
{
  HASSERT( metaClass );
  index.remove(metaClass->className);
  remove(metaClass);
}
  

KLChildMeta *KLMetaRegList::findMetaClass(const char *className) const
{
  HASSERT( className );
  return index.find(className);
}

//////////////////////////////////////////////////////////////////////////////////////////
// Dump device
//////////////////////////////////////////////////////////////////////////////////////////

KLDumpDevice::KLDumpDevice(QIODevice &file) : QTextStream(&file)
{
}


void KLDumpDevice::readFile()
{
  lineBuff.clear();
  QString line;
  while(!eof()) {
    line = QTextStream::readLine();
    line = line.stripWhiteSpace();

    if(line.isEmpty())
      continue;

    if(line[0] == '#' ) // Comment
      continue;

    lineBuff.append(line);
  } 
  linestart = 0;
}


bool KLDumpDevice::readLine( QString &string, ulong &line, ulong &level)
{
  if( line >= lineBuff.count() )
    return false;
  string = lineBuff.at(line);
  if( string[0] == '{' ) 
    level++;
  if( string[0] == '}' ) {
    HASSERT( level );
    level--;
  }
   
  line++;
  return level != 0;
}

QString KLDumpDevice::readEntry(const char *entryname, const char *def)
{
  QString linestr;
  QString key = entryname;
  key += "=";
  ulong elen = key.length();
  ulong line = linestart, level = 0;
  while( readLine( linestr, line, level ) ) {
    if( (level == 1) && (linestr[0] != '{') ) { // Only if it is our level
      if( !stricmp(linestr.left(elen),key) ) {
        linestr.remove(0,elen);
        return linestr;
      }
    }
  } // while
  return def;
}

void KLDumpDevice::writeComment(const char *comment)
{
  *this << indent << "# " << comment << endl;
}

void KLDumpDevice::writeEntry(const char *entryname, const char *entrydata)
{
  *this << indent << entryname << "=" << entrydata << endl;
}

void KLDumpDevice::writeLevelStart(const char *levelname)
{
  *this << indent << "{ " << levelname << endl;
  indent += "  "; 
}

void KLDumpDevice::writeLevelEnd()
{
  indent = indent.left(indent.length() - 2);
  *this << indent << "}" << endl;
}

QString KLDumpDevice::readEntryConv(const char *entryname, const char *def)
{
  QString res = readEntry(entryname,def);
  // Translate \n to chr(\n)
  // Translate \t to chr(\t)
  return res;
}

void KLDumpDevice::writeEntryConv(const char *entryname, const char *entrydata)
{
  QString data = entrydata;
  // Translate chr(\n) to \n
  // Translate chr(\t) to \t
  writeEntry(entryname,data);
}

