/*
 * KMLOFax
 *
 * A utility to process facsimiles received with the ELSA
 * MicroLink(tm) Office or MicroLink(tm) ISDN Office modem.
 *
 * Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen.
 */

#include <unistd.h>
#include <stdlib.h>

#include <qglobal.h>
#include <qdir.h>

#include <kapp.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "downloaddlg.h"
#include "global.h"
#include "filters.h"



#define MOD_STAT_IDLE         0
#define MOD_STAT_WAIT4RESET   1
#define MOD_STAT_WAIT4MODEL   2
#define MOD_STAT_WAIT4DIR     3
#define MOD_STAT_WAIT4DNL     4
#define MOD_STAT_WAIT4DEL     5

#define MODEL_UNKNOWN         0
#define MODEL_OFFICE          1
#define MODEL_ISDN_OFFICE     2

#define ID_MODEL_OFFICE       "MicroLink Office"
#define ID_MODEL_ISDN_OFFICE  "MicroLink ISDN Office"



DownloadDlg::DownloadDlg(QWidget *parent, const char *name) : QProgressDialog(parent, name, false)
{
	config = kapp->config();
	
	modem = new Modem(this, "modem");
	CHECK_PTR(modem);
	init();

	setMinimumDuration(0);
	setAutoReset(false);
	setAutoClose(false);

	connect(modem, SIGNAL(gotLine(const char *)), SLOT(fetchModemLine(const char *)));
	connect(modem, SIGNAL(gotXBlock(const uchar *, int)), SLOT(fetchXModemBlock(const uchar *, int)));
	connect(modem, SIGNAL(xmodemDone(bool)), SLOT(fetchXModemDone(bool)));
	connect(modem, SIGNAL(timeout()), SLOT(fetchTimeout()));
	connect(this, SIGNAL(cancelled()), SLOT(cancel()));

	setCaption(i18n("Download"));
}


DownloadDlg::~DownloadDlg()
{
	reset();
}


bool DownloadDlg::startDownload()
{
	static int speeds[] = { 19200, 38400, 57600, 115200, 230400 };
	QString dev;
	int speed;

	if (status)
		return false;

	config->setGroup(ID_PREF_GROUP_GENERAL);
	load_new = config->readBoolEntry(ID_PREF_GEN_LOADNEW, PREF_GEN_LOADNEW);
	delete_fax = config->readBoolEntry(ID_PREF_GEN_DELETEFAX, PREF_GEN_DELETEFAX);

	config->setGroup(ID_PREF_GROUP_MODEM);
	dev = config->readEntry(ID_PREF_MOD_DEVICE, PREF_MOD_DEVICE);
	speed = config->readNumEntry(ID_PREF_MOD_SPEED, PREF_MOD_SPEED);
	if ((speed < 0) || (speed > 4))
		speed = PREF_MOD_SPEED;
	
	modem->setDevice(dev);
	modem->setSpeed(speeds[speed]);
	modem->setData(8);
	modem->setParity('N');
	modem->setStop(1);

	if (!modem->open()) {
		KMessageBox::sorry(this, i18n("Cannot open modem device."), i18n("Modem Error"));
		return false;
	}
	if (!modem->dsrOn()) {
		KMessageBox::sorry(this, i18n("Modem is off."), i18n("Modem Error"));
		modem->close();
		return false;
	}
	if (!modem->ctsOn()) {
		KMessageBox::sorry(this, i18n("Modem is busy."), i18n("Modem Error"));
		modem->close();
		return false;
	}

	modem->writeLine("");
	usleep(250000);
	modem->flush();
	modem->writeLine("ATZ");

	status = MOD_STAT_WAIT4RESET;

	return true;
}


void DownloadDlg::cancel()
{
	QProgressDialog::cancel();

	if (status == MOD_STAT_WAIT4DNL)
		modem->abortXModem();
	else {
		reset();
		emit done();
	}
}



void DownloadDlg::fetchModemLine(const char *line)
{

	if (!strcmp(line, "ERROR")) {
		QString command;

		switch (status) {
			case MOD_STAT_WAIT4RESET:
				command = "ATZ";
				break;
			case MOD_STAT_WAIT4MODEL:
				command = "ATI6";
				break;
			case MOD_STAT_WAIT4DIR:
				command = "AT$JDIR[=,FAX[,N]]";
				break;
			case MOD_STAT_WAIT4DNL:
				command = "AT$JDNL=\"filename\"";
				break;
			case MOD_STAT_WAIT4DEL:
				command = "AT$JDEL=\"filename\"";
				break;
			default:
				break;
		}
		status = MOD_STAT_IDLE;
		KMessageBox::sorry(this, i18n("Error after modem command '%1'.").arg(command), i18n("Modem Error"));
		emit message(QString("%1: %2").arg(i18n("Error")).arg(i18n("Modem command error.")));
		reset();
		emit done();
		return;
	}
	
	switch (status) {
		case MOD_STAT_WAIT4RESET:					/* "ATZ" sent	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);
				model = MODEL_UNKNOWN;
				modem->writeLine("ATI6");
				status = MOD_STAT_WAIT4MODEL;
			}
			break;
		case MOD_STAT_WAIT4MODEL:					/* "ATI6" sent */
			if (!strcmp(line, "ATI6"))			/* Discard echo */
				return;
			if (!strcmp(line, ID_MODEL_OFFICE))
				model = MODEL_OFFICE;
			else if (!strcmp(line, ID_MODEL_ISDN_OFFICE))
				model = MODEL_ISDN_OFFICE;
			else if (!strcmp(line, "OK")) {
				if (model == MODEL_UNKNOWN) {
					status = MOD_STAT_IDLE;
					KMessageBox::sorry(this, i18n("Unknown modem model \"%1\".").arg(line), i18n("Modem Error"));
					emit message(QString("%1: %2").arg(i18n("Error")).arg(i18n("Unkown modem model.")));
					reset();
					emit done();
					return;
				}
				usleep(250000);
				if (model == MODEL_OFFICE)
					modem->writeLine(load_new ? "AT$JDIR=,FAX,N" : "AT$JDIR=,FAX");
				if (model == MODEL_ISDN_OFFICE)
					modem->writeLine("AT$JDIR");
				status = MOD_STAT_WAIT4DIR;
			}
			break;
		case MOD_STAT_WAIT4DIR:					/* "AT$JDIR[=,FAX[,N]]" sent	*/
			char buff[81], *p;
			fitem_t *fitem;
			char *type, *flags, *datetime, *sizes, *clip;
			int year, month, day, hour, minute, second, size;

			if (!strcmp(line, "OK")) {
				usleep(250000);
				count = 0;
				status = MOD_STAT_WAIT4DNL;
				downloadFile();
				break;
			}

			strncpy(buff, line, 80);
			buff[80] = 0;

			if (!(type = strchr(buff, ',')))
				break;
			p = type;
			type++;
			while ((p > buff) && (*(p-1) == ' '))
				p--;
			*p = 0;

			while (*type == ' ')
				type++;

			if (!(flags = strchr(type, ',')))
				break;
			*flags = 0;
			flags++;
			
			if (!(datetime = strchr(flags, ',')))
				break;
			*datetime = 0;
			datetime++;
			
			if (!(p = strchr(datetime, ',')))
				break;
			p++;
			
			if (!(sizes = strchr(p, ',')))
				break;
			*sizes = 0;
			sizes++;
			
			if ((clip = strchr(sizes, ','))) {
				*clip = 0;
				clip++;
			}

			if (model == MODEL_ISDN_OFFICE) {
				if (strcmp(type, "TIFF"))
					break;
				if (load_new && !strchr(type, 'N'))
					break;
			}

			if (model == MODEL_OFFICE) {
				char timestr[13];

				if (sscanf(datetime, "%d.%d.%d, %d:%d", &day, &month, &year, &hour, &minute) != 5)
					break;
				sprintf(timestr, "%04d%02d%02d%02d%02d", year, month, day, hour, minute);
				if (!strncmp(buff, timestr, 12) && (buff[12] >= '0') && (buff[12] <= '9') &&  (buff[13] >= '0') && (buff[13] <= '9'))
					second = (int)(buff[12] & 0x0f) * 10 + (int)(buff[13] & 0x0f);
				else
					second = 0;
			}
			
			if (model == MODEL_ISDN_OFFICE) {
				if (sscanf(datetime, " %d.%d.%d, %d:%d:%d", &day, &month, &year, &hour, &minute, &second) != 6)
					break;
			}
			if (sscanf(sizes, " %d", &size) != 1)
				break;
			
			fitem = (fitem_t *)malloc(sizeof(fitem_t));
			CHECK_PTR(fitem);
			fitem->next = fitems;
			fitems = fitem;
			strncpy(fitems->name, buff, 20);
			fitems->name[20] = 0;
			fitems->year = year;
			fitems->month = month;
			fitems->day = day;
			fitems->hour = hour;
			fitems->minute = minute;
			fitems->second = second;
			fitems->size = size;
			nfiles++;
			break;
		case MOD_STAT_WAIT4DNL:					/* XModem confirmed	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);

				if (delete_fax) {
					modem->writeLine(QString("AT$JDEL=\"%1\"").arg(fitems->name));
					status = MOD_STAT_WAIT4DEL;
				}
				else {
					setProgress(fitems->size + 1);
					fitem = fitems;
					fitems = fitems->next;
					free(fitem);
					downloadFile();
				}
			}
			break;
		case MOD_STAT_WAIT4DEL:					/* Delete confirmed	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);
				setProgress(fitems->size + 1);
				fitem = fitems;
				fitems = fitems->next;
				free(fitem);
				status = MOD_STAT_WAIT4DNL;
				downloadFile();
			}
		default:
			break;
	}
}


void DownloadDlg::fetchXModemBlock(const uchar *block, int size)
{
	int diff;

	if (file.isOpen() && ((diff = fitems->size - progress()) > 0)) {
		if (size > diff)
			size = diff;
		file.writeBlock((const char *)block, size);
		setProgress(progress () + size);
	}
}


void DownloadDlg::fetchXModemDone(bool success)
{
	QString s;

	if (file.isOpen()) {
		file.close();
		setLabelText(i18n("Waiting for confirmation..."));
		if (success) {
			QDir d;
			QString fname = fitems->name;

			if (fname.right(4) == ".FAX")
				fname.replace(fname.length()-3, 3, "tif");

			if (model == MODEL_OFFICE) {
				MLO2TiffFilter filter;

				filter.setFile(QString(TEMP_DIR) + "/" + fitems->name);
				filter.setTime(fitems->year, fitems->month, fitems->day, fitems->hour, fitems->minute, fitems->second);
				filter.convertFile(QString(INBOX_DIR) + "/" + fname);
				d.remove(expandPath(QString(TEMP_DIR) + "/" + fitems->name));
			}
			else {
				Tiff2TiffFilter filter;

				filter.setFile(QString(TEMP_DIR) + "/" + fitems->name);
				filter.convertFile(QString(INBOX_DIR) + "/" + fname);
				d.remove(expandPath(QString(TEMP_DIR) + "/" + fitems->name));
			}
			emit gotFile(fname);
			if (model == MODEL_ISDN_OFFICE)
				modem->timerStart(2);
		}
		else {
			emit message(QString("%1: %2").arg(i18n("Error")).arg(i18n("Download failed.")));

			reset();
			emit done();
		}
	}
}


void DownloadDlg::fetchTimeout()
{
	QString s;
	
	if (status == MOD_STAT_WAIT4DNL) {
		modem->flush();
		fetchModemLine("OK");
		return;
	}

	KMessageBox::sorry(this, i18n("Modem response timeout."), i18n("Modem Error"));

	emit message(QString("%1: %2").arg(i18n("Error")).arg(i18n("Modem response timeout.")));

	reset();
	emit done();
}


void DownloadDlg::init()
{
	load_new = false;
	delete_fax = false;
	
	status = MOD_STAT_IDLE;
	model = MODEL_UNKNOWN;

	fitems = 0;
	nfiles = 0;
	count = 0;
}


void DownloadDlg::reset()
{
	fitem_t *p;

	QProgressDialog::reset();

	modem->close();

	while (fitems) {
		p = fitems;
		fitems = fitems->next;
		free(p);
	}

	if (file.isOpen())
		file.close();

	init();

	hide();
}


void DownloadDlg::downloadFile()
{
	QString s;

	if (fitems) {
		count++;
		file.setName(expandPath(QString(TEMP_DIR) + "/" + fitems->name));
		if (!file.open(IO_WriteOnly)) {
			KMessageBox::sorry(this, i18n("Cannot create file."), i18n("File Error"));
			emit message(QString("%1: %2!").arg(i18n("Error")).arg(i18n("Download failed")));
			reset();
			emit done();

			return;
		}
		modem->writeLine(QString("AT$JDNL=\"%1\"").arg(fitems->name));
		QProgressDialog::reset();
		setLabelText(i18n("Receiving file %1 of %2...").arg(count).arg(nfiles));
		setTotalSteps(fitems->size);
		setProgress(0);
		modem->receiveXModem(model == MODEL_OFFICE);	// firmware problem (no crc allowed)!
	}
	else {
		if (nfiles)
			emit message(i18n("Received %1 file(s).").arg(nfiles));
		else
			emit message(i18n("No facsimile in modem."));

		reset();
		emit done();
	}
}
