/*
 * KMLOFax
 *
 * A utility to process facsimiles received with the ELSA
 * MicroLink(tm) Office or MicroLink(tm) ISDN Office modem.
 *
 * Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <qglobal.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qstrlist.h>
#include <qmap.h>
#include <qsize.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qprogressdialog.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
#include <ksimpleconfig.h>
#include <kconfig.h>
#include <kseparator.h>
#include <kmessagebox.h>

#include "faxview.h"
#include "global.h"
#include "mlofile.h"
#include "senderaliases.h"
#include "filters.h"



QPopupMenu *moveMenu = 0;
QPopupMenu *copyMenu = 0;

static QMap<int, FolderViewItem *> folderMap;


FaxViewItem::FaxViewItem(QListView *parent, const char *name) : QListViewItem(parent, name)
{
	m_pages = 0;
	m_size = 0;
	m_new = false;
}


FaxViewItem::~FaxViewItem()
{
}


void FaxViewItem::setName(const QString& name_)
{
	m_name = name_;
}


void FaxViewItem::setTime(const QDateTime& time_)
{
	m_time = time_;
	setText(0, KGlobal::locale()->formatDate(time_.date(), true) + " (" + KGlobal::locale()->formatTime(time_.time(), true) + ")");
}


void FaxViewItem::setSender(const QString& sender_)
{
	m_sender = sender_;
	refreshAlias();
}


void FaxViewItem::setPages(int pages_)
{
	QString s;
	
	m_pages = pages_;
	setText(2, s.setNum(pages_));
}


void FaxViewItem::setSize(uint size_)
{
	m_size = size_;
	setText(3, KGlobal::locale()->formatNumber((double)size_, 0));
}


void FaxViewItem::setNew(bool new_)
{
	m_new = new_;
	setPixmap(4, m_new ? UserIcon("yes") : UserIcon("no"));
}


void FaxViewItem::setParams(const QString& name_, const QDateTime& time_, const QString& sender_, int pages_, uint size_, bool new_)
{
	setName(name_);
	setTime(time_);
	setSender(sender_);
	setPages(pages_);
	setSize(size_);
	setNew(new_);
}


void FaxViewItem::refreshAlias()
{
	setText(1, senderaliases->alias(m_sender));
}


QString FaxViewItem::key(int column, bool) const
{
	QString s;

	if (column == 0) {
		s.sprintf("%04d%02d%02d%02d%02d%02d", m_time.date().year(), m_time.date().month(), m_time.date().day(), m_time.time().hour(), m_time.time().minute(), m_time.time().second());
		return s;
	}
	if (column == 2) {
		s.sprintf("%10d", m_pages);
		return s;
	}
	if (column == 3) {
		s.sprintf("%10d", m_size);
		return s;
	}

	return text(column);
}



FaxView::FaxView(QWidget *parent, const char *name) : KListView(parent, name)
{
	has_selection = false;

	config = kapp->config();

	addColumn(i18n("Date (Time)"), 0);
	addColumn(i18n("Sender"), 0);
	addColumn(i18n("Pages"), 0);
	addColumn(i18n("Size"), 0);
	addColumn(i18n("New"), 0);
	
	setColumnAlignment(2, AlignVCenter | AlignHCenter);
	setColumnAlignment(3, AlignVCenter | AlignRight);
	
	setMinimumSize(sizeHint());
	setSelectionModeExt(Extended);
	setSorting(0, false);
	setAllColumnsShowFocus(true);
	setShowSortIndicator(true);

	context_menu = new QPopupMenu(this, "contextmenu");
	CHECK_PTR(context_menu);
	context_menu->insertItem(SmallIcon("viewmag"), i18n("&View..."), ID_FILE_VIEW);
	context_menu->insertItem(SmallIcon("fileprint"), i18n("&Print..."), ID_FILE_PRINT);
	context_menu->insertItem(SmallIcon("filesave"), i18n("&Export..."), ID_FILE_EXPORT);
	context_menu->insertItem(SmallIcon("mail_send"), i18n("&Mail..."), ID_FILE_MAIL);
	context_menu->insertItem(SmallIcon("trashcan_empty"), i18n("Move To &Trash"), ID_FILE_TRASH);
	context_menu->insertItem(i18n("&Sender Alias..."), ID_FILE_SENDER_ALIAS);
	context_menu->insertSeparator();
	context_menu->insertItem(i18n("Move To"), moveMenu);
	context_menu->insertItem(i18n("Copy To"), copyMenu);
	
	connect(this, SIGNAL(currentChanged(QListViewItem *)), SLOT(currentChangedSlot(QListViewItem *)));
	connect(this, SIGNAL(selectionChanged()), SLOT(selectionChangedSlot()));
	connect(this, SIGNAL(doubleClicked(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(this, SIGNAL(returnPressed(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(this, SIGNAL(rightButtonPressed(QListViewItem *, const QPoint&, int)), SLOT(rightButtonPressedSlot(QListViewItem *, const QPoint&, int)));

	connect(context_menu, SIGNAL(activated(int)), SLOT(activatedSlot(int)));

	connect(senderaliases, SIGNAL(applyButtonPressed()), SLOT(refreshAliases()));

	setMinimumSize(40, 30);

	readOptions();
}


FaxView::~FaxView()
{
}


void FaxView::readOptions()
{
	config->setGroup("FaxWindow Options");
	
	setColumnWidth(0, config->readNumEntry("DateWidth", 140));
	setColumnWidth(1, config->readNumEntry("SenderWidth", 160));
	setColumnWidth(2, config->readNumEntry("PagesWidth", 50));
	setColumnWidth(3, config->readNumEntry("SizeWidth", 60));
	setColumnWidth(4, config->readNumEntry("NewWidth", 40));
}


void FaxView::saveOptions()
{
	config->setGroup("FaxWindow Options");
	
	config->writeEntry("DateWidth", columnWidth(0));
	config->writeEntry("SenderWidth", columnWidth(1));
	config->writeEntry("PagesWidth", columnWidth(2));
	config->writeEntry("SizeWidth", columnWidth(3));
	config->writeEntry("NewWidth", columnWidth(4));
}


void FaxView::insertFaxItem(const QString& name, const QDateTime& time, const QString& sender, int pages, uint size, bool isnew)
{
	FaxViewItem *item;
	
	item = new FaxViewItem(this);
	item->setParams(name, time, sender, pages, size, isnew);
}


void FaxView::takeFaxItem(const QString& name, const QDateTime& time, const QString& sender, int pages, uint size, bool isnew)
{
	FaxViewItem *item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		if (item->name() == name)
			break;
	
	if (item)
		item->setParams(name, time, sender, pages, size, isnew);
	else
		insertFaxItem(name, time, sender, pages, size, isnew);
}


void FaxView::deleteFaxItem(const QString& name)
{
	FaxViewItem *item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		if (item->name() == name)
			break;

	if (item)
		delete item;
}


void FaxView::setCurrentFaxNew(bool isnew)
{
	if (currentItem())
		((FaxViewItem *)currentItem())->setNew(isnew);
}


bool FaxView::currentFaxNew()
{
	if (currentItem())
		return ((FaxViewItem *)currentItem())->isNew();

	return false;
}


QString FaxView::currentFaxName()
{
	if (currentItem())
		return ((FaxViewItem *)currentItem())->name();

	return QString::null;
}


QString FaxView::currentFaxSender()
{
	if (currentItem())
		return ((FaxViewItem *)currentItem())->sender();

	return QString::null;
}


bool FaxView::faxNew(const QString& name)
{
	FaxViewItem *item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		if (item->name() == name)
			return item->isNew();

	return true;
}


void FaxView::rightButtonPressedSlot(QListViewItem *, const QPoint& point, int)
{
	if (has_selection)
		context_menu->popup(point);
}


void FaxView::currentChangedSlot(QListViewItem *item)
{
	emit hasCurrent(item != 0);
}


void FaxView::selectionChangedSlot()
{
	QListViewItem *item;

	has_selection = false;

	for (item = firstChild(); (item); item = item->nextSibling())
		if (item->isSelected()) {
			has_selection = true;
			break;
		}

	emit hasSelection(has_selection);
}


void FaxView::executedSlot(QListViewItem *)
{
	emit command(ID_FILE_VIEW);
}


void FaxView::activatedSlot(int id_)
{
	emit command(id_);
}


void FaxView::refreshAliases()
{
	FaxViewItem *item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		item->refreshAlias();
}




EditFolder::EditFolder(QWidget *parent, const char *name) : KDialog (parent, name, true)
{
	QVBoxLayout *vbox;
	QHBoxLayout *hbox;
	QLabel *label;
	KSeparator *separator;
	QPushButton *ok, *cancel;
	QSize size;

	vbox = new QVBoxLayout(this, marginHint());
	CHECK_PTR(vbox);

	vbox->addStretch();

	hbox = new QHBoxLayout();
	vbox->addLayout(hbox);

	label = new QLabel(i18n("&Name:"), this);
	label->setMinimumSize(label->sizeHint());
	hbox->addWidget(label);

	m_name = new QLineEdit(this);
	label->setBuddy(m_name);
	m_name->setMinimumWidth(200);
	m_name->setFixedHeight(m_name->sizeHint().height());
	hbox->addWidget(m_name);

	vbox->addStretch();
	
	vbox->addSpacing(marginHint());

	separator = new KSeparator(KSeparator::HLine, this);
	vbox->addWidget(separator);
	
	hbox = new QHBoxLayout();
	vbox->addLayout(hbox);

	ok = new QPushButton(i18n("&OK"), this);
	ok->setDefault(true);
	cancel = new QPushButton(i18n("&Cancel"), this);

	size = ok->sizeHint().expandedTo(cancel->sizeHint());

	ok->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addStretch();
	hbox->addWidget(ok);
//	hbox->addSpacing(spacingHint());
	hbox->addWidget(cancel);

	vbox->activate();

	connect(ok, SIGNAL(clicked()), SLOT(accept()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	m_name->setFocus();
}


EditFolder::~EditFolder()
{
}



FolderViewItem::FolderViewItem(QListView *parent, Type type, const char *name) : QListViewItem(parent, name)
{
	init(type);
}


FolderViewItem::FolderViewItem(QListViewItem *parent, Type type, const char *name) : QListViewItem(parent, name)
{
	init(type);
}


FolderViewItem::~FolderViewItem()
{
}


void FolderViewItem::init(Type type)
{
	m_type = type;

	if (m_type == Root) {
		setPixmap(0, SmallIcon("kmlofax"));
		setText(0, i18n("Facsimiles"));
	}
	else if (m_type == Inbox) {
		setPixmap(0, SmallIcon("inbox"));
		setText(0, i18n("Inbox"));
	}
	else if (m_type == Trash) {
		setText(0, i18n("Trash"));
	}
	else if (m_type == User) {
		setPixmap(0, SmallIcon("folder"));
	}
	setFiles(0);
}


void FolderViewItem::setName(const QString& name_)
{
	if (m_type == User)
		setText(0, name_);
}


QString FolderViewItem::name()
{
	return text(0);
}


QString FolderViewItem::path()
{
	if (m_type == Root)
		return FOLDERS_DIR;
	else if (m_type == Inbox)
		return INBOX_DIR;
	else if (m_type == Trash)
		return TRASH_DIR;

	return ((FolderViewItem *)parent())->path() + "/" + name();
}


void FolderViewItem::setFiles(int files_)
{
	QString s;
	
	m_files = files_;

	if (m_type == Root)
		return;

	setText(1, s.setNum(m_files));
	if (m_type == Trash)
		setPixmap(0, (files_) ? SmallIcon("trashcan_full") : SmallIcon("trashcan_empty"));
}

	
void FolderViewItem::markCurrent(bool current)
{
	if (m_type == Inbox)
		setPixmap(0, current ? SmallIcon("inbox_open") : SmallIcon("inbox"));
	else if (m_type == User)
		setPixmap(0, current ? SmallIcon("folder_open") : SmallIcon("folder"));
}


FolderView::FolderView(QWidget *parent, const char *name) : KListView(parent, name)
{
	moveMenu = new QPopupMenu();
	copyMenu = new QPopupMenu();

	config = kapp->config();

	inbox_folder = 0;
	trash_folder = 0;
	current_folder = 0;

	addColumn(i18n("Folder"), 0);
	addColumn(i18n("Items"), 0);
	setColumnAlignment(1, AlignVCenter | AlignRight);
	setMinimumSize(sizeHint());
	setSorting(-1);
	setAllColumnsShowFocus(true);
	
	context_menu = new QPopupMenu(this, "foldercontext");
	CHECK_PTR(context_menu);
	context_menu->insertItem(SmallIcon("folder_new"), i18n("&New Folder..."), ID_FOLDER_NEW);
	context_menu->insertItem(i18n("&Rename..."), ID_FOLDER_RENAME);
	context_menu->insertSeparator();
//	context_menu->insertItem(i18n("&Empty"), ID_FOLDER_EMPTY);
	context_menu->insertItem(SmallIcon("remove"), i18n("Re&move"), ID_FOLDER_REMOVE);
	
	connect(this, SIGNAL(currentChanged(QListViewItem *)), SLOT(currentChangedSlot(QListViewItem *)));
	connect(this, SIGNAL(rightButtonPressed(QListViewItem *, const QPoint&, int)), SLOT(rightButtonPressedSlot(QListViewItem *, const QPoint&, int)));
	connect(context_menu, SIGNAL(activated(int)), SLOT(activatedSlot(int)));

	init();

	readOptions();
}


FolderView::~FolderView()
{
	delete moveMenu;
	moveMenu = 0;
	delete copyMenu;
	copyMenu = 0;
}


void FolderView::readOptions()
{
	config->setGroup("FolderWindow Options");
	
	setColumnWidth(0, config->readNumEntry("FolderWidth", 100));
	setColumnWidth(1, config->readNumEntry("ItemsWidth", 38));
}


void FolderView::saveOptions()
{
	config->setGroup("FolderWindow Options");
	
	config->writeEntry("FolderWidth", columnWidth(0));
	config->writeEntry("ItemsWidth", columnWidth(1));
}


void FolderView::setFiles(FolderViewItem *item)
{
	KSimpleConfig *sconfig;
	int number;

	sconfig = new KSimpleConfig(expandPath(item->path() + "/" + KMLOFAX_FAXDIR), true);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries", 0);
	delete sconfig;
	
	item->setFiles(number);
}


void FolderView::init()
{
	FolderViewItem *root_item = new FolderViewItem(this, FolderViewItem::Root);

	init(root_item, expandPath(FOLDERS_DIR));

	trash_folder = new FolderViewItem(root_item, FolderViewItem::Trash);
	inbox_folder = new FolderViewItem(root_item, FolderViewItem::Inbox);

	root_item->setOpen(true);

	QListViewItemIterator it(firstChild());
	while (it.current()) {
		setFiles((FolderViewItem *)it.current());
		it++;
	}

	rebuildFolderMenus();
}


void FolderView::init(FolderViewItem *parent, const QString& dir)
{
	QDir d(dir);
	const QFileInfoList *files;
	FolderViewItem *item;

	d.setFilter(QDir::Dirs | QDir::NoSymLinks);
	files = d.entryInfoList();

	if (files) {
		QFileInfoListIterator it(*files);
		QFileInfo *f;
		while ((f = it.current()) != 0) {
			++it;
			if (f->fileName().at(0) != '.') {
				item = new FolderViewItem(parent, FolderViewItem::User);
				item->setName(f->fileName());
				init(item, f->filePath());
			}
		}
	}
}


QPopupMenu *FolderView::rebuildFolderMenu(FolderViewItem *item, QPopupMenu *menu_, bool move)
{
	int id;

	if (move)
		connect(menu_, SIGNAL(activated(int)), SLOT(moveSlot(int)));
	else
		connect(menu_, SIGNAL(activated(int)), SLOT(copySlot(int)));

	FolderViewItem *child = (FolderViewItem *)item->firstChild();
	while (child) {
		if (child->type() == FolderViewItem::Inbox)
			id = menu_->insertItem(SmallIcon("inbox"), child->name());
		else if (child->type() == FolderViewItem::Trash)
			id = menu_->insertItem(SmallIcon("trashcan_empty"), child->name());
		else
			id = menu_->insertItem(SmallIcon("folder"), child->name());

		folderMap.insert(id, child);

		if (child->firstChild()) {
			menu_->insertItem(SmallIcon("folder_open"), i18n("Subfolders of %1").arg(child->name()), rebuildFolderMenu(child, new QPopupMenu(), move));
		}

		child = (FolderViewItem *)child->nextSibling();
	}

	return menu_;
}


void FolderView::rebuildFolderMenus()
{
	folderMap.clear();
	moveMenu->clear();
	copyMenu->clear();

	(void)rebuildFolderMenu((FolderViewItem *)firstChild(), moveMenu, true);
	(void)rebuildFolderMenu((FolderViewItem *)firstChild(), copyMenu, false);
}


void FolderView::createFolder()
{
	EditFolder edit(this);
	edit.setCaption(i18n("New Folder"));

	if (edit.exec()) {
		QString name = edit.name();
		if (!name.isEmpty()) {
			QDir d(expandPath(current_folder->path() + "/" + name));
			if (d.exists()) {
				KMessageBox::sorry(this, i18n("Folder exists."), i18n("File Error"));
				return;
			}
			if (!d.mkdir(d.path())) {
				KMessageBox::sorry(this, i18n("Could not create folder."), i18n("File Error"));
				return;
			}
			FolderViewItem *item = new FolderViewItem(current_folder, FolderViewItem::User);
			item->setName(name);
			rebuildFolderMenus();
		}
	}
}


void FolderView::renameFolder()
{
	EditFolder edit(this);
	edit.setCaption(i18n("Rename Folder"));
	edit.setName(current_folder->name());

	if (edit.exec()) {
		QString name = edit.name();
		if (!name.isEmpty()) {
			QDir d(expandPath(((FolderViewItem *)current_folder->parent())->path() + "/" + name));
			if (d.exists()) {
				KMessageBox::sorry(this, i18n("Folder exists."), i18n("File Error"));
				return;
			}
			if (!d.rename(expandPath(current_folder->path()), d.path())) {
				KMessageBox::sorry(this, i18n("Could not rename folder."), i18n("File Error"));
				return;
			}
			current_folder->setName(name);
			rebuildFolderMenus();
		}
	}
}


void FolderView::removeFolder()
{
	if ((current_folder->files() > 0) || (currentItem()->childCount() > 0)) {
		KMessageBox::sorry(this, i18n("Folder is not empty."), i18n("File Error"));
		return;
	}
	QDir d;
	d.remove(expandPath(current_folder->path() + "/" + KMLOFAX_FAXDIR));
	if (!d.rmdir(expandPath(current_folder->path()))) {
		KMessageBox::sorry(this, i18n("Could not remove folder."), i18n("File Error"));
		return;
	}

	delete current_folder;

	rebuildFolderMenus();
//	current_folder = 0;
}


void FolderView::rightButtonPressedSlot(QListViewItem *, const QPoint& point, int)
{
	if (current_folder)
		context_menu->popup(point);
}


void FolderView::currentChangedSlot(QListViewItem *item)
{
	if (current_folder)
		current_folder->markCurrent(false);

	current_folder = (FolderViewItem *)item;

	if (current_folder) {
		current_folder->markCurrent(true);
		FolderViewItem::Type ftype = current_folder->type();
		context_menu->setItemEnabled(ID_FOLDER_NEW, (ftype == FolderViewItem::Root) || (ftype == FolderViewItem::User));
		context_menu->setItemEnabled(ID_FOLDER_RENAME, ftype == FolderViewItem::User);
//		context_menu->setItemEnabled(ID_FOLDER_EMPTY, ftype != FolderViewItem::Root);
		context_menu->setItemEnabled(ID_FOLDER_REMOVE, ftype == FolderViewItem::User);
		emit hasCurrent(ftype == FolderViewItem::Root, ftype == FolderViewItem::User);
	}
}


void FolderView::moveSlot(int id_)
{
	if (folderMap.contains(id_))
		emit move(folderMap[id_]);
}


void FolderView::copySlot(int id_)
{
	if (folderMap.contains(id_))
		emit copy(folderMap[id_]);
}


void FolderView::activatedSlot(int id_)
{
	switch (id_) {
		case ID_FOLDER_NEW:
			createFolder();
			break;
		case ID_FOLDER_RENAME:
			renameFolder();
			break;
		case ID_FOLDER_REMOVE:
			removeFolder();
			break;
		default:
			break;
	}
}



FaxFiler::FaxFiler(QWidget *parent, const char *name) : QSplitter(parent, name)
{
	config = kapp->config();
	current_folder = 0;

	folderlist = new FolderView(this);
	faxlist = new FaxView(this);

	connect(folderlist, SIGNAL(selectionChanged(QListViewItem *)), SLOT(folderChangedSlot(QListViewItem *)));
	connect(folderlist, SIGNAL(hasCurrent(bool, bool)), SLOT(folderHasCurrentSlot(bool, bool)));
	
	connect(faxlist, SIGNAL(hasCurrent(bool)), SLOT(faxHasCurrentSlot(bool)));
	connect(faxlist, SIGNAL(hasSelection(bool)), SLOT(faxHasSelectionSlot(bool)));
	connect(faxlist, SIGNAL(command(int)), SLOT(commandSlot(int)));
	
	connect(folderlist, SIGNAL(move(FolderViewItem *)), SLOT(moveCallback(FolderViewItem *)));
	connect(folderlist, SIGNAL(copy(FolderViewItem *)), SLOT(copyCallback(FolderViewItem *)));

	readOptions();
}


FaxFiler::~FaxFiler()
{
}


void FaxFiler::setInboxFolder()
{
	folderlist->setCurrentItem((QListViewItem *)folderlist->inboxFolder());
}


void FaxFiler::readOptions()
{
	folderlist->readOptions();
	faxlist->readOptions();

	config->setGroup("FilerWindow Options");

	if (config->hasKey("Sizes"))
		setSizes(config->readIntListEntry("Sizes"));
	else {
		QValueList<int> def_sizes;
		def_sizes.append(142);
		def_sizes.append(610);
		setSizes(def_sizes);
	}
}


void FaxFiler::saveOptions()
{
	folderlist->saveOptions();
	faxlist->saveOptions();
	
	config->setGroup("FilerWindow Options");

	config->writeEntry("Sizes", sizes());
}


void FaxFiler::saveCurrentFolder()
{
	KSimpleConfig *sconfig;
	QString s;
	FaxViewItem *item;
	int i;

	if (!current_folder)
		return;

	QFile::remove(expandPath(currentPath() + "/" + KMLOFAX_FAXDIR));
	sconfig = new KSimpleConfig(expandPath(currentPath() + "/" + KMLOFAX_FAXDIR), false);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	sconfig->writeEntry("Entries", faxlist->childCount());
	item = (FaxViewItem *)faxlist->firstChild();
	for (i = 0; (item); i++) {
		s.sprintf("Entry %d", i);
		sconfig->setGroup(s);
		sconfig->writeEntry("FileName", item->name());
		sconfig->writeEntry("New", item->isNew());
		sconfig->writeEntry("Time", item->time());
		sconfig->writeEntry("Sender", item->sender());
		sconfig->writeEntry("Pages", item->pages());
		sconfig->writeEntry("Size", item->size());
		item = (FaxViewItem *)item->nextSibling();
	}
	sconfig->sync();
	delete sconfig;
}


void FaxFiler::takeFaxItem(FolderViewItem *folder, const QString& name, bool isnew)
{
	TiffFile tiffile;
	KSimpleConfig *sconfig;
	int i, number;
	QString group;

	sconfig = new KSimpleConfig(expandPath(folder->path() + "/" + KMLOFAX_FAXDIR));
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries", 0);
	for (i = 0; i < number; i++) {
		group = QString("Entry %1").arg(i);
		if (sconfig->hasGroup(group)) {
			sconfig->setGroup(group);
			if (sconfig->readEntry("FileName", "") == name)
				break;
		}
	}
	if (i == number) {
		number++;
		sconfig->setGroup("General");
		sconfig->writeEntry("Entries", number);
		sconfig->setGroup(QString("Entry %1").arg(number-1));
		folder->setFiles(number);
	}
	sconfig->writeEntry("New", isnew);
	sconfig->writeEntry("FileName", name);
	tiffile.setName(folder->path() + "/" + name);
	if (tiffile.open(IO_ReadOnly)) {
		sconfig->writeEntry("Time", tiffile.time());
		sconfig->writeEntry("Sender", tiffile.sender());
		sconfig->writeEntry("Pages", tiffile.pages());
		sconfig->writeEntry("Size", tiffile.size());
		if (current_folder == folder)
			faxlist->takeFaxItem(name, tiffile.time(), tiffile.sender(), tiffile.pages(), tiffile.size(), isnew);
		tiffile.close();
	}
	delete sconfig;
}


void FaxFiler::newFaxItem(const QString& name)
{
	takeFaxItem(folderlist->inboxFolder(), name, true);
}


void FaxFiler::moveFax(const QString& name, FolderViewItem *to, bool copy)
{
	bool is_new = faxlist->faxNew(name);

	if (copy) {
		QFile source(expandPath(currentPath() + "/" + name));
		QFile target(expandPath(to->path() + "/" + name));
		char buffer[4096];
		int result;
		if (!source.open(IO_ReadOnly))
			return;
		if (!target.open(IO_WriteOnly)) {
			source.close();
			return;
		}
		while ((result = source.readBlock(buffer, 4096)) > 0)
			target.writeBlock(buffer, result);
		target.close();
		source.close();
	}
	else {
		QDir d;
		d.remove(expandPath(to->path() + "/" + name));
		if (!d.rename(expandPath(currentPath() + "/" + name), expandPath(to->path() + "/" + name)))
			return;
	}

	takeFaxItem(to, name, is_new);
}


void FaxFiler::moveFax(FolderViewItem *to, bool copy)
{
	FaxViewItem *item;
	bool remove = false;

	if ((!current_folder) || (!to) || (current_folder == to))
		return;

	item = (FaxViewItem *)faxlist->firstChild();
	while (item) {
		if (item->isSelected()) {
			moveFax(item->name(), to, copy);
			remove = !copy;
		}
		if (remove) {
			FaxViewItem *item2 = item;
			item = (FaxViewItem *)item->nextSibling();
			delete item2;
			remove = false;
		}
		else
			item = (FaxViewItem *)item->nextSibling();
	}
	
	if (!copy)
		saveCurrentFolder();

	folderlist->setFiles(current_folder);
}


void FaxFiler::setFolder(FolderViewItem *item)
{
	KSimpleConfig *sconfig;
	int i, number = 0;
	QString group;

	current_folder = item;

	faxlist->clear();
	
	sconfig = new KSimpleConfig(expandPath(current_folder->path() + "/" + KMLOFAX_FAXDIR), true);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");
	for (i = 0; i < number; i++) {
		group = QString("Entry %1").arg(i);
		if (sconfig->hasGroup(group)) {
			sconfig->setGroup(group);
			faxlist->insertFaxItem(sconfig->readEntry("FileName"), sconfig->readDateTimeEntry("Time"), sconfig->readEntry("Sender"), sconfig->readNumEntry("Pages"), sconfig->readNumEntry("Size"), sconfig->readBoolEntry("New"));
		}
	}
	delete sconfig;

	emit hasCurrentFax(false);
}


QString FaxFiler::currentPath()
{
	if (current_folder)
		return current_folder->path();

	return QString::null;
}


QString FaxFiler::currentFaxName()
{
	return faxlist->currentFaxName();
}


void FaxFiler::setCurrentFaxNew(bool isnew)
{
	faxlist->setCurrentFaxNew(isnew);
	setFaxNew(faxlist->currentFaxName(), isnew);
}


bool FaxFiler::currentFaxNew()
{
	return faxlist->currentFaxNew();
}


QString FaxFiler::currentFaxSender()
{
	return faxlist->currentFaxSender();
}


void FaxFiler::deleteFax()
{
	if (current_folder == folderlist->trashFolder()) {
		FaxViewItem *item, *item2;
		bool new_fax = false;

		item = (FaxViewItem *)faxlist->firstChild();
		while (item) {
			if (item->isSelected() && !item->isNew()) {
				QDir d;
				d.remove(expandPath(current_folder->path() + "/" + item->name()));
				item2 = item;
				item = (FaxViewItem *)item->nextSibling();
				delete item2;
			}
			else {
				if (item->isNew())
					new_fax = true;
				item = (FaxViewItem *)item->nextSibling();
			}
		}

		saveCurrentFolder();
		folderlist->setFiles(current_folder);

		if (new_fax)
			KMessageBox::sorry(this, i18n("One or more facsimiles are marked as new\nand have not been deleted.\nPlease view or print them first."), i18n("Warning"));
	}
	else
		moveFax(folderlist->trashFolder(), false);
}


void FaxFiler::selectAllFax()
{
	faxlist->selectAll(true);
}

	
void FaxFiler::setFaxNew(const QString& name, bool isnew)
{
	KSimpleConfig *sconfig;
	int i, number = 0;
	QString group;

	if (name.isEmpty())
		return;

	sconfig = new KSimpleConfig(expandPath(current_folder->path() + "/" + KMLOFAX_FAXDIR), false);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");
	for (i = 0; i < number; i++) {
		group = QString("Entry %1").arg(i);
		if (sconfig->hasGroup(group)) {
			sconfig->setGroup(group);
			if (sconfig->readEntry("FileName") == name) {
				sconfig->writeEntry("New", isnew);
				break;
			}
		}
	}
	delete sconfig;
}


int FaxFiler::rebuildFolder()
{
	int res;
	
	res = rebuildFolder(current_folder);
	
//	setFolder(folderlist->name());

	return res;
}


void FaxFiler::updateFromPre21()
{
	KSimpleConfig *sconfig;
	int i, number = 0;
	QString group, fname;
	MLO2TiffFilter filter;
	
	config->setGroup("General");
	if (config->readBoolEntry("Update21Done", false))
		return;
	
	sconfig = new KSimpleConfig(expandPath(KMLOFAX_FAXDIR), true);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");

	if (!number) {
		config->writeEntry("Update21Done", true);
		return;
	}

	i = KMessageBox::warningYesNoCancel(this,
		i18n("There are %1 old facsimile files.\n" \
		     "Should they be converted into the new file\n" \
		     "format to be used with KMLOFax %2?").arg(number).arg(VERSION),
		i18n("Question"));

	if (i == KMessageBox::Cancel)
		return;

	config->writeEntry("Update21Done", true);

	if (i == KMessageBox::No)
		return;

	QProgressDialog progress(i18n("Converting files..."), i18n("Abort"), number, this, "progress", true);

	for (i = 0; i < number; i++) {
		progress.setProgress(i);
		kapp->processEvents();
		if (progress.wasCancelled())
			break;
		group = QString("Entry %1").arg(i);
		if (sconfig->hasGroup(group)) {
			sconfig->setGroup(group);
			fname = sconfig->readEntry("FileName");
			filter.setFile(fname);
			filter.setTime(mloDateSplit(fname));
			if (fname.right(4) == ".FAX")
				fname.replace(fname.length()-3, 3, "tif");
			if (filter.convertFile(QString(INBOX_DIR) + "/" + fname))
				takeFaxItem(folderlist->inboxFolder(), fname, sconfig->readBoolEntry("New"));
		}
	}

	progress.setProgress(number);

	delete sconfig;
}


void FaxFiler::newFolder()
{
	folderlist->createFolder();
}


void FaxFiler::renameFolder()
{
	folderlist->renameFolder();
}


void FaxFiler::emptyFolder()
{
}


void FaxFiler::removeFolder()
{
	folderlist->removeFolder();
}


int FaxFiler::rebuildFolder(FolderViewItem *folder)
{
	TiffFile tifffile;
	int number = 0;
	KSimpleConfig *sconfig;

	QFile::remove(expandPath(folder->path() + "/" + KMLOFAX_FAXDIR));
	
	sconfig = new KSimpleConfig(expandPath(folder->path() + "/" + KMLOFAX_FAXDIR), false);
	CHECK_PTR(sconfig);
	
	QDir d(expandPath(folder->path()));
	const QFileInfoList *files;
	
	d.setNameFilter("*.tif");
	files = d.entryInfoList();

	if (files) {
		kapp->setOverrideCursor(waitCursor);
		QFileInfoListIterator it(*files);
		QFileInfo *f;
		while ((f=it.current()) != 0) {
			++it;
			tifffile.setName(folder->path() + "/" + f->fileName());
			if (!(tifffile.open(IO_ReadOnly)))
				continue;
			sconfig->setGroup(QString("Entry %1").arg(number++));
			sconfig->writeEntry("FileName", f->fileName());
			sconfig->writeEntry("Time", tifffile.time());
			sconfig->writeEntry("New", false);
			sconfig->writeEntry("Sender", tifffile.sender());
			sconfig->writeEntry("Pages", tifffile.pages());
			sconfig->writeEntry("Size", tifffile.size());
			tifffile.close();
		}
		kapp->restoreOverrideCursor();
	}

	sconfig->setGroup("General");
	sconfig->writeEntry("Entries", number);
	delete sconfig;

	folder->setFiles(number);

	return number;
}


void FaxFiler::moveCallback(FolderViewItem *to)
{
	moveFax(to, false);
}


void FaxFiler::copyCallback(FolderViewItem *to)
{
	moveFax(to, true);
}


void FaxFiler::folderChangedSlot(QListViewItem *item)
{
	setFolder((FolderViewItem *)item);
}


void FaxFiler::folderHasCurrentSlot(bool isRoot, bool isUser)
{
	emit hasCurrentFolder(isRoot, isUser);
}


void FaxFiler::faxHasCurrentSlot(bool yesno)
{
	emit hasCurrentFax(yesno);
}


void FaxFiler::faxHasSelectionSlot(bool yesno)
{
	moveMenu->setEnabled(yesno);
	copyMenu->setEnabled(yesno);
}


void FaxFiler::commandSlot(int id_)
{
	emit command(id_);
}
