/*
 * KMLOFax
 *
 * A utility to process facsimiles received with the ELSA
 * MicroLink(tm) Office or MicroLink(tm) ISDN Office modem.
 *
 * Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen.
 */

#include <stdlib.h>
#include <time.h>
#include <string.h>

#include <qprogressdialog.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qsize.h>
#include <qfont.h>
#include <qpushbutton.h>
#include <qtooltip.h>

#include <kapp.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kseparator.h>

#include "maildlg.h"
#include "filters.h"
#include "mlofile.h"
#include "senderaliases.h"
#include "global.h"




MailClient::MailClient() : DwSmtpClient()
{
	connected = false;
	srand(time(0));
}


MailClient::~MailClient()
{
	if (connected)
		disconnect();
}


bool MailClient::SendDataMult(const DwString& aStr)
{
	return SendDataMult(aStr.data(), aStr.length());
}


bool MailClient::SendDataMult(const char *aBuf, int aBufLen)
{
	return PSend(aBuf, aBufLen) == aBufLen;
}


void MailClient::createBoundary()
{
	int i;

	strcpy(boundary, "0_=_vwxy");
	for (i=8; i < 39; i++)
		boundary[i] = 65 + (int)(26.0 * rand() / (RAND_MAX + 1.0));
	boundary[39] = 0;
}


bool MailClient::connect(const char *server, int port)
{
	if (Open(server, port))
		if (Helo() == 250)
			connected = true;
		else
			Close();

	return connected;
}


bool MailClient::sendHeader(const char *from,  const char *to, const char *subj)
{
	DwString msg;

	createBoundary();

	if (Mail(from) != 250)
		return false;

	if (Rcpt(to) != 250)
		return false;

	if (Data() != 354)
		return false;

	msg.clear();

	if (*subj) {
		msg.append("Subject: ");
		msg.append(subj);
		msg.append("\r\012");
	}

	msg.append("X-Mailer: KMLOFax ");
	msg.append(VERSION);
	msg.append("\r\012MIME-Version: 1.0\r\012");
	msg.append("Content-Type: multipart/mixed;\r\012 boundary=\"");
	msg.append(boundary);
	msg.append("\"\r\012\r\012");

	return SendDataMult(msg);
}


bool MailClient::sendBody(const char *body)
{
	DwString msg;
	char buff[1000];
	int i = 0, j = 0;

	msg.assign("--");
	msg.append(boundary);
	msg.append("\r\012Content-Type: text/plain; charset=iso-8859-1\r\012");
	msg.append("Content-Transfer-Encoding: 8bit\r\012\r\012");

	if (!SendDataMult(msg))
		return false;

	while (body[i]) {
		buff[j++] = body[i++];
		if ((!body[i]) || (buff[j-1] == '\n') || (j > 996)) {
			if (buff[j-1] == '\n')
				j--;
			if ((j == 1) && (buff[0] == '.'))
				buff[j++] = '.';
			buff[j++] = '\r';
			buff[j++] = '\012';
			buff[j] = 0;
			if (!SendDataMult(buff, j))
				return false;
			j = 0;
		}
	}

	return true;
}


bool MailClient::sendFile(FILE *stream)
{
	DwString msg;
	char buff[258];
	int i;
	QProgressDialog *prog;

	fseek(stream, 0, SEEK_END);
	
	msg.assign("--");
	msg.append(boundary);
	msg.append("\r\012Content-Type: application/postscript; name=\"facsimile.ps\"\r\012");
	msg.append("Content-Transfer-Encoding: 7bit\r\012");
	msg.append("Content-Disposition: attachment; filename=\"facsimile.ps\"\r\012\r\012");

	if (!SendDataMult(msg)) {
		return false;
	}

	prog = new QProgressDialog(i18n("Sending mail..."), i18n("Cancel"), (int)ftell(stream), 0, "mailprogress", true);
	CHECK_PTR(prog);
	prog->setCaption(i18n("Mail Sending"));

	rewind(stream);
	prog->setProgress(0);
	
	while (fgets(buff, 256, stream)) {
		if (prog->wasCancelled()) {
			delete prog;
			return false;
		}

		i = strlen(buff);
		if (i &&  buff[i-1] == '\n')
			i--;
		buff[i++] = '\r';
		buff[i++] = '\012';
		buff[i] = 0;
		if (!SendDataMult(buff, i)) {
			delete prog;
			return false;
		}

		prog->setProgress((int)ftell(stream));
	}

	delete prog;

	msg.assign("\r\012--");
	msg.append(boundary);
	msg.append("--\r\012");

	return SendData(msg) == 250;
}


void MailClient::disconnect()
{
	Quit();
	Close();
}



MailDlg::MailDlg(QWidget *parent, const char *name) : KDialog (parent, name, true)
{
	QVBoxLayout *vbox;
	QHBoxLayout *hbox;
	QGroupBox *gb;
	QGridLayout *grid;
	QLabel *label;
	QPushButton *help, *send, *cancel;
	QSize size;
	QFont font;
	KSeparator *separator;

	config = kapp->config();
	
	vbox = new QVBoxLayout(this, marginHint());

	gb = new QGroupBox(this);

	grid = new QGridLayout(gb, 2, 2, marginHint());

	label = new QLabel(i18n("&To:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 0);

	header_to = new QLineEdit(gb);
	label->setBuddy(header_to);
	font = header_to->font();
	font.setFamily("courier");
	header_to->setFont(font);
	header_to->setMinimumSize(header_to->sizeHint());
	grid->addWidget(header_to, 0, 1);

	label = new QLabel(i18n("&Subject:"), gb);
	label->setMinimumSize(label->sizeHint());

	grid->addWidget(label, 1, 0);

	header_subject = new QLineEdit(gb);
	label->setBuddy(header_subject);
	header_subject->setFont(font);
	header_subject->setMinimumSize(header_subject->sizeHint());
	grid->addWidget(header_subject, 1, 1);

	grid->setColStretch(1, 1);

	grid->activate();

	vbox->addWidget(gb);

	body = new QMultiLineEdit(this);
	body->setFont(font);
	
	vbox->addWidget(body, 1);

	label = new QLabel(QString("%1: facsimile.ps (application/postscript)").arg(i18n("Attachment")), this);
	label->setMinimumSize(label->sizeHint());
	label->setFrameStyle(QFrame::Panel|QFrame::Sunken);
	
	vbox->addWidget(label);

	separator = new KSeparator(KSeparator::HLine, this);
	vbox->addWidget(separator);
	
	hbox = new QHBoxLayout();
	vbox->addLayout(hbox);

	help = new QPushButton(i18n("&Help"), this);
	send = new QPushButton(i18n("&Send"), this);
	send->setDefault(true);
	cancel = new QPushButton(i18n("&Cancel"), this);

	size = help->sizeHint().expandedTo(send->sizeHint().expandedTo(cancel->sizeHint()));

	help->setFixedSize(size);
	send->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addWidget(help);
	hbox->addStretch();
	hbox->addWidget(send);
	hbox->addWidget(cancel);

	vbox->activate();

	config->setGroup(ID_PREF_GROUP_GENERAL);
	if (config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS)) {
		QToolTip::add(header_to, i18n("Email address of the receiver"));
		QToolTip::add(header_subject, i18n("Subject of the email"));
		QToolTip::add(body, i18n("Body of the email"));
	}

	connect(help, SIGNAL(clicked()), SLOT(slotHelp()));
	connect(send, SIGNAL(clicked()), SLOT(slotSend()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	resize(400, 300);

	setCaption(i18n("Mail"));
}


MailDlg::~MailDlg()
{
}


void MailDlg::mailFax(const QString& name)
{
	QString sender;
	TiffFile tifffile;
	
	tifffile.setName(name);
	if (tifffile.open(IO_ReadOnly)) {
		sender = SENDER_ALIAS(tifffile.sender());
		tifffile.close();
	}
	
	m_name = name;

	if (sender.isEmpty())
		header_subject->setText(i18n("Facsimile"));
	else
		header_subject->setText(i18n("Facsimile from %1").arg(sender));
	
	body->setText(i18n("This is a facsimile file forwarded with KMLOFax."));
}


void MailDlg::slotHelp()
{
	kapp->invokeHelp("MAIL-HELP");
}


void MailDlg::slotSend()
{
	MailClient mclient;
	QString from, server, name, org, email, reply;
	int port;
	Tiff2PSFilter filter;
	FILE *f;
	
	config->setGroup(ID_PREF_GROUP_MAIL);
	server = config->readEntry(ID_PREF_MAIL_SERVER, PREF_MAIL_SERVER);
	port = config->readNumEntry(ID_PREF_MAIL_PORT, PREF_MAIL_PORT);
	name = config->readEntry(ID_PREF_MAIL_NAME, PREF_MAIL_NAME);
	org = config->readEntry(ID_PREF_MAIL_ORG, PREF_MAIL_ORG);
	email = config->readEntry(ID_PREF_MAIL_EMAIL, PREF_MAIL_EMAIL);
	reply = config->readEntry(ID_PREF_MAIL_REPLY, PREF_MAIL_REPLY);
	
	if (server.isEmpty()) {
		KMessageBox::sorry(this, i18n("No SMTP server specified."), i18n("Mail Error"));
		return;
	}

	if (email.isEmpty()) {
		KMessageBox::sorry(this, i18n("No sender specified."), i18n("Mail Error"));
		return;
	}

	if (header_to->text().isEmpty()) {
		KMessageBox::sorry(this, i18n("No receiver specified."), i18n("Mail Error"));
		return;
	}

	if (!mclient.connect(server.latin1(), port)) {
		KMessageBox::sorry(this, i18n("Cannot connect to server."), i18n("SMTP Error"));
		return;
	}

	from = name;

	if (name.isEmpty())
		from = email;
	else {
		from += " <";
		from += email;
		from += ">";
	}

	if (!mclient.sendHeader(from.latin1(), header_to->text().latin1(), header_subject->text().latin1())) {
		KMessageBox::sorry(this, i18n("Cannot send mail header."), i18n("SMTP Error"));
		mclient.disconnect();
		return;
	}

	if (!mclient.sendBody(body->text().latin1())) {
		KMessageBox::sorry(this, i18n("Cannot send mail body."), i18n("SMTP Error"));
		mclient.disconnect();
		return;
	}

	if (!(f = tmpfile())) {
		KMessageBox::sorry(this, i18n("Cannot create temporary file."), i18n("SMTP Error"));
		mclient.disconnect();
		return;
	}
	
	filter.setFile(m_name);
	
	if (!filter.convertFile(f)) {
		fclose(f);
		mclient.disconnect();
		return;
	}

	if (!mclient.sendFile(f)) {
		KMessageBox::sorry(this, i18n("Cannot send file."), i18n("SMTP Error"));
	}

	mclient.disconnect();

	accept();
}
