/*
 * KMLOFax
 *
 * A utility to process facsimiles received with the ELSA
 * MicroLink(tm) Office or MicroLink(tm) ISDN Office modem.
 *
 * Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen.
 * PostScript(R) is a registered trademark of Adobe Systems Incorporated.
 */

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <qlayout.h>
#include <qgroupbox.h>
#include <qtooltip.h>
#include <qsize.h>
#include <qdir.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kseparator.h>

#include "printdlg.h"
#include "global.h"
#include "mlofile.h"
#include "filters.h"


#include <qlistview.h>
#include <qtextstream.h>
#include <qheader.h>
#include <ctype.h>


#if defined(QT_CUPS_SUPPORT)
#include <cups/cups.h>
#endif


// -- Taken from qprintdialog.cpp, Qt-2.2.3 (begin) --------------------------

static void perhapsAddPrinter(QListView *printers, const QString& name, QString host, QString comment)
{
	const QListViewItem *i = printers->firstChild();

	while (i && (i->text(0) != name))
		i = i->nextSibling();
	if (i)
		return;
	if (host.isEmpty())
		host = qApp->translate("QPrintDialog", "locally connected");
	(void)new QListViewItem(printers, name.simplifyWhiteSpace(), host.simplifyWhiteSpace(), comment.simplifyWhiteSpace());
}


static void parsePrintcap(QListView *printers)
{
	QFile printcap(QString::fromLatin1("/etc/printcap"));
	if (!printcap.open(IO_ReadOnly))
		return;

	char * line = new char[1025];
	line[1024] = '\0';

	QString printerDesc;
	int lineLength = 0;

	while (!printcap.atEnd() && (lineLength=printcap.readLine(line, 1024)) > 0) {
		if (*line == '#') {
			*line = '\0';
			lineLength = 0;
		}
		if (lineLength >= 2 && line[lineLength-2] == '\\') {
			line[lineLength-2] = '\0';
			printerDesc += QString::fromLocal8Bit(line);
		}
		else {
			printerDesc += QString::fromLocal8Bit(line);
			printerDesc = printerDesc.simplifyWhiteSpace();
			int i = printerDesc.find(':');
			QString printerName, printerComment, printerHost;
			if (i >= 0) {
				// have : want |
				int j = printerDesc.find('|');
				printerName = printerDesc.left(j > 0 ? j : i);
				if (j > 0) {
					// try extracting a comment from the aliases...
					printerComment = qApp->translate("QPrintDialog", "Aliases: ");
					printerComment += printerDesc.mid(j+1, i-j-1);
					j = printerComment.length();
					while (j > 0) {
						j--;
						if (printerComment[j] == '|')
							printerComment[j] = ',';
					}
				}
				// then look for a real comment
				j = i+1;
				while (printerDesc[j].isSpace())
					j++;
				if (printerDesc[j] != ':') {
					printerComment = printerDesc.mid(i, j-i);
					printerComment = printerComment.simplifyWhiteSpace();
				}
				// look for lprng psuedo all printers entry
				i = printerDesc.find(QRegExp(": *all *="));
				if (i >= 0)
					printerName = "";
				// look for signs of this being a remote printer
				i = printerDesc.find(QRegExp(QString::fromLatin1(": *rm *=")));
				if (i >= 0) {
					// point k at the end of remote host name
					while (printerDesc[i] != '=')
						i++;
					while (printerDesc[i] == '=' || printerDesc[i].isSpace())
						i++;
					j = i;
					while (j < (int)printerDesc.length() && printerDesc[j] != ':')
						j++;
						
					// and stuff that into the string
					printerHost = printerDesc.mid(i, j-i);
				}
			}
			if (printerName.length())
				perhapsAddPrinter(printers, printerName, printerHost, printerComment);
			// chop away the line, for processing the next one
			printerDesc = "";
		}
	}
	delete[] line;
}


// solaris, not 2.6
static void parseEtcLpPrinters(QListView *printers)
{
	QDir lp(QString::fromLatin1("/etc/lp/printers"));
	const QFileInfoList *dirs = lp.entryInfoList();
	if (!dirs)
		return;

	QFileInfoListIterator it(*dirs);
	QFileInfo *printer;
	QString tmp;
	while ((printer = it.current()) != 0) {
		++it;
		if (printer->isDir()) {
			tmp.sprintf("/etc/lp/printers/%s/configuration", printer->fileName().ascii());
			QFile configuration(tmp);
			char *line = new char[1025];
			QRegExp remote(QString::fromLatin1("^Remote:"));
			QRegExp contentType(QString::fromLatin1("^Content types:"));
			QString printerHost;
			bool canPrintPostscript = FALSE;
			if (configuration.open(IO_ReadOnly)) {
				while (!configuration.atEnd() && configuration.readLine(line, 1024) > 0) {
					if (remote.match(QString::fromLatin1(line)) == 0) {
						const char *p = line;
						while (*p != ':')
							p++;
						p++;
						while (isspace(*p))
							p++;
						printerHost = QString::fromLocal8Bit(p);
						printerHost = printerHost.simplifyWhiteSpace();
					}
					else if (contentType.match(QString::fromLatin1(line)) == 0) {
						char *p = line;
						while (*p != ':')
							p++;
						p++;
						char *e;
						while (*p) {
							while (isspace(*p))
								p++;
							if (*p) {
								char s;
								e = p;
								while (isalnum(*e))
									e++;
								s = *e;
								*e = '\0';
								if (!qstrcmp(p, "postscript") || !qstrcmp(p, "any"))
									canPrintPostscript = TRUE;
								*e = s;
								if (s == ',')
									e++;
								p = e;
							}
						}
					}
				}
				if (canPrintPostscript)
					perhapsAddPrinter(printers, printer->fileName(), printerHost, QString::fromLatin1(""));
			}
			delete[] line;
		}
	}
}


// solaris 2.6
static char *parsePrintersConf(QListView *printers)
{
	QFile pc(QString::fromLatin1("/etc/printers.conf"));
	if (!pc.open(IO_ReadOnly))
		return 0;

	char *line = new char[1025];
	line[1024] = '\0';

	QString printerDesc;
	int lineLength = 0;

	char *defaultPrinter = 0;

	while (!pc.atEnd() && (lineLength=pc.readLine(line, 1024)) > 0) {
		if (*line == '#') {
			*line = '\0';
			lineLength = 0;
		}
		if (lineLength >= 2 && line[lineLength-2] == '\\') {
			line[lineLength-2] = '\0';
			printerDesc += QString::fromLocal8Bit(line);
		}
		else {
			printerDesc += QString::fromLocal8Bit(line);
			printerDesc = printerDesc.simplifyWhiteSpace();
			int i = printerDesc.find(':');
			QString printerName, printerHost, printerComment;
			if (i >= 0) {
				// have : want |
				int j = printerDesc.find('|', 0);
				if (j >= i)
					j = -1;
				printerName = printerDesc.mid(0, j < 0 ? i : j);
				if (printerName == QString::fromLatin1("_default")) {
					i = printerDesc.find(QRegExp(QString::fromLatin1(": *use *=")));
					while (printerDesc[i] != '=')
						i++;
					while (printerDesc[i] == '=' || printerDesc[i].isSpace())
						i++;
					j = i;
					while (j < (int)printerDesc.length() && printerDesc[j] != ':' && printerDesc[j] != ',')
						j++;
					// that's our default printer
					defaultPrinter = qstrdup(printerDesc.mid(i, j-i).ascii());
					printerName = "";
					printerDesc = "";
				}
				else if (printerName == QString::fromLatin1("_all")) {
					// skip it.. any other cases we want to skip?
					printerName = "";
					printerDesc = "";
				}

				if (j > 0) {
					// try extracting a comment from the aliases...
					printerComment = qApp->translate("QPrintDialog", "Aliases: ");
					printerComment += printerDesc.mid(j+1, i-j-1);
					for (j=printerComment.length(); j>-1; j--)
						if (printerComment[j] == '|')
							printerComment[j] = ',';
				}
				// look for signs of this being a remote printer
				i = printerDesc.find(QRegExp(QString::fromLatin1(": *bsdaddr *=")));
				if (i >= 0) {
					// point k at the end of remote host name
					while (printerDesc[i] != '=')
						i++;
					while (printerDesc[i] == '=' || printerDesc[i].isSpace())
						i++;
					j = i;
					while (j < (int)printerDesc.length() && printerDesc[j] != ':' && printerDesc[j] != ',')
						j++;
					// and stuff that into the string
					printerHost = printerDesc.mid(i, j-i);
					// maybe stick the remote printer name into the comment
					if (printerDesc[j] == ',') {
						i = ++j;
						while (printerDesc[i].isSpace())
							i++;
						j = i;
						while (j < (int)printerDesc.length() && printerDesc[j] != ':' && printerDesc[j] != ',')
							j++;
						if (printerName != printerDesc.mid(i, j-i)) {
							printerComment = QString::fromLatin1("Remote name: ");
							printerComment += printerDesc.mid(i, j-i);
						}
					}
				}
			}
			if (printerComment == ":")
				printerComment = ""; // for cups
			if (printerName.length())
				perhapsAddPrinter(printers, printerName, printerHost, printerComment);
			// chop away the line, for processing the next one
			printerDesc = "";
		}
	}
	delete[] line;
	return defaultPrinter;
}


// HP-UX
static void parseEtcLpMember(QListView *printers)
{
	QDir lp(QString::fromLatin1("/etc/lp/member"));
	if (!lp.exists())
		return;
	const QFileInfoList *dirs = lp.entryInfoList();
	if (!dirs)
		return;

	QFileInfoListIterator it(*dirs);
	QFileInfo *printer;
	QString tmp;
	while ((printer = it.current()) != 0) {
		++it;
		// uglehack.
		// I haven't found any real documentation, so I'm guessing that
		// since lpstat uses /etc/lp/member rather than one of the
		// other directories, it's the one to use.  I did not find a
		// decent way to locate aliases and remote printers.
		if (printer->isFile())
			perhapsAddPrinter(printers, printer->fileName(), qApp->translate("QPrintDialog","unknown"), QString::fromLatin1(""));
	}
}

// IRIX 6.x
static void parseSpoolInterface(QListView *printers)
{
	QDir lp(QString::fromLatin1("/usr/spool/lp/interface"));
	if (!lp.exists())
		return;
	const QFileInfoList * files = lp.entryInfoList();
	if (!files)
		return;

	QFileInfoListIterator it(*files);
	QFileInfo *printer;
	while ((printer = it.current()) != 0) {
		++it;

		if (!printer->isFile())
			continue;

		// parse out some information
		QFile configFile(printer->filePath());
		if (!configFile.open(IO_ReadOnly))
			continue;

		QCString line(1025);
		QString hostName;
		QString hostPrinter;
		QString printerType;

		QRegExp typeKey(QString::fromLatin1("^TYPE="));
		QRegExp hostKey(QString::fromLatin1("^HOSTNAME="));
		QRegExp hostPrinterKey(QString::fromLatin1("^HOSTPRINTER="));
		int length;

		while (!configFile.atEnd() && (configFile.readLine(line.data(), 1024)) > 0) {
			if (typeKey.match(line, 0, &length) == 0) {
				printerType = line.mid(length, line.length()-length);
				printerType = printerType.simplifyWhiteSpace();
			}
			if (hostKey.match(line, 0, &length) == 0) {
				hostName = line.mid(length, line.length()-length);
				hostName = hostName.simplifyWhiteSpace();
			}
			if (hostPrinterKey.match(line, 0, &length) == 0) {
				hostPrinter = line.mid(length, line.length()-length);
				hostPrinter = hostPrinter.simplifyWhiteSpace();
			}
		}
		configFile.close();

		printerType = printerType.stripWhiteSpace();
		if (!printerType.isEmpty() && qstricmp(printerType.ascii(), "postscript"))
			continue;

		if (hostName.isEmpty() || hostPrinter.isEmpty()) {
			perhapsAddPrinter(printers, printer->fileName(), QString::fromLatin1(""), QString::fromLatin1(""));
		}
		else {
			QString comment = QString::fromLatin1("Remote name: ");
			comment += hostPrinter;
			perhapsAddPrinter(printers, printer->fileName(), hostName, comment);
		}
	}
}


// Every unix must have its own.  It's a standard.  Here is AIX.
static void parseQconfig(QListView *printers)
{
	QFile qconfig(QString::fromLatin1("/etc/qconfig"));
	if (!qconfig.open(IO_ReadOnly))
		return;

	QTextStream ts(&qconfig);
	QString line;

	QString stanzaName; // either a queue or a device name
	bool up = TRUE; // queue up?  default TRUE, can be FALSE
	QString remoteHost; // null if local
	QString deviceName; // null if remote

	QRegExp newStanza(QString::fromLatin1("^[0-z][0-z]*:$"));

	// our basic strategy here is to process each line, detecting new
	// stanzas.  each time we see a new stanza, we check if the
	// previous stanza was a valid queue for a) a remote printer or b)
	// a local printer.  if it wasn't, we assume that what we see is
	// the start of the first stanza, or that the previous stanza was
	// a device stanza, or that there is some syntax error (we don't
	// report those).

	do {
		line = ts.readLine();
		bool indented = line[0].isSpace();
		line = line.simplifyWhiteSpace();

		if (indented && line.contains('=')) { // line in stanza

			int i = line.find('=');
			QString variable = line.left(i).simplifyWhiteSpace();
			QString value=line.mid(i+1, line.length()).simplifyWhiteSpace();
			if (variable == QString::fromLatin1("device"))
				deviceName = value;
			else if (variable == QString::fromLatin1("host"))
				remoteHost = value;
			else if (variable == QString::fromLatin1("up"))
				up = !(value.lower() == QString::fromLatin1("false"));
		}
		else if (line[0] == '*') { // comment
			// nothing to do
		}
		else if (ts.atEnd() || // end of file, or beginning of new stanza
			(!indented && line.contains(newStanza))) {
			if (up && stanzaName.length() > 0 && stanzaName.length() < 21) {
				if (remoteHost.length()) // remote printer
					perhapsAddPrinter(printers, stanzaName, remoteHost, QString::null);
				else if (deviceName.length()) // local printer
					perhapsAddPrinter(printers, stanzaName, QString::null, QString::null);
			}
			line.truncate(line.length()-1);
			if (line.length() >= 1 && line.length() <= 20)
				stanzaName = line;
			up = TRUE;
			remoteHost = QString::null;
			deviceName = QString::null;
		}
		else {
			// syntax error?  ignore.
		}
	} while(!ts.atEnd());
}


static char *parseCupsOutput(QListView *printers)
{
	char *defaultPrinter = 0;
#if defined(QT_CUPS_SUPPORT)
	int nd;
	cups_dest_t *d;
	nd = cupsGetDests(&d);
	if (nd < 1)
		return 0;

	int n = 0;
	while (n < nd) {
		perhapsAddPrinter(printers, d[n].name, qApp->translate("QPrintDialog", "Unknown Location"), 0);
		if (d[n].is_default && !defaultPrinter)
			defaultPrinter = qstrdup(d[n].instance);
		n++;
	}
#else
	Q_UNUSED(printers);
#endif
	return defaultPrinter;
}
// -- Taken from qprintdialog.cpp, Qt-2.2.3 (end) ----------------------------



PrintDlg::PrintDlg(QWidget *parent, const char *name) : KDialog (parent, name, true)
{
	QBoxLayout *vbox, *hbox, *fvbox, *fhbox, *fhbox2;
	QRadioButton *rb;
	QGroupBox *gb;
	QPushButton *help, *cancel;
	QLabel *l;
	QSize size;

	config = kapp->config();

	pid = 0;
	to_printer = true;
	print_range = false;

	vbox = new QVBoxLayout(this, 12, 0);

	destination = new QButtonGroup(this);
	destination->setTitle(i18n("Print Destination"));
	fvbox = new QVBoxLayout(destination, 12, 0);
	fvbox->addSpacing(8);
	rb = new QRadioButton(i18n("Print to printer:"), destination);
	rb->setMinimumSize(rb->sizeHint());
	rb->setChecked(true);
	fvbox->addWidget(rb);
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	fhbox->addSpacing(19);

// -- Taken from qprintdialog.cpp, Qt-2.2.3 (begin) --------------------------

	printers = new QListView(destination);
	printers->setAllColumnsShowFocus(true);
	printers->addColumn(i18n("Printer"), 125);
	printers->addColumn(i18n("Host"), 125);
	printers->addColumn(i18n("Comment"), 150);
	printers->setFrameStyle(QFrame::WinPanel | QFrame::Sunken);
	
	char * etcLpDefault = 0;

	etcLpDefault = parseCupsOutput(printers);
	if (printers->childCount() == 0) {
	// we only use other schemes when cups fails.
		parsePrintcap(printers);
		parseEtcLpMember(printers);
		parseSpoolInterface(printers);
		parseQconfig(printers);

		QFileInfo f;
		f.setFile(QString::fromLatin1("/etc/lp/printers"));
		if (f.isDir()) {
			parseEtcLpPrinters(printers);
			QFile def(QString::fromLatin1("/etc/lp/default"));
			if (def.open(IO_ReadOnly)) {
				if (etcLpDefault)
					delete[] etcLpDefault;
				etcLpDefault = new char[1025];
				def.readLine(etcLpDefault, 1024);
				char *p = etcLpDefault;
				while(p && *p) {
					if (!isprint(*p) || isspace(*p))
						*p = 0;
					else
						p++;
				}
			}
		}

		f.setFile(QString::fromLatin1("/etc/printers.conf"));
		if (f.isFile()) {
			char * def = parsePrintersConf(printers);
			if (def) {
				if (etcLpDefault)
					delete[] etcLpDefault;
				etcLpDefault = def;
			}
		}
	}

	// all printers hopefully known.  try to find a good default
	QString dollarPrinter;
	{
		char * t;
		t = getenv("PRINTER");
		if (!t || !*t)
			t = getenv("LPDEST");
		dollarPrinter = QString::fromLatin1(t);
	}
	int quality = 0;

	// bang the best default into the listview
	const QListViewItem *lvi = printers->firstChild();
	printers->setCurrentItem((QListViewItem *)lvi);
	while (lvi) {
		QRegExp ps1(QString::fromLatin1("[^a-z]ps[^a-z]"));
		QRegExp ps2(QString::fromLatin1("[^a-z]ps$"));
		QRegExp lp1(QString::fromLatin1("[^a-z]lp[^a-z]"));
		QRegExp lp2(QString::fromLatin1("[^a-z]lp$"));
		if (quality < 4 && lvi->text(0) == dollarPrinter) {
			printers->setCurrentItem((QListViewItem *)lvi);
			quality = 4;
		}
		else if (quality < 3 && etcLpDefault && lvi->text(0) == QString::fromLatin1(etcLpDefault)) {
			printers->setCurrentItem((QListViewItem *)lvi);
			quality = 3;
		}
		else if (quality < 2 && (lvi->text(0) == QString::fromLatin1("ps") ||
			ps1.match(lvi->text(2)) > -1 || ps2.match(lvi->text(2)) > -1)) {
			printers->setCurrentItem((QListViewItem *)lvi);
			quality = 2;
		}
		else if (quality < 1 && (lvi->text(0) == QString::fromLatin1("lp") ||
			lp1.match(lvi->text(2)) > -1 || lp2.match(lvi->text(2)) > -1)) {
			printers->setCurrentItem((QListViewItem *)lvi);
			quality = 1;
		}
		lvi = lvi->nextSibling();
	}
	if (printers->currentItem())
		printers->setSelected(printers->currentItem(), true);

	if (etcLpDefault)			// Avoid purify complaint
		delete[] etcLpDefault;

// -- Taken from qprintdialog.cpp, Qt-2.2.3 (end) ----------------------------

	int h = fontMetrics().height();
	if (printers->firstChild())
		h = printers->firstChild()->height();
	printers->setMinimumSize(printers->sizeHint().width(), printers->header()->height() + 3 * h);
	fhbox->addWidget(printers, 3);

	fvbox->addSpacing(6);
	
	rb = new QRadioButton(i18n("Print to file:"), destination);
	rb->setMinimumSize(rb->sizeHint());
	fvbox->addWidget(rb);
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	fhbox->addSpacing(19);
	file = new QLineEdit(destination);
	file->setMinimumWidth(file->sizeHint().width());
	file->setFixedHeight(file->sizeHint().height());
	file->setText(QDir::currentDirPath() + "/facsimile.ps");
	file->setEnabled(false);
	fhbox->addWidget(file);

	fhbox->addSpacing(6);

	browse = new QPushButton(destination);
	browse->setPixmap(SmallIcon("fileopen"));
	browse->setAutoDefault(false);
	browse->setFixedSize(browse->sizeHint()+QSize(6, 6));	
	browse->setEnabled(false);
	fhbox->addWidget(browse);
	
	fvbox->activate();
	vbox->addWidget(destination, 1);

	vbox->addSpacing(12);
	
	hbox = new QHBoxLayout();
	vbox->addLayout(hbox);

	gb = new QGroupBox(i18n("Printer Settings"), this);
	fvbox = new QVBoxLayout(gb, 12, 0);
	fvbox->addSpacing(8);

	level2 = new QCheckBox(i18n("Language Level 2"), gb);
	level2->setMinimumSize(level2->sizeHint());
	fvbox->addWidget(level2);
	fvbox->addSpacing(8);

	interpolate = new QCheckBox(i18n("Interpolate"), gb);
	interpolate->setMinimumSize(interpolate->sizeHint());
	fvbox->addWidget(interpolate);

	hbox->addWidget(gb);

	hbox->addSpacing(12);
			
	gb = new QGroupBox(i18n("Paper Format"), this);
	fvbox = new QVBoxLayout(gb, 12, 0);
	fvbox->addSpacing(8);
	
	orientation = new QComboBox(false, gb);
	orientation->insertItem(i18n("Portrait"), PAPER_PORTRAIT);
	orientation->insertItem(i18n("Landscape"), PAPER_LANDSCAPE);
	fvbox->addWidget(orientation);

	fvbox->addSpacing(8);
	
	paper_format = new QComboBox(false, gb);
	paper_format->insertItem(QString("%1 (210297 %2)").arg(i18n("A4")).arg(i18n("mm")), PAPER_A4);
	paper_format->insertItem(QString("%1 (257364 %2)").arg(i18n("B4")).arg(i18n("mm")), PAPER_B4);
	paper_format->insertItem(QString("%1 (182257 %2)").arg(i18n("B5")).arg(i18n("mm")), PAPER_B5);
	paper_format->insertItem(QString("%1 (811 %2)").arg(i18n("Letter")).arg(i18n("inches")), PAPER_LETTER);
	paper_format->insertItem(QString("%1 (814 %2)").arg(i18n("Legal")).arg(i18n("inches")), PAPER_LEGAL);
	paper_format->insertItem(QString("%1 (710 %2)").arg(i18n("Executive")).arg(i18n("inches")), PAPER_EXECUTIVE);
	fvbox->addWidget(paper_format);

	fvbox->activate();

	hbox->addWidget(gb);

	vbox->addSpacing(12);

	gb = new QGroupBox(i18n("Options"), this);
	fhbox = new QHBoxLayout(gb, 12, 2);

	fvbox = new QVBoxLayout();
	fhbox->addLayout(fvbox);

	fvbox->addSpacing(8);
	
	range = new QButtonGroup(this);
	range->hide();
	prt_all = new QRadioButton(i18n("Print all"), gb);
	prt_all->setChecked(true);
	range->insert(prt_all, 0);
	fvbox->addWidget(prt_all);
	prt_range = new QRadioButton(i18n("Print range"), gb);
	range->insert(prt_range, 1);
	fvbox->addWidget(prt_range);
	
	fhbox2 = new QHBoxLayout();
	fvbox->addLayout(fhbox2);

	fhbox2->addSpacing(19);
	
	from_page_label = new QLabel(i18n("From page:"), gb);
	from_page_label->setEnabled(false);
	fhbox2->addWidget(from_page_label);
	from_page = new QSpinBox(1, 999, 1, gb);
	from_page->setEnabled(false);
	from_page_label->setBuddy(from_page);
	from_page->setValue(1);
	from_page->setMinimumWidth(from_page->sizeHint().width());
	fhbox2->addWidget(from_page, 1);

	fhbox2 = new QHBoxLayout();
	fvbox->addLayout(fhbox2);

	fhbox2->addSpacing(19);
	
	to_page_label = new QLabel(i18n("To page:"), gb);
	to_page_label->setEnabled(false);
	fhbox2->addWidget(to_page_label);
	to_page = new QSpinBox(1, 999, 1, gb);
	to_page->setEnabled(false);
	to_page_label->setBuddy(to_page);
	to_page->setValue(999);
	to_page->setMinimumWidth(to_page->sizeHint().width());
	fhbox2->addWidget(to_page, 1);

	fhbox->addSpacing(25);

	fvbox = new QVBoxLayout();
	fhbox->addLayout(fvbox);

	fvbox->addSpacing(8);
	
	fhbox2 = new QHBoxLayout();
	fvbox->addLayout(fhbox2);
	
	l = new QLabel(i18n("Number of copies:"), gb);
	fhbox2->addWidget(l);
	copies = new QSpinBox(1, 99, 1, gb);
	l->setBuddy(copies);
	copies->setValue(1);
	copies->setMinimumWidth(copies->sizeHint().width());
	fhbox2->addWidget(copies, 1);

	size = from_page_label->sizeHint().expandedTo(to_page_label->sizeHint()).expandedTo(l->sizeHint());
	from_page_label->setMinimumSize(size);
	to_page_label->setMinimumSize(size);
	l->setMinimumSize(size.width() + 19, size.height());
	
	fvbox->addStretch(1);
	
	ribbons = new QCheckBox(i18n("Print FAX ribbons"), gb);
	ribbons->setMinimumSize(ribbons->sizeHint());
	fvbox->addWidget(ribbons);

	fhbox->activate();

	vbox->addWidget(gb);

	vbox->addSpacing(12);
	
	hbox = new QHBoxLayout(12);
	vbox->addLayout(hbox);

	help = new QPushButton(i18n("&Help"), this);
	ok = new QPushButton(i18n("&OK"), this);
	ok->setDefault(true);
	cancel = new QPushButton(i18n("&Cancel"), this);
	size = help->sizeHint().expandedTo(ok->sizeHint().expandedTo(cancel->sizeHint()));

	help->setFixedSize(size);
	ok->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addWidget(help);
	hbox->addStretch();
	hbox->addWidget(ok);
	hbox->addWidget(cancel);

	vbox->activate();

	config->setGroup(ID_PREF_GROUP_GENERAL);
	if (config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS)) {
		QToolTip::add(browse, i18n("Select file to print to"));
		QToolTip::add(level2, i18n("Use PostScript Language Level 2"));
		QToolTip::add(interpolate, i18n("Interpolate for smoother output"));
		QToolTip::add(orientation, i18n("Orientation of the paper"));
		QToolTip::add(paper_format, i18n("Paper format used by the printer"));
		QToolTip::add(prt_all, i18n("Print the whole document"));
		QToolTip::add(prt_range, i18n("Print only the specified range"));
		QToolTip::add(from_page, i18n("First page to print"));
		QToolTip::add(to_page, i18n("Last page to print"));
		QToolTip::add(copies, i18n("Number of copies to print"));
		QToolTip::add(ribbons, i18n("Print FAX ribbons"));
	}

	readOptions();

	connect(destination, SIGNAL(clicked(int)), SLOT(destinationSelected(int)));
	connect(file, SIGNAL(textChanged(const QString&)), SLOT(fileNameChanged(const QString&)));
	connect(browse, SIGNAL(clicked()), SLOT(slotBrowse()));
	connect(level2, SIGNAL(toggled(bool)), SLOT(level2Selected(bool)));
	connect(range, SIGNAL(clicked(int)), SLOT(rangeSelected(int)));
	connect(from_page, SIGNAL(valueChanged(int)), SLOT(fromPageChanged(int)));
	connect(to_page, SIGNAL(valueChanged(int)), SLOT(toPageChanged(int)));
	connect(help, SIGNAL(clicked()), SLOT(slotHelp()));
	connect(ok, SIGNAL(clicked()), SLOT(slotOk()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	setCaption(i18n("Print"));

	resize(minimumSize());
}


PrintDlg::~PrintDlg()
{
	if (pid) {
		(void)::kill(pid, 6);
		(void)::wait(0);
		pid = 0;
	}
}


void PrintDlg::printFax(const QString& name)
{
	TiffFile mlofile;

	mlofile.setName(name);
	if (!mlofile.open(IO_ReadOnly)) {
		KMessageBox::sorry(this, i18n("Cannot open facsimile file."), i18n("File Error"));
		return;
	}

	max_page = mlofile.pages();
	mlofile.close();

	from_page->setRange(1, max_page);
	to_page->setRange(1, max_page);
	from_page->setValue(1);
	to_page->setValue(max_page);

	m_name = name;
}


void PrintDlg::readOptions()
{
	QListViewItem *item;
	QString prt, host;

	config->setGroup(ID_PREF_GROUP_PRINT);
	prt = config->readEntry(ID_PREF_PRT_PRINTER);
	host = config->readEntry(ID_PREF_PRT_HOST);
	for (item = printers->firstChild(); (item); item = item->nextSibling()) {
		if ((prt == item->text(0)) && (host == item->text(1))) {
			printers->setCurrentItem(item);
			printers->setSelected(item, true);
			break;
		}
	}
	if (printers->currentItem())
		printers->ensureItemVisible(printers->currentItem());
	level2->setChecked(config->readBoolEntry(ID_PREF_PRT_LEVEL2, PREF_PRT_LEVEL2));
	interpolate->setChecked(config->readBoolEntry(ID_PREF_PRT_INTERPOL, PREF_PRT_INTERPOL));
	interpolate->setEnabled(level2->isChecked());
	orientation->setCurrentItem(config->readNumEntry(ID_PREF_PRT_ORIENT, PREF_PRT_ORIENT));
	paper_format->setCurrentItem(config->readNumEntry(ID_PREF_PRT_PAPER, PREF_PRT_PAPER));
	copies->setValue(config->readNumEntry(ID_PREF_PRT_COPIES, PREF_PRT_COPIES));
	ribbons->setChecked(config->readBoolEntry(ID_PREF_PRT_RIBBONS, PREF_PRT_RIBBONS));
}


void PrintDlg::saveOptions()
{
	QListViewItem *item;

	config->setGroup(ID_PREF_GROUP_PRINT);
	item = printers->currentItem();
	if (item) {
		config->writeEntry(ID_PREF_PRT_PRINTER, item->text(0));
		config->writeEntry(ID_PREF_PRT_HOST, item->text(1));
	}
	config->writeEntry(ID_PREF_PRT_LEVEL2, level2->isChecked());
	config->writeEntry(ID_PREF_PRT_INTERPOL, interpolate->isChecked());
	config->writeEntry(ID_PREF_PRT_ORIENT, orientation->currentItem());
	config->writeEntry(ID_PREF_PRT_PAPER, paper_format->currentItem());
	config->writeEntry(ID_PREF_PRT_COPIES, copies->value());
	config->writeEntry(ID_PREF_PRT_RIBBONS, ribbons->isChecked());
}


void PrintDlg::slotBrowse()
{
	QString s = file->text();

	if (s.isEmpty())
		s = QDir::currentDirPath();

	s = KFileDialog::getSaveFileName(s, i18n("*.ps|PostScript Files (*.ps)\n*|All Files (*)"), this);
	if (!s.isEmpty())
		file->setText(s);
}


void PrintDlg::slotHelp()
{
	kapp->invokeHelp("PRINT-HELP");
}


void PrintDlg::slotOk()
{
	Tiff2PSFilter *filter;
	FILE *ps;

	filter = new Tiff2PSFilter(m_name);
	filter->setFormat(paper_format->currentItem(), orientation->currentItem() == PAPER_LANDSCAPE);
	filter->setLevel2(level2->isChecked());
	filter->setInterpolate(interpolate->isChecked());
	filter->setRange(from_page->value(), to_page->value());
	filter->setCopies(copies->value());
	filter->setRibbons(ribbons->isChecked());
	
	config->setGroup(ID_PREF_GROUP_PRINT);
	filter->setMargins(
		config->readNumEntry(ID_PREF_PRT_LMARGIN, PREF_PRT_LMARGIN),
		config->readNumEntry(ID_PREF_PRT_RMARGIN, PREF_PRT_RMARGIN),
		config->readNumEntry(ID_PREF_PRT_TMARGIN, PREF_PRT_TMARGIN),
		config->readNumEntry(ID_PREF_PRT_BMARGIN, PREF_PRT_BMARGIN)
	);

	
	if (to_printer) {
		QListViewItem *l = printers->currentItem();
		if (l) {
			int fds[2];
			if (pipe(fds) != 0) {
					// Could not open pipe to print
			}
			pid = fork();
			if (pid == 0) {
				if (fork() > 0)
					exit(0);
				dup2(fds[0], 0);
				
				int i;
#if defined(_SC_OPEN_MAX)
				i = (int)sysconf(_SC_OPEN_MAX);
#elif defined(_POSIX_OPEN_MAX)
				i = (int)_POSIX_OPEN_MAX;
#elif defined(OPEN_MAX)
				i = (int)OPEN_MAX;
#else
				i = QMAX(256, fds[0]);
#endif
				while(--i > 0)
					::close(i);

				QString lparg = QString("-d") + l->text(0);
				QString lprarg = QString("-P") + l->text(0);

				(void)execlp("lp", "lp", lparg.ascii(), 0);
				(void)execlp("lpr", "lpr", lprarg.ascii(), 0);
				(void)execl("/bin/lp", "lp", lparg.ascii(), 0);
				(void)execl("/bin/lpr", "lpr", lprarg.ascii(), 0);
				(void)execl("/usr/bin/lp", "lp", lparg.ascii(), 0);
				(void)execl("/usr/bin/lpr", "lpr", lprarg.ascii(), 0);
				::close(0);
				(void)sleep(1);
				::exit(0);
			}
			::close(fds[0]);
			ps = fdopen(fds[1], "w");
			filter->convertFile(ps);
			fclose(ps);
/*			
			if (ps <= 0)
				KMessageBox::sorry(this, i18n("Cannot execute print command."), i18n("Printer Error"));
			else {
				filter->convertFile(ps);
				::close(ps);
			}
*/
		}
	}
	else {
		if ((ps = fopen(file->text().local8Bit(), "w")) == 0) {
			KMessageBox::sorry(this, i18n("Cannot create file for output."), i18n("Printer Error"));
		}
		else {
			kapp->setOverrideCursor(waitCursor);
			filter->convertFile(ps);
			fclose(ps);
			kapp->restoreOverrideCursor();
		}
	}		

	delete filter;

	saveOptions();

	accept();
}


void PrintDlg::destinationSelected(int id_)
{
	to_printer = (id_ == 0);
	
	printers->setEnabled(to_printer);
	file->setEnabled(!to_printer);
	if (to_printer) {
		if (file->hasFocus() || browse->hasFocus())
			printers->setFocus();
	}
	else
		file->setFocus();
	browse->setEnabled(!to_printer);
	ok->setEnabled(to_printer ? true : !file->text().isEmpty());
}


void PrintDlg::fileNameChanged(const QString& text)
{
	ok->setEnabled(!text.isEmpty());
}


void PrintDlg::level2Selected(bool on)
{
	interpolate->setEnabled(on);
}


void PrintDlg::rangeSelected(int id_)
{
	bool print_range = (id_ != 0);

	from_page_label->setEnabled(print_range);
	from_page->setEnabled(print_range);
	to_page_label->setEnabled(print_range);
	to_page->setEnabled(print_range);
}


void PrintDlg::fromPageChanged(int fp)
{
	to_page->setRange(fp, max_page);
}


void PrintDlg::toPageChanged(int tp)
{
	from_page->setRange(1, tp);
}
