/*
 *  KSeg
 *  Copyright (C) 1999 Ilya Baran
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 ibaran@mit.edu
 */


#include "G_ref.H"
#include "KSegDocument.H"
#include "KSegConstruction.H"
#include "G_object.H"
#include <qptrdict.h>
#include "G_drawstyle.H"
#include <qpainter.h>
#include "G_undo.H"
#include <hash_set>

G_ref::G_ref(KSegDocument *inDoc)
  : label(this)
{
  doc = inDoc;
  where = NULL;
  exists = visible = true;
  labelVisible = given = final = deleted = selected = false;
  drawstyle = G_drawstyle::match();
}

G_ref::~G_ref()
{
  drawstyle->deleteReference();
  if(where != NULL && !deleted) delete where;
}

void G_ref::create(G_Type inType, int inDescendType, const G_refs &inParents, KSegDocument *inDoc)
{

  type = inType;
  descendType = inDescendType;

  parents = inParents;

  doc = inDoc;

  drawstyle->deleteReference();
  drawstyle = inDoc->getDefaultDrawstyle();
  drawstyle->addReference();

  if(type == G_LOOP) visible = false;

  createObject();

  update();

  labelVisible = false;
  label.setString(doc->autoLabel(this));
}


void G_ref::createObject()
{
  deleted = false;
  selected = false;

  if(doc) {
    doc->addRef(this);
    doc->addUndo(new G_undoCreate(this));
  }

  int i;
  for(i = 0; i < (int)parents.count(); i++) {
    parents[i]->children.append(this);
  }

  //create new object
  switch(type) {
  case G_POINT:
    where = new G_pointObject(this);
    break;
  case G_LINE:
    where = new G_lineObject(this);
    break;
  case G_SEGMENT:
    where = new G_segmentObject(this);
    break;
  case G_RAY:
    where = new G_rayObject(this);
    break;
  case G_CIRCLE:
    where = new G_circleObject(this);
    break;
  case G_ARC:
    where = new G_arcObject(this);
    break;
  case G_LOCUS:
    where = new G_locusObject(this);
    break;
  case G_MEASURE:
    where = new G_measureObject(this);
    break;
  case G_CALCULATE:
    where = new G_calculateObject(this);
    break;
  case G_LOOP:
    where = NULL;
    break;
  default:
    qFatal("Unknown type %d!\n", type);
    where = NULL;
    break;
  }

  exists = true;
}


void G_ref::remove()
{
  int i;

  ASSERT(children.count() == 0);

  //undo stuff:
  doc->addUndo(new G_undoDelete(this));

  doc->delRef(this);

  for(i = 0; i < (int)parents.count(); i++) {
    parents[i]->children.removeRef(this);
  }

  if(where) delete where;

  deleted = true;
}


void G_ref::reconstrain(int inDescendType, const G_refs &inParents)
{
  int i;

  if(inDescendType == descendType && inParents == parents) return;

  for(i = 0; i < (int)parents.count(); i++) {
    parents[i]->children.removeRef(this);
  }

  parents = inParents;

  for(i = 0; i < (int)parents.count(); i++) {
    parents[i]->children.append(this);
  }

  descendType = inDescendType;

  doc->topSortAllRefs();
}


#define ADD_REF(_ref) if(!added[(void *)(_ref)]) { refs.append((G_ref *)(_ref));  \
  if(refs.count() > Primes[addedIndex]) added.resize(Primes[++addedIndex]); \
  added.insert((void *)(_ref), &dummy); }  // yeah, this "discards const" but nothing will happen to it...

//return true if inRef is an ancestor of this.
bool G_ref::isAncest(const G_ref *inRef) const
{
  G_refs refs;
  int dummy;
  QPtrDict<int> added(Primes[0]);
  int addedIndex = 0;
  
  int i, j;

  if(this == inRef) return false; // strict ancestorship only

  ADD_REF(inRef)

  for(i = 0; i < (int)refs.count(); i++) {
    for(j = 0; j < (int)refs[i]->getChildren().count(); j++) {
      if(refs[i]->getChildren()[j] == this) return true;

      ADD_REF(refs[i]->getChildren()[j])
    }
  } 

  return false;

}

bool G_ref::isDescend(const G_ref *inRef) const
{
  return inRef->isAncest(this);
}

void G_ref::update()
{
  int i;

  if(type == G_LOOP) { visible = false; return; }

  for(i = 0; i < (int)parents.count(); i++) {
    if(!parents[i]->getExists()) {
      if(type != G_LOCUS || descendType != G_OBJECT_LOCUS || i != 1) {
	exists = false;
	return;
      }
    }
  }

  exists = true;

  where->update();
}


bool G_ref::isDirectParent(int which)
{
  if(type & G_VALUE || parents[which]->type & G_VALUE) return true;

  if(descendType == G_REFLECTED) return true;
  
  if(descendType == G_TRANSLATED && which > 0) return false;
  if(descendType == G_TRANSLATED && which == 0) return true;
  
  if((descendType == G_ROTATED || descendType == G_SCALED) && which > 1)
    return false;
  if((descendType == G_ROTATED || descendType == G_SCALED) && which <= 1)
    return true;
  
  if(type == G_LOCUS) {
    if(descendType == G_OBJECT_LOCUS && which != 1) return false;
    return true;
  }
  
  if(type == G_POINT || type == G_SEGMENT || type == G_LOCUS)
    return true;
  
  if(type == G_LINE) {
    if(descendType == G_TWOPOINTS_LINE) return true;
    if(parents[which]->type == G_POINT) return true;
    else return false;
  }

  if(type == G_RAY) return true;

  if(type == G_CIRCLE) {
    if(descendType == G_CENTERPOINT_CIRCLE) return true;
    if(parents[which]->type == G_POINT) return true;
    else return false;
  }
  
  if(type == G_ARC) return true;
  
  if(type == G_POLYGON) return true;
  
  return false;
}


void G_ref::drawLabel(QPainter &p)
{
  if(labelVisible) {
    ASSERT( (type & (G_POINT | G_CURVE)) );

    label.draw(p, *drawstyle, selected);
  }
}

void G_ref::changeDrawstyle(G_drawstyle *d)
{
  if(drawstyle != d) {
    doc->addUndo(new G_undoChangeDrawstyle(this));
    setDrawstyle(d);

    if(!labelVisible) {
      QPixmap tmp(1, 1);
      QPainter tmpp(&tmp);
      label.draw(tmpp, *d, false);
    }
    
    if(type & G_TEXT && !isDrawn()) {
      QPixmap tmp(1, 1);
      QPainter tmpp(&tmp);
      where->draw(tmpp);
    }

  }
  else {
    d->deleteReference();
  }
}


//which types may be substituted for this type if this is a given
int G_ref::whatCanItBe()
{
  static std::hash_set<G_ref *> refsConsidered; // prevents infinite recursion

  //if it's a measurement or calculation, it can be either
  if(type & G_VALUE) return G_VALUE;
  
  //otherwise, if it's not some curve, it can only be itself
  if(!(type & G_CURVE)) return type;

  if(refsConsidered.count(this)) return G_CURVE; //if it's already been considered

  int soFar = G_CURVE; //stores what we haven't eliminated yet
  //now go through the children and eliminate possibilities
  int i;
  for(i = 0; i < (int)children.size(); ++i) {
    if(soFar == type) { //if we eliminated all possibilities
      return type;
    }

    //only a segment can be used for scaling
    if(type == G_SEGMENT && (children[i]->type & G_GEOMETRIC) &&
       children[i]->descendType == G_SCALED && children[i]->parents[0] != this) {
      soFar &= G_SEGMENT;
      continue;
    }

    //for transforms, determine possibilities recursively
    if(children[i]->type == type && IS_TRANSFORM(children[i]->descendType)) {
      refsConsidered.insert(this);
      soFar &= children[i]->whatCanItBe();
      refsConsidered.erase(this);
      continue;
    }

    //check point children
    if(children[i]->type == G_POINT) {
      //only segments have mps
      if(children[i]->descendType == G_MID_POINT) return G_SEGMENT;
      if(children[i]->descendType == G_INTERSECTION2_POINT) {
	if(type & G_STRAIGHT == 0) { //if it's a curve, it must stay a curve
	  soFar &= (G_CURVE - G_STRAIGHT);
	}
      }
    }

    //check line children
    if(children[i]->type == G_LINE) {
      if(children[i]->descendType == G_PERPENDICULAR_LINE || 
	 children[i]->descendType == G_PARALLEL_LINE) {
	soFar &= G_STRAIGHT;
	continue;
      }
    }

    if(children[i]->type == G_CIRCLE && //only a segment can be a radius of a circle
       children[i]->descendType == G_CENTERRADIUS_CIRCLE) return G_SEGMENT;

    if(children[i]->type == G_MEASURE) {
      if(children[i]->descendType == G_LENGTH_MEASURE) {
	soFar &= (G_ARC | G_CIRCLE | G_SEGMENT);
	continue;
      }
      if(children[i]->descendType == G_RADIUS_MEASURE) {
	soFar &= (G_ARC | G_CIRCLE);
	continue;
      }
      if(children[i]->descendType == G_RATIO_MEASURE) return G_SEGMENT;
      if(children[i]->descendType == G_SLOPE_MEASURE) {
	soFar &= G_STRAIGHT;
	continue;
      }
    }

    //finally check loops recursively
    if(children[i]->type == G_LOOP) {
      ASSERT(doc->isConstruction());
      KSegConstruction *d = (KSegConstruction *)doc;

      int j;
      for(j = 0; j < int(children[i]->parents.size()); ++j) {
	if(children[i]->parents[j] != this) continue;

	//now the current object corresponds to the jth given in the loop that is
	//the ith child.  Check if it can do that.
	refsConsidered.insert(this);
	soFar &= d->getGiven()[j]->whatCanItBe();
	refsConsidered.erase(this);
      }
    }
  }

  return soFar;
}




QDataStream &operator<<(QDataStream &stream, G_ref &ref)
{
  //parents and draw style have already been saved.

  short info = 0; //saves space

  //the first five bits is the object type
  int tmp = qRound((log(double(ref.type)) / log(2.)));

  info |= (tmp & 31); //bits 0-4

  //the next four bits is the descend type
  info |= ((ref.descendType & 15) << 5); //bits 5-8

  //the next four bits are the visible, label visible, given, and final flags
  info |= (ref.visible << 9); //bit 9
  info |= (ref.labelVisible << 10); //bit 10
  info |= (ref.given << 11); //bit 11
  info |= (ref.final << 12); //bit 12

  stream << info;

  if(ref.type == G_LOOP) return stream;

  stream << ref.label;

  ref.where->save(stream);

  return stream;
}


QDataStream &operator>>(QDataStream &stream, G_ref &ref)
{
  //parents, draw style, and the document have already been loaded.

  short info = 0; //saves space

  stream >> info;

  //the first five bits is the object type
  ref.type = (G_Type)(1 << (info & 31));
  info >>= 5;
  
  //the next four bits is the descend type
  ref.descendType = (info & 15);
  info >>= 4;

  //the next four bits are the visible, label visible, given, and final flags
  ref.visible = info & 1;
  ref.labelVisible = info & 2;
  ref.given = info & 4;
  ref.final = info & 8;

  ref.createObject();

  if(ref.type == G_LOOP) return stream;

  stream >> ref.label;

  ref.where->load(stream);

  return stream;
}



QString G_ref::getNameFromType(G_Type type)
{
  if(type == G_POINT) return QString("Point");
  if(type == G_SEGMENT) return QString("Segment");
  if(type == G_RAY) return QString("Ray");
  if(type == G_LINE) return QString("Line");
  if(type == G_CIRCLE) return QString("Circle");
  if(type == G_ARC) return QString("Arc");
  if(type == G_POLYGON) return QString("Polygon");
  if(type == G_CIRCLEINTERIOR) return QString("Circle Interior");
  if(type == G_ARCSECTOR) return QString("Arc Sector");
  if(type == G_ARCSEGMENT) return QString("Arc Segment");
  if(type == G_LOCUS) return QString("Locus");
  if(type == G_MEASURE) return QString("Measurement");
  if(type == G_CALCULATE) return QString("Calculation");
  if(type == G_ANNOTATION) return QString("Annotation");

  return QString("Unknown");
}
