#include <kwvdialwidget.h>
#include <kwvdialwidget.moc>

KwvdialWidget::KwvdialWidget(QWidget *parent, const char *name, WvConf &_cfg)
	: KTabCtl(parent,name), dirty(FALSE),
		cfg(_cfg),oldbytes(0),oldavg(0),oldrcv(0),oldtmt(0)
{
    logtab = new QWidget(this,"logtab");
    loglayout = new QGridLayout(logtab,1,1,5,-1);
    logwindow = new QListBox(logtab,"logwindow");
    loglayout->addWidget(logwindow,0,0);
    addTab(logtab,"&Log");
    
    graphtab = new QWidget(this,"graphtab");
    graphlayout = new QGridLayout(graphtab,10,5,5,-1);

    graph = new KGraph(graphtab,"KGraph");
    graphlayout->addMultiCellWidget(graph,0,9,0,2);
    
    recv = new KLed(KLed::green,KLed::Off,KLed::Flat,KLed::Circular,graphtab,"recv");
    recvl = new QLabel(recv,"Receiving:",graphtab);
    graphlayout->addWidget(recvl,0,3);
    graphlayout->addWidget(recv,0,4);   

    trns = new KLed(KLed::red,KLed::Off,KLed::Flat,KLed::Circular,graphtab,"trns");
    trnsl = new QLabel(trns,"Transmitting:",graphtab);
    graphlayout->addWidget(trnsl,1,3);
    graphlayout->addWidget(trns,1,4);
    
    bytein = new QLabel(NULL,"0000",graphtab);
    byteinl = new QLabel(NULL,"Bytes In:",graphtab);
    byteout = new QLabel(NULL,"0000",graphtab);
    byteoutl = new QLabel(NULL,"Bytes Out:",graphtab);
    graphlayout->addWidget(byteinl,2,3);
    graphlayout->addWidget(bytein,2,4);
    graphlayout->addWidget(byteoutl,3,3);
    graphlayout->addWidget(byteout,3,4);

    speedl = new QLabel(NULL,"Average Rate:",graphtab);
    speed = new QLabel(NULL,"0 Kb/Sec",graphtab);
    graphlayout->addWidget(speedl,4,3);
    graphlayout->addWidget(speed,4,4);

    addTab(graphtab,"&Stats");

    configtab = new QWidget(this,"configtab");
    configlayout = new QGridLayout(configtab,10,5,5,-1);
    
    profilel = new QLabel(NULL,"Profile:",configtab);
    profile = new QComboBox(configtab,"profile");
    profiletypel = new QLabel(NULL,"Connection Type:",configtab);
    profiletype = new QComboBox(configtab,"type");
    configlayout->addWidget(profilel,0,0);
    configlayout->addMultiCellWidget(profile,0,0,1,2);
    configlayout->addWidget(profiletypel,0,3);
    configlayout->addWidget(profiletype,0,4);
    
    usernamel = new QLabel(NULL,"Name:",configtab);
    username = new QLineEdit(configtab);
    passwordl = new QLabel(NULL,"Password:",configtab);
    password = new QLineEdit(configtab);
    password->setEchoMode(QLineEdit::Password);
    usetokenpassword = new QCheckBox("Use Passwd Token",configtab);
    configlayout->addWidget(usernamel,1,0);
    configlayout->addMultiCellWidget(username,1,1,1,2);
    configlayout->addWidget(passwordl,2,0);
    configlayout->addWidget(password,2,1);
    configlayout->addWidget(usetokenpassword,2,2);
    
    areal = new QLabel(NULL,"Area Code:",configtab);
    area = new QLineEdit(configtab);
    telnuml = new QLabel(NULL,"Telephone:",configtab);
    telnum = new QLineEdit(configtab);
    configlayout->addWidget(areal,3,0);
    configlayout->addWidget(area,3,1);
    configlayout->addWidget(telnuml,4,0);
    configlayout->addWidget(telnum,4,1);
    connect(area,SIGNAL(textChanged(const char *)),SLOT(setdirty()));
    connect(telnum,SIGNAL(textChanged(const char *)),SLOT(setdirty()));
    
    baudl = new QLabel(NULL,"Baud Rate:",configtab);
    baud = new QLineEdit(configtab);
    configlayout->addWidget(baudl,1,3);
    configlayout->addWidget(baud,1,4);
    
    callwtg = new QCheckBox("Disable Call Waiting",configtab);
    dialtone = new QCheckBox("Do not Wait for Dial Tone",configtab);
    dialasld = new QCheckBox("Dial as Long Distance",configtab);
    pulsetone = new QCheckBox("Dial as Pulse",configtab);
    stupidmode = new QCheckBox("Emulate Windows Dial up Networking",configtab);
    newpppd = new QCheckBox("New PPPD (Post 2.3.0)",configtab);
    autoredial = new QCheckBox("Redial on Disconnect?",configtab);
    configlayout->addMultiCellWidget(callwtg,2,2,3,4);
    configlayout->addMultiCellWidget(dialtone,3,3,3,4);
    configlayout->addMultiCellWidget(dialasld,4,4,3,4);
    configlayout->addMultiCellWidget(pulsetone,5,5,3,4);
    configlayout->addMultiCellWidget(stupidmode,6,6,3,4);
    configlayout->addMultiCellWidget(newpppd,7,7,3,4);
    configlayout->addMultiCellWidget(autoredial,8,8,3,4);
    
    currencyl = new QLabel(NULL,"Currency:",configtab);
    currency = new QComboBox(configtab);
    ratel = new QLabel(NULL,"Cost/Min:",configtab);
    rate = new QLineEdit(configtab);
    configlayout->addWidget(currencyl,5,0);
    configlayout->addMultiCellWidget(currency,5,5,1,2);
    configlayout->addWidget(ratel,6,0);
    configlayout->addMultiCellWidget(rate,6,6,1,2);
    
    extrainitl = new QLabel(NULL,"Init String:",configtab);
    extrainit = new QLineEdit(configtab);
    configlayout->addWidget(extrainitl,7,0);
    configlayout->addMultiCellWidget(extrainit,7,7,1,2);

    volumel = new QLabel(NULL,"Modem Volume:",configtab);
    volume = new QSlider(0,3,1,1,QSlider::Horizontal,configtab,"Volume");
    configlayout->addWidget(volumel,8,0);
    configlayout->addMultiCellWidget(volume,8,8,1,2);

    detectbutton = new QPushButton("Detect Modem",configtab,"detectModem");
    profilebutton = new QPushButton("New Profile",configtab,"NewProfile");
    profdelbutton = new QPushButton("Delete This Profile",configtab,"DelProfile");
    applybutton = new QPushButton("Apply Changes",configtab,"Apply");
    resetbutton = new QPushButton("Reset",configtab,"Reset");
    configlayout->addWidget(detectbutton,9,0);
    configlayout->addWidget(profilebutton,9,1);
    configlayout->addWidget(profdelbutton,9,2);
    configlayout->addWidget(applybutton,9,3);
    configlayout->addWidget(resetbutton,9,4);
    
    connect(this,SIGNAL(tabSelected(int)),SLOT(setup(int)));

    connect(detectbutton,SIGNAL(clicked()),SLOT(wvautoconf()));
    connect(profilebutton,SIGNAL(clicked()),SLOT(newprof()));
    connect(profdelbutton,SIGNAL(clicked()),SLOT(deleteprof()));
    connect(applybutton,SIGNAL(clicked()),SLOT(applyprof()));
    connect(resetbutton,SIGNAL(clicked()),SLOT(resetprof()));
    connect(profiletype,SIGNAL(activated(int)),SLOT(redrawtab()));
    connect(profile,SIGNAL(activated(int)),SLOT(chgdsp(int)));
    
    connect(username,SIGNAL(textChanged(const char *)),SLOT(setdirty()));
    connect(password,SIGNAL(textChanged(const char *)),SLOT(setdirty()));
    connect(baud,SIGNAL(textChanged(const char *)),SLOT(setdirty()));
    connect(callwtg,SIGNAL(toggled(bool)),SLOT(setdirty()));
    connect(dialtone,SIGNAL(toggled(bool)),SLOT(setdirty()));
    connect(dialasld,SIGNAL(toggled(bool)),SLOT(setdirty()));
    connect(pulsetone,SIGNAL(toggled(bool)),SLOT(setdirty()));
    connect(stupidmode,SIGNAL(toggled(bool)),SLOT(setdirty()));
    connect(newpppd,SIGNAL(toggled(bool)),SLOT(setdirty()));
    connect(autoredial,SIGNAL(toggled(bool)),SLOT(setdirty()));
    
    addTab(configtab,"&Configure");
    
    show();
}

KwvdialWidget::~KwvdialWidget()
{
}

void KwvdialWidget::_begin_line()
{
    buffer.zap();
}

void KwvdialWidget::_mid_line(const char *str, size_t len)
{
    // Don't include newline that Avery puts in as Default ;)
    if (str[len-1]=='\n')
	len--;
    buffer.put(str,len);
}

void KwvdialWidget::_end_line()
{
    buffer.put("",1);
    logwindow->insertItem((char *)buffer.get(buffer.used()));
    logwindow->setBottomItem(logwindow->count()-1);
}

void KwvdialWidget::update(int timeonline)
{
    FILE *netproc;
    char bf[256],prefix[50];
    long rcv=0, tmt=0, avg=0;
    char *tmp=NULL;
    long tmp1=0,tmp2=0;

    netproc = fopen("/proc/net/dev","r");
    if (netproc)
    {
	do
	{
	    fgets(bf,254,netproc);
	    sscanf(bf,"%s",prefix);
	    if (!strcmp(prefix, "ppp0:"))
	    {
		sscanf(bf,"%s%li%*i%*i%*i%*i%*i%*i%*i%li",prefix,&rcv,&tmt);
	    }
	} while (!feof(netproc));
    tmp = (char *)malloc(sizeof(bf));
    sprintf(tmp,"%li",rcv);
    bytein->setText(tmp);
    sprintf(tmp,"%li",tmt);
    byteout->setText(tmp);
    fclose(netproc);
    }

    if (!(timeonline%5))
    {
    	tmp1=(tmt+rcv-oldbytes)/5;
	avg=((oldavg*95) + (tmp1*5))/100;
	// Yes, We KNOW it's wrong, but you try dividing something by 1024
	// and packing it properly into 3 "decimal" (read 1024ths) places
    	tmp1=avg/1000;
	tmp2=avg%1000;
	sprintf(tmp,"%li.%03li Kb/Sec",tmp1,tmp2);
	speed->setText(tmp);
    	oldbytes=tmt+rcv;
    	oldavg=avg;
    }
    free(tmp);
    if ( rcv > oldrcv )
	recv->on();
    else
	recv->off();

    if ( tmt > oldtmt )
	trns->on();
    else
	trns->off();
    graph->graphon();
    graph->addpoint(rcv-oldrcv,tmt-oldtmt);
    graph->repaint();
    fprintf(stderr,"Receive Difference: %li\n",rcv-oldrcv);
    fprintf(stderr,"Transmit Difference: %li\n",tmt-oldtmt);
    oldrcv=rcv;
    oldtmt=tmt;
}

void KwvdialWidget::statsoff()
{
	speed->setText("0 Kb/Sec");
	bytein->setText("0000");
	byteout->setText("0000");
	recv->off();
	trns->off();
	graph->reset();
	graph->repaint();
}

void KwvdialWidget::changedisplay(WvString sect)
{
    char *cptr;
    const char *tmp;
    WvString tmpnum;
    int i;
    
    profiletype->clear();
    profiletype->insertItem("Dial");
    profiletype->insertItem("ISDN");
    profiletype->insertItem("PPPOE");
    
    currency->clear();
    currency->insertItem("Dollar");
    currency->insertItem("Euro");
    currency->insertItem("Pound");
    currency->insertItem("Franc");
    currency->insertItem("Peso");
    currency->insertItem("Ruble");
    currency->insertItem("DeutchMark");
    
    rate->clear();
    rate->setText("0.00");
    
    username->setText(cfg.get(sect,"Username"));
    password->setText(cfg.get(sect,"Password"));
    tmp = cfg.get(sect,"Phone");
    if (strstr(cfg.get(sect,"Dial Type",""),"ISDN"))
    {
	// This is an ISDN number so:
	// First check and see if there is a & and if so
	// Read all text up to the & and put it in area, throw away
	// the & and read all text up to the /0 and put it in telnum
	
	profiletype->setCurrentItem(1);
	areal->setText("SPID#1");
	telnuml->setText("SPID#2");
    }
    else
    {
	if (tmp != NULL)    
	{
	    if (!strncmp(tmp,"1-",2))
	    {
		dialasld->setChecked(TRUE);
		tmp += 2;
		if (strchr(tmp,'-'))
		{
		    tmpnum = tmp;
		    cptr = strchr(tmpnum.edit(), '-');
		    telnum->setText(cptr + 1);
		    *cptr = 0;
		    QString *temp = new QString(tmpnum);
		    area->setText(*temp);
		} else {
		    telnum->setText(tmp);
		    area->setText("");
		}
	    } else {
		area->setText("");
		telnum->setText(tmp);
	    }
	}
	areal->setText("Area Code");
	telnuml->setText("Telephone");
	profiletype->setCurrentItem(0);
    }
    if (strstr(cfg.get(sect,"Dial Prefix",""),"*70,,"))
        callwtg->setChecked(TRUE);
    else
        callwtg->setChecked(FALSE);
    
    if (strstr(cfg.get(sect,"Auto Redial",""),"yes"))
	autoredial->setChecked(TRUE);
    else
        autoredial->setChecked(FALSE);
    
    if (strstr(cfg.get(sect,"Dial Command",""),"ATDP"))
	pulsetone->setChecked(TRUE);
    else
        pulsetone->setChecked(FALSE);
    
    if (strstr(cfg.get(sect,"Stupid Mode",""),"yes"))
	stupidmode->setChecked(TRUE);
    else
        stupidmode->setChecked(FALSE);
    
    if (strstr(cfg.get(sect,"New PPPD",""),"yes"))
	newpppd->setChecked(TRUE);
    else
        newpppd->setChecked(FALSE);
    
    if (strstr(cfg.get(sect,"Init3",""),"ATX1"))
	dialtone->setChecked(TRUE);
    else
        dialtone->setChecked(FALSE);
    
    if (strstr(cfg.get(sect,"Init4",""),"ATM0"))
        volume->setValue(0);
    else if (strstr(cfg.get(sect,"Init4",""),"ATM1L1"))
	volume->setValue(1);
    else if (strstr(cfg.get(sect,"Init4",""),"ATM1L2"))
	volume->setValue(2);
    else if (strstr(cfg.get(sect,"Init4",""),"ATM1L3"))
	volume->setValue(3);
    
    baud->setText(cfg.get(sect,"Baud"));

    if (cfg.get(sect,"Rate") != NULL)
    {
	rate->setText(cfg.get(sect,"Rate"));
    }

    if (cfg.get(sect,"Currency") != NULL)
    {
	for (i=0;i<=currency->count();i++)
	{
	    if (strcmp(currency->text(i),cfg.get(sect,"Currency")))
		break;
	}
	currency->setCurrentItem(i-1);
    }

    if (cfg.get(sect,"Init5") !=NULL)
	extrainit->setText(cfg.get(sect,"Init5"));

    dirty=FALSE;    
}

void KwvdialWidget::setup(int id)
{
    if (id==2)
    {
	profile->clear();     
	profiletype->clear();
	currency->clear();
	
	WvConf::Iter i(cfg);
	
	for (i.rewind();i.next(); )
        {
	    QString *temp=new QString(i().name);
	    CHECK_PTR(temp);
	    profile->insertItem(*temp);
	}
	if (profile->count() > 0)    
        {
	    WvString sect(profile->currentText());
            changedisplay(sect);
	} else {        // Call the AutoConf SLOT
	    wvautoconf();
        }
    } else {
	if (dirty)
	{
	        switch(QMessageBox::warning(this,"KWvDial",
				    "You have unsaved Changes\nClick OK to save them\n",
				    QMessageBox::Ok,QMessageBox::Cancel))
        	{        case QMessageBox::Ok:
				applyprof();				
	    			break;
			 case QMessageBox::Cancel:
				dirty=FALSE;
	    			break;
		}
        }
    }
}

void KwvdialWidget::redrawtab()
{
    if ( profiletype->currentItem() == 0 )
    {
	areal->setText("Area Code");
        telnuml->setText("Telephone");
    } else {
	areal->setText("SPID#1");
    	telnuml->setText("SPID#2");
    }
    configtab->repaint();	
}

void KwvdialWidget::applyprof()
{
    WvString InitStr1;
    WvString sect(profile->currentText());
    WvString phone;
    bool blank_area = (strlen(area->text()) == 0);
    

    if ((strlen(username->text()) != 0) && (strlen(password->text()) != 0) &&
        (strlen(telnum->text()) != 0))
    {
	if (dialtone->isChecked())
	    InitStr1 = "ATX1";
	else
            InitStr1 = "ATX4";
	
	if ( strcmp(profiletype->currentText(),"Dial") )
	{
		if (dialasld->isChecked())
            		phone = WvString("1-%s%s%s", (const char *)area->text(), blank_area ? "" : "-", 
			     (const char *)telnum->text());
		else
            		phone = WvString("%s%s%s", (const char *)area->text(), blank_area ? "" : "-", 
			     (const char *)telnum->text());
        } else {
		// We need to do format checking here - make sure the number
		// starts with a 9, and ends with a 0, etc.
		phone = WvString("%s%s",blank_area ? "" : (const char *)area->text(),(const char *)telnum->text());
	}
	cfg.set(sect,"Phone", phone);
	cfg.set(sect,"Username", username->text());
	cfg.set(sect,"Password", password->text());
	cfg.set(sect,"Init3", InitStr1);
	cfg.set(sect,"Baud",baud->text());
	cfg.set(sect,"Rate",rate->text());
	cfg.set(sect,"Currency",currency->currentText());
	cfg.set(sect,"Init5",extrainit->text());

        if (callwtg->isChecked())
            cfg.set(sect,"Dial Prefix","*70,,");
	else
            cfg.set(sect,"Dial Prefix","");
        if (stupidmode->isChecked())
            cfg.set(sect,"Stupid Mode","yes");
	else
            cfg.set(sect,"Stupid Mode","no");
        if (autoredial->isChecked())
            cfg.set(sect,"Auto Redial","yes");
	else
            cfg.set(sect,"Auto Redial","no");
	if (newpppd->isChecked())
            cfg.set(sect,"New PPPD","yes");
        else
            cfg.set(sect,"New PPPD","no");         
	if (pulsetone->isChecked())
            cfg.set(sect,"Dial Command","ATDP");
	else
            cfg.set(sect,"Dial Command","ATDT");  
	
        switch(volume->value())
        {
        case 0:
            cfg.set(sect,"Init4","ATM0");
            break;
        case 1:
            cfg.set(sect,"Init4","ATM1L1");
            break;
        case 2:
            cfg.set(sect,"Init4","ATM1L2");
            break;
        case 3:
            cfg.set(sect,"Init4","ATM1L3");
        }

	cfg.set(sect,"Dial Type",profiletype->currentText());

	dirty = FALSE;

    } else {
        QMessageBox::warning(this,"KWvDial",
			     "Please make sure that\nThe Name,Password and Telephone\nfields are filled in\nfor this profile");
    }
    repaint();
    cfg.flush();
}

void KwvdialWidget::resetprof()
{
    changedisplay((const char *)profile->currentText());
    repaint();
}

void KwvdialWidget::deleteprof()
{
    WvString sect(profile->currentText());
    if ( strncmp(sect,"Dialer Defaults",15) != 0)
    {
        WvString pretemp = WvString("Are you sure you want to delete\n %s ?\n",sect);
	QString *temp = new QString(pretemp);
        CHECK_PTR(temp);
        switch(QMessageBox::warning(this,"KWvDial",*temp,
				    QMessageBox::Ok,QMessageBox::Cancel))
        {        case QMessageBox::Ok:
	    cfg.delete_section(sect);
	    profile->removeItem(profile->currentItem());
	    setup(2);
	    cfg.flush();
	    break;
	case QMessageBox::Cancel:
	    break;
        }    
    } else {
        QMessageBox::warning(this,"KWvDial", 
		             "Sorry, not allowed to delete Default section\n");    
    }
}

void KwvdialWidget::newprof()
{
    KwvdialNew temp(NULL,"NewProfile");
    if (temp.exec())
    {	
	if (strlen(temp.newprofile->text()) != 0)
	{	
	    profile->insertItem(temp.newprofile->text());
	    profile->setCurrentItem(profile->count()-1);
	    
	    callwtg->setChecked(FALSE);
	    dialtone->setChecked(FALSE);
	    autoredial->setChecked(FALSE);
	    dialasld->setChecked(FALSE);
	    pulsetone->setChecked(FALSE);
	    stupidmode->setChecked(FALSE);
	    
	    if (strstr(cfg.get("Dialer Defaults","New PPPD",""),"yes"))
		newpppd->setChecked(TRUE);
	    else
		newpppd->setChecked(FALSE);
	    
	    baud->setText(cfg.get("Dialer Defaults","Baud"));
	    username->clear();
	    password->clear();
	    area->clear();
	    telnum->clear();
	    dirty = TRUE;
	}	
    } else {
	QMessageBox::warning(this,"KWvDial","New Profile Didn't work");
    }
    
    repaint();
}

void KwvdialWidget::chgdsp(int id)
{    

	if (dirty)
	{
	        switch(QMessageBox::warning(this,"KWvDial",
				    "You have unsaved Changes\nClick OK to save them\n",
				    QMessageBox::Ok,QMessageBox::Cancel))
        	{        case QMessageBox::Ok:
				applyprof();				
	    			break;
			 case QMessageBox::Cancel:
	    			break;
		}
		dirty=FALSE;
        }

	WvString sect(profile->text(id));
	changedisplay(sect);
} 

void KwvdialWidget::setdirty()
{
	dirty=TRUE;
}

void KwvdialWidget::wvautoconf()
{
    WvLog log("KWvDial",WvLog::Info);

    showTab(0);

    WvModemScanList l;
    while (!l.isdone())
        l.execute();

    if (l.count() < 1)
    {
        log("\n\nSorry, no modem was detected! "
            "Is it in use by another program?\n\n"
            "If your modem is correctly installed and should have "
            "been located,\nplease send mail to "
            "wvdial@worldvisions.ca.\n");
        return;
    }

    WvModemScanList::Iter i(l);

    i.rewind(); i.next();
    WvModemScan &m = i;
    WvString fn = m.filename(), init = m.initstr();

    log("\nFound a modem on %s.\n", fn);
    log("\nWriting Default Section\n");

    static char s[]="Dialer Defaults";
    cfg.set(s, "Modem", fn);
    cfg.setint(s, "Baud", m.maxbaud());
    cfg.set(s, "Init1", "ATZ");
    cfg.set(s, "Init2", init);

    log("AutoConfigure Done\n");	
    setup(2);
    
}

/* Below this line is the New Profile Dialog Box Class - pretty simple,
 * Which is why I didn't give it it's own file - not worth it.
 */

KwvdialNew::KwvdialNew(QWidget *parent = 0, const char *name = 0)
    : QDialog(parent,name,TRUE)
{
    ok = new QPushButton( "Ok", this );
    ok->setDefault(TRUE);
    connect( ok, SIGNAL(clicked()), SLOT(accept()));
    
    newprofile = new QLineEdit(this);
    newprofilel = new QLabel(newprofile,"Profile Name:",this);
    newprofilelayout = new QGridLayout(this,2,3,5,-1);
    newprofilelayout->addWidget(newprofilel,0,0);
    newprofilelayout->addMultiCellWidget(newprofile,0,0,1,2);
    newprofilelayout->addWidget(ok,1,1);
    
    resize(250,70);
}

KwvdialNew::~KwvdialNew()
{
}
