/*******************************************************/
/* methods for a net-widget                            */
/*                                                     */
/* this contains the current active circuit widget,    */
/* manages various modes, status, drawing and io       */
/*                                                     */
/* Andreas Rostin                                      */
/* 27.10.97                                            */
/*******************************************************/
#include <qpainter.h>
#include <qlistbox.h>
#include <qstring.h>
#include <qdir.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qpopupmenu.h>
#include <qtimer.h>
#include <qwidget.h>
#include <qframe.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qdialog.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qpixmap.h>
#include <qlabel.h>
#include <qcombobox.h>

#include <netw.h>
#include <klogic.h>
#include <klogicIO.h>
#include <xnet.h>
#include <xdevice.h>
#include <selection.h>
#include <propOsz.h>
#include <propDev.h>
#include <propName.h>
#include <propPwr.h>
#include <propText.h>
#include <dlgEqu.h>

#include "netw.moc"
//********************************************
//* static methods of NetWidget              *
//********************************************

int NetWidget::STATsimStepTime = 40;
int NetWidget::STATsimBurst = 1;
QTimer NetWidget::simTimer;
int NetWidget::actualDevFunc = 0;
int NetWidget::drawmode = 0;
int NetWidget::simmode = 0;

// static method
int NetWidget::simTime()
{
	return STATsimStepTime;
}

// static method
void NetWidget::setSimTime(int newi)
{
	if ((newi >= MIN_SIMSTEPTIME) && (newi <=  MAX_SIMSTEPTIME)) {
		STATsimStepTime = newi;
		if (simTimer.isActive())
			simTimer.changeInterval(STATsimStepTime);
	}
}

// static method
int NetWidget::simBurst()
{
	return STATsimBurst;
}

// static method
void NetWidget::setSimBurst(int newi)
{
	if ((newi >= MIN_SIMBURST) && (newi <=  MAX_SIMBURST))
		STATsimBurst = newi;
}

// static method
// set current object for dropping
void NetWidget::setDevice(int newdev)
{
	actualDevFunc = newdev;
}

// static method
// set current drawing-mode
void NetWidget::setDrawMode(int newmode)
{
	drawmode = newmode;
}

// static method
// get current drawing-mode
int NetWidget::getDrawMode()
{
	return drawmode;
}

// static method
// change between permanent and stepping simulation mode
void NetWidget::setSimMode(int newmode)
{
	simmode = newmode;

	// permanent simulation mode
	if (simmode == MODE_SIM_MULT) {
		simTimer.start(STATsimStepTime);
	}
	else {
		simTimer.stop();
	}
}

// static method
// return current simulation mode
int NetWidget::getSimMode()
{
	return simmode;
}

//********************************************
//* methods of NetWidget                     *
//********************************************
NetWidget::NetWidget(QWidget *hparent, QWidget *parent) : QFrame(hparent)
{
	client = 0;

	setBackgroundMode(PaletteBase);    // set widgets background
	setBackgroundColor(white);
	setFrameStyle( QFrame::Panel | QFrame::Raised );
	resize(VIRT_SCREEN_SIZE_X, VIRT_SCREEN_SIZE_Y);                  // set default size in pixels

	activeNet = new XDeviceNet();
	activeDev = (XDevice *)NULL;
	activeWire = (XWire *)NULL;
	activeSelection = new Selection();
	_parent = parent;

	// drawing mode
	setDrawMode(NetWidget::MODE_DRAW);
	setSimMode(NetWidget::MODE_SIM_MULT);
	// device to drop when in drop-mode
	actualDevFunc = 0;

	// popup for the right mouse key (on devices)
	rpop = new QPopupMenu();
	connect(rpop, SIGNAL (activated (int)), SLOT (rmenuCallback(int)));

	// popup for the right mouse key (on selections)
	rselpop = new QPopupMenu();
	rselpop->insertItem(klocale->translate("Copy"), RPOPSEL_COPY);
	rselpop->insertItem(klocale->translate("Paste"), RPOPSEL_PASTEAT);
	rselpop->insertItem(klocale->translate("Cut"), RPOPSEL_CUT);
	connect(rselpop, SIGNAL (activated (int)), SLOT (rmenuSelCallback(int)));

	changed = 0;
	movecursor = 0;
	rmenuActive = 0;

	connect(&simTimer, SIGNAL(timeout()), this, SLOT(simStep()));
	connect(this, SIGNAL(netContentChanged()), parent, SLOT(repaintClients()));
}

NetWidget::NetWidget(QWidget *hparent, QWidget *parent, XDeviceNet *net)
	: QFrame(hparent)
{
	client = 1;

	setBackgroundMode(PaletteBase);    // set widgets background
	setBackgroundColor(white);
	setFrameStyle( QFrame::Panel | QFrame::Raised );
	resize(VIRT_SCREEN_SIZE_X, VIRT_SCREEN_SIZE_Y);                // set default size in pixels

	activeNet = net;
	activeDev = (XDevice *)NULL;
	activeWire = (XWire *)NULL;
	activeSelection = new Selection();
	_parent = hparent;

	// drawing mode
	setDrawMode(NetWidget::MODE_DRAW);
	// device to drop when in drop-mode
	actualDevFunc = 0;

	// popup for the right mouse key (on devices)
	rpop = new QPopupMenu();
	connect(rpop, SIGNAL (activated (int)), SLOT (rmenuCallback(int)));

	// popup for the right mouse key (on selections)
	rselpop = new QPopupMenu();
	rselpop->insertItem(klocale->translate("Copy"), RPOPSEL_COPY);
	rselpop->insertItem(klocale->translate("Paste"), RPOPSEL_PASTEAT);
	rselpop->insertItem(klocale->translate("Cut"), RPOPSEL_CUT);
	connect(rselpop, SIGNAL (activated (int)), SLOT (rmenuSelCallback(int)));

	changed = 0;
	movecursor = 0;
	rmenuActive = 0;

	connect(&simTimer, SIGNAL(timeout()), this, SLOT(simStep()));
	connect(this, SIGNAL(netContentChanged()), parent, SLOT(repaintClients()));
}

NetWidget::~NetWidget()
{
	if (client) return;

	delete activeNet;
	delete activeSelection;
}

void NetWidget::mousePressEvent( QMouseEvent *e )
{	QPainter p;

	if (e->button() == LeftButton) {
		if (rmenuActive) {
			rmenuActive = 0;
			return;
		}

		if (drawmode == MODE_DRAW) {
			// wire clicked? set active wire
			activeWire = activeNet->containsWire(e->pos());
			if (activeWire) {
				return;
			}

			// device clicked? set active device
			activeDev = activeNet->containsDev(e->pos());
			if (activeDev) {
				activeDev->setOrigin();
				return;
			}

			// nothing clicked? new wire
			p.begin(this);
			activeWire = activeNet->newWire(&p, e->pos());
			p.end();
			if (activeWire) {
				emit netContentChanged();
				return;
			}
			p.end();
		}
		if (drawmode == MODE_DROP) {
			// move device
			activeDev = activeNet->containsDev(e->pos());
			if (activeDev) {
				activeDev->setOrigin();
				p.begin(this);
				activeNet->drawDev(this, &p, activeDev);
				p.end();
				// redraw client widgets with the same content as this
				emit netContentChanged();
				changed = 1;
				return;
			}
			// create new device
			activeDev = activeNet->newDevice(actualDevFunc, e->pos().x(), e->pos().y());
			if (!activeDev && (actualDevFunc == XDevice::fIN)) {
				QMessageBox::warning(_parent, "device failure", "unable to add more named inputs\nresize this device");
				emit netContentChanged();
				changed = 1;
				return;
			}
			if (!activeDev && (actualDevFunc == XDevice::fOUT)) {
				QMessageBox::warning(_parent, "device failure", "unable to add more named outputs\nresize this device");
				emit netContentChanged();
				changed = 1;
				return;
			}
			if (activeNet->containsDev(activeDev)) {
				deleteDev(0);
				emit netContentChanged();
				changed = 1;
				return;
			}
			activeDev->setOrigin();
			p.begin(this);
			activeNet->drawDev(this, &p, activeDev);
			p.end();
			emit netContentChanged();
			if (activeDev->drawGraph() && activeDev->graphEnabled())
				emit graphChanged();
			return;
		}
		if (drawmode == MODE_SEL) {
			if (activeSelection->getStatus() == SEL_EXISTS && activeSelection->onFrame(e->pos())) {
				// selection move mode
				activeSelection->beginMove(e->pos());
			} else if (activeSelection->getStatus() == SEL_EXISTS &&  NULL != (activeDev = activeNet->containsDev(e->pos()))) {
				if (activeSelection->contains(activeDev)) {
					// selection move mode
					activeSelection->beginMove(e->pos());
				} else {
					// select device, then selection move mode
					activeSelection->addTemp(activeDev);
					activeSelection->beginMove(e->pos());
					p.begin(this);
					activeSelection->erase(&p);
					activeNet->drawAll(this, &p);
					activeSelection->draw(&p);
					p.end();
				}
			} else {
				// new selection?
				activeSelection->setNet(activeNet);

				// select/deselect single device
				activeDev = activeNet->containsDev(e->pos());
				if (activeDev) {
					if (activeSelection->contains(activeDev)) activeSelection->remove(activeDev);
					else activeSelection->add(activeDev);
					activeDev = (XDevice *)NULL;
					activeWire = (XWire *)NULL;
					p.begin(this);
					activeSelection->erase(&p);
					activeNet->drawAll(this, &p);
					activeSelection->draw(&p);
					p.end();
					return;
				}

				// select/deselect single wire
				activeWire = activeNet->containsWire(e->pos());
				if (activeWire) {
					if (activeSelection->contains(activeWire)) activeSelection->remove(activeWire);
					else activeSelection->add(activeWire);
					activeDev = (XDevice *)NULL;
					activeWire = (XWire *)NULL;
					p.begin(this);
					activeSelection->erase(&p);
					activeNet->drawAll(this, &p);
					activeSelection->draw(&p);
					p.end();
					return;
				}

				// group selections
				p.begin(this);
				activeSelection->erase(&p);
				p.end();
				activeSelection->beginFrame(e->pos());
			}
			p.begin(this);
			activeSelection->erase(&p);
			activeNet->drawAll(this, &p);
			activeSelection->draw(&p);
			p.end();
			return;
		}
	}

	if (e->button() == RightButton) {
		// right clicked in selection mode? open popup to choose operation
		if (drawmode == MODE_SEL) {
			activeSelection->at(e->pos());
			selRightSelMenu(e);
			rmenuActive = 1;
			return;
		}

		// device right clicked? open popup to chooce operation
		if (NULL != (activeDev = activeNet->containsDev(e->pos()))) {
			selRightMenu(e);
			changed = 1;
			rmenuActive = 1;
			return;
		}

		// right clicked twice?
		if (rmenuActive) {
			rmenuActive = 0;
			return;
		}

		// wire right clicked? remove node 
		p.begin(this);
		if (activeNet->removeNode(&p, e->pos())) {
			activeNet->drawAll(this, &p);
			activeWire = (XWire *)NULL;
			emit netContentChanged();
			changed = 1;
			p.end();
			return;
		}
		p.end();
	}
}

void NetWidget::mouseReleaseEvent( QMouseEvent *e)
{	int ret;
	QPainter p;

	if (e->button() == RightButton) return;

	if (drawmode == MODE_SEL) {
		if (activeSelection->getStatus() == SEL_START) {	// making frames
			activeSelection->endFrame();
			return;
		}
		if (activeSelection->getStatus() == SEL_MOVE) {	// moving frames or select devices
			activeSelection->endMove();

			if (! activeSelection->hasMoved()) {
				// select/deselect single device when group was not moved
				activeDev = activeNet->containsDev(e->pos());
				if (activeDev) {
					if (activeSelection->isTemp(activeDev)) activeSelection->fixTemp(activeDev);
					else activeSelection->remove(activeDev);
					activeDev = (XDevice *)NULL;
					activeWire = (XWire *)NULL;
					p.begin(this);
					activeSelection->erase(&p);
					activeNet->drawAll(this, &p);
					activeSelection->draw(&p);
					p.end();
					activeDev = (XDevice *)NULL;
					return;
				}
			}
			activeDev = activeNet->containsDev(e->pos());
			if (activeDev) {
				// look if some wires want to drop nodes..
				activeDev->garbageCollection();
				activeDev = (XDevice *)NULL;
			}
			return;
		}
		return;
	}

	// check for collision with another device
	if (activeDev) {
		//look for devices lying in the device 
		if (activeNet->containsDev(activeDev)) {
			activeDev->toOrigin();
			p.begin(this);
			activeNet->drawDev(this, &p, activeDev);
			p.end();
		}

		// if device was not moved and it is interactive, toggle its status
		if (activeDev->isInteractive() && activeDev->isOrigin(e->pos())) {
			activeDev->toggleStaticInput();
			p.begin(this);
			activeNet->drawStatus(this, &p);
			p.end();
		}

		// look if some wires want to drop nodes..
		activeDev->garbageCollection();

		emit netContentChanged();
		changed = 1;
	}
	if (activeWire) {
		// check for connection to device or another wire
		ret = activeNet->checkConnection(activeWire);
		switch (ret) {
		case NFAIL:
			QMessageBox::warning(_parent,
				klocale->translate("connection failure"),
				klocale->translate("only one output per wire allowed"));
			break;
		case DFAILMAXI:
			QMessageBox::warning(_parent,
				klocale->translate("connection failure"),
				klocale->translate("maximum number of inputs exeeded"));
			break;
		case DFAILMAXO:
			QMessageBox::warning(_parent,
				klocale->translate("connection failure"),
				klocale->translate("maximum number of outputs exeeded"));
			break;
		case WSHORT:
			QMessageBox::warning(_parent,
				klocale->translate("connection failure"),
				klocale->translate("short cut\n"));
			break;
		default:
			break;
		}
		p.begin(this);
		activeWire->erase(&p);
		activeNet->drawAll(this, &p);
		activeNet->releaseWire(&p, activeWire);
		p.end();
		emit netContentChanged();
		changed = 1;
	}
	activeDev = (XDevice *)NULL;
	activeWire = (XWire *)NULL;
}

void NetWidget::mouseMoveEvent( QMouseEvent *e )
{	QPainter p;

	if (drawmode == MODE_SEL) {
		if (FALSE == hasMouseTracking()) setMouseTracking(TRUE);
		if (activeSelection->getStatus() == SEL_EXISTS && activeSelection->onFrame(e->pos())) {
			if (!movecursor) {
				movecursor = 1;
				setCursor(sizeAllCursor);
				return;
			}
		} else {
			if (movecursor) {
				movecursor = 0;
				setCursor(arrowCursor);
				return;
			}
			if (activeSelection->getStatus() == SEL_START) {		// making frames
				activeSelection->changeFrame(e->pos());
				p.begin(this);
				activeSelection->erase(&p);
				activeNet->drawAll(this, &p);
				activeSelection->draw(&p);
				p.end();
				emit netContentChanged();
				return;
			}
			if (activeSelection->getStatus() == SEL_MOVE) {		// moving frames
				p.begin(this);
				activeSelection->moveFrame(e->pos(), &p);
				activeNet->drawAll(this, &p);
				p.end();
				emit netContentChanged();
				return;
			}
		}
		return;
	}
	if (TRUE == hasMouseTracking()) {
		activeDev = (XDevice *)NULL;
		activeWire = (XWire *)NULL;
		setMouseTracking(FALSE);
	}

	if (rmenuActive) return;

	// move wire/device
	if (activeDev) {
		if (activeDev->setPos(e->pos())) {
			p.begin(this);
			activeNet->drawDev(this, &p, activeDev);
			p.end();
			emit netContentChanged();
		}
	}
	if (activeWire) {
		p.begin(this);
		activeWire->erase(&p);
		activeWire->updateNode(e->pos());
		activeNet->drawAll(this, &p);
		p.end();
		emit netContentChanged();
	}
}

void NetWidget::mouseDoubleClickEvent( QMouseEvent *e )
{
	// device clicked?
	activeDev = activeNet->containsDev(e->pos());
	if (activeDev) {
		if (activeDev->type() == XDevice::fSWI) {
			activeDev = (XDevice *)NULL;
			return;
		}

		// switch to sub circuit?
		if (toSubNet()) {
			emit netContentChanged();
			return;
		}

		// open device properties?
		if (openProp()) {
			emit netContentChanged();
			return;
		}
	}
}

// always call whith repaint=FALSE
void NetWidget::paintEvent(QPaintEvent *)
{	int x, y;
	int x1s, y1s, x2s, y2s;
	QPainter p;

	x1s = visi.x() / GRID * GRID - GRID;
	x2s = (visi.width() + visi.x()) / GRID * GRID + GRID;
	y1s = visi.y() / GRID * GRID - GRID;
	y2s = (visi.height() +  visi.y()) / GRID * GRID + GRID;

	if (!isActiveWindow()) erase();

	p.begin(this);
	p.setPen(black);
	for(x = x1s ; x < x2s ;x += GRID)
		for(y = y1s; y < y2s; y += GRID)
			p.drawPoint(x, y);
	activeNet->drawAll(this, &p);
	activeSelection->draw(&p);
	p.end();
}

int NetWidget::printNet(QPainter *p)
{
	activeNet->drawAll((QPaintDevice *)NULL, p);
	return 1;
}

// right mouse button popup entries
void NetWidget::selRightMenu(QMouseEvent *e)
{
	if (!activeDev) return;

	rpop->clear();
	rpop->insertItem(klocale->translate("properties"), RPOP_PROP);
	if (activeDev->type() == XDevice::fNET) {
		rpop->insertItem(klocale->translate("open"), RPOP_SUB);
		rpop->insertItem(klocale->translate("open in window"), RPOP_SUBW);
		rpop->insertItem(klocale->translate("display graph"), RPOP_SHOWGRAPH);
	}
	if (activeDev->type() == XDevice::fEQU) {
		rpop->insertItem(klocale->translate("equations"), RPOP_EQU);
	}
	if (activeDev->drawGraph()) {
		if (activeDev->graphEnabled()) {
			rpop->insertItem(klocale->translate("hide graph"), RPOP_GRAPH);
		} else {
			rpop->insertItem(klocale->translate("show graph"), RPOP_GRAPH);
		}
	}

	rpop->insertSeparator();
	rpop->insertItem(klocale->translate("remove"), RPOP_DEL);

	rpop->popup(mapToGlobal(e->pos()));
}

// callback for right mouse button popup
void NetWidget::rmenuCallback(int val)
{
	switch(val) {
		case RPOP_DEL:
			deleteDev(1);
			break;
		case RPOP_PROP:
			openProp();
			break;
		case RPOP_EQU:
			showDeviceEquations();
			break;
		case RPOP_SUB:
			toSubNet();
			break;
		case RPOP_SUBW:
			toSubNetW();
			break;
		case RPOP_GRAPH:
			if (!activeDev) return;
			if (activeDev->graphEnabled()) activeDev->enableGraph(0);
			else activeDev->enableGraph(1);
			emit graphChanged();
			break;
		case RPOP_SHOWGRAPH:
			if (activeNet->devIsNet(activeDev))
				emit showGraph(activeNet->devIsNet(activeDev));
			break;
		default:
			break;
	}
	emit netContentChanged();
	changed = 1;
	rmenuActive = 0;
}

// right mouse button popup entries on selections
void NetWidget::selRightSelMenu(QMouseEvent *e)
{	//(entries are static)
	rselpop->popup(mapToGlobal(e->pos()));
}

//callback for right mouse button popup on selections
// also called vom outside (mainw.ccp/menu)
void NetWidget::rmenuSelCallback(int val)
{	QPainter p;

	switch(val) {
		case RPOPSEL_CUT:
			p.begin(this);
			activeSelection->cut(&p, activeNet);
			activeNet->drawAll(this, &p);
			p.end();
			emit netContentChanged();
			emit graphChanged();
			changed = 1;
			break;
		case RPOPSEL_COPY:
			activeSelection->copy(activeNet);
			break;
		case RPOPSEL_PASTE:
			activeSelection->at(QPoint(40, 10));
		case RPOPSEL_PASTEAT:
			p.begin(this);
			activeSelection->paste(&p, activeNet);
			activeSelection->erase(&p);
			activeNet->drawStatus(this, &p);
			activeNet->drawAll(this, &p);
			activeSelection->draw(&p);
			p.end();
			emit netContentChanged();
			emit graphChanged();
			changed = 1;
			break;
		default:
			break;
	}
	rmenuActive = 0;
}

// remove old selections
void NetWidget::removeSelection()
{	QPainter p;

	p.begin(this);
	activeSelection->remove(&p, 1);
	activeSelection->erase(&p);
	activeNet->drawAll(this, &p);
	activeSelection->draw(&p);
	p.end();
}

// remove device
void NetWidget::deleteDev(int ask)
{	QPainter p;
	int simw_rel = 1;

	if (activeDev->type() == XDevice::fNET) {
		if (ask) {
			if (1 == QMessageBox::warning(_parent, klocale->translate("remove device"),
					klocale->translate("remove entire sub-circuit?"), klocale->translate("OK"), klocale->translate("Cancel")))
			{
				activeDev = (XDevice *)NULL;
				return;
			}
		}
		emit netContentChanged();
		emit netDeleted(activeNet->devIsNet(activeDev));
	}
	if (!activeDev->drawGraph() ||  !activeDev->graphEnabled())
		simw_rel = 0;

	p.begin(this);
	activeNet->deleteDevice(&p, activeDev);
	activeNet->drawAll(this, &p);
	p.end();
	activeDev = (XDevice *)NULL;

	if (simw_rel) emit graphChanged();
}

// open device properties?
int NetWidget::openProp()
{	QPainter p;

	// open properties for oszillators
	if (activeDev->type() == Device::fOSZ) {
		OszProp *dlg = new OszProp(_parent, klocale->translate("oscillator properties"), activeNet, activeDev);
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		dlg->show();
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		delete dlg;
		activeDev = (XDevice *)NULL;
		changed = 1;
		emit graphChanged();
		return 1;
	}

	// open properties for text devices
	if (activeDev->type() == Device::fTXT) {
		TextProp *dlg = new TextProp(_parent, klocale->translate("text label"), activeNet, activeDev);
		dlg->show();
		delete dlg;
		activeDev = (XDevice *)NULL;
		changed = 1;
		return 1;
	}

	// open properties for power sources
	if (activeDev->type() == Device::fPWR) {
		PwrProp *dlg = new PwrProp(_parent, klocale->translate("device properties"), activeNet, activeDev);
		dlg->show();
		delete dlg;
		p.begin(this);
		activeNet->drawStatus(this, &p);
		p.end();
		activeDev = (XDevice *)NULL;
		changed = 1;
		emit graphChanged();
		return 1;
	}

	// open properties for inputs and ouputs
	if ((activeDev->type() == Device::fIN) ||
	    (activeDev->type() == Device::fOUT) ||
	    (activeDev->type() == Device::fSWI)) {
		NameProp *dlg = new NameProp(_parent, klocale->translate("device name"), activeNet, activeDev);
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		dlg->show();
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		
		delete dlg;
		activeDev = (XDevice *)NULL;
		changed = 1;
		emit graphChanged();
		return 1;
	}

	// open dialog with common device properties
	DevProp *dlg = new DevProp(_parent, klocale->translate("device properties"),this ,activeNet, activeDev);
	connect(dlg, SIGNAL(delayChange()), this, SLOT(delayChanged()));
	if (simmode == MODE_SIM_MULT) simTimer.stop();
	dlg->show();
	if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);

	disconnect(dlg, SIGNAL(delayChange()), this, SLOT(delayChanged()));
	delete dlg;
	activeDev = (XDevice *)NULL;
	changed = 1;
	emit graphChanged();
	return 1;
}

// switch to sub circuit in a new window?
int NetWidget::toSubNetW()
{	XDeviceNet *new_activeNet;

	if (!activeDev) return 0;

	new_activeNet = activeNet->devIsNet(activeDev);
	if (new_activeNet) {
		// remove the current selection before switching to a new circuit
		removeSelection();

		emit createNewWidget(new_activeNet);
		return 1;
	}
	return 0;
}

// switch to sub circuit?
int NetWidget::toSubNet()
{	XDeviceNet *new_activeNet;

	new_activeNet = activeNet->devIsNet(activeDev);
	if (new_activeNet) {
		// remove the current selection before switching to a new circuit
		removeSelection();

		activeNet = new_activeNet;
		activeDev = (XDevice *)NULL;
		activeWire = (XWire *)NULL;
		// signal mainw info about parent or not (changes toolbar)
		if (activeNet->parent())
			emit netChanged(1);
		else
			emit netChanged(0);
		repaint(TRUE);
		return 1;
	}
	return 0;
}

// switch to parent of active net
void NetWidget::toParentNet()
{	XDeviceNet *new_activeNet;

	// remove the current selection before switching to a new circuit
	removeSelection();

	new_activeNet = activeNet->parent();
	if (!new_activeNet) {
		warning("no parent net!?");
		return;
	}
	activeNet = new_activeNet;
	activeWire = (XWire *)NULL;
	activeDev = (XDevice *)NULL;
	repaint(TRUE);
	// signal mainw info about having a parent (changes toolbar)
	if (activeNet->parent())
		emit netChanged(1);
	else
		emit netChanged(0);
}

int NetWidget::saveNet(QString filename)
{	XDeviceNet *_net;
	int ret, sub = 0;
	klogicIO *_io;

	_net = activeNet->rootParent();
	if (_net != activeNet) {	
		ret = QMessageBox::warning(_parent,
				klocale->translate("save circuit"),
				klocale->translate("the current circuit is a sub circuit.\nsave it as a.."),
				klocale->translate("main circuit"),
				klocale->translate("sub circuit"));
		if (ret) {
			_net = activeNet;
			sub = 1;
		}
	}
	else {
		changed = 0;
		sub = 0;
	}

	_io = new klogicIO(filename, _net);
	if (sub) {
		_io->setSubFilename();
		ret = _io->writeNet(_net);
	} else
		ret = _io->writeNet();
	delete _io;
	if (ret <= 0) {
		QMessageBox::warning(_parent,
			klocale->translate("save error"),
			klocale->translate("unable to write file"));
		return 0;
	}
	return 1;
}

// load file
int NetWidget::openNet(QString filename, int init=0)
{	XDeviceNet *_net;
	int ret;
	klogicIO *_io;
	QPainter p;
	int create_sub = 0;

	_io = new klogicIO(filename);
	ret = _io->checkSubNet();
	if (ret < 0) {
		QMessageBox::warning(_parent,
			klocale->translate("read error"),
			klocale->translate("unable to read file"));
		return 0;
	}
	if (ret == 1) {
		create_sub = 1;
		_net = activeNet;
	} else {
		if (changed && !activeNet->rootParent()->empty()) {
			if (QMessageBox::Yes != QMessageBox::warning(_parent,
				klocale->translate("new circuit"), 
				klocale->translate("delete current circuit?"), 
				QMessageBox::Yes, 
				QMessageBox::No))
				return 1;
		}
		_net = activeNet->rootParent();
		activeNet = _net;
		activeDev = (XDevice *)NULL;
		activeWire = (XWire *)NULL;

		p.begin(this);
		activeNet->deleteNet(&p);
		p.end();
		changed = 0;
	}

	if (simmode == MODE_SIM_MULT) simTimer.stop();
	_io->setNet(_net);
	ret = _io->readNet(create_sub);

	if (ret <= 0) {
		QMessageBox::warning(_parent,
			klocale->translate("read error"),
			klocale->translate("unable to read file"));
		return 0;
	}
	if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);

	repaint(TRUE);

	// signal mainw info about having a parent or not (changes toolbar)
	if (activeNet->parent())
		emit netChanged(1);
	else
		emit netChanged(0);
	delete _io;
	emit graphChanged();
	if (!init) emit netContentChanged();

	p.begin(this);
	activeNet->drawStatus(this, &p);
	p.end();

	return 1;
}

int NetWidget::newNet()
{	QPainter p;

	if ((changed && !activeNet->empty() &&
			QMessageBox::Yes == (QMessageBox::warning(_parent,
			klocale->translate("new circuit"), 
			klocale->translate("delete current circuit?"), 
			QMessageBox::Yes,
			QMessageBox::No))) || (!changed)) {
		p.begin(this);
		activeNet->deleteNet(&p);
		p.end();
		repaint(TRUE);
		changed = 0;
		emit netContentChanged();
		emit graphChanged();
		return 1;
	}
	return 0;
}

// display dialog with the equations of the current circuit
void NetWidget::showCircuitEquations()
{
	if (!activeNet->checkCircuitNames()) {
		if (QMessageBox::No == (QMessageBox::warning(this,
				"Parse circuit equations",
				"There are some non unique device names \nwithin your circuit ... \n\nrename them automaticly?",
				QMessageBox::Yes,
				QMessageBox::No)))
			return;
		activeNet->unifyCircuitNames();
	}
	DlgEqu *equation_dialog = new DlgEqu(this, klocale->translate("Circuit Equation Editor"), activeNet);
	equation_dialog->show();
	delete equation_dialog;
}

// display dialog with the equations of the active device
void NetWidget::showDeviceEquations()
{
	if (!activeDev) return;
	DlgEqu * equation_dialog = new DlgEqu(this, klocale->translate("Device Equation Editor"), activeDev);
	equation_dialog->show();
	delete equation_dialog;
}

// retrieve the active XDeviceNet
XDeviceNet *NetWidget::getActive()
{
	return activeNet;
}

// check if given net is this or a parent of this
int NetWidget::contained(XDeviceNet *rootnet)
{	XDeviceNet *_rootnet = activeNet;

	if (_rootnet == rootnet) return 1;
	while(NULL != (_rootnet = _rootnet->parent())) {
		if (_rootnet == rootnet) return 1;
	}
	return 0;
}

// private slot
// perform a simulation step
void NetWidget::simStep()
{	QPainter p;

	if (!client) {
		// process a simulation step
		activeNet->rootParent()->Burst(STATsimBurst);
		emit simStepped();
	}
	p.begin(this);
	// draw wires and devices with changed image
	activeNet->drawStatus(this, &p);
	p.end();
}

// public slot
// update visible rect, emitted by MainWidget
void NetWidget::visible(QRect newvisi)
{
	visi = newvisi;
}

// public slot
void NetWidget::delayChanged()
{
	activeNet->rootParent()->calcDelay();
}

// apply device defaults to all existing devices
void NetWidget::applyDefaults()
{	XDeviceNet *toplevel = activeNet->rootParent();

	toplevel->applyDefaults();
	repaint(TRUE);
}

