/*************************************************/
/* methods for class Wire                        */
/*                                               */
/* managing the nodes of a wire                  */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qpainter.h>

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <xwire.h>
#include <klogic.h>

/***************************************************/
/* methods of Wire                                 */
/***************************************************/
Wire::Wire()
{	from_node = (QPoint *)NULL;
	current_node = (QPoint *)NULL;
	to_node = (QPoint *)NULL;
	o_node.setX(0);
	o_node.setY(0);
	o_start_node.setX(0);
	o_start_node.setY(0);
	o_end_node.setX(0);
	o_end_node.setY(0);
	o_ins_node1.setX(0);
	o_ins_node1.setY(0);
	o_ins_node2.setX(0);
	o_ins_node2.setY(0);
	MustErase = 0;
	color = Qt::black;
	input1 = (QPoint *)NULL;
	input2 = (QPoint *)NULL;
	output = (QPoint *)NULL;
	io_p1 = (QPoint *)NULL;
	io_p2 = (QPoint *)NULL;
}

Wire::Wire(QPoint src)
{	align(&src);
	QPoint *p = new QPoint(src);
	wire.Append(p);
	from_node = (QPoint *)NULL;
	current_node = p;
	to_node = (QPoint *)NULL;
	o_node = src;
	o_start_node.setX(0);
	o_start_node.setY(0); 
	o_end_node.setX(0);
	o_end_node.setY(0); 
	o_ins_node1.setX(0);
	o_ins_node1.setY(0);
	o_ins_node2.setX(0);
	o_ins_node2.setY(0);
	MustErase = 0;
	color = Qt::black;
	input1 = (QPoint *)NULL;
	input2 = (QPoint *)NULL;
	output = (QPoint *)NULL;
	io_p1 = (QPoint *)NULL;
	io_p2 = (QPoint *)NULL;
}

Wire::~Wire()
{	list<QPoint> *l = wire.First();
	QPoint *pt;

	while(l && (NULL != (pt = l->Get()))) {
		if (!nodeIsReference(pt))
			delete pt;
                l = l->Next();
	}
	wire.Destroy();
	io_p.Destroy();
	selnode.Destroy();
}

// node clicked? then lock the node
list<QPoint> *Wire::contains(QPoint p)
{	list<QPoint> *l = wire.First();

	align(&p);

	// if point is in a node, lock node 
	while(l) {
		if (*(l->Get()) == p) {
			lockNode(l);
			return l;
		}
		l = l->Next();
	}
	return (list<QPoint> *)NULL;
}

// (selection method) select nodes contained in rect or all nodes
int Wire::select(QRect r, int all)
{	list<QPoint> *l = wire.First();
	int ret = 0;

	while(l) {
		if (all || r.contains(*l->Get())) {
			if (!selnode.With(l->Get())) selnode.Append(l->Get());
			ret = 1;
		} else {
			selnode.Destroy(l->Get());
		}
		l = l->Next();
	}
	return ret;
}

// (selection method) unselect all nodes
int Wire::isSelected()
{
	if (selnode.First()) return 1;
	return 0;
}

// (selection method) unselect all nodes
void Wire::unselect()
{
	selnode.Destroy();
}

// return minimum x cood over all selected nodes
int Wire::getMinSelX()
{	list<QPoint> *l = selnode.First();
	int ret;

	if (!l) return 0;
	else ret = l->Get()->x();

	while(l) {
		if (l->Get()->x() < ret) ret = l->Get()->x();
		l = l->Next();
	}
	return ret;
}

// return minimum x cood over all selected nodes
int Wire::getMaxSelX()
{	list<QPoint> *l = selnode.First();
	int ret;

	if (!l) return 0;
	else ret = l->Get()->x();

	while(l) {
		if (l->Get()->x() > ret) ret = l->Get()->x();
		l = l->Next();
	}
	return ret;
}

// return minimum y cood over all selected nodes
int Wire::getMinSelY()
{	list<QPoint> *l = selnode.First();
	int ret;

	if (!l) return 0;
	else ret = l->Get()->y();

	while(l) {
		if (l->Get()->y() < ret) ret = l->Get()->y();
		l = l->Next();
	}
	return ret;
}

// return minimum y cood over all selected nodes
int Wire::getMaxSelY()
{	list<QPoint> *l = selnode.First();
	int ret;

	if (!l) return 0;
	else ret = l->Get()->y();

	while(l) {
		if (l->Get()->y() > ret) ret = l->Get()->y();
		l = l->Next();
	}
	return ret;
}

int Wire::lockSelectedNode()
{	current_node = (QPoint *)NULL;
	from_node = (QPoint *)NULL;
	to_node = (QPoint *)NULL;

	if (selnode.First()) current_node = selnode.First()->Get();

	if (current_node) {
		o_node = *current_node;
		return 1;
	}
	return 0;
}

// move all selected nodes
void Wire::addPos(int dx, int dy)
{	list<QPoint> *l = selnode.First();
	QPoint *pt;

	while(l) {
		pt = l->Get();
		// do not move nodes connected to a device
		if (pt != input1 && pt != input2 && pt != output) {
			// do not move nodes wich are references
			if (pt != io_p1 && pt != io_p2) {
				pt->setX(pt->x() + dx);
				pt->setY(pt->y() + dy);
			}
		}
		l = l->Next();
	}
}

// creates new node, if not i/o and not end-node
// otherwise locks the node
int Wire::createNode(QPoint p)
{	list<QPoint> *l;
	QPoint *pt;

	// method contains() locks the node, if wire contains p
	if (NULL != (l = contains(p))) {
		pt = l->Get();

		// new node, if no successor and not connected, otherwise existing node
		if (l->Prev() && !l->Next()) {
			if ((pt != input1) && (pt != input2) && (pt != output) && (pt != io_p1) && (pt != io_p2)) {
				current_node = new QPoint(*pt);
				return lockNode(wire.Append(current_node));
			}
		}

		// new node, if no successor and not connected, otherwise existing node
		if (!l->Prev() && l->Next()) {
			if ((pt != input1) && (pt != input2) && (pt != output) && (pt != io_p1) && (pt != io_p2)) {
				current_node = new QPoint(*pt);
				return lockNode(wire.Insert(current_node, 0, pt));
			}
		}

		// new node, if wire with single node
		if (!l->Prev() && !l->Next()) {
			current_node = new QPoint(*pt);
			return lockNode(wire.Append(current_node));
		}

		// somewhat in between the wire..
		return 1;
	}
	return 0;
}

// locks point by content: used for import only!
// tries only to lock the first and the last node!
int Wire::lockNode(QPoint pt)
{	list <QPoint> *f = wire.First();
	QPoint *fp = f->Get();
	list <QPoint> *l = wire.Last();
	QPoint *lp = l->Get();

	if (*fp == pt && io_p1 != fp && io_p2 != fp)
		return lockNode(f);

	if (*lp == pt && io_p1 != lp && io_p2 != lp)
		return lockNode(l);

	if (*fp == pt)
		return lockNode(f);

	if (*lp == pt)
		return lockNode(l);

	return NODE_NOLOCK;
}

// lock a node
int Wire::lockNode(QPoint *pt)
{	list<QPoint> *l = wire.With(pt);

	if (l) return lockNode(l);
	return NODE_NOLOCK;
}

// lock a node in the list of nodes
int Wire::lockNode(list<QPoint> *l)
{	int ret = 0;

	current_node = l->Get();
	if (!l->Prev() && !l->Next()) {
		from_node = (QPoint *)NULL;
		to_node = (QPoint *)NULL;
		ret = NODE_SINGLE;
	}
	else if (l->Prev() && !l->Next()) {
		from_node = l->Prev()->Get();
		to_node = (QPoint *)NULL;
		ret = NODE_FRONT;
	}
	else if (!l->Prev() && l->Next()) {
		from_node = l->Next()->Get();
		to_node = (QPoint *)NULL;
		ret = NODE_FRONT;
	}
	else if (l->Prev() && l->Next()) {
		from_node = l->Prev()->Get();
		to_node = l->Next()->Get();
		ret = NODE_MIDDLE;
	}
	o_node = *current_node;
	return ret;
}

// lock the start node of the wire
int Wire::lockLastNode()
{	current_node = (QPoint *)NULL;
	from_node = (QPoint *)NULL;
	to_node = (QPoint *)NULL;

	if (wire.First()) current_node = wire.First()->Get();

	if (current_node) {
		o_node = *current_node;
		return 1;
	}
	return 0;
}

// remove the node from the list
// (does not destroy the node)
void Wire::cutNode()
{	if (current_node) {
		wire.Destroy(current_node);
		io_p.Destroy(current_node);
		selnode.Destroy(current_node);
		current_node = (QPoint *)NULL;
	}
}

// remove the node from the list
// (destroys the node)
void Wire::removeNode()
{
	if (current_node) {
		wire.Destroy(current_node);
		io_p.Destroy(current_node);
		selnode.Destroy(current_node);
		delete current_node;
		current_node = (QPoint *)NULL;
	}
}

void Wire::releaseNode()
{	from_node = (QPoint *)NULL;
	current_node = (QPoint *)NULL;
	to_node = (QPoint *)NULL;
}

int Wire::countNodes()
{
	return wire.counter();
}

int Wire::isPart(QPoint *pt)
{	if (wire.With(pt)) return 1;
	return 0;
}

// check connection between wires
// given point lies near a node or a line of this wire
// if connection required, connect it (insert ref, lock node, set input/output)
QPoint *Wire::checkConnection(QPoint *pt)
{	list<QPoint> *l1 = (list<QPoint> *)NULL;
	list<QPoint> *l2 = (list<QPoint> *)NULL;
	int der_eq;
	int inbetw_eq;

	// first of all, check if point lies near a point of this wire
	if (NULL != (l1 = contains(*pt))) {
		// never connect input- or output nodes!
		if ((current_node == input1) || (current_node == input2) || (current_node == output))
			return (QPoint *)NULL;

		// insert foreign node into wire, append pt before the existing own node
		wire.Insert(pt, 0, l1->Get());
		io_p.Append(pt);
		// remove the existing (currently locked) node!
		if (!activeIsReference() && !activeIsForeign()) {
			removeNode();
		}
		l1 = wire.With(pt);
		if (!l1) fatal("Wire::checkConnection(..) node not inserted??");
		lockNode(l1);
	  	return current_node;
	}

	// check each line..
	l1 = wire.First();
	l2 = wire.First()->Next();
	while (l1 && l2) {
		// look if line goes through the given point
		der_eq = derive(*l1->Get(), *l2->Get(), *pt);
		// if it is so, look for coordinates..
		if (der_eq) {
			inbetw_eq = inBetween(*l1->Get(), *l2->Get(), *pt);
			if (inbetw_eq) {
				o_ins_node1 = *l1->Get();
				o_ins_node2 = *l2->Get();

				// insert foreign node, append pt between the existing two own nodes
				wire.Insert(pt, 0, l2->Get());
				io_p.Append(pt);
				l1 = wire.With(pt);
				if (!l1) fatal("Wire::checkConnection(..) node not inserted??");
				lockNode(l1);
				return current_node;
			}
		}
		l1 = l2;
		l2 = l2->Next();
	}
	return (QPoint *)NULL;
}

// internal: derivation pt1:pt2 ~= pt1:pt?
int Wire::derive(QPoint pt1, QPoint pt2, QPoint pt)
{	double d1, d2, alpha, beta;

	long dx1, dy1, dx2, dy2;

	// reduce influence of node-distance by factor for pt1
	dx1 = pt1.x() - pt2.x();
	dy1 = pt1.y() - pt2.y();
	dx2 = pt1.x() + dx1 - pt.x();
	dy2 = pt1.y() + dy1 - pt.y();

	if (dy1 == 0) d1 = 0.0;
	else if (dx1 != 0) d1 = (double)dy1 / (double)dx1;
	else d1 = 999999.0;
	if (dy2 == 0) d2 = 0.0;
	else if (dx2 != 0) d2 = (double)dy2 / (double)dx2;
	else d2 = 999999.0;

	alpha = atan(d1);
	beta = atan(d2);

	if ((alpha >= beta - 0.05) && (alpha <= beta + 0.05)) return 1;
	return 0;
}

// internal: point between line(-rect)?
int Wire::inBetween(QPoint pt1, QPoint pt2, QPoint pt)
{	QRect r;

	if (pt1.x() > pt2.x()) {
		r.setX(pt2.x());
		r.setWidth(pt1.x() - pt2.x());
	}
	else {
		r.setX(pt1.x() - 1);
		r.setWidth(pt2.x() - pt1.x() + 2);
        }
	if (pt1.y() > pt2.y()) {
		r.setY(pt2.y());
		r.setHeight(pt1.y() - pt2.y());
	}
	else {
		r.setY(pt1.y() - 1);
		r.setHeight(pt2.y() - pt1.y() + 2);
	}
	if (r.contains(pt)) return 1;
	return 0;
}

QPoint Wire::getActive()
{	return *current_node;
}

QPoint *Wire::getActiveNode()
{	return current_node;
}

// returns index of active node (wire-connection; conn1 or conn2)
int Wire::getActiveConnIndex()
{	if (!current_node) return 0;
	if (current_node == io_p1) return 1;
	if (current_node == io_p2) return 2;
	return 0;
}

// returns 1, if active node is the input-node
int Wire::activeIsInput()
{
	if (!current_node) return 0;
	if (current_node == input1) return 1;
	if (current_node == input2) return 2;
	return 0;
}

// returns 1, if active node is the input-node
int Wire::nodeIsInput(QPoint *pt)
{
	if (!pt) return 0;
	if (pt == input1) return 1;
	if (pt == input2) return 2;
	return 0;
}

// returns 1, if active node is the output-node
int Wire::activeIsOutput()
{	if (current_node == output) return 1;
	return 0;
}

// returns 1, if active node is the output-node
int Wire::nodeIsOutput(QPoint *pt)
{	if (pt == output) return 1;
	return 0;
}

// returns 1, if active node is start- or end-point of wire
int Wire::activeIsEnd()
{	if ((current_node == wire.First()->Get()) || 
	    (current_node == wire.Last()->Get())) return 1;
	return 0;
}

// returns 1, if active node is reference of a connection
int Wire::activeIsReference()
{
	if ((current_node == io_p1) || (current_node == io_p2)) return 1;
	return 0;
}

// returns 1, if active node is a connection node
//            which another wire posesses
int Wire::activeIsForeign()
{
	if (io_p.With(current_node)) return 1;
	return 0;
}

// returns 1 if point is reference of a connection (first or last node)
int Wire::nodeIsReference(QPoint *pt)
{	if ((pt == io_p1) || (pt == io_p2)) {
		// lock the point
		current_node = pt;
		o_node = *current_node;
		return 1;
	}
	return 0;
}

// returns 1, if node is reference of a connection (first or last node of another wire)
int Wire::nodeIsForeign(QPoint *pt)
{
	if (io_p.With(pt)) return 1;
	return 0;
}

// draw wire, erase active parts before
void Wire::drawImage(QPainter *p)
{	list<QPoint> *l = wire.First();
	int f = 0;

	p->setPen(color);
	while(l && l->Get() && l->Next() && l->Next()->Get()) {
		// draw selected nodes green
		if (selnode.With(l->Get()) || selnode.With(l->Next()->Get())) {
			if (!f) {
				f = 1;
				p->setPen(Qt::green);
			}
		}
		else if (f) {
			f = 0;
			p->setPen(color);
		}
		p->drawLine(*(l->Get()), *(l->Next()->Get()));
		l = l->Next();
	}
	if (io_p1) drawSolderedPoint(p, Qt::black, *io_p1);
	if (io_p2) drawSolderedPoint(p, Qt::black, *io_p2);
}

// erase whole wire
void Wire::eraseWire(QPainter *p)
{	list<QPoint> *l = wire.First();
	QPoint *p1, *p2;

	if (io_p1) drawSolderedPoint(p, Qt::white, *io_p1);
	if (io_p2) drawSolderedPoint(p, Qt::white, *io_p2);
	while(l && l->Next()) {
		p1 = l->Get();
		p2 = l->Next()->Get();
		p->setPen(Qt::white);
		p->drawLine(*p1, *p2);
		p->setPen(Qt::black);
		restoreArry(p, p1->x(), p1->y(), p2->x(), p2->y());
		l = l->Next();
	}
}

// erase first, active, and last part of wire
void Wire::erase(QPainter *p)
{	QPoint pt2;
	list<QPoint> *first = wire.First();
	QPoint *firstp = (QPoint *)NULL;
	if (first) firstp = first->Get();
	list<QPoint> *last = wire.Last();
	QPoint *lastp = (QPoint *)NULL;
	if (last) lastp = last->Get();

	if ((!firstp) || (!lastp)) return;

	if (o_start_node.isNull()) o_start_node = *firstp;
	if (o_end_node.isNull()) o_end_node = *lastp;

	// node inserted: erase old part of wire
	if (!o_ins_node1.isNull() && !o_ins_node2.isNull()) {
		// delete old wire-part and restore grid
		p->setPen(Qt::white);
		p->drawLine(o_ins_node1, o_ins_node2);
		p->setPen(Qt::black);
		restoreArry(p, o_ins_node1.x(), o_ins_node1.y(), o_ins_node2.x(), o_ins_node2.y());
		o_ins_node1.setX(0);
		o_ins_node1.setY(0);
		o_ins_node2.setX(0);
		o_ins_node2.setY(0);
	}

	// erase old active part of wire
	if (!o_node.isNull()) {
		// delete old wire-part and restore grid
		if (from_node) {
			p->setPen(Qt::white);
			p->drawLine(*from_node, o_node);
			p->setPen(Qt::black);
			restoreArry(p, from_node->x(), from_node->y(), o_node.x(), o_node.y());
		}
		if (to_node) {
			p->setPen(Qt::white);
			p->drawLine(o_node, *to_node);
			p->setPen(Qt::black);
			restoreArry(p, to_node->x(), to_node->y(), o_node.x(), o_node.y());
		}
		// remove connection-point
		if (current_node && ((current_node == io_p1) || (current_node == io_p2)))
			drawSolderedPoint(p, Qt::white, o_node);
		o_node.setX(0);
		o_node.setY(0);
	}

	if (o_start_node != *firstp) {
		// remove connection-point
		if ((firstp == io_p1) || (firstp == io_p2))
			drawSolderedPoint(p, Qt::white, o_start_node);
		// erase first part of wire if changed
		// value of old start-node (first node) is in o_start_node
		if (first && first->Next()) {
			pt2 = *first->Next()->Get();
			// erase old wire-part
			if (!o_start_node.isNull()) {
				p->setPen(Qt::white);
				p->drawLine(o_start_node, pt2);
			}
			// restore grid
			p->setPen(Qt::black);
			restoreArry(p, o_start_node.x(), o_start_node.y(), pt2.x(), pt2.y());
		}
		o_start_node = *firstp;
	}

	if (o_end_node != *lastp) {
		// remove connection-point
		if ((lastp == io_p1) || (lastp == io_p2))
			drawSolderedPoint(p, Qt::white, o_end_node);
		// erase last part of wire if changed
		// value of last end-node (last node) is in o_end_node
		if (last && last->Prev()) {
			pt2 =*last->Prev()->Get();
			// erase old wire-part
			if (!o_end_node.isNull()) {
				p->setPen(Qt::white);
				p->drawLine(o_end_node, pt2);
			}
			// restore grid
			p->setPen(Qt::black);
			restoreArry(p, o_end_node.x(), o_end_node.y(), pt2.x(), pt2.y());
		}
		o_end_node = *lastp;
	}
}

// draw active part of wire
void Wire::draw(QPainter *p)
{
	if (current_node) {
		p->setPen(color);
		if (from_node) p->drawLine(*from_node, *current_node);
		if (to_node) p->drawLine(*current_node, *to_node);
	}
}

// draw connection between wires
void Wire::drawSolderedPoint(QPainter *p, QColor c, QPoint pt)
{	p->setPen(c);
	p->setBrush(c);
	p->drawPie(pt.x() - 2, pt.y() - 2, 4, 4, 0, 5760);
}

void Wire::restoreArry(QPainter *p, int x1, int y1, int x2, int y2)
{	int i, j, a, b, c, d;
	if (x1 < x2) {
		a = x1;
		b = x2;
	}
	else {
		b = x1;
		a = x2;
	}
	if (y1 < y2) {
		c = y1;
		d = y2;
	}
	else {
		d = y1;
		c = y2;
	}
	for(i=a;i <= b;i += GRID) {
		for(j=c; j <= d; j += GRID) {
			p->drawPoint(i,j);
		}
	}
}

void Wire::updateNode(QPoint p)
{
	align(&p);

	if (!current_node) fatal(NOACTIVE);
	o_node = *current_node;
	*current_node = p;
}

QPoint *Wire::align(QPoint *p)
{	p->setX((p->x() + GRIDHALF) / GRID * GRID);
	p->setY((p->y() + GRIDHALF) / GRID * GRID);
	return p;
}

void Wire::setColor(QColor c)
{
	color = c;
}

// remove useless nodes
void Wire::garbageCollection()
{	list<QPoint> *lp = wire.First();
	QPoint *last = (QPoint *)NULL;
	QPoint *act = (QPoint *)NULL;
	int modified = 0;
	int flag1, flag2, flag3, flag4;

	while(lp && lp->Get()) {
		act = lp->Get();
		if (last && (*last == *act)) {
			// try to remove point "act"
			flag1 = nodeIsReference(act);
			flag2 = nodeIsForeign(act);
			flag3 = nodeIsInput(act);
			flag4 = nodeIsOutput(act);
			if (!flag1 && !flag2 && !flag3 && !flag4) {
				// remove point act
				lp->Destroy(act);
				delete act;
				modified = 1;
			}

			// try to remove point "last"
			else {
				flag1 = nodeIsReference(last);
				flag2 = nodeIsForeign(last);
				flag3 = nodeIsInput(last);
				flag4 = nodeIsOutput(last);
				if (!flag1 && !flag2 && !flag3 && !flag4) {
					// remove point last
					lp->Destroy(last);
					delete last;
					modified = 1;
				}
			}

		}
		if (modified) {
			modified = 0;
			last = (QPoint *)NULL;
			lp = wire.First();
		} else {
			last = lp->Get();
			lp = lp->Next();
		}
	}
}
	
// returns a string with all nodes like this: "{x1:y1:x2:y2:x3:...}"
QString Wire::wire2string(int selonly, int dx = 0, int dy = 0)
{	list<QPoint> *l = wire.First();
	QString ret;
	char buf[1024];
	int first = 1;

	ret.truncate(0);

	while(l && l->Get()) {
		if ((selonly && selnode.With(l->Get())) || (!selonly)) {
			if (first) {
				sprintf(buf, "{%d:%d}", l->Get()->x() - dx, l->Get()->y() - dy);
				first = 0;
			}
			else {
				sprintf(buf, ",{%d:%d}", l->Get()->x() - dx, l->Get()->y() - dy);
			}
			ret.append(buf);
		}
		l = l->Next();
	}
	return ret;
}

// makes a wire from a string containing points
// return 0 if unsuccessful
int Wire::string2wire(const char *swire, int dx = 0, int dy = 0)
{	QPoint pt(0,0);
	int nodecounter = 0;
	char *buf = (char *)swire;
	char *posa, *pose;

	while(NULL != (posa = strchr(buf, '{'))) {
		if (!(pose = strchr(buf, ':'))) return 0;
		*pose = '\0';
		pt.setX(atoi(++posa) + dx);
		posa = pose + 1;
		if (!(pose = strchr(posa, '}'))) return 0;
		*pose = '\0';
		pt.setY(atoi(posa) + dy);
		buf = pose + 1;

		current_node = new QPoint(pt);
		wire.Append(current_node);
		nodecounter++;
	}
	if (nodecounter > 1)
		return 1;
	else
		return 0;
}

// set the active node as input-node, if it's an endpoint
int Wire::connectInput()
{
	if (!input1) {
		if(current_node && (!to_node || !from_node)) {
			input1 = current_node;
			return WOK1;
		}
		fatal(NOACTIVECONN);
	}
	if (!input2) {
		if(current_node && (!to_node || !from_node)) {
			input2 = current_node;
			return WOK2;
		}
		fatal(NOACTIVECONN);
	}
	return WFAIL;
}

// reset input-node
void Wire::disconnectInput(int idx)
{
	if (idx == 1) {
		if (input2) {
			input1 = input2;
			input2 = (QPoint *)NULL;
		}
		else {
			input1 = (QPoint *)NULL;
		}
	}
	else if (idx == 2) input2 = (QPoint *)NULL;
	else fatal("no locked input node for disconnection"); 
}

// set the active node as output-node, if it's an endpoint
int Wire::connectOutput()
{
	if (input1 && input2) return WFATAL;

	if (!output) {
		if(current_node && (!to_node || !from_node)) {
			output = current_node;
			return WOK;
		}
		fatal(NOACTIVECONN);
	}
	return WFAIL;
}

// reset output-node
void Wire::disconnectOutput()
{	output = (QPoint *)NULL;
}

// set io-point for connection to another wire
int Wire::connectWire(int idx)
{	if (io_p1 && io_p2) return WFATAL;

	if (current_node) {
		if (!io_p1 && (idx == 1)) {
			io_p1 = current_node;
			io_p.Append(current_node);
			return idx;
		}
		if (!io_p2 && (idx == 2)) {
			io_p2 = current_node;
			io_p.Append(current_node);
			return idx;
		}
	}
	return WFATAL;
}

void Wire::disconnectWire(int idx)
{	if (idx == 1) {
		io_p.Destroy(io_p1);
		io_p1 = (QPoint *)NULL;
	}
	if (idx == 2) {
		io_p.Destroy(io_p2);
		io_p2 = (QPoint *)NULL;
	}
}

// change position of input-node (device-connection nodes) on the currently locked node
void Wire::setInputPosition(QPoint newp)
{
	if (!current_node) return;
	if (current_node == input1) *input1 = newp;
	else if (current_node == input2) *input2 = newp;
}

// change position of input-node (device-connection nodes)
void Wire::setInputPosition(QPoint newp, int idx)
{
	if ((idx == 1) && input1) *input1 = newp;
	else if ((idx == 2) && input2) *input2 = newp;
}

// change position of output-node (device-connection nodes)
void Wire::setOutputPosition(QPoint newp)
{
	if (output) *output = newp;
}

// get position of input-node (device-connection nodes)
QPoint Wire::getInputPosition(int idx, int selonly = 0)
{
	QPoint p(0,0);
	if ((idx == 1) &&  input1) {
		if (!selonly) return *input1;
		if (selnode.With(input1)) return *input1;
	}
	if ((idx == 2) &&  input2) {
		if (!selonly) return *input2;
		if (selnode.With(input2)) return *input2;
	}
	return p;
}

// get position of input-node (device-connection nodes)
QPoint Wire::getOutputPosition(int selonly = 0)
{
	QPoint p(0,0);
	if (output) {
		if (!selonly) return *output;
		if (selnode.With(output)) return *output;
	}
	return p;
}

// get position of input/output-node (wire-connection nodes)
QPoint Wire::getIOPosition(int idx, int selonly = 0)
{	QPoint p(0,0);
	if (idx == 1 && io_p1) {
		if (!selonly) return *io_p1;
		if (selnode.With(io_p1)) return *io_p1;
	}
	if (idx == 2 && io_p2) {
		if (!selonly) return *io_p2;
		if (selnode.With(io_p2)) return *io_p2;
	}
	return p;
}

