#ifndef __KPCOMM_H__
#define __KPCOMM_H__

#include "kpmessages.h"

#include <qqueue.h>

#include <ksock.h>

/**
 * Pass blocks data using shared memory and
 *  control info using sockets.
 *
 * There are no ACKs right now.
 * 
 * In the future, I may add data-via-socket to facilitate remote display
 *  of data with KPComm.
 *
 * @author David Sweet <dsweet@kde.org>
 **/
class SHMComm : public KSocket
{
 Q_OBJECT
 public:
 enum {NoDataMessage=-1}; 

 /**
  * Some messages used by KPComm.  Make your messages have values
  *  greater than 100 so you won't interfere with future versions of
  *  KPComm.
  **/
 enum {NoMessage=-1, NOP=1, SHMDelete=2};

  /**
   * Create a socket or take over a KSocket.
   * @param socketfilename Something like /tmp/kpcomm.pid will do.
   **/
  SHMComm (const char *socketfilename);

  /**
   * The server will need to:
   *    int newsocket = dup (ksocket->socket());
   *    kpcomm = new KPComm (newsocket);
   *    delete ksocket;
   * to transfer the socket passed to slotAccept (KSocket *ksocket).
   **/
  SHMComm (int socket);

  /**
   * This sends a control message + large data.
   * Message types are in the enums. <i>data</i> is anything.
   **/
  void send (int message, int option, const void *data, int datasize);

  /**
   * This sends a control message + an integer option.
   **/
  void send (int message, int option=0);

  /**
   * Call this when you receive a @ref messageWaiting() signal.
   * <i>data</i> gets filled with the data.
   **/
  void * get (int &message, int &option, int &datasize, int &id);

  /**
   * Call this to free up the data when you are done with it.
   *  pass the <i>data</i> and <i>id</i> you got from @ref get(). 
   **/
  void releaseData (void *data, int id);

  public slots:
    void slotReadMessage (KSocket *);
    void slotWriteMessage (KSocket *);

 signals:
  void messageWaiting (int);

 private:
  int curmessage;
  int cursendmessage;
  struct SPacket
    { int shmid; int size; int option; int message; };
  QQueue<SPacket> sendpackets;
  QQueue<SPacket> recvpackets;

  void init (void);

};

/**
 * KPComm is used by both sides. 
 *  One side listens with KServerSocket.  Messages 
 *  can be passed either way.
 *
 * @author David Sweet
 **/
class KPComm : public SHMComm
{
 public:

  /**
   * Create a socket or take over a KSocket.
   * @param socketfilename Something like /tmp/kpcomm.pid will do.
   **/
  KPComm (const char *socketfilename);

  /**
   * The server will need to:
   *    int newsocket = dup (ksocket->socket());
   *    kpcomm = new KPComm (newsocket);
   *    delete ksocket;
   * to transfer the socket passed to slotAccept (KSocket *ksocket).
   **/
  KPComm (int socket);

  /**
   * Send a matrix description.
   *
   * (convenience method)
   **/
  void sendMatrixDescription (int matrixid, int nrows, int ncols, const char *name);

  /**
   * Send matrix data.
   *
   * (convenience method)
   **/
  void sendMatrix (int matrixid, int nrows, int ncols, double *matrix);

};


#endif

