/**
    KPlotServer:  GUI + KPComm server for linking to RLab, Perl, etc.
 
    Copyright (C) 1999, 2000 by David Sweet


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
 
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
 
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    

    Please read the file COPYING in this directory for information about 
     the GNU GPL.

*/                                      




//NOTES:
//   o  If you want to access a sprite (to hide() it, for example),
//       you should save the pointer to it when you create it.  
//   o  The plot owns the matrices you pass to it and will delete them
//       upon exit (via setAutoDelete() in its QList<KPMatrix>

#include <unistd.h>
#include <stdio.h>

#include <math.h>
#include <iostream>

#include <qfile.h>
#include <qkeycode.h>
#include <qlayout.h>
#include <qwhatsthis.h>

#include <kapp.h>
#include <kstddirs.h>
#include <kaboutdialog.h>
#include <krun.h>
#include <kfiledialog.h>
#include <kglobal.h>
#include <ktoolbar.h>
#include <ktoolbarradiogroup.h>
#include <kstatusbar.h>
#include <kiconloader.h>
#include <kimgio.h>
#include <kmsgbox.h>
#include <kurl.h>
#include <kprogress.h>
#include <kmenubar.h>
#include <kdebug.h>
#include <ksock.h>
#include <klocale.h>

#include <kplotw.h>
#include <kpsaxis.h>
#include <kps2dparametric.h>
#include <kps2doptions.h>
#include <kpstext.h>
#include <kporder.h>
#include <kplineddlg.h>
#include <kpcolortable.h>
#include <kplineddlg.h>
//#include <kpspline.h>
#include <kputil.h>

#include "zoomcursor_xbm.h"

#include <kpcomm.h>
#include <kpmessages.h>

#include "kplotserver.moc"


//Toolbar
const int ID_PRINT=1;
const int ID_SELECT=3;
const int ID_ZOOM=4;
const int ID_ZOOMOUT=5;

//Menubar
const int ID_PTYPES=10;

//Statusbar
const int S_XCOORD=1;
const int S_YCOORD=2;
const int S_MAIN=3;

//Context menu(s)
const int ID_SPRITE_DELETE=99;

//Need to get this from the command line
#define SOCKNAME "/tmp/kplotcomm"

#define AREA 0

#define KPLOTSERVER_VERSION "KPlotServer 0.0.4"

KPlotServer::KPlotServer (const char *_socketname, const char *name) : KTMainWindow (name)
{
  plotw = new KPlotW (this);//, 0);//, QFrame::Panel | QFrame::Raised);

  socketname = _socketname;

  setCaption (KPLOTSERVER_VERSION);
    
  KMenuBar *menu = menuBar();
  file = new QPopupMenu;
  //  file->insertItem ("&Import Matrix...", this, SLOT(slotImport()),		    CTRL+Key_I);
  //  idaddmatrix=file->insertItem ("&Add Matrix...", this, SLOT(slotAdd()),		    CTRL+Key_A);
  //  idreload=file->insertItem ("&Reload...", this, SLOT(slotReload()),		    CTRL+Key_R);
  idprint=file->insertItem ("&Print...", plotw, SLOT(slotPrint()));
  idprintdraft=file->insertItem ("Print &Draft...", plotw, SLOT(slotPrintDraft()));
  file->insertSeparator();
  file->insertItem ("&Quit", this, SLOT(slotQuit()), 
		    CTRL+Key_Q);
  menu -> insertItem ("&File", file);

  file->setItemEnabled (idprint, false);
  file->setItemEnabled (idprintdraft, false);
  //  file->setItemEnabled (idaddmatrix, false);
  //  file->setItemEnabled (idreload, false);

  edit = new QPopupMenu;
  ptypes = new QPopupMenu;
  ptypes->insertItem ("Parametric", KPlotW::Parametric2D);
  ptypes->insertItem ("Matrix Map", KPlotW::Map2D);
  ptypes->insertItem ("Bar Graph", KPlotW::Bars);
  connect ( ptypes, SIGNAL (activated (int)),
	    SLOT (slotPlotType (int)) );

  idptypes = edit->insertItem ("&Plot type", ptypes);
  idcoords = edit->insertItem ( "Change &coordinates...", this,
				SLOT (slotCoordsDialog()) );
  idautocoords = edit->insertItem ( "Automatic Coordinates", this,
				    SLOT (slotAutoCoords()) );
  //  idspline = edit->insertItem ("Fit With &Spline...", this, SLOT (slotAddSpline()));
  //  edit->setItemEnabled (idspline, false);
  edit->setItemEnabled (idcoords, false);
  edit->setItemEnabled (idptypes, false);
  edit->setItemEnabled (idautocoords, false);
  menu->insertItem ("&Edit", edit);

  

  menu->insertSeparator();

  QPopupMenu *help = helpMenu();
  menu->insertItem ( "&Help", help );
  connect ( help, SIGNAL (activated(int)),
	    this, SLOT (slotAboutBox (int)) );
  QWhatsThis::add (plotw, "This is the plotting area.");


  KToolBar *kt = toolBar ();
  QPixmap pixmap;
  /*  pixmap = Icon( "fileopen.xpm" );
  kt->insertButton(pixmap, ID_OPEN, SIGNAL (clicked()), this,
		   SLOT (slotImport()),
		   true, "Import Matrix...");   
  */

  pixmap = BarIcon( "fileprint.xpm" );
  kt->insertButton(pixmap, ID_PRINT, SIGNAL (clicked()), plotw,
			SLOT (slotPrint()),
			false, "Print Plot (HQ)...");

  pixmap = BarIcon ( "viewmag-.xpm");
  kt->insertButton(pixmap, ID_ZOOMOUT,
		   SIGNAL(clicked()), this,
		   SLOT (slotZoomOut()),
		   false,
		   "Zoom out");

  //Tools
  kt->insertSeparator();
  
  //This exclamation should be an arrow cursor!
  pixmap = BarIcon ("exclamation.xpm");
  kt->insertButton(pixmap, ID_SELECT, false,
		   "Select/move tool");
  kt->setToggle (ID_SELECT);
  pixmap = BarIcon ( "viewmag+.xpm");
  kt->insertButton(pixmap, ID_ZOOM, false,
		   "Zoom tool");
  kt->setToggle (ID_ZOOM);

  KToolBarRadioGroup *kr = new KToolBarRadioGroup (kt);
  kr->addButton (ID_SELECT);
  kr->addButton (ID_ZOOM);

  connect (kt, SIGNAL (toggled (int)),
	   this, SLOT (slotToolBarToggle(int)) );

  //My zoom cusor (magnifying glass)
  QBitmap zb (zoomcursor_width, zoomcursor_height, zoomcursor_bits, TRUE);
  zoomcursor = new QCursor (zb, zb.createHeuristicMask(), 7, 7);
  

  ///////////////
  KStatusBar *ks = statusBar ();
  ks->setInsertOrder (KStatusBar::RightToLeft);
  //  ks->setBorderWidth (3);
  ks->insertItem ("000000000000", S_YCOORD);
  ks->insertItem ("000000000000", S_XCOORD);
  ks->insertItem (" ", S_MAIN);
  ks->changeItem ("", S_XCOORD);
  ks->changeItem ("", S_YCOORD);

  //Emits the signal if a mouse button is pressed/held down.
  plotw->view()->setEmitMouseMove ();
  connect (plotw->view(), SIGNAL (mouseAt (int,int)),
	   this, SLOT (slotMouseCoords (int,int)) );
  

  setView (plotw);
  setFocusPolicy (QWidget::ClickFocus);

  char tmp [1024];
  dirpath = getcwd (tmp, 1024);

  tool=SelectTool;

  plot=0;
  resize (400,300);

  startServer();
}

const int idaboutkplot = 2;
void
KPlotServer::slotAboutBox (int id)
{
  if (id!=idaboutkplot)
    return;

  KAboutDialog aboutbox (this);

  QString aboutkp =
    "KPlot: User-friendly plotting for KDE\n"
    "This is KPlotServer, which provides a GUI for programs which\n"
    "cannot or will not provide their own\n\n"
    //    "You can use KPlot to easily and quickly create attractive,\n"
    //    " publication-quality scientific plots."   // Not quite yet
    "KPlot is built around KPlotW, a 2D plotting widget.\n"
    "Notable features:\n"
    "    * WYSIWYG formatting\n"
    "    * point-and-click plot editing\n"
    "    * responsive GUI\n\n"
    "Please send bug reports and feature requests to the above email address.";
  
  aboutbox.setAuthor ("David Sweet", "dsweet@kde.org",
	     "http://www.chaos.umd.edu/~dsweet/KDE", aboutkp);

  aboutbox.setVersion (KPLOTSERVER_VERSION);

  KImageIO::registerFormats();
  locate ("appdata", "kplogo.png");
  QPixmap p;
  p.load(locate ("appdata", "kplogo.png"));
  aboutbox.setLogo (p);

  aboutbox.adjust();
  aboutbox.exec();

}


void
KPlotServer::startServer (void)
{
  KServerSocket *server = new KServerSocket (socketname);
  connect ( server, SIGNAL (accepted(KSocket *)),
	    this, SLOT (slotAccept(KSocket *)) );
  kDebugInfo (AREA, "KPComm SERVER STARTED\n");
}

void
KPlotServer::slotAccept (KSocket *socket)
{
  kDebugInfo (AREA, "ACCEPT\n");
  int newsocket = dup (socket->socket());
  kpcomm = new KPComm (newsocket);
  delete socket;
  connect ( kpcomm, SIGNAL (messageWaiting (int)),
            this, SLOT (slotMessageWaiting (int)) );
  kDebugInfo (AREA, "ACCEPT DONE\n");
}

void
KPlotServer::slotMessageWaiting (int)
{
  void *data=0;
  int message, datasize, msgid, option;
  data=(void *)kpcomm->get (message, option, datasize, msgid);
  kDebugInfo (AREA, "message is [%d]  datasize=[%d]\n", message, datasize);
  kDebugInfo (AREA, "option is [%d]  data=[%d]\n", option, (int) data);

  if (data==0) //no message
    return;


  //Process messages here
  switch (message)
    {
    case KPCMatrixDesc:
      {
	kDebugInfo (AREA, "KPCMatrixDesc\n");
	SKPCMatrixDesc *matdesc = (SKPCMatrixDesc *)data;
	kDebugInfo (AREA, "[%d] [%d]\n",
		matdesc->nrows, matdesc->ncols);
	kDebugInfo (AREA, "[%s]\n", matdesc->name);

	struct SKPMatrixEntry *m = new SKPMatrixEntry;
	m->nrows = matdesc->nrows;
	m->ncols = matdesc->ncols;
	m->name = matdesc->name;
	m->id = option;
	matrices.append (m);
      }
      break;

    case KPCMatrix:
      {
	kDebugInfo (AREA, "KPCMatrix\n");
	SKPMatrixEntry *me;


	SKPMatrixEntry fme;


	fme.id = option;
	
	if (matrices.find (&fme)!=-1)
	  {
	    me=matrices.current();
	    //	    printf ("r=%d c=%d\n",		    me->nRows,		    me->nCols);

	    me->matrix = 
	      new KPMatrix ((double *)data, me->ncols,
			    me->nrows);
	    
	    //TEST -- create should be a different call
	    kDebugInfo (AREA, "CREATE\n");
	    if (plot!=0)
	      delete plot;

	    plot = createPlot (me->matrix, -1);
	    updateMenusEtc();
	  }
	else
	  kDebugInfo (AREA, "Invalid matrix id.\n");


      }
      break;

    default:
      kDebugInfo (AREA, "Unrecognized message: %d\n", message);
    };

  
  kpcomm->releaseData (data, msgid);
}


void
KPlotServer::slotMouseCoords (int x, int y)
{
  //Update the coordinates in the statusbar.
  switch (plot->whatami())
    {
    case RTTI_KPS2DParametric:
      {
	QString qs;
	qs.sprintf ("x=%4g", plot->coords()->xo_to_c (x));
	statusBar()->changeItem ((const char *)qs, S_XCOORD);
	qs.sprintf ("y=%4g", plot->coords()->yo_to_c (y));
	statusBar()->changeItem (qs, S_YCOORD);
      }
      break;
    case RTTI_KPS2DMap:
      {
	unsigned int row = (int) plot->coords()->yo_to_c (y) + 1;
	unsigned int col = (int) plot->coords()->xo_to_c (x) + 1;
	if (col>=plot->coords()->cxmin() &&
	    col<=plot->coords()->cxmax() &&
	    row>=plot->coords()->cymax() &&
	    row<=plot->coords()->cymin())
	  {
	    QString qs;
	    qs.sprintf ("row=%u", row);
	    statusBar()->changeItem (qs, S_XCOORD);
	    qs.sprintf ("col=%u", col);
	    statusBar()->changeItem ((const char *)qs, S_YCOORD);
	  }
      }
      break;
    }
}

/*
void
KPlotServer::slotReload()
{
  //Save the plot coords
  KPCoords c;
  c=*plot->coords();

  switch (plot->whatami())
    {
    case RTTI_KPS2DParametric:
      {
	KPS2DParametric *p=(KPS2DParametric *)plot;
	
	int n = filenames.count();	
	unsigned int i=0, imax=p->nSeries();

	//Save the plot options
	KPS2DParametricOptions po [imax];
	for (i=0; i<imax; i++)
	  po [i] = p->options (i);

	delete plot;

	KPMatrix *m = loadMatrix (filenames.at(0));
	if (!m)
	  return;
	plot = createPlot (m, KPlotW::Parametric2D);
	p=(KPS2DParametric *)plot;

	
	if (n>1)
	  {
	    for (int i=1, is=0; i<n; i++)
	      {
		if (filenames.at(i)[0]!='\0')
		  {
		    QString qs ("Reloading ");
		    qs += filenames.at(i);
		    statusBar()->changeItem ((const char *)qs, S_MAIN);
	
		    m = loadMatrix (filenames.at(i));
		    if (m)
		      {
			p->addMatrix (m);
			KPCoords c = KPS2DParametric::autoCoords (m, p->coords());
			
			p->setCoords (c);
		      }
		  }
		else
		  {
		  //		    spline (splines.at(is)->series, splines.at(is)->detail);
		    is++;
		  }
	      }
	    
	  }
	//restore the plot options
	for (i=0; i<imax; i++)
	  p->setOptions (po [i], i);
      }
      break;
    default:
      {
	int ptype = plot->whatami();
	delete plot;
	KPMatrix *m = loadMatrix (filenames.at(0));
	if (!m)
	  return;
	plot = createPlot (m, ptype);
      }
    }
  
  //Restore the coords.
  plot->setCoords (c);

  plot->update();

  QString qs ("Reloaded all matrices");
  statusBar()->changeItem ((const char *)qs, S_MAIN);  
}


void
KPlotServer::slotAdd()
{
  //  KFileDialog fd (dirpath);
  //  fd.setCaption("Add Matrix");

  QString filename = KFileDialog::getOpenFileName(dirpath);

  if (filename.isNull())
    return;

  KURL url (filename);
  dirpath = url.directory();

  KPMatrix *m=loadMatrix (filename);
  if (!m)
    return;

  filenames.append (filename);
  ((KPS2DParametric *)plot)->addMatrix (m);
  KPCoords c = KPS2DParametric::autoCoords (m, plot->coords());

  plot->setCoords (c);
  plot->update();

  niceCaption();
  QString qs ("Added ");
  qs += filename;
  statusBar()->changeItem ((const char *)qs, S_MAIN);  

}
*/

void
KPlotServer::keyPressEvent (QKeyEvent *ke)
{
  switch (ke->key())
    {
    case Key_Delete:
      //      if (ke->state() & (ShiftButton | ControlButton | AltButton))
	{
	  if (plotw->view()->focused()!=plot)
	    delete plotw->view()->focused();
	}
	break;
    default:
      ke->ignore();
    };
}


/*
void
KPlotServer::slotAddSpline()
{
  //Only works with KPS2DParametric!

  KPLinedDlg seriesdlg ("Fit With Spline (Beizer) Curve",
			"Series Number:", "1", this);

  // Better dialog: choose /series/ from combo box, enter /detail/
  //  as an integer (w/a validator).

  if (seriesdlg.exec())
    {
      int series = seriesdlg.text().toInt() - 1;
      KPS2DParametric *p= (KPS2DParametric *)plot;
      if (series<0 || series>p->nMatrices())
	{
	  QString r;
	  r.sprintf ("Invalid series number.\nValid series are in"
		     " [%d, %d].\n", 0, p->nMatrices());
	  KMsgBox::message (this, "Error",
			    r, KMsgBox::STOP, "OK");
	  return;
	}
      //Later /detail/ will be configurable.  Right now it's 10.
      int detail = 10;
      spline (series, detail);
      SSplineRec *sr = new SSplineRec;
      sr->series = series;
      sr->detail = detail;
      splines.append (sr);
      filenames.append ("");
    }
      

}
 */

/*
void
KPlotServer::spline (int series, int detail)
{
  KPS2DParametric *p= (KPS2DParametric *)plot;

  KPSpline *s = new KPSpline (p->matrix (series), detail);
  unsigned int nm = p->addMatrix (s);
  p->setPointStyle (Points, nm);
  p->setConnected (true, nm);
  QString qs;
  qs.sprintf ("Spline for series %d", series);
  p->setSeriesName (qs, nm);
  p->setColor (gray, nm);
    
}
*/


void
KPlotServer::niceCaption (void)
{
  if (filenames.count()==1)
    {
      KURL url (filenames.at(0));
      QString caption (url.filename());
      caption = caption + " - " + KPLOTSERVER_VERSION;
      
      setCaption (caption);
    }
  else
    {
      QString caption ("Multiple files");
      caption = caption + " - " + KPLOTSERVER_VERSION;
      
      setCaption (caption);
    }
  
}

 
KPS2D *
KPlotServer::createPlot (KPMatrix *m, int ptype)
{
  KPS2D *p=0;

  if (ptype == -1)  //auto
    {
      printf ("AUTO:  %d %d  %d %f\n",
	      m->nRows(), m->nCols(),
	      abs (m->nRows()-m->nCols()),
	      .25*kpmax((int)m->nRows(), (int)m->nCols()) );
      
      if (m->nCols()==1)
	ptype=KPlotW::Bars;
      else
	{
	  if (abs (m->nRows()-m->nCols()) <
	      .25*kpmax((int)m->nRows(), (int)m->nCols()) )  //square-ish
	    ptype = KPlotW::Map2D;
	  else
	    ptype = KPlotW::Parametric2D;
	}
    }

  printf ("PLOT TYPE = %d\n", ptype);

  //Create plot
  switch (ptype)
    {
    case KPlotW::Map2D:
	{
	  //Color table (outside access later)
	  double mx = 1e-300;
	  double mn = 1e300;
	  unsigned int c;
	  for (c=0;c<m->nCols();c++)
	    {
	      if (m->max(c)>mx)
		mx = m->max(c);
	      if (m->min(c)<mn)
		mn = m->min(c);
	    }
	  if (mx==mn)
	    mx=mn+1;

	  KPColorTable colortable (mn, mx);
	  double w = mx-mn;
	  double di = w/255;
	  for (double i=mn;i<=mx;i+=di)
	    {
	      int c=(int) (255.*((i-mn)/w));
	      colortable.addColor (i, QColor (c, c, 255));
	    }

	  p = plotw->addPlot(KPlotW::Map2D, m, 0, &colortable);

	}
	break;
    default:
      p = plotw->addPlot(ptype, m);
    }

  
  //Set up plot


  switch (ptype)
    {
    case KPlotW::Parametric2D:
      {
	p->addCoordFrame();

	((KPS2DParametric *)p)->setConnected (false);
	if (m->nRows()>=100)
	  ((KPS2DParametric *)p)->setPointStyle (Points);
	else
	  ((KPS2DParametric *)p)->setPointStyle (FilledCircles);
	
	
	p->installDefaultRMBMenu ();
	
      }
      break;
    case KPlotW::Map2D:
      p->addCoordFrame();
      p->installDefaultRMBMenu ();
      break;

    case KPlotW::Bars:
      p->addCoordFrame();
      p->installDefaultRMBMenu ();
      break;
    }
  
  p_rmb=p->rmbMenu();
  p_rmb->insertSeparator (0);
  p_rmb->insertItem ("Text Label", this, SLOT (slotAddText()),
		     0, -1, 0);
  
  axesRMB(p);
  p->setMovableX (false);
  p->setMovableY (false);
  p->setDrawFrame (false);


  p->update();	
  //  cout << "E" << endl;

  return p;
}


void
KPlotServer::slotAddText()
{
  //There should be a method in KPlotW for this --
  // we shouldn't need to access the view directly.
  KPSText *t = new KPSText (plotw->view(), "Text Label", .25, .1, ZTEXT);
  t->installDefaultRMBMenu();

  QPopupMenu *rmb = t->rmbMenu();
  rmb->insertItem ("Remove Text Label", ID_SPRITE_DELETE, 0);
  connect ( rmb, SIGNAL (activated (int)),
	    t, SLOT (slotPopupRedirect (int)) );
  connect ( t, SIGNAL (popupRedirect (KPSprite *, int)),
	    this, SLOT (slotSpriteDelete (KPSprite *, int)) );

  t->update();
}

void
KPlotServer::axesRMB(KPS2D *plot)
{
  QPopupMenu *rmb;
  bool upd=false;

  if (plot->xAxis())
    {
      if ( (rmb = plot->xAxis()->rmbMenu()) )
	{
	  rmb->insertItem ("Remove X axis ", ID_SPRITE_DELETE, 0);
	  connect ( rmb, SIGNAL (activated (int)),
		    plot->xAxis(), SLOT (slotPopupRedirect (int)) );
	  connect ( plot->xAxis(), SIGNAL (popupRedirect (KPSprite *, int)),
		    this, SLOT (slotSpriteDelete (KPSprite *, int)) );
	  
	  upd=true;
	}
    }

  if (plot->yAxis())
    {
      if ( (rmb = plot->yAxis()->rmbMenu()) )
	{
	  rmb->insertItem ("Remove Y axis ", ID_SPRITE_DELETE, 0);
	  connect ( rmb, SIGNAL (activated (int)),
		    plot->yAxis(), SLOT (slotPopupRedirect (int)) );
	  connect ( plot->yAxis(), SIGNAL (popupRedirect (KPSprite *, int)),
		    this, SLOT (slotSpriteDelete (KPSprite *, int)) );
	  
	  plot->yAxis()->update();
	  upd=true;
	}
    }
}

void
KPlotServer::slotSpriteDelete (KPSprite *s, int id)
{
  if (id==ID_SPRITE_DELETE)
    {
      todel=s;
      //Not the most elegant solution, but we can't delete
      // the sprite if we've been called from one of its methods.
      QTimer::singleShot ( 0, this, 
			   SLOT(slotSpriteDelete2 ()) );
    }
}

void
KPlotServer::slotSpriteDelete2 ()
{
  delete todel;
  plot->update();
}

void
KPlotServer::slotPointStyle(int idstyle)
{
  
  for (int i=0;i<npointstyles;i++)
    {
      if (stylelookup [i] == idstyle)
	{
	  ((KPS2DParametric *)plot)->setPointStyle (i);
	  break;
	}
    }

  plot->update();
}

void
KPlotServer::slotPointSize ()
{
  /*  KPS2DParametric *pp = (KPS2DParametric *)plot;

  QString ps;
  ps.setNum (pp->pointSize());

  KPLinedDlg dlg ("Point Size", "Size:", ps, plotw->view());
  if (dlg.exec())
    pp->setPointSize (dlg.text().toDouble());
  */

}

void
KPlotServer::slotFrame ()
{
  KPS2DParametric *pp = (KPS2DParametric *)plot;

  pp->setDrawFrame (!pp->drawFrame());
  p_rmb->setItemChecked (idframe, pp->drawFrame());
  pp->update();
}

void
KPlotServer::slotConnected ()
{
  KPS2DParametric *pp = (KPS2DParametric *)plot;

  pp->setConnected (!pp->connected());
  p_rmb->setItemChecked (idconnected, pp->connected());
  pp->update();
}

void
KPlotServer::slotPlotType(int id)
{
  //  if (plot)
    {
      KPMatrix *m = plot->matrix();
      delete plot;
      plot = createPlot (m, id);
      updateMenusEtc ();
      zoomout.clear();
    }
}


void
KPlotServer::updateMenusEtc (void)
{
  if (plot)
    {
      toolBar()->setItemEnabled (ID_PRINT, true);
      //      toolBar()->setItemEnabled (ID_RELOAD, true);
      toolBar()->setItemEnabled (ID_SELECT, true);
      toolBar()->setItemEnabled (ID_ZOOM, true);

      file->setItemEnabled (idprint, true);
      file->setItemEnabled (idprintdraft, true);
      //      file->setItemEnabled (idreload, true);

      edit->setItemEnabled (idptypes, true);
      //      edit->setItemEnabled (idcoords, true);
      

      if (plot->whatami()==RTTI_KPS2DParametric)
	{
	  //	  file->setItemEnabled (idaddmatrix, true);
	  //	  edit->setItemEnabled (idspline, true);
	  //	  edit->setItemEnabled (idautocoords, true);
	}
      else
	{
	  //	  file->setItemEnabled (idaddmatrix, false);
	  //	  edit->setItemEnabled (idspline, false);
	}

    }

}

void
KPlotServer::slotQuit ()
{
  close();
}

void
KPlotServer::slotToolBarToggle(int id)
{
  switch (id)
    {
    case ID_SELECT:
      if (toolBar()->isButtonOn(id))
	{
	  plotw->setCursor (arrowCursor);
	  tool=SelectTool;
	}
      break;
    case ID_ZOOM:
      if (toolBar()->isButtonOn(id))
	{
	  
	  plotw->setCursor (*zoomcursor);
	  tool=ZoomTool;
	  plotw->view()->zoomTool(true);
	  connect ( plotw->view(), SIGNAL(zoomedRect(QRect)),
		    this, SLOT (slotNewCoords(QRect)) );
	}
      else
	{
	  plotw->view()->zoomTool(false);
	  disconnect ( plotw->view(), SIGNAL(zoomedRect(QRect)),
		       this, SLOT (slotNewCoords(QRect)) );
	}
    }
}

void
KPlotServer::slotNewCoords(QRect r)
{
  double xmin=0, xmax=0, ymin=0, ymax=0;
  double xcross, ycross;

  //Store old coords
  KPCoords *cc = new KPCoords (*plot->coords());
  zoomout.append (cc);
  toolBar()->setItemEnabled (ID_ZOOMOUT, true);
  
  switch (plot->whatami())
    {
    case RTTI_KPS2DMap:
      xmin = floor (plot->coords()->xo_to_c (r.x()));
      xmax = ceil (plot->coords()->xo_to_c (r.x()+r.width()));
      ymin = ceil (plot->coords()->yo_to_c (r.y()+r.height()));
      ymax = floor (plot->coords()->yo_to_c (r.y()));
      break;
      //    case RTTI_KPS2DParametric:
    default:
      xmin = plot->coords()->xo_to_c (r.x());
      xmax = plot->coords()->xo_to_c (r.x()+r.width());
      ymin = plot->coords()->yo_to_c (r.y()+r.height());
      ymax = plot->coords()->yo_to_c (r.y());
      break;
    }

  xcross = plot->coords()->ccrossx();
  ycross = plot->coords()->ccrossy();
  
  if (xcross<xmin)
    xcross=xmin;
  else if(xcross>xmax)
    xcross = xmax;

  if (ycross<ymin)
    ycross=ymin;
  else if(ycross>ymax)
    ycross = ymax;

  printf ("NC %f %f %f %f  %f %f\n",
	  xmin, xmax, ymin, ymax,
	  xcross, ycross);
  KPCoords newcoords (xmin, xmax, ymin, ymax, xcross, ycross,
		      plot->coords()->oX0(), plot->coords()->oY0(),
		      plot->coords()->owidth(), plot->coords()->oheight());
  plot->setCoords (newcoords);
  zoomout.clear();
  plot->update();
}

void
KPlotServer::slotZoomOut()
{
  if (zoomout.count())
    {
      plot->setCoords (*zoomout.at(zoomout.count()-1));
      plot->update();
      zoomout.remove (zoomout.count()-1);
    }
  if (!zoomout.count())
    toolBar()->setItemEnabled (ID_ZOOMOUT, false);

}


void
KPlotServer::slotCoordsDialog ()
{
  coordsdlg = new QDialog (0);
  QGridLayout *layout = new QGridLayout (coordsdlg, 2, 4, 10);
 
  coordsp = new KPCoordsProp (plot->coords(), coordsdlg);
  layout->addMultiCellWidget (coordsp, 0, 0, 0, 3);
  QPushButton *b = new QPushButton ("&OK", coordsdlg);
  b->setMinimumSize (b->size());
  b->setMaximumSize (b->size());
  b->setDefault(true);
  connect ( b, SIGNAL (clicked()),
           this, SLOT (slotOKCoords()) );
  layout->addWidget (b, 1, 1);

  b = new QPushButton ("&Apply", coordsdlg);
  b->setMinimumSize (b->size());
  b->setMaximumSize (b->size());
  connect ( b, SIGNAL (clicked()),
            this, SLOT (slotApplyCoords()) );
  layout->addWidget (b, 1, 2);

  b = new QPushButton ("&Close", coordsdlg);
  b->setMinimumSize (b->size());
  b->setMaximumSize (b->size());
  connect ( b, SIGNAL (clicked()),
            this, SLOT (slotCloseCoords()) );
  layout->addWidget (b, 1, 3);

  coordsdlg->show();

  edit->setItemEnabled (idcoords, false);
}                        

void
KPlotServer::slotOKCoords()
{
  slotApplyCoords();
  slotCloseCoords();
}

void
KPlotServer::slotCloseCoords()
{
  delete coordsdlg;
  edit->setItemEnabled (idcoords, true);
}

KPlotServer::~KPlotServer()
{
  //  delete zoomcursor;
}



void
KPlotServer::slotApplyCoords()
{
  plot->setCoords (KPCoords (coordsp));
  plot->update();
}

void
KPlotServer::slotAutoCoords()
{
  //for parametric only now
  plot->setCoords ( ((KPS2DParametric *)plot)->autoCoords ());
  plot->update();
}

