/**
 * KPS2DMap
 * David Sweet
 *
 * A KPSprite which plots a color map of a matrix.  It optionally performs
 *  interpolation.
 *
 **/


#ifndef __KPS2DMAP_H__
#define __KPS2DMAP_H__

#include "kps2d.h"
#include "kpmatrix.h"
#include "kpcoords.h"
#include "kpcolortable.h"

#include "kpswhatami.h"

class KPS2DMapProp;

class KPS2DMap : public KPS2D
{
 Q_OBJECT

 public:
  enum Interpolation {None, BilinearVertex};
  /**
   * The coords now tell us what row and column ranges to
   *  plot.
   * The KPColorTable tells how matrix cell values map to colors.
   **/
  KPS2DMap (KPView *view, const KPCoords &, KPMatrix *, 
	    const KPColorTable &,
	    double x, double y, double w, double h, int z=0);

  KPS2DMap (KPView *view, const KPCoords &,
	    const KPColorTable &,
	    double x, double y, double w, double h, int z=0);

  /**
   * Find a good set of coordinates for this plot.
   **/
  virtual KPCoords autoCoords (const KPCoords *c = 0);

  /**
   * Installs the default RMB menu.
   **/
  void installDefaultRMBMenu (void);

  /**
   * Set the interpolation type.
   **/
  void setInterpolation (Interpolation i);

  /**
   * Get the interpolation type.
   **/
  Interpolation interpolation (void) {return interp;}

  /**
   * (reimplmented) See KPS2D.
   **/
  void setCoords (const KPCoords &);

  /**
   * Set to draw the grid separating the cells.
   **/
  void drawGrid (bool b=true) { bdrawgrid = b; }

  /**
   * Is the grid being drawn? 
   **/
  bool grid (void) { return bdrawgrid; }

  /**
   * Get the colorTable.
   **/
  KPColorTable colorTable (void) { return colortable; }

  /**
   * Set the colorTable.
   **/
  void setColorTable (KPColorTable _kpct) { colortable = _kpct; }

  /**
   * This KPSprite offers a properties widget.
   **/
  QWidget *propWidget (QWidget *parent, const char *name=0);

  /**
   * Set the grid's logical size.
   **/
  void setGridSize (double size);

  /**
   * Get the grid size.
   **/
  double gridSize (void) { return gridsize0; }

  /**
   * I am a KPS2DMap.
   **/
  virtual int whatami() const { return RTTI_KPS2DMap; }

 public slots:
  void slotProperties ();

 protected:
  bool bdrawgrid;
  double gridsize0;
  int gridsize;
  KPS2DMapProp *propwid;

  virtual void reDraw (QPainter *);
  void computeSizes(void);
  //  virtual void setRect();

  virtual void resizeEvent (QResizeEvent *);
  /**
   * Draw a single patch.  This performs the appropriate
   *  interpolation.
   **/
  virtual void drawPatch (QPainter *p,
			  int x, int y,
			  int width, int height, 
			  unsigned int row, unsigned int col,
			  unsigned int nrow);
  
  inline QColor getR (unsigned int i, unsigned int j, unsigned int ni);
  inline QColor getD (unsigned int i, unsigned int j, unsigned int ni);
  inline QColor getRD (unsigned int i, unsigned int j, unsigned int ni);
  inline QColor bilinInterp (double dx, double dx, 
			   QColor f00, QColor f01,
			   QColor f10, QColor f11);

 private:
  QPopupMenu *rmb;
  KPColorTable colortable;
  Interpolation interp;
  int rdi, rdni, rdj, rdpatches;
  double rdndx, rdndy;
};




#include <stdlib.h>

#include <qcheckbox.h>

#include <qbuttongroup.h>

class KLineEdit;

class KPS2DMapProp : public QWidget
{
  Q_OBJECT

  public:
   KPS2DMapProp (KPS2DMap *, QWidget *parent, const char *name, WFlags f=0);

  /**
   * Get the interpolation type.
   **/
  KPS2DMap::Interpolation interpolation (void) {return interp;}

  /**
   * Is the grid being drawn? 
   **/
  bool grid (void) { return cbgrid->isChecked(); }

  /**
   * Set interpolation type.
   **/
  void setInterpolation (KPS2DMap::Interpolation);

  /**
   * Get the colorTable.
   **/
  KPColorTable colorTable (void) { return kpct; }

  /**
   * Get the grid's logical size.
   **/
  double gridSize (void) { return atof (lesize->text())/100.; }



 protected:
 QCheckBox *cbgrid;
 QButtonGroup *bg;
 KPS2DMap::Interpolation interp;
 KPColorTable kpct;
 KLineEdit *lesize;
 QLabel *gridsize;

 protected slots:
    void slotInterpolation (int);
    void slotDisable();
};

#endif
