/***************************************************************************
                          seticontainer.h  -  version 0.3 - 2001.02.25
                             -------------------
    begin                : Fri Nov 17 2000
    copyright            : (C) 2000 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SETICONTAINER_H
#define SETICONTAINER_H

#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>

#include <sys/types.h>
#include <signal.h>
#include <unistd.h>

/** This structures stores all relevant data of a spike. */
struct SpikeScore {
                  /** The spike power. */
                  double  power;
                  /** The score of the spike. */
                  double  score;
                  /** An integer value representing the spike. */
                  int     bin;
                  /** The fft index value. */
                  int     fft_index;
                  /** The fft length (number of data points). */
                  int     fft_len;
                  /** The chirp rate at which this spike was found. See
                    * @ref StateFileData for a detailed explanation. */
                  double  chirprate;
                  /** The sky position (right ascension). */
                  double  ra;
                  /** The sky position (declination). */
                  double  dec;
                  /** The name of work unit in which the spike was found. */
                  QString wu_name;
                  };

/** This structures stores all relevant data of a gaussian. */
struct GaussianScore {
                     /** The gaussian power. */
                     double  power;
                     /** The score of the gaussian. This value is determined
                       * calculating (power/chisq). */
                     double  score;
                     /** An integer value representing the gaussian. */
                     int     bin;
                     /** Chi-squared value representing the quality of the
                       * gaussian fit. Lower values are better. */
                     double  chisq;
                     /** The fitted mean value. */
                     double  true_mean;
                     /** Sigma represents the width of the gaussian. */
                     double  sigma;
                     /** The fft index value. */
                     int     fft_index;
                     /** The fft length (number of data points). */
                     int     fft_len;
                     /** The chirp rate at which this gaussian was found. See
                       * @ref StateFileData for a detailed explanation. */
                     double  chirprate;
                     /** This array stores the gaussian shape. */
                     double  data[64];
                     /** The sky position (right ascension). */
                     double  ra;
                     /** The sky position (declination). */
                     double  dec;
                     /** The name of work unit in which the gaussian was
                       * found. */
                     QString wu_name;
                     };

/** This structures stores all relevant data of a pulse. Pulses are spikes
  * that are repeated many times.
  */
struct PulseScore {
                  /** The pulse power. */
                  double  power;
                  /** The score of the pulse. */
                  double  score;
                  /** The mean value of the data. */
                  double  mean;
                  /** The period of the pulse in seconds. */
                  double  period;
                  /** Details unknown. */
                  int     freq_bin;
                  /** Details unknown. */
                  double  time_bin;
                  /** The fft length (number of data points). */
                  int     fft_len;
                  /** The chirp rate at which this pulse was found. See
                    * @ref StateFileData for a detailed explanation. */
                  double  chirprate;
                  /** This array stores the pulse profile. */
                  unsigned short int data[512];
                  /** The sky position (right ascension). */
                  double  ra;
                  /** The sky position (declination). */
                  double  dec;
                  /** The name of work unit in which the pulse was found. */
                  QString wu_name;
                  };

/** This structures stores all relevant data of a triplet. Triplets are
  * three evenly spaced spikes. */
struct TripletScore {
                    /** The triplet power. */
                    double  power;
                    /** The score of the triplet. Currently the score is
                      * identical to the power. */
                    double  score;
                    /** The mean value of the data. */
                    double  mean;
                    /** The period of the triplet in seconds. */
                    double  period;
                    /** Details unknown. */
                    double  bperiod;
                    /** Marks the position of the first pulse of the triplet
                      * in the data set. */
                    int     tpotind0_0;
                    /** Always identical to @em tpotind0_0. */
                    int     tpotind0_1;
                    /** Marks the position of the second pulse of the triplet
                      * in the data set. */
                    int     tpotind1_0;
                    /** Always identical to @em tpotind1_0. */
                    int     tpotind1_1;
                    /** Marks the position of the third pulse of the triplet
                      * in the data set. */
                    int     tpotind2_0;
                    /** Always identical to @em tpotind2_0. */
                    int     tpotind2_1;
                    /** Details unknown. */
                    int     freq_bin;
                    /** Details unknown. */
                    double  time_bin;
                    /** Details unknown. */
                    double  scale;
                    /** The fft length (number of data points). */
                    int     fft_len;
                    /** The chirp rate at which this pulse was found. See
                      * @ref StateFileData for a detailed explanation. */
                    double  chirprate;
                    /** This array stores the triplet profile. */
                    unsigned short int data[512];
                    /** The sky position (right ascension). */
                    double  ra;
                    /** The sky position (declination). */
                    double  dec;
                    /** The name of work unit in which the triplet was found. */
                    QString wu_name;
                    };

/** The WUScore structure collects all spike, gaussian, pulse, and triplet
  * data of a work unit. */
struct WUScore  {
                /** A @ref SpikeScore structure with the spike data. */
                SpikeScore    spike;
                /** A @ref GaussianScore structure with the gaussian data. */
                GaussianScore gaussian;
                /** A @ref PulseScore structure with the pulse data. */
                PulseScore    pulse;
                /** A @ref TripletScore structure with the triplet data. */
                TripletScore  triplet;
                };

/** The StateFileData structure contains all information that can be found
  * in the `state.sah' file. */
struct StateFileData {
                     /** The number of processed Fast Fourier
                       * Transformations at the time of the last write to the
                       * state file. The FFT decomposes a signal taken over a
                       * certain period of time into its frequency components.
                       */
                     int     ncfft;
                     /** The chirp (or drift) rate in Hz/s. Assuming that the
                       * extraterrestrial signals are coming from a moving
                       * object (relative to the Earth), the signals are
                       * received with a certain frequency change. This is
                       * the well-known Doppler effect, which can also be
                       * observed e.g. for an approaching car horn. The amount
                       * of the frequency shift depends on the relative
                       * velocity of the extraterrestrial object with respect
                       * to Earth. Unfortunately we don't exactly know what the
                       * relative velocity is (since it varies from object to
                       * object), so the client needs to search a range of
                       * values (-50 Hz/s to +50 Hz/s for version 3.03).
                       */
                     double  cr;
                     /** The FFT length at the time of the last write to the
                       * state file. */
                     int     fl;
                     /** The CPU time in seconds. */
                     double  cpu;
                     /** The percentage done by the client. This is a value
                       * between 0 and 1. */
                     double  prog;
                     /** Used by the client to pick up the processing at the
                       * point where it stopped at shut-down. */
                     int     potfreq;
                     /** Used by the client to pick up the processing at the
                       * point where it stopped at shut-down. */
                     int     potactivity;
                     /** Represents the size of the file `outfile.sah', which
                       * contains signals to be returned to the server. */
                     int     outfilepos;
                     /** The maximum signals found in the work unit. */
                     WUScore max;
                     };

/** The StateFileData structure contains all user data. All the
  * information is  retrieved from the `user_info.sah' file. */
struct UserInfoData {
                    /** The user's unique ID number. */
                    int     id;
                    /** The user's key (?). */
                    int     key;
                    /** The user's email address. */
                    QString email_addr;
                    /** The name of the user. */
                    QString name;
                    /** The url address of the user's homepage (if given). */
                    QString url;
                    /** The user's home country. */
                    QString country;
                    /** The postal code of the user's home town. */
                    int     postal_code;
                    /** This flag decides whether the user's name is allowed
                      * to be shown on the SETI@home web pages. */
                    bool    show_name;
                    /** This flag decides whether the user's email address is
                      * allowed to be shown on the SETI@home web pages. */
                    bool    show_email;
                    /** Details unknown. */
                    int     venue;
                    /** This string contains the time and date the user
                      * registered with SETI@home. */
                    QString register_time;
                    /** This string contains the time and date the last
                      * work unit was sent to the user. This information is
                      * not updated any more. */
                    QString last_wu_time;
                    /** This string contains the time and date the user has
                      * sent back the last completed work unit to the
                      * server. */
                    QString last_result_time;
                    /** The number of work units sent to the user.  This
                      * information is not updated any more. */
                    int     nwus;
                    /** The number of results the user has sent back to the
                      * server. Equivalent to the number of completed work
                      * units. */
                    int     nresults;
                    /** The total CPU time (in seconds) which was needed to
                      * complete all the work units. */
                    double  total_cpu;
                    /** Details unknown. */
                    int     params_index;
                    };

/** This structure holds all work-unit-related data. All the information is
  * retrieved from the `work_unit.sah' file. */
struct WorkUnitData {
                    /** What has to be done with the data? Always "seti". */
                    QString     task;
                    /** This is an internal version number (not the one of
                      * the client!). */
                    int         version;
                    /** The name of the work unit. */
                    QString     name;
                    /** An internal value. Details unknown. */
                    QString     data_type;
                    /** An internal value. Details unknown. */
                    int         data_class;
                    /** An internal version number. Has to do with the
                      * splitter that splits the data stream from Arecibo
                      * into handy work units. */
                    QString     splitter_version;
                    /** This is the right ascension (RA) start value. Defines
                      * where the telescope has started to scan the sky for
                      * this work unit. */
                    double      start_ra;
                    /** This is the declination start value. Defines
                      * where the telescope has started to scan the sky for
                      * this work unit. */
                    double      start_dec;
                    /** This is the right ascension (RA) end value. Defines
                      * where the telescope has stopped to scan the sky for
                      * this work unit. */
                    double      end_ra;
                    /** This is the declination end value. Defines
                      * where the telescope has stopped to scan the sky for
                      * this work unit. */
                    double      end_dec;
                    /** The angle range (together with the above RA and
                      * declination values) defines the part of the sky that
                      * has been scanned for this work unit. A large angle
                      * range means that the telescope has swept very quickly
                      * between the start and end position. */
                    double      angle_range;
                    /** This string contains the date and time this work
                      * unit was recorded. */
                    QString     time_recorded;
                    /** The center frequency of the subband covered by this
                      * work unit. */
                    double      subband_center;
                    /** The base frequency of the subband covered by this
                      * work unit. */
                    double      subband_base;
                    /** The rate at which the data are sampled. */
                    double      subband_sample_rate;
                    /** Details unknown. */
                    int         fft_len;
                    /** Details unknown. */
                    int         ifft_len;
                    /** The SETI@home search covers a 2.5 MHz band. This
                      * frequency band is broken down into 256 subbands.
                      * This value tells you the subband number of this work
                      * unit. */
                    int         subband_number;
                    /** This string contains an identifier for the receiver.
                      * "ao1420" is the Arecibo Telescope. */
                    QString     receiver;
                    /** The number of samples in the work unit. */
                    int         nsamples;
                    /** Details unknown. */
                    QString     tape_version;
                    /** The number of positions that define the path of the
                      * telescope over the sky. */
                    int         num_positions;
                    /** A string list containing the coordinates that define
                      * the path of the telescope. */
                    QStringList coordinates;
                    };

/**
  * The SetiContainer class is a C++ class that allows convenient access to
  * all the information in the state files of a SETI@home client. It
  * periodically examines these files and updates its own data members
  * accordingly. Other programs can use this class to extract information
  * about the progress of the client, the found signals, or the work unit
  * being processed. Additional functions offer more advanced info (like
  * the amount of TeraFlops in a work unit, estimated time of completion,
  * etc). The SetiContainer class is based on the Qt toolkit, and extensively
  * uses Qt's Signal/Slot technique.
	*
	* Using the SetiContainer is very simple. Here is a short code snippet:
	*
	* <PRE>
	* Seticontainer* sec = new SetiContainer("/home/crunchking/setiathome/");
	*
	* double  progress = sec->progress();
	* QString wuname   = sec->wuName();
	* cout << wuname << ": " << progress << " %";
	* </PRE>
	*
	* This snippet creates a SetiContainer class which monitors the SETI@home
	* client in the given directory. The name of the work unit and the client's
	* progress are read and printed to stdout. By default, the SetiContainer
	* class refreshes its data every 30 seconds. To change the refresh interval
	* to 60 seconds, just do
	*
	* <PRE>
	* sec->setRefreshInterval(60);
	* </PRE>
	*
	* SetiContainer offers more advanced functions: it calculates the amount
	* of floating point operations (Flops) required to process the WU in
	* dependence on the angle range, calculates the estimated time to complete
	* the work unit, or converts the given time values from seconds into a
	* nicely formatted string.
	*
	* Another way of using this class is to inherit from it, and to extend it
	* with all the features you would like to have. Have a look at the
	* Ksetiwatch source code (http://ksetiwatch.sourceforge.net) to see how it
	* is done.
	*
	* All information from the client's output files is organized in a logical
	* manner using dedicated structures. For example, there are the
	* @ref WorkUnitData structure, the @ref UserInfoData structure, or the
	* @ref WUScore structure (which itself consists of four other structures:
	* @ref SpikeScore, @ref GaussianScore, @ref PulseScore, and
	* @ref TripletScore). By proper naming of these structures in your code,
	* the resulting code is very readable.
	* Here is an example; consider we have two WUScore structures, one holds
	* the record signals, the other contains some newly found signals:
	*
	* <PRE>
	* WUScore record;
	* WUScore new;
	* ...
	* ...
	* ...
	* if(new.pulse.score > record.pulse.score)
	*   {
	*   record.pulse = new.pulse;
	*   }
	* ...
	* </PRE>
	*
	* This piece of code checks if the score value of the new pulse is larger
	* than that of the record pulse, and -if this is true- copies the entire
	* pulse data to the record WUScore.
	*
	* SetiContainer makes extensive use of Qt's signal/slot mechanism. For
	* example, the progressIncreased() signal is emitted whenever the client
	* has updated its state file with new data, or the newPulse() signal is
	* emitted when a new high pulse signal was found. Connecting to a signal
	* looks like this:
	*
	* <PRE>
	* connect(this,     SIGNAL(progressIncreased()),
	*         mywidget, SLOT(slotUpdateTimeData()));
	* </PRE>
	*
	* Using this approach, the GUI elements are only updated when required,
	* resulting in very efficient code.
	*
	* In addition, the SetiContainer class provides a sophisticated way to
	* check the state of the SETI@home client. It can distinguish between
	* four different states: Running, Stopped, Finished, and Loading.
	* Whenever the state changes, the stateChanged() signal is
	* emitted.
	*
  * @author Gordon Machel <gmachel@users.sourceforge.net>
  */
class SetiContainer : public QObject
{
Q_OBJECT

public:
  /** Constructs a Seti container class.
    *
    * @param dir The working folder of the SETI@home client.
    * @param refresh The refresh interval. This class looks every @em refresh
    *                seconds if the data needs to be updated. Default is
    *                30 secs.
    */
  SetiContainer(const QString& dir, int refresh=30);
  /** The destructor. */
  ~SetiContainer();
  /** Sets the refresh interval to @em refresh seconds.
    *
    * @param refresh The refresh interval in seconds.
    */
  /** Returns the the SETI@home directory. */
  QString directory() const {return setiDirectory;}
  /** Sets the location's directory. */
  void setDirectory(const QString& d);
  /** Sets the refresh interval for this location.
    *
    * @param refresh The refresh interval in seconds.
    */
  void setRefreshInterval(int refresh);
  /** Returns the current refresh interval time in seconds. */
  int refreshInterval() {return(refreshIval/1000);}
  /** Returns the number of the currently processed Fast Fourier
    * Transformation.
    */
  int fftNumber() {return(stfData.ncfft);}
  /** Returns the length (in data points) of the currently processed Fast
    * Fourier Transformation.
    */
  int fftLength() {return(stfData.fl);}
  /** Returns the current chirp rate (or drift rate) in Hz/s. */
  double chirpRate() {return(stfData.cr);}
  /** Returns the cpu time (in seconds) spent on the current work unit. */
  double cpuTime() {return(stfData.cpu);}
  /** Returns the size of the file `outfile.sah' which collects data about
    * interesting signals to be returned to the SETI@home server. If this
    * value has increased the client has found a new interesting signal.
    */
  int outfileSize() {return(stfData.outfilepos);}
	/** This is a parameter used internally by the SETI@home client. */
	int potFrequency() {return(stfData.potfreq);}
	/** This is a parameter used internally by the SETI@home client. */
	int potActivity() {return(stfData.potactivity);}
	/** Returns the current percentage done by the client. */
	double progress() {return(100.0*stfData.prog);}
  /** Returns the client's progress rate (in % per hour) based on the
    * actual percentage done and the corresponding cpu time.
    */
  double progressRate();
  /** Returns the estimated time (in seconds) needed to finish the work
    * unit.
    */
  double remainingTime();
  /** Returns the estimated time needed to finish the work unit as a
    * formatted string.
    */
  QString remainingTimeAsString();
	/** Returns a WUScore structure containing the highest signals found in
	  * the current work unit so far. See @ref WUScore for details about
	  * the members of the WUScore structure.
	  */
	WUScore wuScore() {return(stfData.max);}
	/** Returns a SpikeScore structure containing the highest spike found in
	  * the current work unit so far. See @ref SpikeScore for details about
	  * the members of the SpikeScore structure.
	  */
	SpikeScore spikeScore() {return(stfData.max.spike);}
	/** Returns a GaussianScore structure containing the highest gaussian found
	  * in the current work unit so far. See @ref GaussianScore for details
	  * about the members of the GaussianScore structure.
	  */
	GaussianScore gaussianScore() {return(stfData.max.gaussian);}
	/** Returns a PulseScore structure containing the highest pulse found
	  * in the current work unit so far. See @ref PulseScore for details
	  * about the members of the PulseScore structure.
	  */
	PulseScore pulseScore() {return(stfData.max.pulse);}
	/** Returns a TripletScore structure containing the highest triplet found
	  * in the current work unit so far. See @ref TripletScore for details
	  * about the members of the TripletScore structure.
	  */
	TripletScore tripletScore() {return(stfData.max.triplet);}
	/** Returns the user's id number. */
	int userID() {return(uifData.id);}
	/** Returns the user's id key. */
	int key() {return(uifData.key);}
	/** Returns the user's email address. */
	QString userEmailAddress() {return(uifData.email_addr);}
	/** Returns the user's name. */
	QString userName() {return(uifData.name);}
	/** Returns the user's homepage address. */
	QString userHomepage() {return(uifData.url);}
	/** Returns the user's country. */
	QString userCountry() {return(uifData.country);}
	/** Returns the user's postal code. */
	int userPostalCode() {return(uifData.postal_code);}
	/** If true the user's name is allowed to be shown on the SETI@home
	  * web pages.
	  */
	bool showName() {return(uifData.show_name);}
	/** If true the user's email address is allowed to be shown on the
	  * SETI@home web pages.
	  */
	bool showEmail() {return(uifData.show_email);}
  /** Returns the venue parameter (whatever it is good for). */
  int venue() {return(uifData.venue);}
  /** Returns the time the user registered with SETI@home as a plain
    * string (as stored in user_info.sah).
    */
  QString registerTime() {return(uifData.register_time);}
  /** Returns the date and time the user registered with SETI@home in
    * a nicely formatted string. An empty string is returned when the
    * information could not be found.
    */
  QString registerTimeString();
  /** Returns the time the last work unit was sent to the user as plain
    * string (as stored in user_info.sah). Note: this information is
    * not updated any more.
    */
  QString lastWUTime() {return(uifData.last_wu_time);}
  /** Returns the time the user has sent his/her last work unit result to
    * the SETI@home server as a plain string (as stored in user_info.sah).
    */
  QString lastResultTime() {return(uifData.last_result_time);}
  /** Returns the date and time the last result was sent in a nicely
    * formatted string. An empty string is returned when the information
    * could not be found.
    */
  QString lastResultTimeString();
  /** Returns the number of work units sent to the user. Note: this
    * information is not updated any more.
    */
  int numberOfWUs() {return(uifData.nwus);}
  /** Returns the number of results the user has sent back to the SETI@home
    * server.
    */
  int numberOfResults() {return(uifData.nresults);}
	/** Returns the total cpu time (in seconds) the user has processed so far.
	  */
	double totalCPUTime() {return(uifData.total_cpu);}
  /** Returns the average CPU time (in seconds) required to process one
    * work unit.
    */
  double averageCPUTime();
	/** Returns a parameter index value. No idea what this is good for. */
	double paramsIndex() {return(uifData.params_index);}
	/** Returns the task identifier string. We are all in the "seti" business,
	  * aren't we?
	  */
	QString task() {return(wufData.task);}
	/** Returns an internal version number. This is not the version number of
	  * the SETI@home client.
	  */
	int wuVersion() {return(wufData.version);}
	/** Returns the name of the work unit. */
	QString wuName() {return(wufData.name);}
	/** Returns the data type of the attached work unit data. */
	QString wuDataType() {return(wufData.data_type);}
	/** Returns the data class of the attached work unit data. */
	int wuDataClass() {return(wufData.data_class);}
	/** Returns the version number of the splitter (software?). */
	QString splitterVersion() {return(wufData.splitter_version);}
	/** Returns the right ascension (RA) value where the telescope has
	  * started to scan the sky for this work unit.
	  */
	double startRA() {return(wufData.start_ra);}
	/** Returns the declination value where the telescope has
	  * started to scan the sky for this work unit.
	  */
	double startDec() {return(wufData.start_dec);}
	/** Returns the right ascension (RA) value where the telescope has
	  * stopped to scan the sky for this work unit.
	  */
	double endRA() {return(wufData.end_ra);}
	/** Returns the declination value where the telescope has
	  * stopped to scan the sky for this work unit.
	  */
	double endDec() {return(wufData.end_dec);}
	/** Returns the angle range which the telescope has scanned for this
	  * work unit.
	  */
	double angleRange() {return(wufData.angle_range);}
  /** Calculates the amount of floating point operations that need to be
    * processed to finish a work unit depending on the angle range. This
    * function is based on the work of Lawrence Kirby and Roelof Engelbrecht.
    *
    * @param ar      The angle range of the work unit.
    * @param version The version of the SETI@home client as an integer value
    *                (300 is "3.00", 303 is "3.03"). If omitted, this method
    *                uses the version number found in `version.sah'.
    *
    * @return The number of floating point operations in TeraFlops.
    */
  double teraFlops(double ar, int version=0);
  /** Returns the progress rate in MegaFlops/s. */
  double megaFlopsPerSecond();
	/** Returns the time the work unit was recorded (as stored in
	  * in the work_unit.sah file).
	  */
	QString timeRecorded() {return(wufData.time_recorded);}
  /** Returns the date and the time the work unit was recorded in a nicely
    * formatted string. An empty string is returned when the information
    * could not be found.
    */
  QString timeRecordedString();
	/** Returns the subband center frequency in GHz. */
	double centerFrequency() {return(wufData.subband_center/1.0e9);}
	/** Returns the subband base frequency in GHz. */
	double baseFrequency() {return(wufData.subband_base/1.0e9);}
	/** Returns the sample rate in Hz. */
	double sampleRate() {return(wufData.subband_sample_rate);}
	/** This parameter is used internally by the SETI@home client. */
	int wuFFTLength() {return(wufData.fft_len);};
	/** This parameter is used internally by the SETI@home client. */
	int wuIFFTLength() {return(wufData.ifft_len);}
	/** Returns the subband number. */
	int subbandNumber() {return(wufData.subband_number);}
	/** Returns the identifier string of the facility that recorded
	  * the work unit. "ao1420" defines the Arecibo Radio Telescope.
	  */
	QString receiver() {return(wufData.receiver);}
	/** Returns the number of samples in the work unit. */
	int numberOfSamples() {return(wufData.nsamples);}
	/** Returns the version number of the tape storing the work unit. */
	QString tapeVersion() {return(wufData.tape_version);}
	/** Returns the number of sky coordinates that define the path of
	  * the telescope over the sky when recording the work unit.
	  */
	int numberOfCoords() {return(wufData.num_positions);}
	/** Returns a string list with all the sky coordinates that define
	  * the path of the telescope over the sky when the work
	  * unit was recorded.
	  */
	QStringList coords() {return(wufData.coordinates);}
  /** Returns the version number of the SETI@home client. It is
    * returned as an integer formed of both the major and the minor
    * version number. Example: for client version 3.03, a 303 is
    * returned.
    */
  int clientVersion() {return(sahVersion);}
  /** Checks if the file represented by @em sah exists.
    *
    * @param sah Can be one of the following: @p SC_StateFile,
    *            @p SC_UserInfoFile, @p SC_WorkUnitFile, @p SC_ResultFile,
    *            @p SC_ResultHeaderFile, or @p SC_WtempFile.
    */
  bool exists(int sah);
  /** Checks if the SETI@home client is running by trying to signal
    * the process @em pid.
    *
    * @param pid The process id of the client. If omitted the pid will be
    * read from `pid.sah'.
    * @param checkCmdLine If true, this method checks whether the
    * process defined by @em pid is really that of a SETI@home client.
    *
    * @return true if client is running, false if it is stopped or on error.
    */
  bool isClientRunning(pid_t pid = 0, bool checkCmdLine = false);
  /** Returns the state of the location. */
  int clientState() const {return cltState;}
  /** Sets the state of the location. Setting it to -1, triggers a refresh
    * during the next update cycle. */
  void setClientState(int st) {cltState = st;}
  /** This function allows low-level access to the entries in the S@h
    * output files. It reads the entry @em e from the file @em sah. Use this
    * function only if you know what you are doing!
    *
    * @param sah The file from which the entry shall be read. You can choose
    *            from four values: SC_StateFile, SC_UserInfoFile,
    *            SC_WorkUnitFile, and SC_ResultFile.
    * @param e   The entry which shall be retrieved.
    */
  QString readEntry(int sah, const QString& e);
	/** Initializes all members of the WUScore structure (except for the
    * data arrays).
    *
    * @param score A pointer to the WUScore structure to be cleared.
	  */
	static void initWUScore(WUScore* score);
  /** Converts the time (in seconds) to a string.
    *
    * @param time The time in seconds.
    * @param hms  Defines the kind of presentation. If true, the time will be
    *             given in hours:minutes:seconds, if false in
    *             days:hours:minutes:seconds.
    */
	static QString convertTime(double time, bool hms);
  /**
   * Status definition values
   */
  enum SetiStatus {Stopped, Running, Finished, Loading};
  /** The identifiers for the different output files of the SETI@home
    * client.
    */
  enum sahID {SC_StateFile, SC_UserInfoFile,
              SC_WorkUnitFile, SC_ResultFile,
              SC_ResultHeaderFile, SC_WtempFile};
  	
public slots:
  /** Updates the data from all SETI@home files.
    *
    * @param sig If true, Qt's signals are allowed to be emitted when
    *            necessary.
    *
    * @return 0 in case of success, 1 on error.
    */
  int updateSahData(bool sig=true);

protected:
  int      refreshIval;
  int      sahVersion;
  QString  setiDirectory;
  QString  stfFileName;
  QString  wufFileName;
  QString  uifFileName;
  QString  rsfFileName;
  QString  rhdFileName;
  QString  wtpFileName;
  QStringList stateFile;
  QStringList workunitFile;
  QStringList userinfoFile;
  QString  stfTimeStamp;
  QString  wufTimeStamp;
  QString  uifTimeStamp;
  StateFileData stfData;
  UserInfoData  uifData;
  WorkUnitData  wufData;
  int cltState;
  int cltPid;

protected: // Protected methods
  /** Reads the state file and puts it into a string list.
    *
    * @return 0 in case of success, 1 on error.
    */
  int readStateFile();
  /** Reads the work unit file and puts it into a string list.
    *
    * @return 0 in case of success, 1 on error.
    */
  int readWorkUnitFile();
  /** Reads the user info file and puts it into a string list.
    *
    * @return 0 in case of success, 1 on error.
    */
  int readUserInfoFile();
  /** Reads a SETI@home sah file into a string list.
    *
    * @param file  The file name.
    * @param list  A pointer to a QStrList where the lines are stored.
    * @param endId The end-of-file identifier. If a line identical to
    *              this string is found, reading stops. When this parameter
    *              is omitted, the entire file is read.
    *
    * @return 0 in case of success, 1 on error.
    */
  int readSahFile(const QString& file, QStringList& list,
                  const QString& endId=0);
	/** Reads an entry given by @em e from file @em fn. Reads it directly from
	  * disk.
	  *
	  * @param fn The file name.
	  * @param e  The entry identifier string.
	  *
	  * @return The corresponding entry, or an empty string if the entry could
	  *         not be found.
	  */
	QString readEntry(const QString& fn, const QString& e);
	/** Reads an entry given by @em e from the string list @em list.
	  *
	  * @param list A pointer to the string list containing a sah file.
	  * @param e    The entry identifier string.
	  *
	  * @return The corresponding entry, or an empty string if the entry could
	  *         not be found.
	  */
	QString readEntry(QStringList& list, const QString& e);
  /** Initializes all state file data members. */
  void initStateFileData();
  /** Initializes all work unit data members. */
  void initWorkUnitData();
  /** Initializes all user info data members. */
  void initUserInfoData();
  /** Scans the state file and puts the data in the StateFileData
    * structure.
    *
    * @param sig If true, Qt's signals are allowed to be emitted when
    *            necessary.
    */
  void scanStateFile(bool sig=true);
  /** Scans the state file for spike data.
    *
    * @param sig If true, Qt's signals are allowed to be emitted when
    *            necessary.
    */
  void scanSpikeData(bool sig=true);
  /** Scans the state file for basic Gaussian data.
    *
    * @param sig If true, Qt's signals are allowed to be emitted when
    *            necessary.
    */
  void scanGaussianData(bool sig=true);
  /** Scans the state file for basic pulse data.
    *
    * @param sig If true, Qt's signals are allowed to be emitted when
    *            necessary.
    */
  void scanPulseData(bool sig=true);
  /** Scans the state file for basic triplet data.
    *
    * @param sig If true, Qt's signals are allowed to be emitted when
    *            necessary.
    */
  void scanTripletData(bool sig=true);
  /** Scans the state file for Gaussian graph data ('bg_pot xx'). */
  void scanGaussianGraphData();
  /** Scans the state file for pulse graph data ('bp_pot'). */
  bool scanPulseGraphData();
  /** Scans the state file for triplet graph data ('bt_pot'). */
  bool scanTripletGraphData();
  /** Scans the user info file and puts the data in the UserInfoData
      structure. */
  void scanUserInfoData();
  /** Scans the state file and puts the data in the stateFileData
      structure. */
  void scanWorkUnitData();
  /** Returns the version number of the SETI@home client. It is
    * returned as an integer formed of both the major and the minor
    * version number. Example: for client version 3.03, a 303 is
    * returned.
    */
  int readClientVersion();
  /** Fills all data members (all structs) with zeros and reads the client's
    * data from disk. */
  void initAllData();
  /** The event handler for timer events. This method is called when
    * the internal refresh timer triggers the event to synchronize
    * the internal data structures with the SETI@home files.
    */
  virtual void timerEvent(QTimerEvent* e);
  /** Checks the status of the client.
    *
    * @return One of the following values:
    * @em Stopped, @em Running, @em Finished, or @em Loading.
    */
  int checkClientState();
  /** Retrieves the pid of the S@h client.
    *
    * @return The pid, or -1 on error.
    */
  int getClientPid();
  /** This function is called every two minutes by a timer event. It checks
    * if the client has made any progress within the two minutes, and switches
    * the state to Stopped if nothing has happened. This is needed for passive
    * monitoring of remote clients on other computers, where no valid process
    * id could be obtained.
    *
    * @return The current state of the client.
    */
  int checkClientStatePassive();

private:
  int    refTimer;
  int    sttTimer1;
  int    sttTimer2;
  double oldProgress;

signals: // Signals
  /** This signal is emitted when the client has updated the state file
    * with new data.
    */
  void progressIncreased();
  /** This signal is emitted when the progress value has been found to
    * be decreased. This is a rather unusual event, and should only
    * happen when the client has started to reprocess the current
    * work unit from the beginning (probably after updating the S@h
    * client).
    */
  void progressDecreased();
	/** This signal is emitted when a new high spike has been found in the
	  * work unit.
	  *
	  * @param sps The SpikeScore structure with the new data.
	  */
	void newSpike(SpikeScore sps);
	/** This signal is emitted when a new high gaussian has been found in the
	  * work unit.
	  *
	  * @param gas The GaussianScore structure with the new data.
	  */
	void newGaussian(GaussianScore gas);
	/** This signal is emitted when a new high pulse has been found in the
	  * work unit.
	  *
	  * @param pus The PulseScore structure with the new data.
	  */
	void newPulse(PulseScore pus);
	/** This signal is emitted when a new high triplet has been found in the
	  * work unit.
	  *
	  * @param trs The TripletScore structure with the new data.
	  */
	void newTriplet(TripletScore trs);
	/** This signal is emitted when a new work unit has been loaded.
	  *
	  */
	void newWorkUnit(WorkUnitData wud);
	/** This signal is emitted when the state of the client has changed.
	  *
	  * @param state    The new state.
	  * @param loadsize If state is 'Loading', this parameter passes the current
	  *                 file size of the work unit being loaded. For all other
	  *                 states it is zero (not used).
	  */
	void stateChanged(int state, int loadsize=0);

};

#endif
