/* 
 *  Copyright (C) 1999-2001 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program may be distributed under the terms of the Q Public
 * License as defined by Trolltech AS of Norway and appearing in the
 * file LICENSE.QPL included in the packaging of this file.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */


#include <qdir.h>
#include <qtextstream.h>
#include <qmessagebox.h>
#include <qsplitter.h>
#include <qtimer.h>
#include <qregexp.h>
#include <qfiledialog.h>
#include <kapp.h>
#include <kaccel.h>
#include <kiconloader.h>
#include <kkeydialog.h>
#include <kfiledialog.h>
#include <kmenubar.h>
#include <klocale.h>
#include <kconfig.h>
#if QT_VERSION >= 200
#include <dcopclient.h>
#include <qcstring.h>
#include <qvaluelist.h>
#include <qsplitter.h>
#include <kglobal.h>
#include <ktoolbarbutton.h>
#else
#include <knewpanner.h>
#endif
#include <stdlib.h>
#include <unistd.h>
#include "logdlg.h"
#include "diffdlg.h"
#include "resolvedlg.h"
#include "annotatedlg.h"
#include "commitdlg.h"
#include "updatedlg.h"
#include "checkoutdlg.h"
#include "tagdlg.h"
#include "mergedlg.h"
#include "historydlg.h"
#include "updateview.h"
#include "protocolview.h"
#include "cvsprogressdlg.h"
#include "repositorydlg.h"
#include "settingsdlg.h"
#include "changelogdlg.h"
#include "misc.h"
#include "cervisiaapp.h"
#include "../config.h"

#include "toplevel.h"
#include "toplevel.moc"


#if QT_VERSION < 200
#define QCString QString
#else
#define Icon BarIcon
#endif


TopLevel::TopLevel(const char *name)
    : KTMainWindow( name )
{
    (void) statusBar();
#if QT_VERSION < 200
    statusBar()->insertItem("", 0);
#endif
#if 0
    timer = new QTimer(this);
    connect( timer, SIGNAL(timeout()),
	     statusBar(), SLOT(clear()) );
#endif

    KConfig *config = capp->config();
    config->setGroup("LookAndFeel");
    bool splitHorz = config->readBoolEntry("SplitHorizontally");

#if QT_VERSION >= 200
    splitter = new QSplitter(splitHorz? QSplitter::Horizontal : QSplitter::Vertical, this);
#else
    splitter = new KNewPanner(this, "panner",
                              splitHorz? KNewPanner::Vertical : KNewPanner::Horizontal);
#endif
 
    update = new UpdateView(splitter);
    update->setFocus();
    connect( update, SIGNAL(rightButtonPressed(QListViewItem*, const QPoint&, int)),
             this, SLOT(popupRequested()) );
    connect( update, SIGNAL(doubleClicked(QListViewItem*)),
             this, SLOT(slotOpen()) );

    protocol = new ProtocolView(splitter);

#if QT_VERSION < 200
    splitter->activate(update, protocol);
#endif
    setView(splitter, false);
    setupAccel();
    setupMenuBar();
    setupToolBar();
    updateAccel();
}


TopLevel::~TopLevel ()
{
    delete file;
    delete view;
    delete options;
    delete help;
}


void TopLevel::setupAccel()
{
    accel = new KAccel(this);

    accel->insertItem( i18n("Open File"),
		       "Open", CTRL+Key_O );
    accel->insertItem( i18n("Update Files"),
		       "Update", CTRL+Key_U );
    accel->insertItem( i18n("Status"),
                       "Status", Key_F5 );
    accel->insertItem( i18n("Commit file"),
		       "Commit", Key_NumberSign );
    accel->insertItem( i18n("Add File to Repository"),
		       "Add", Key_Plus );
    accel->insertItem( i18n("Remove File from Repository"),
		       "Remove", Key_Minus );
    accel->insertItem( i18n("Browse log file"),
		       "Log", CTRL+Key_L );
    accel->insertItem( i18n("View Annotate"),
		       "Annotate", CTRL+Key_A );
    accel->insertItem( i18n("Diff against repository"),
                       "Diff", CTRL+Key_D );
    
    accel->connectItem( "Open",     this, SLOT(slotOpen()) );
    accel->connectItem( "Update",   this, SLOT(slotUpdate()) );
    accel->connectItem( "Status",   this, SLOT(slotStatus()) );
    accel->connectItem( "Commit",   this, SLOT(slotCommit()) );
    accel->connectItem( "Add",      this, SLOT(slotAdd()) );
    accel->connectItem( "Remove",   this, SLOT(slotRemove()) );
    accel->connectItem( "Log",      this, SLOT(slotBrowseLog()) );
    accel->connectItem( "Annotate", this, SLOT(slotAnnotate()) );
    accel->connectItem( "Diff",     this, SLOT(slotDiff()) );
#if QT_VERSION >= 200
    accel->connectItem( KStdAccel::Quit, this, SLOT(slotExit()) );
#else
    accel->connectItem( KAccel::Quit, this, SLOT(slotExit()) );
#endif

    accel->readSettings();
}


void TopLevel::updateAccel()
{
    accel->changeMenuAccel( file, file_open,   "Open" );
    accel->changeMenuAccel( file, file_update, "Update" );
    accel->changeMenuAccel( file, file_status, "Status" );
    accel->changeMenuAccel( file, file_commit, "Commit" );
    accel->changeMenuAccel( file, file_add,    "Add" );
    accel->changeMenuAccel( file, file_remove, "Remove" );
    accel->changeMenuAccel( view, view_log,    "Log" );
    accel->changeMenuAccel( view, view_anno,   "Annotate" );
    accel->changeMenuAccel( view, view_diff,   "Diff" );
#if QT_VERSION >= 200
    accel->changeMenuAccel( file, file_exit,   KStdAccel::Quit );
#else
    accel->changeMenuAccel( file, file_exit,   KAccel::Quit );
#endif
}


void TopLevel::setupMenuBar()
{
    //
    // File menu
    //
    file = new QPopupMenu();
    
    file_sandbox = file->insertItem( i18n("O&pen Sandbox..."),
                                     this, SLOT(slotOpenSandbox()) );
    file_recent  = file->insertItem( i18n("O&pen recent..."),
                                     recent = new QPopupMenu() );
    file_chlog   = file->insertItem( i18n("&Insert ChangeLog entry..."),
                                     this, SLOT(slotChangeLog()) );
    
    file->insertSeparator();
    
    file_update = file->insertItem( i18n("&Update"),
				    this, SLOT(slotUpdate()) );
    file_status = file->insertItem( i18n("&Status"),
				    this, SLOT(slotStatus()) );

    file->insertSeparator();
    
    file_open    = file->insertItem( Icon("fileopen.xpm"), i18n("&Open..."),
                                     this, SLOT(slotOpen()) );
    file_resolv  = file->insertItem( i18n("Reso&lve..."),
                                     this, SLOT(slotResolve()) );

    file->insertSeparator();
    
    file_commit = file->insertItem( i18n("&Commit..."),
				    this, SLOT(slotCommit()) );
    file_add    = file->insertItem( i18n("&Add to Repository..."),
				    this, SLOT(slotAdd()) );
    file_addbin = file->insertItem( i18n("Add &binary..."),
                                    this, SLOT(slotAddBinary()) );

    file_remove = file->insertItem( i18n("&Remove from Repository..."),
				    this, SLOT(slotRemove()) );

    file->insertSeparator();

    file_exit   = file->insertItem( Icon("exit.xpm"), i18n("&Quit"),
				    this, SLOT(slotExit()) );

    //
    // View menu
    //
    view = new QPopupMenu();
    
    view_log    = view->insertItem( i18n("Browse &Log..."),
				    this, SLOT(slotBrowseLog()) );
#if 0
    view_mlog   = view->insertItem( i18n("Browse Multi-File Log..."),
				    this, SLOT(slotBrowseMultiLog()) );
#endif
    view_anno   = view->insertItem( i18n("&Annotate..."),
				    this, SLOT(slotAnnotate()) );
    view_diff   = view->insertItem( i18n("&Difference to Repository..."),
				    this, SLOT(slotDiff()) );
    view_change = view->insertItem( i18n("Last &Change..."),
                                    this, SLOT(slotLastChange()) );
    view_history= view->insertItem( i18n("&History..."),
                                    this, SLOT(slotHistory()) );
    

    view->insertSeparator();

    view_unfold = view->insertItem( i18n("&Unfold file tree"),
				    update, SLOT(unfoldTree()) );
#if QT_VERSION < 200
    view_desel  = view->insertItem( i18n("De&select All"),
				    update, SLOT(deselectAll()) );
#endif

    //
    // Advanced menu
    //
    advanced = new QPopupMenu();

    advanced_tag    = advanced->insertItem( i18n("&Tag/Branch..."),
                                            this, SLOT(slotCreateTag()) );
    advanced_deltag = advanced->insertItem( i18n("&Delete Tag..."),
                                            this, SLOT(slotDeleteTag()) );
    advanced_uptag  = advanced->insertItem( i18n("&Update to Tag/Date..."),
                                            this, SLOT(slotUpdateToTag()) );
    advanced_uphead = advanced->insertItem( i18n("Update to &HEAD"),
                                            this, SLOT(slotUpdateToHead()) );
    advanced_merge  = advanced->insertItem( i18n("&Merge..."),
                                            this, SLOT(slotMerge()) );

    advanced->insertSeparator();

    advanced_addwatch  = advanced->insertItem( i18n("&Add watch..."),
                                               this, SLOT(slotAddWatch()) );
    advanced_remwatch  = advanced->insertItem( i18n("&Remove watch..."),
                                               this, SLOT(slotRemoveWatch()) );
    advanced_watchers  = advanced->insertItem( i18n("&Show watchers"),
                                               this, SLOT(slotShowWatchers()) );
    
    advanced->insertSeparator();

    advanced_patch     = advanced->insertItem( i18n("Create &patch against repository"),
                                               this, SLOT(slotMakePatch()) );

    //
    // Repository menu
    //
    repo = new QPopupMenu();

    repo_checkout = repo->insertItem( i18n("&Checkout..."),
				      this, SLOT(slotCheckout()) );
    repo_import   = repo->insertItem( i18n("&Import..."),
                                      this, SLOT(slotImport()) );

    repo->insertSeparator();
    
    repo_repos    = repo->insertItem( i18n("&Repositories..."),
                                      this, SLOT(slotRepositories()) );

    //
    // Options menu
    //
    options = new QPopupMenu();

    options_create      = options->insertItem( i18n("Create &directories on update"),
                                               this, SLOT(slotCreateDirs()) );
    
    options_prune       = options->insertItem( i18n("&Prune empty directories on update"),
                                               this, SLOT(slotPruneDirs()) );
    
    options_updaterecur = options->insertItem( i18n("&Update recursively"),
                                               this, SLOT(slotUpdateRecursive()) );
    
    options_commitrecur = options->insertItem( i18n("&Commit and remove recursively"),
                                               this, SLOT(slotCommitRecursive()) );

    options_docvsedit   = options->insertItem( i18n("Do cvs &edit automatically when necessary"),
                                               this, SLOT(slotDoCVSEdit()) );

    options->insertSeparator();
    
    options_sett        = options->insertItem( i18n("&Settings..."),
                                               this, SLOT(slotSettings()) );
    
    options_keys        = options->insertItem( i18n("Configure &keybindings..."),
                                               this, SLOT(slotConfigureKeys()) );
    
    help = new QPopupMenu();

    help->insertItem( i18n("&Contents"),
		      this, SLOT(slotHelp()) );
    help->insertSeparator();
    help->insertItem( i18n("CVS &Info"),
		      this, SLOT(slotCVSInfo()) );
    help->insertItem( i18n("&About Cervisia"),
		      this, SLOT(aboutCervisia()) );

    KMenuBar *menubar = menuBar();
    menubar->insertItem( i18n("&File"), file );
    menubar->insertItem( i18n("&View"), view );
    menubar->insertItem( i18n("&Advanced"), advanced );
    menubar->insertItem( i18n("&Repository"), repo );
    menubar->insertItem( i18n("&Options"), options );
    menubar->insertSeparator();
    menubar->insertItem( i18n("&Help"), help );

    connect( file, SIGNAL(aboutToShow()), SLOT(updateMenu()) );
    connect( view, SIGNAL(aboutToShow()), SLOT(updateMenu()) );
    connect( advanced, SIGNAL(aboutToShow()), SLOT(updateMenu()) );
    connect( recent, SIGNAL(aboutToShow()), SLOT(fillRecentPopup()) );
    connect( recent, SIGNAL(activated(int)), SLOT(recentActivated(int)) );
}


void TopLevel::setupToolBar()
{
    KToolBar *toolbar = toolBar();

    toolbar->insertButton( Icon("up.xpm"), 101, SIGNAL(clicked()),
			   this, SLOT(slotCommit()),
			   true, i18n("Commit") );

    toolbar->insertButton( Icon("down.xpm"), 102, SIGNAL(clicked()),
			   this, SLOT(slotUpdate()),
			   true, i18n("Update") );

    toolbar->insertSeparator();
    
    toolbar->insertButton( Icon("fileopen.xpm"), 103, SIGNAL(clicked()),
			   this, SLOT(slotOpen()),
			   true, i18n("Open file") );
    
    toolbar->insertButton( Icon("help.xpm"), 104, SIGNAL(clicked()),
			   this, SLOT(slotHelp()),
			   true, i18n("Help") );

    toolbar->insertSeparator();
    
    toolbar->insertButton( Icon("stop.xpm"), 42, SIGNAL(clicked()),
                           protocol, SLOT(cancelJob()),
                           false, i18n("Stop") );
}

void TopLevel::popupRequested()
{
    QPopupMenu pop;
    int id_open   = pop.insertItem( i18n("&Open..."),
                                    this, SLOT(slotOpen()) );
    int id_resolv = pop.insertItem( i18n("Reso&lve..."),
                                    this, SLOT(slotResolve()) );
    int id_commit = pop.insertItem( i18n("&Commit..."),
                                    this, SLOT(slotCommit()) );
    int id_add    = pop.insertItem( i18n("&Add to Repository..."),
                                    this, SLOT(slotAdd()) );
    int id_remove = pop.insertItem( i18n("&Remove from Repository..."),
                                    this, SLOT(slotRemove()) );
    bool single = update->hasSingleSelection();
    pop.setItemEnabled(id_open, single);
    pop.setItemEnabled(id_resolv, single);
    bool nojob = !toolBar()->getButton(42)->isEnabled();
    pop.setItemEnabled(id_commit, nojob);
    pop.setItemEnabled(id_add, nojob);
    pop.setItemEnabled(id_remove, nojob);
    pop.exec(QCursor::pos());
}


void TopLevel::updateMenu()
{
    bool single = update->hasSingleSelection();
    toolBar()->getButton(103)->setEnabled(single);
    file->setItemEnabled(file_open, single);
    file->setItemEnabled(file_resolv, single);
    view->setItemEnabled(view_log, single);
    view->setItemEnabled(view_anno, single);
    view->setItemEnabled(view_diff, single);
    view->setItemEnabled(view_change, single);

    bool nojob = !toolBar()->getButton(42)->isEnabled();
    toolBar()->getButton(101)->setEnabled(nojob);
    toolBar()->getButton(102)->setEnabled(nojob);
    file->setItemEnabled(file_update, nojob);
    file->setItemEnabled(file_status, nojob);
    file->setItemEnabled(file_commit, nojob);
    file->setItemEnabled(file_add, nojob);
    file->setItemEnabled(file_addbin, nojob);
    file->setItemEnabled(file_remove, nojob);
    advanced->setItemEnabled(advanced_tag, nojob);
    advanced->setItemEnabled(advanced_deltag, nojob);
    advanced->setItemEnabled(advanced_uptag, nojob);
    advanced->setItemEnabled(advanced_uphead, nojob);
    advanced->setItemEnabled(advanced_merge, nojob);
    advanced->setItemEnabled(advanced_addwatch, nojob);
    advanced->setItemEnabled(advanced_remwatch, nojob);
    advanced->setItemEnabled(advanced_watchers, nojob);
    repo->setItemEnabled(repo_checkout, nojob);
    repo->setItemEnabled(repo_import, nojob);
}


void TopLevel::aboutCervisia()
{
    QString aboutstr;
    aboutstr.sprintf( i18n("Cervisia %s\n\n"
      	                   "Copyright (C) 1999-2001\n"), VERSION );
    aboutstr +=	i18n("Bernd Gehrmann <bernd@physik.hu-berlin.de>\n\n"
                     "This program may be distributed under the terms of the Q Public\n"
                     "License as defined by Trolltech AS of Norway and appearing in the\n"
                     "file LICENSE.QPL included in the packaging of this file.\n\n"
                     "This program is distributed in the hope that it will be useful,\n"
                     "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
                     "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.");
    QMessageBox::about(0, "Cervisia", aboutstr);
}


void TopLevel::slotOpenSandbox()
{
#if QT_VERSION >= 200
    QString dirname = KFileDialog::getExistingDirectory(QDir::homeDirPath(), this,
                      i18n("Open Sandbox")); 
#else
    QString dirname = KDirDialog::getDirectory(QDir::homeDirPath(), this);
#endif
    if (dirname.isEmpty())
        return;

    openSandbox(dirname);
}


void TopLevel::slotChangeLog()
{
    // Modal dialog
    ChangeLogDialog *l = new ChangeLogDialog();
    if (l->readFile(sandbox + "/ChangeLog"))
        {
            if (l->exec())
                changelogstr = l->message();
        }
    
    delete l;
}


void TopLevel::slotOpen()
{
    QString filename;
    update->getSingleSelection(&filename);
    if (!filename.isEmpty())
	{
            if (opt_doCVSEdit)
                {
                    if (!QFileInfo(filename).isWritable())
                        {
                            CvsProgressDialog l("Edit", this);
                            l.setCaption("CVS Edit");
                            QString cmdline = cvsClient() + " edit ";
                            cmdline += quote(filename);
                            if (!l.execCommand(sandbox, repository, cmdline, "edit") ||
                                !QFileInfo(filename).isWritable())
                                return;
                        }
                }
            
            KConfig *config = capp->config();
            config->setGroup("Communication");
#if 0
            bool usedcop = config->readBoolEntry("UseDCOP", false);
            QCString client = config->readEntry("DCOPClient", "KDevelop").latin1();
            QCString object = config->readEntry("DCOPObject", "Application").latin1();
            if (usedcop)
                {
                    capp->dcopClient()->attach();
                    capp->dcopClient()->registerAs("cervisia");
                    if (!capp->dcopClient()->isApplicationRegistered(client))
                        usedcop = false;
                }
            if (usedcop)
                {
                    QByteArray params;
                    QDataStream stream(params, IO_WriteOnly);
                    stream << QFileInfo(filename).absFilePath();
                    capp->dcopClient()->send(client, object,
                                             "openDocument(QString)", params);
                }
            else
                {
                    QString command = config->readEntry("Editor", "kwrite");
                    command += " ";
                    command += quote(filename);
                    command += "&";
                    ::system(command);
                }
#else
	    QString command = config->readEntry("Editor", "kwrite");
            command += " ";
	    command += quote(filename);
	    command += "&";
	    ::system(command);
#endif
	}
}


void TopLevel::slotResolve()
{
    QString filename;
    update->getSingleSelection(&filename);
    if (filename.isEmpty())
        return;

    // Non-modal dialog
    ResolveDialog *l = new ResolveDialog();
    if (l->parseFile(filename))
        l->show();
    else
        delete l;
}

void TopLevel::slotUpdate()
{
    updateOrStatus(false, "");
}


void TopLevel::slotStatus()
{
    updateOrStatus(true, "");
}


void TopLevel::slotUpdateToTag()
{
    UpdateDialog *l = new UpdateDialog(this);

    if (l->exec())
        {
            QString tagopt;
            if (l->byTag())
                {
                    tagopt = "-r ";
                    tagopt += l->tag();
                }
            else
                {
                    tagopt = "-D ";
                    tagopt += quote(l->date());
                }
            tagopt += " ";
            updateOrStatus(false, tagopt);
        }
    delete l;
}


void TopLevel::slotUpdateToHead()
{
    updateOrStatus(false, "-A ");
}


void TopLevel::slotMerge()
{
    MergeDialog *l = new MergeDialog(this);

    if (l->exec())
        {
            QString tagopt;
            if (l->byBranch())
                {
                    tagopt = "-j ";
                    tagopt += l->branch();
                }
            else
                {
                    tagopt = "-j ";
                    tagopt += l->tag1();
                    tagopt += " -j ";
                    tagopt += l->tag2();
                }
            tagopt += " ";
            updateOrStatus(false, tagopt);
        }
    delete l;
}


void TopLevel::slotCommit()
{
    commitOrAddOrRemove(CommitDialog::Commit);
}


void TopLevel::slotAdd()
{
    commitOrAddOrRemove(CommitDialog::Add);
}


void TopLevel::slotAddBinary()
{
    commitOrAddOrRemove(CommitDialog::AddBinary);
}


void TopLevel::slotRemove()
{
    commitOrAddOrRemove(CommitDialog::Remove);
}


void TopLevel::updateOrStatus(bool noact, QString extraopt)
{
    QStrList list = update->multipleSelection();
    if (list.isEmpty())
        return;
    
    update->prepareJob(opt_updateRecursive,
                       noact? UpdateView::UpdateNoAct : UpdateView::Update);

    QString cmdline;
    if (noact)
        cmdline = cvsClient() + " -n update ";
    else
        cmdline = cvsClient() + " update ";
    if (opt_updateRecursive)
        cmdline += "-R ";
    else
        cmdline += "-l ";
    if (opt_createDirs)
        cmdline += "-d ";
    if (opt_pruneDirs)
        cmdline += "-P ";
    cmdline += extraopt;
    cmdline += joinLine(list);
    cmdline += " 2>&1";
    
    if (protocol->startJob(sandbox, repository, cmdline))
        {
            showJobStart(cmdline);
            connect( protocol, SIGNAL(receivedLine(QString)), update, SLOT(processUpdateLine(QString)) );
            connect( protocol, SIGNAL(jobFinished(bool)), update, SLOT(finishJob(bool)) );
            connect( protocol, SIGNAL(jobFinished(bool)), this, SLOT(slotJobFinished(bool)) );
        }
}


void TopLevel::commitOrAddOrRemove(CommitDialog::ActionType action)
{
    QStrList list = update->multipleSelection();
    if (list.isEmpty())
        return;
    
    // modal dialog
    CommitDialog *l = new CommitDialog(action, this);
    if (action == CommitDialog::Commit)
        {
            l->setLogMessage(changelogstr);
            l->setLogHistory(sandbox, repository, recentCommits);
        }
    l->setFileList(list);
    
    if (l->exec())
        {
            QString cmdline;
	    switch (action)
		{
		case CommitDialog::Commit:
		    {
                        recentCommits.insert(0, l->logMessage());
                        while (recentCommits.count() > 30)
                            recentCommits.removeLast();
                        update->prepareJob(opt_commitRecursive, UpdateView::Commit);
			cmdline = cvsClient() + " commit ";
                        if (opt_commitRecursive)
                            cmdline += "-R ";
                        else
                            cmdline += "-l ";
			cmdline += "-m ";
			cmdline += quote(l->logMessage());
			cmdline += " ";
		    }
		    break;
		case CommitDialog::Add:
                    update->prepareJob(false, UpdateView::Add);
		    cmdline = cvsClient() + " add ";
		    break;
		case CommitDialog::AddBinary:
                    update->prepareJob(false, UpdateView::Add);
		    cmdline = cvsClient() + " add -kb ";
		    break;
		case CommitDialog::Remove:
                    update->prepareJob(opt_commitRecursive, UpdateView::Remove);
		    cmdline = cvsClient() + " remove -f ";
                    if (opt_commitRecursive)
                        cmdline += "-R ";
                    else
                        cmdline += "-l ";
		    break;
		}
	    cmdline += joinLine(list);
            cmdline += " 2>&1";
            
            if (protocol->startJob(sandbox, repository, cmdline))
                {
                    showJobStart(cmdline);
                    switch (action)
                        {
                        case CommitDialog::Commit:
                            connect( protocol, SIGNAL(jobFinished(bool)),
                                     update, SLOT(finishJob(bool)) );
                        case CommitDialog::Add:
                        case CommitDialog::AddBinary:
                        case CommitDialog::Remove:
                            connect( protocol, SIGNAL(jobFinished(bool)),
                                     update, SLOT(finishJob(bool)) );
                        }
                    connect( protocol, SIGNAL(jobFinished(bool)),
                             this, SLOT(slotJobFinished(bool)) );
                }
        }
    
    delete l;
}


void TopLevel::showJobStart(QString cmdline)
{
    toolBar()->setItemEnabled(42, true);
#if QT_VERSION >= 200
    statusBar()->message(cmdline);
#else
    statusBar()->changeItem(cmdline, 0);
#endif
    updateMenu();
}


void TopLevel::slotJobFinished(bool)
{
    toolBar()->setItemEnabled(42, false);
#if QT_VERSION >= 200
    statusBar()->message(i18n("Done"));
#else
    statusBar()->changeItem(i18n("Done"), 0);
#endif
    updateMenu();
}


void TopLevel::slotExit()
{
    (void) queryExit();
    kapp->quit();
}


void TopLevel::slotBrowseLog()
{
    QString filename;
    update->getSingleSelection(&filename);
    if (filename.isEmpty())
        return;

    // Non-modal dialog
    LogDialog *l = new LogDialog();
    if (l->parseCvsLog(sandbox, repository, filename))
        l->show();
    else
        delete l;
}


#if 0
void TopLevel::slotBrowseMultiLog()
{
    QStrList list = update->multipleSelection();
    if (!list.isEmpty())
	{
            // Non-modal dialog
	    MultiLogDialog *l = new MultiLogDialog();
	    if (l->parseCvsLog(".", list))
                l->show();
            else
                delete l;
	}
}
#endif


void TopLevel::slotAnnotate()
{
    QString filename;
    update->getSingleSelection(&filename);
    
    if (filename.isEmpty())
        return;

    // Non-modal dialog
    AnnotateDialog *l = new AnnotateDialog();
    if (l->parseCvsAnnotate(sandbox, repository, filename, ""))
        l->show();
    else
        delete l;
}


void TopLevel::slotDiff()
{
    QString filename;
    update->getSingleSelection(&filename);
    
    if (filename.isEmpty())
        return;

    // Non-modal dialog
    DiffDialog *l = new DiffDialog();
    if (l->parseCvsDiff(sandbox, repository, filename, "", ""))
        l->show();
    else
        delete l;
}


void TopLevel::slotAddWatch()
{
    addOrRemoveWatch(WatchDialog::Add);
}


void TopLevel::slotRemoveWatch()
{
    addOrRemoveWatch(WatchDialog::Remove);
}


void TopLevel::addOrRemoveWatch(WatchDialog::ActionType action)
{
    QStrList list = update->multipleSelection();
    if (list.isEmpty())
        return;

    WatchDialog *l = new WatchDialog(action, this);

    if (l->exec() && l->events() != WatchDialog::None)
        {
            QString cmdline = cvsClient() + " watch ";
            if (action == WatchDialog::Add)
                cmdline += "add ";
            else
                cmdline += "remove ";
            
            WatchDialog::Events events = l->events();
            if (events != WatchDialog::All)
                {
                    if (events & WatchDialog::Commits)
                        cmdline += "-a commit ";
                    if (events & WatchDialog::Edits)
                        cmdline += "-a edit ";
                    if (events & WatchDialog::Unedits)
                        cmdline += "-a unedit ";
                }
            
            cmdline += joinLine(list);
                    
            if (protocol->startJob(sandbox, repository, cmdline))
                {
                    showJobStart(cmdline);
                    connect( protocol, SIGNAL(jobFinished(bool)), this, SLOT(slotJobFinished(bool)) );
                }
        }
        
    delete l;
}


void TopLevel::slotShowWatchers()
{
    QStrList list = update->multipleSelection();
    if (list.isEmpty())
        return;

    QString cmdline = cvsClient() + " watchers ";
    cmdline += joinLine(list);
                    
    if (protocol->startJob(sandbox, repository, cmdline))
        {
            showJobStart(cmdline);
            connect( protocol, SIGNAL(jobFinished(bool)), this, SLOT(slotJobFinished(bool)) );
        }
}


void TopLevel::slotMakePatch()
{
    CvsProgressDialog l("Diff", this);
    l.setCaption("CVS Diff");
    QString cmdline = "cvs diff -uR";
    if (!l.execCommand(sandbox, repository, cmdline, "diff"))
        return;

    QString filename = KFileDialog::getSaveFileName();
    if (filename.isEmpty())
        return;

    QFile f(filename);
    if (!f.open(IO_WriteOnly))
	{
	    QMessageBox::information(this, "Cervisia",
				     i18n("Could not open file for writing."));
	    return;
	}
    QTextStream t(&f);
    QCString line;
    while (l.getOneLine(&line))
        t << line << '\n';

    f.close();
}


void TopLevel::slotImport()
{
    importOrCheckout(CheckoutDialog::Import);
}


void TopLevel::slotCheckout()
{
    importOrCheckout(CheckoutDialog::Checkout);
}


void TopLevel::importOrCheckout(CheckoutDialog::ActionType action)
{
    CheckoutDialog *l = new CheckoutDialog(action, this);

    if (l->exec())
        {
            QString cmdline = "cd ";
            cmdline += l->workingDirectory();
            cmdline += " && " + cvsClient() + " -d ";
            cmdline += l->repository();
            if (action == CheckoutDialog::Checkout)
                {
                    cmdline += " checkout -P ";
                    cmdline += l->module();
                }
            else
                {
                    cmdline += " import";
                    if (l->importBinary())
                        cmdline += " -kb";
                    QString ignore = l->ignoreFiles().stripWhiteSpace();
                    if (!ignore.isEmpty())
                        {
                            cmdline += " -I ";
                            cmdline += quote(ignore);
                        }
                    cmdline += " -m ";
                    cmdline += (QString("\"") + "" + "\" "); // log message?
                    cmdline += l->module();
                    cmdline += " ";
                    cmdline += l->vendorTag();
                    cmdline += " ";
                    cmdline += l->releaseTag();
                }
            
            if (protocol->startJob(sandbox, repository, cmdline))
                {
                    showJobStart(cmdline);
                    connect( protocol, SIGNAL(jobFinished(bool)), this, SLOT(slotJobFinished(bool)) );
                }
        }
        
    delete l;
}


void TopLevel::slotRepositories()
{
    RepositoryDialog *l = new RepositoryDialog(this);
    l->show();
}


void TopLevel::slotCreateTag()
{
    createOrDeleteTag(TagDialog::Create);
}


void TopLevel::slotDeleteTag()
{
    createOrDeleteTag(TagDialog::Delete);
}


void TopLevel::createOrDeleteTag(TagDialog::ActionType action)
{
    QStrList list = update->multipleSelection();
    if (list.isEmpty())
        return;

    TagDialog *l = new TagDialog(action, this);

    if (l->exec())
        {
            QString cmdline = cvsClient() + " tag ";
            if (action == TagDialog::Delete)
                cmdline += "-d ";
            else if (l->branchTag())
                cmdline += "-b ";
            cmdline += l->tag();
            cmdline += " ";
            cmdline += joinLine(list);
                    
            if (protocol->startJob(sandbox, repository, cmdline))
                {
                    showJobStart(cmdline);
                    connect( protocol, SIGNAL(jobFinished(bool)), this, SLOT(slotJobFinished(bool)) );
                }
        }
        
    delete l;
}



void TopLevel::slotLastChange()
{
    QString filename, revA, revB;
    update->getSingleSelection(&filename, &revA);
    if (filename.isEmpty())
        return;
    
    int pos, lastnumber;
    bool ok;
    if ( (pos = revA.findRev('.')) == -1
         || (lastnumber=revA.right(revA.length()-pos-1).toUInt(&ok), !ok) )
        {
            QMessageBox::information(this, "Cervisia", i18n("The revision looks invalid."));
            return;
        }
    if (lastnumber == 0)
        {
            QMessageBox::information(this, "Cervisia", i18n("The is the first revision on the branch."));
            return;
        }
    revB = revA.left(pos+1) + QString().setNum(lastnumber-1);

    // Non-modal dialog
    DiffDialog *l = new DiffDialog();
    if (l->parseCvsDiff(sandbox, repository, filename, revB, revA))
        l->show();
    else
        delete l;
}


void TopLevel::slotHistory()
{
    // Non-modal dialog
    HistoryDialog *l = new HistoryDialog();
    if (l->parseHistory(sandbox, repository))
        l->show();
    else
        delete l;
}


void TopLevel::slotCreateDirs()
{
    opt_createDirs = !opt_createDirs;
    options->setItemChecked(options_create, opt_createDirs);
}


void TopLevel::slotPruneDirs()
{
    opt_pruneDirs = !opt_pruneDirs;
    options->setItemChecked(options_prune, opt_pruneDirs);
}


void TopLevel::slotUpdateRecursive()
{
    opt_updateRecursive = !opt_updateRecursive;
    options->setItemChecked(options_updaterecur, opt_updateRecursive);
}


void TopLevel::slotCommitRecursive()
{
    opt_commitRecursive = !opt_commitRecursive;
    options->setItemChecked(options_commitrecur, opt_commitRecursive);
}


void TopLevel::slotDoCVSEdit()
{
    opt_doCVSEdit = !opt_doCVSEdit;
    options->setItemChecked(options_docvsedit, opt_doCVSEdit);
}


void TopLevel::slotSettings()
{
    SettingsDialog *l = new SettingsDialog(this);
    l->show();

#if QT_VERSION >= 200
    KConfig *config = capp->config();
    config->setGroup("LookAndFeel");
    bool splitHorz = config->readBoolEntry("SplitHorizontally");
    splitter->setOrientation(splitHorz? QSplitter::Horizontal : QSplitter::Vertical);
#endif
}


void TopLevel::slotConfigureKeys()
{
    if (KKeyDialog::configureKeys(accel))
	updateAccel();
}


void TopLevel::slotHelp()
{
    statusBar()->message(i18n("Invoking help on Cervisia"), 2000);
    capp->invokeHTMLHelp("", "");
}


void TopLevel::slotCVSInfo()
{
    statusBar()->message(i18n("Invoking help on CVS"), 2000);
#if QT_VERSION >= 200
    KApplication::startServiceByDesktopName("khelpcenter", QString("info:/cvs/Top"));
#else
    if ( fork() == 0 )	
	{
	    execl("/bin/sh", "/bin/sh", "-c", "kdehelp 'info:(cvs)'", 0);
	}
#endif
}


void TopLevel::openSandbox(QString dirname)
{
    QFileInfo fi1(dirname);
    QString sandboxpath = fi1.absFilePath();
    
    // Nice side-effect of this: Deleted sandboxes are removed
    recentFiles.remove(sandboxpath);

    QFileInfo fi2(sandboxpath + "/CVS");
    if (!fi2.exists() || !fi2.isDir())
        {
            QMessageBox::information(this, "Cervisia", i18n("This is not a CVS directory."));
            return;
        }

    recentFiles.insert(0, sandboxpath);
    while (recentFiles.count() > 10)
        recentFiles.removeLast();

    changelogstr = "";
    sandbox = sandboxpath;
    repository = "";

    QFile f(sandbox + "/CVS/Root");
    if (f.open(IO_ReadOnly)) 
	{ 
	    QTextStream t(&f); 
	    repository = t.readLine();
	} 
    

#if QT_VERSION >= 200
    setCaption(sandbox + "(" + repository + ")");
#else
    setCaption("Cervisia - " + sandbox + "(" + repository + ")");
#endif
    QDir::setCurrent(sandbox);
    update->openDirectory(sandbox);

    KConfig *config = capp->config();
    config->setGroup("General");
    bool dostatus = config->readBoolEntry(repository.contains(":")?
                                     "StatusForRemoteRepos" : "StatusForLocalRepos", false);
    if (dostatus)
        {
            update->setSelected(update->firstChild(), true);
            updateOrStatus(true, "");
        }
}


void TopLevel::fillRecentPopup()
{
    recent->clear();
    QStrListIterator it(recentFiles);
    int n = 0;
    for (; it.current(); ++it)
        recent->insertItem(it.current(), n++);
}


void TopLevel::recentActivated(int n)
{
    openSandbox(recentFiles.at(n));
}


#if 0
void TopLevel::parseStatus(QString pathname, QStrList list)
{
    char buf[512];
    QString command;
    QString dirpath;
    QString name, statusstr, version;
    enum { Begin, File, FileSep, WorkRev, FinalSep } state;

    QString line = joinLine(list);
    
    command = "cd ";
    command += pathname;
    command += " && " + cvsClient() + " status -l ";
    command += line;
    //    command += " 2>&1";

    FILE *f = popen(command, "r");
    if (!f)
	return;

    state = Begin;
    while (fgets(buf, sizeof buf, f))
	{
	    QCString line = buf;
	    chomp(&line);
            //	    DEBUGOUT( "Line: " << line );
	    switch (state)
		{
		case Begin:
		    if (line.left(22) == "cvs status: Examining ")
			dirpath = line.right(line.length()-22);
		    state = File;
                    //		    DEBUGOUT( "state = file" );
		    break;
		case File:
		    if (line.length() > 32 &&
			line.left(6) == "File: " &&
			line.mid(24, 8) == "Status: ")
			{
			    name = line.mid(6, 18).stripWhiteSpace();
			    if (dirpath != ".")
				name.prepend("/").prepend(dirpath);
			    statusstr = line.right(line.length()-32)
				.stripWhiteSpace();
			    state = FileSep;
                            //			    DEBUGOUT( "state = FileSep" );
			}
		    break;
		case FileSep:
		    if (!line.isEmpty()) ; // Error
		    state = WorkRev;
                    //		    DEBUGOUT( "state = WorkRev" );
		    break;
		case WorkRev:
		    if (line.left(21) == "   Working revision:\t")
			{
			    int pos;
			    version = line.right(line.length()-21);
			    if ( (pos = version.find(" ")) != -1 )
				version.truncate(pos);
			    state = FinalSep;
                            //			    DEBUGOUT( "state = FinalSep" );
			}
		    break;
		case FinalSep:
		    if (line == "")
			{
                            //			    DEBUGOUT( "Adding: " << name <<
                            //				      "Status: " << statusstr << "Version: " << version );
			    UpdateView::Status status = UpdateView::Unknown;
			    if (statusstr == "Up-to-date")
				status = UpdateView::UpToDate;
			    else if (statusstr == "Locally Modified")
				status = UpdateView::LocallyModified;
			    else if (statusstr == "Locally Added")
				status = UpdateView::LocallyAdded;
			    else if (statusstr == "Locally Removed")
				status = UpdateView::LocallyRemoved;
			    else if (statusstr == "Needs Checkout")
				status = UpdateView::NeedsUpdate;
			    else if (statusstr == "Needs Patch")
				status = UpdateView::NeedsPatch;
			    else if (statusstr == "Needs Merge")
				status = UpdateView::NeedsMerge;
			    else if (statusstr == "File had conflicts on merge")
				status = UpdateView::Conflict;
                            //			    update->addEntry(status, name /*, version*/);
			    state = Begin;
                            //			    DEBUGOUT( "state = Begin" );
			}
		}
	}
    pclose(f);
}
#endif


bool TopLevel::queryExit()
{
    KConfig *config = capp->config();
    
    config->setGroup("Recent sandboxes");
    for (int i = 1; recentFiles.count(); ++i)
        {
            QString str;
            str.setNum(i);
            str.prepend("Sandbox");
            config->writeEntry(str, recentFiles.at(0));
            recentFiles.removeFirst();
        }
    
    config->setGroup("Main window");
    config->writeEntry("Customized", true);
    config->writeEntry("Size", size());
    
    config->setGroup("Diff dialog");
    DiffDialog::saveOptions(config);
    config->setGroup("Log dialog");
    LogDialog::saveOptions(config);
    config->setGroup("Resolve dialog");
    ResolveDialog::saveOptions(config);
    config->setGroup("Commit dialog");
    CommitDialog::saveOptions(config);
    config->setGroup("ChangeLog dialog");
    ChangeLogDialog::saveOptions(config);
    config->setGroup("Annotate dialog");
    AnnotateDialog::saveOptions(config);
    config->setGroup("Checkout dialog");
    CheckoutDialog::saveOptions(config);
    config->setGroup("History dialog");
    HistoryDialog::saveOptions(config);
    config->setGroup("Repository dialog");
    RepositoryDialog::saveOptions(config);
    config->setGroup("AddRepository dialog");
    AddRepositoryDialog::saveOptions(config);

    config->setGroup("Session");
    saveProperties(config);

    config->sync();
    return true;
}


void TopLevel::restorePseudo(QString dirname)
{
    KConfig *config = capp->config();

    config->setGroup("Recent sandboxes");
    for (int i = 1; ; ++i)
        {
            QString str;
            str.setNum(i);
            str.prepend("Sandbox");
            QString entry = config->readEntry(str);
            if (entry.isEmpty())
                break;
            recentFiles.append(entry);
        }
    
    config->setGroup("Main window");
    if (config->readEntry("Customized"))
        resize(config->readSizeEntry("Size"));

    config->setGroup("Diff dialog");
    DiffDialog::loadOptions(config);
    config->setGroup("Log dialog");
    LogDialog::loadOptions(config);
    config->setGroup("Resolve dialog");
    ResolveDialog::loadOptions(config);
    config->setGroup("Commmit dialog");
    CommitDialog::loadOptions(config);
    config->setGroup("ChangeLog dialog");
    ChangeLogDialog::loadOptions(config);
    config->setGroup("Annotate dialog");
    AnnotateDialog::loadOptions(config);
    config->setGroup("Checkout dialog");
    CheckoutDialog::loadOptions(config);
    config->setGroup("History dialog");
    HistoryDialog::loadOptions(config);
    config->setGroup("Repository dialog");
    RepositoryDialog::loadOptions(config);
    config->setGroup("AddRepository dialog");
    AddRepositoryDialog::loadOptions(config);

    config->setGroup("Session");
    if (!dirname.isEmpty())
        config->writeEntry("Current Directory", dirname);
    readProperties(config);
}

 
void TopLevel::readProperties(KConfig *config)
{
    // Unfortunately, the KConfig systems sucks and we have to live
    // with all entries in one group for session management.
    
    opt_createDirs = config->readBoolEntry("Create Dirs", true);
    options->setItemChecked(options_create, opt_createDirs);
    
    opt_pruneDirs = config->readBoolEntry("Prune Dirs", true);
    options->setItemChecked(options_prune, opt_pruneDirs);
    
    opt_updateRecursive = config->readBoolEntry("Update Recursive", false);
    options->setItemChecked(options_updaterecur, opt_updateRecursive);

    opt_commitRecursive = config->readBoolEntry("Commit Recursive", false);
    options->setItemChecked(options_commitrecur, opt_commitRecursive);

    opt_doCVSEdit = config->readBoolEntry("Do cvs edit", false);
    options->setItemChecked(options_docvsedit, opt_doCVSEdit);

#if QT_VERSION >= 200
    int splitterpos1 = config->readNumEntry("Splitter Pos 1", 0);
    int splitterpos2 = config->readNumEntry("Splitter Pos 2", 0);
    if (splitterpos1)
        {
            QValueList<int> sizes;
            sizes << splitterpos1;
            sizes << splitterpos2;
            splitter->setSizes(sizes);
        }
#else
    int separatorpos = config->readNumEntry("SeparatorPos", 0);
    if (separatorpos)
        splitter->setSeparatorPos(separatorpos);
#endif

    QString currentDir = config->readEntry("Current Directory");
    if (!currentDir.isEmpty())
        openSandbox(currentDir);
}


void TopLevel::saveProperties(KConfig *config)
{
    config->writeEntry("Create Dirs", opt_createDirs);
    config->writeEntry("Prune Dirs", opt_pruneDirs);
    config->writeEntry("Update Recursive", opt_updateRecursive);
    config->writeEntry("Commit Recursive", opt_commitRecursive);
    config->writeEntry("Do cvs edit", opt_doCVSEdit);
#if QT_VERSION >= 200
    QValueList<int> sizes = splitter->sizes();
    config->writeEntry("Splitter Pos 1", sizes[0]);
    config->writeEntry("Splitter Pos 2", sizes[1]);
#else
    config->writeEntry("SeparatorPos", splitter->separatorPos());
#endif
    config->writeEntry("Current Directory", sandbox);
}


// Local Variables:
// c-basic-offset: 4
// End:
