/*
 *  Copyright (C) 1999-2001 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program may be distributed under the terms of the Q Public
 * License as defined by Trolltech AS of Norway and appearing in the
 * file LICENSE.QPL included in the packaging of this file.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */


#include <qpainter.h>
#include <qscrollbar.h>
#include <qpixmap.h>
#include <kconfig.h>
#include <kdebug.h>

#include "diffview.moc"

#include "diffview.h"
#include "diffmodel.h"
#include "difference.h"
#include "generalsettings.h"
#include "kdiffview.h"

class DiffViewItem
{
public:
	QString            line;
	Difference::Type   type;
	bool               inverted;
	int                lineno;
	int                id;
};

int DiffViewItemList::compareItems( QCollection::Item item1, QCollection::Item item2 )
{
	return (static_cast<DiffViewItem*>(item1)->lineno
	     == static_cast<DiffViewItem*>(item2)->lineno)? 0 : 1;
}

static const int BORDER = 7;

DiffView::DiffView( GeneralSettings* settings, bool withlinenos,
                    QWidget *parent, const char *name )
	: QTableView( parent, name, WNorthWestGravity | WRepaintNoErase )
{
	m_settings = settings;
	connect( m_settings, SIGNAL( settingsChanged() ), this, SLOT( repaint() ) );

	setNumRows(0);
	setNumCols( 1 + (withlinenos?1:0) );
	setBackgroundMode( PaletteBase );
	setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding ) );
	setWFlags( WResizeNoErase );

	QFontMetrics fm(font());
	setCellHeight(fm.lineSpacing());
	setCellWidth(0);
	textwidth = 0;

	m_tabWidth = 8;

	items.setAutoDelete(true);
	linenos = withlinenos;
}

void DiffView::setFont(const QFont &font)
{
	QTableView::setFont(font);
	QFontMetrics fm(font);
	setCellHeight(fm.lineSpacing());
}

void DiffView::setTabWidth(uint tabWidth)
{
	m_tabWidth = tabWidth;
}

int DiffView::minScrollId()
{
	int row = ( height() / 2 ) / cellHeight(0);
	if ( row >= 0 && row < (int)items.count() ) {
		DiffViewItem* item = items.at( row );
		if( item ) return ( item->id );
	}
	return 0;
}

int DiffView::maxScrollId()
{
	int row = numRows() - ( height() / 2 ) / cellHeight(0);
	if ( row >= 0 && row < (int)items.count() ) {
		DiffViewItem* item = items.at( row );
		if( item ) return ( item->id );
	}
	return 0;
}

int DiffView::pageStep()
{
	return ( viewHeight() - cellHeight(0) ) / cellHeight(0);
}

int DiffView::getScrollId()
{
	int row = findRow( height() / 2 );
	if ( row >= 0 && row < (int)items.count() ) {
		DiffViewItem* item = items.at( row );
		if( item ) return ( item->id );
	}
	return 0;
}

void DiffView::scrollToId( int id )
{
	DiffViewItem* item;
	for( item = items.first(); item != 0; item = items.next() ) {
		if( item->id >= id ) {
			setCenterOffset( items.at() );
			return;
		}
	}

	kdDebug() << "DiffView: scrollToId: id " << id << " not found" << endl;
}

// *offset methods are only for views withlineno
void DiffView::removeAtOffset(int offset)
{
    items.remove(offset);
    setNumRows(numRows()-1);
}

void DiffView::insertAtOffset(const QString &line, Difference::Type type, int offset)
{
    DiffViewItem *item = new DiffViewItem;
    item->line = line;
    item->type = type;
    item->lineno = -1;
    item->inverted = false;
    items.insert(offset, item);
    setNumRows(numRows()+1);
}

void DiffView::setCenterOffset(int offset)
{
//    if (!rowIsVisible(offset))
//	{
	    int visiblerows = viewHeight()/cellHeight(0);
	    setTopCell( QMAX(0, offset - visiblerows/2) );
//	}
}

void DiffView::addLine( const QString &line, Difference::Type type, int lineno, int id )
{
    QString spaces = QString::fromLatin1("                ").left(m_tabWidth);
    int pos = -1;
    QString str = line;
    while ((pos = str.find('\t')) != -1)
        str.replace((uint)pos, 1, spaces);

    QFontMetrics fm(font());
    textwidth = QMAX(textwidth, fm.width(line));

    DiffViewItem *item = new DiffViewItem;
    item->line = str;
    item->type = type;
    item->lineno = lineno;
	item->id = id;
    item->inverted = false;
    items.append(item);
    setNumRows(numRows()+1);
}

QString DiffView::stringAtOffset(int offset)
{
    if (offset >= (int)items.count())
	{
	    kdDebug() << "Internal error: lineAtOffset" << endl;
	}
    return items.at(offset)->line;
}

int DiffView::count()
{
    return items.count();
}

int DiffView::findLine( int lineno )
{
    int offset;
    DiffViewItem tmp;
    tmp.lineno = lineno;
    if ( (offset = items.find(&tmp)) == -1)
	{
	    kdDebug() << "Internal Error: Line " << lineno << " not found" << endl;
	    return -1;
	}
    return offset;
}

void DiffView::setInverted( int lineno, bool inverted )
{
    int offset;
    if ( (offset = findLine(lineno)) != -1)
	items.at(offset)->inverted = inverted;
}

void DiffView::setCenterLine( int lineno )
{
    int offset;
    if ( (offset = findLine(lineno)) != -1)
	setCenterOffset(offset);
}

QString DiffView::stringAtLine(int lineno)
{
    int pos;
    if ( (pos = findLine(lineno)) != -1 )
        return items.at(pos)->line;
    else
        return QString();
}

int DiffView::cellWidth( int col )
{
	if (col == 0 && linenos) {
		QFontMetrics fm(font());
		return fm.width("10000");
	} else {
		int rest = linenos ? cellWidth(0) : 0;
		return QMAX(textwidth, viewWidth()-rest);
	}
}

int DiffView::cellHeight( int row )
{
	return QTableView::cellHeight( row );
}

void DiffView::paintCell(QPainter *p, int row, int col)
{
	DiffViewItem* item = items.at(row);

	int width = cellWidth(col);
	int height = cellHeight(row);

	QColor backgroundColor = white;
	int align = AlignLeft;
	int innerborder = 0;
	QString str;

	backgroundColor = m_settings->getColorForDifferenceType( item->type, item->inverted );

	QFont oldFont(p->font());
	if ( item->type == Difference::Separator ) {
		backgroundColor = gray;
		if ( col == (linenos?1:0) )
			str = item->line;
		QFont f(oldFont);
		f.setBold(true);
		p->setFont(f);
	} else if ( col == 0 && linenos ) {
		if( item->type == Difference::Unchanged ) {
			backgroundColor = QColor(222, 222, 222);
		}
		innerborder = 5;
		align = AlignRight;
		str.setNum(item->lineno);
	} else {
		str = item->line;
	}

	p->setPen(black);
	p->fillRect(0, 0, width, height, backgroundColor);
	if( item->inverted ) {
		if( row == 0 || !items.at( row - 1 )->inverted ) {
			p->drawLine( 0, 0, width, 0 );
		}
		if( (uint)row == items.count() - 1 || !items.at( row + 1 )->inverted ) {
			p->drawLine( 0, height - 1, width, height - 1 );
		}
		QFont f(oldFont);
		f.setBold(true);
		p->setFont(f);
	}
	p->drawText(innerborder, 0, width-2*innerborder, height, align, str);
	p->setFont(oldFont);
}


void DiffView::wheelEvent(QWheelEvent *e)
{
    QApplication::sendEvent(verticalScrollBar(), e);
}

int DiffView::rowPos( int row )
{
	return row * cellHeight( row ) - yOffset();
}

DiffConnectWidget::DiffConnectWidget( DiffModel* model, GeneralSettings* settings,
                                      KDiffView* parent, const char* name)
    : QWidget(parent, name)
{
	m_model = model;
	m_settings = settings;
	m_diffView = parent;
	connect( m_settings, SIGNAL( settingsChanged() ), this, SLOT( repaint() ) );
	setBackgroundMode( NoBackground );
	setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Minimum ) );
}

DiffConnectWidget::~DiffConnectWidget()
{
}

void DiffConnectWidget::setDiffViews( DiffView* left, DiffView* right )
{
	m_leftView = left;
	m_rightView = right;
}

void DiffConnectWidget::setModel( DiffModel* model ) {
	m_model = model;
	repaint();
}

QSize DiffConnectWidget::sizeHint() const
{
	return QSize(50, style().scrollBarExtent().height());
}

void DiffConnectWidget::paintEvent( QPaintEvent* /* e */ )
{
	QPixmap pixbuf(size());
	QPainter paint(&pixbuf, this);
	QPainter* p = &paint;

	p->fillRect( 0, 0, pixbuf.width(), pixbuf.height(), white );

	QListIterator<Difference> differences = QListIterator<Difference>( m_model->getDifferences() );
	for( int i = 0; differences.current(); ++differences, ++i ) {

		int tl, tr, bl, br;
		const Difference* d = differences.current();
		bool selected = (m_diffView->getSelectedItem() == i );

		int rowA = m_leftView->findLine( d->linenoA );
		tl = m_leftView->rowPos( rowA );

		int rowB = m_rightView->findLine( d->linenoB );
		tr = m_rightView->rowPos( rowB );

		rowA = rowA + d->sourceLineCount() - 1;
		bl = m_leftView->rowPos( rowA ) + m_leftView->cellHeight( rowA ) - 1;

		rowB = rowB + d->destinationLineCount() - 1;
		br = m_rightView->rowPos( rowB ) + m_rightView->cellHeight( rowB ) - 1;

		// This happens if it is an add or delete
		if( bl < tl ) bl = tl+1;
		if( br < tr ) br = tr+1;

		QPointArray topBezier = makeTopBezier( tl, tr );
		QPointArray bottomBezier = makeBottomBezier( bl, br );

		p->setPen( m_settings->getColorForDifferenceType( d->type, selected ) );
		p->setBrush( m_settings->getColorForDifferenceType( d->type, selected ) );
		p->drawPolygon ( makeConnectPoly( topBezier, bottomBezier ) );

		if( selected ) {
			p->setPen( black );
			p->drawPolyline( topBezier );
			p->drawPolyline( bottomBezier );
		}

	}

	p->setPen( black );
	p->drawLine( 0,0, 0,pixbuf.height() );
	p->drawLine( pixbuf.width()-1,0, pixbuf.width()-1,pixbuf.height() );

	p->flush();
	bitBlt(this, 0, 0, &pixbuf);
}

QPointArray DiffConnectWidget::makeTopBezier( int tl, int tr )
{
	int l = 0;
	int r = width();
	QPointArray controlPoints;
	controlPoints.setPoints( 4, l,tl, 20,tl, r-20,tr, r,tr );
	return controlPoints.quadBezier();
}

QPointArray DiffConnectWidget::makeBottomBezier( int bl, int br )
{
	int l = 0;
	int r = width();
	QPointArray controlPoints;
	controlPoints.setPoints( 4, r,br, r-20,br, 20,bl, l,bl );
	return controlPoints.quadBezier();
}

QPointArray DiffConnectWidget::makeConnectPoly( const QPointArray& topBezier, const QPointArray& bottomBezier )
{
	QPointArray poly( topBezier.size() + bottomBezier.size() );
	for( uint i = 0; i < topBezier.size(); i++ )
		poly.setPoint( i, topBezier.point( i ) );
	for( uint i = 0; i < bottomBezier.size(); i++ )
		poly.setPoint( i + topBezier.size(), bottomBezier.point( i ) );

	return poly;
}
