/*
    KMLOCfg
    
    A utility to configure the ELSA MicroLink(tm) Office modem.

    Copyright (C) 2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 


#include <qglobal.h>
#include <qkeycode.h>
#include <qpixmap.h>
#include <qsize.h>
#include <qwhatsthis.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
// #include <kmessagebox.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <ktoolbarbutton.h>

#include "kmlocfg.h"
#include "global.h"
#include "configure.h"
#include "loaddlg.h"
#include "setupdlg.h"



KMLOCfg::KMLOCfg(QWidget *parent, const char *name) : KMainWindow(parent, name)
{
	QPixmap pix;
	
	config = kapp->config();
	
	initMenuBar();
	initToolBar();
	initStatusBar();
	initKeyAccel();
	initWhatsThis();
	
	configure = new Configure(this, "configure");
	CHECK_PTR(configure);

	loaddlg = new LoadDlg(this, "loaddlg");
	CHECK_PTR(loaddlg);

	setupdlg = new SetupDlg(this, "setupdlg");
	CHECK_PTR(setupdlg);

	view = new QLabel(this);
	pix = UserIcon("mloffice");
	view->setPixmap(pix);
	view->setFixedSize(view->sizeHint());
	setCentralWidget(view);

	slotNewConfig();
	slotFileNew();

	enableCommand(ID_FILE_UPLOAD, false);
	enableCommand(ID_FILE_EDIT, false);

	connect(configure, SIGNAL(newConfig()), SLOT(slotNewConfig()));

	connect(loaddlg, SIGNAL(downDone(bool)), SLOT(slotFetchDownDone(bool)));
	connect(loaddlg, SIGNAL(upDone(bool)), SLOT(slotFetchUpDone(bool)));

//	resize(sizeHint());
	setFixedSize(minimumSizeHint());
}


KMLOCfg::~KMLOCfg()
{
}


void KMLOCfg::enableCommand(int id_, bool enable)
{
	menuBar()->setItemEnabled(id_, enable);
	toolBar()->setItemEnabled(id_, enable);
}


void KMLOCfg::readOptions()
{
	config->setGroup("MainWindow Options");

	bool toolBarOn = config->readBoolEntry("Show ToolBar", true);
	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, toolBarOn);
	if (!toolBarOn)
		toolBar()->hide();
	bool statusBarOn = config->readBoolEntry("Show StatusBar", true);
	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, statusBarOn);
	if (!statusBarOn)
		statusBar()->hide();
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));
}


void KMLOCfg::saveOptions()
{
	config->setGroup("MainWindow Options");
	
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());
}


void KMLOCfg::initMenuBar()
{
	fileMenu = new QPopupMenu(0, "mainfilemenu");
	CHECK_PTR(fileMenu);
	fileMenu->insertItem(SmallIcon("filedownload"), i18n("&Download"), ID_FILE_DOWNLOAD);
	fileMenu->insertItem(SmallIcon("fileupload"), i18n("&Upload"), ID_FILE_UPLOAD);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("filenew"), i18n("&New Configuration"), ID_FILE_NEW);
	fileMenu->insertItem(SmallIcon("edit"), i18n("&Edit Configuration..."), ID_FILE_EDIT);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("exit"), i18n("&Quit"), ID_FILE_QUIT);

	settingsMenu = new QPopupMenu(0, "mainsettingsmenu");
	CHECK_PTR(settingsMenu);
	settingsMenu->setCheckable(true);
	settingsMenu->insertItem(i18n("Show &Toolbar"), ID_SETTINGS_TOOLBAR);
	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, true);
	settingsMenu->insertItem(i18n("Show &Statusbar"), ID_SETTINGS_STATUSBAR);
	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, true);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(i18n("Save Options"), ID_SETTINGS_SAVEOPTIONS);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(SmallIcon("configure"), i18n("&Configure KMLOCfg..."), ID_SETTINGS_CONFIGURE);
	
	QString aboutstr = QString("KMLOCfg ") + VERSION + i18n(
		"\nCopyright %1 2000 Oliver Gantz <Oliver.Gantz@epost.de>\n\n" \
		"A utility to configure the\n" \
		"ELSA MicroLink(tm) Office modem.\n\n" \
		"This program is free software and is distributed under the terms\n" \
		"of the GNU General Public License.\n\n" \
		"ELSA and MicroLink are trademarks of ELSA AG, Aachen.").arg('');
	
	helpMenu_ = helpMenu(aboutstr);
	
	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), helpMenu_);

	connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(settingsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
}


void KMLOCfg::initToolBar()
{
	toolBar()->insertButton(BarIcon("filedownload"), ID_FILE_DOWNLOAD, true, i18n("Download Configuration"));
	toolBar()->insertButton(BarIcon("fileupload"), ID_FILE_UPLOAD, true, i18n("Upload Configuration"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("filenew"), ID_FILE_NEW, true, i18n("New Configuration"));
	toolBar()->insertButton(BarIcon("edit"), ID_FILE_EDIT, true, i18n("Edit Configuration"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("configure"), ID_SETTINGS_CONFIGURE, true, i18n("Configure KMLOCfg"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("help"), ID_HELP_CONTENTS, SIGNAL(clicked()), this, SLOT(appHelpActivated()), true, i18n("Help"));
	toolBar()->insertButton(BarIcon("exit"), ID_FILE_QUIT, true, i18n("Quit"));
	toolBar()->alignItemRight(ID_FILE_QUIT, true);

	connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
}


void KMLOCfg::initStatusBar()
{
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG, 1);
	statusBar()->setItemAlignment(ID_STATUS_MSG, AlignLeft | AlignVCenter);
}


void KMLOCfg::initKeyAccel()
{
	keyAccel = new KAccel(this);
	CHECK_PTR(keyAccel);

	keyAccel->insertItem(i18n("Download Configuration"), "DownloadConfig", CTRL+Key_D);
	keyAccel->insertItem(i18n("Upload Configuration"), "UploadConfig", CTRL+Key_U);
	
	keyAccel->insertItem(i18n("Show Toolbar"), "ShowToolbar", CTRL+Key_T);
	keyAccel->insertItem(i18n("Show Statusbar"), "ShowStatusbar", CTRL+Key_S);

	keyAccel->readSettings();
	
	keyAccel->connectItem("DownloadConfig", this, SLOT(slotFileDownload()));
	keyAccel->connectItem("UploadConfig", this, SLOT(slotFileUpload()));
	keyAccel->connectItem(KStdAccel::Quit, this, SLOT(slotFileQuit()));
	
	keyAccel->connectItem("ShowToolbar", this, SLOT(slotSettingsShowToolbar()));
	keyAccel->connectItem("ShowStatusbar", this, SLOT(slotSettingsShowStatusbar()));
	
	keyAccel->connectItem(KStdAccel::Help, this, SLOT(appHelpActivated()));

	keyAccel->changeMenuAccel(fileMenu, ID_FILE_DOWNLOAD, "DownloadConfig");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_UPLOAD, "UploadConfig");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_QUIT, KStdAccel::Quit);

	keyAccel->changeMenuAccel(settingsMenu, ID_SETTINGS_TOOLBAR, "ShowToolbar");
	keyAccel->changeMenuAccel(settingsMenu, ID_SETTINGS_STATUSBAR, "ShowStatusbar");
}


void KMLOCfg::initWhatsThis()
{
	QWhatsThis::add(toolBar()->getButton(ID_FILE_DOWNLOAD),
		i18n("Click this button to download the configuration of your modem.<br><br>" \
		"You can also select the command <b>Download</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_SETTINGS_CONFIGURE),
		i18n("Opens the configure dialog.<br><br>" \
		"You can also select the command <b>Configure KMLOCfg</b> from the \"Settings\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_HELP_CONTENTS),
		i18n("Invokes the KDE help system with the KMLOCfg help pages displayed.<br><br>" \
		"You can also select the command <b>Contents</b> from the \"Help\" menu."));
}


void KMLOCfg::commandCallback(int id_)
{
	switch (id_) {
		case ID_FILE_DOWNLOAD:
			slotFileDownload();
			break;
		case ID_FILE_UPLOAD:
			slotFileUpload();
			break;
		case ID_FILE_NEW:
			slotFileNew();
			break;
		case ID_FILE_EDIT:
			slotFileEdit();
			break;
		case ID_FILE_QUIT:
			slotFileQuit();
			break;
		case ID_SETTINGS_TOOLBAR:
			slotSettingsShowToolbar();
			break;
		case ID_SETTINGS_STATUSBAR:
			slotSettingsShowStatusbar();
			break;
		case ID_SETTINGS_SAVEOPTIONS:
			saveOptions();
			break;
		case ID_SETTINGS_CONFIGURE:
			slotSettingsConfigure();
			break;
		default:
			break;
	}
}

		
void KMLOCfg::slotFileDownload()
{
	if (loaddlg->startDownload()) {
		enableCommand(ID_FILE_DOWNLOAD, false);
		enableCommand(ID_FILE_UPLOAD, false);
		slotStatusHelpMsg(i18n("Reading modem's configuration..."));
	}
}


void KMLOCfg::slotFileUpload()
{
	loaddlg->setConfig(setupdlg->getConfig());
	if (loaddlg->startUpload()) {
		enableCommand(ID_FILE_DOWNLOAD, false);
		enableCommand(ID_FILE_UPLOAD, false);
		slotStatusHelpMsg(i18n("Writing modem's configuration..."));
	}
}


void KMLOCfg::slotFileNew()
{
	loaddlg->setDefaultConfig();
	setupdlg->setConfig(loaddlg->getConfig());

	has_config = true;

	enableCommand(ID_FILE_UPLOAD, has_config);
	enableCommand(ID_FILE_EDIT, has_config);
}


void KMLOCfg::slotFileEdit()
{
	if (setupdlg->isVisible())
		setupdlg->raise();
	else
		setupdlg->show();
}


void KMLOCfg::slotFileQuit()
{
	close();
}


void KMLOCfg::slotSettingsShowToolbar()
{
	bool toolBarOn = !settingsMenu->isItemChecked(ID_SETTINGS_TOOLBAR);

	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, toolBarOn);
	if (toolBarOn)
		toolBar()->show();
	else
		toolBar()->hide();

	setFixedSize(minimumSizeHint());
}


void KMLOCfg::slotSettingsShowStatusbar()
{
	bool statusBarOn = !settingsMenu->isItemChecked(ID_SETTINGS_STATUSBAR);

	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, statusBarOn);
	if (statusBarOn)
		statusBar()->show();
	else
		statusBar()->hide();
	
	setFixedSize(minimumSizeHint());
}


void KMLOCfg::slotSettingsConfigure()
{
	if (configure->isVisible())
		configure->raise();
	else
		configure->show();
}


void KMLOCfg::slotStatusMsg(const QString& text)
{
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KMLOCfg::slotStatusHelpMsg(const QString& text)
{
	statusBar()->message(text, 2000);
}


void KMLOCfg::slotNewConfig()
{
	bool tooltips;

	config->setGroup(ID_PREF_GROUP_GENERAL);
	tooltips = config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS);

	configure->addToolTips(tooltips);
	setupdlg->addToolTips(tooltips);
}


void KMLOCfg::slotFetchDownDone(bool success)
{
	if (success) {
		has_config = true;
		setupdlg->setConfig(loaddlg->getConfig());
		slotFileEdit();
	}	

	enableCommand(ID_FILE_DOWNLOAD, true);
	enableCommand(ID_FILE_UPLOAD, has_config);
	enableCommand(ID_FILE_EDIT, has_config);
}


void KMLOCfg::slotFetchUpDone(bool)
{
	enableCommand(ID_FILE_DOWNLOAD, true);
	enableCommand(ID_FILE_UPLOAD, true);
}
