/*
    KMLOCfg
    
    A utility to configure the ELSA MicroLink(tm) Office modem.

    Copyright (C) 2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <stdio.h>
#include <string.h>

#include <qglobal.h>
#include <qtabdialog.h>
#include <qwidget.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qgroupbox.h>
#include <qcombobox.h>
#include <qtooltip.h>
#include <qsize.h>
#include <qslider.h>

#include <kapp.h>
#include <klocale.h>
#include <kglobal.h>

#include "setupdlg.h"
#include "global.h"



static const uchar country_code_tab[] = {
	0x0A, 0x0F, 0x31, 0x3C, 0x3D, 0x04, 0x59, 0x68, 0x69,
	0x74, 0x7B, 0x82, 0x8B, 0xA5, 0xA6, 0xA0, 0xB4, 0xB5
};



SetupDlg::SetupDlg(QWidget *parent, const char *name) : QTabDialog (parent, name)
{
	static const char *country_tab[] = {
		I18N_NOOP("Austria"),I18N_NOOP("Belgium"),I18N_NOOP("Denmark"),I18N_NOOP("Finland"),
		I18N_NOOP("France"),I18N_NOOP("Germany"),I18N_NOOP("Italy"),I18N_NOOP("Liechtenstein"),
		I18N_NOOP("Luxemburg"),I18N_NOOP("Monaco"),I18N_NOOP("Netherlands"),I18N_NOOP("Norway"),
		I18N_NOOP("Portugal"),I18N_NOOP("Sweden"),I18N_NOOP("Switzerland"),I18N_NOOP("Spain"),
		I18N_NOOP("United Kingdom"),I18N_NOOP("USA")
	};
	int i;
	char buf[16];
	QGridLayout *grid;
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *label;
	QGroupBox *gb;
	QSize size;

	gotToolTips = FALSE;

	page1 = new QWidget(this, "setuppage1");
	CHECK_PTR(page1);
	page2 = new QWidget(this, "setuppage2");
	CHECK_PTR(page2);
	page3 = new QWidget(this, "setuppage3");
	CHECK_PTR(page3);
	page4 = new QWidget(this, "setuppage4");
	CHECK_PTR(page4);
	page5 = new QWidget(this, "setuppage5");
	CHECK_PTR(page5);

	addTab(page1, i18n("General"));
	addTab(page2, i18n("Calls"));
	addTab(page3, i18n("Answering Machine"));
	addTab(page4, i18n("Fax && Misc"));
	addTab(page5, i18n("Access"));

	/* Page 1: General */
	size = QSize(150, 0);

	vbox = new QVBoxLayout(page1, 12);

	gb = new QGroupBox(i18n("Firmware"), page1);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	general_firmware = new QLabel(" ", gb);
	general_firmware->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	general_firmware->setMinimumSize(general_firmware->sizeHint());
	svbox->addWidget(general_firmware);

	svbox->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Internal Modem Time"), page1);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Date, Time:"), gb);
	label->setFixedWidth(label->sizeHint().width());
	label->setMinimumHeight(label->sizeHint().height());
	shbox->addWidget(label);
	general_time = new QLabel(" ", gb);
	general_time->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	general_time->setMinimumSize(general_time->sizeHint());
	shbox->addWidget(general_time);

	general_sync = new QCheckBox(i18n("Synchronize modem time"), gb);
	general_sync->setMinimumSize(general_sync->sizeHint());
	svbox->addWidget(general_sync);

	svbox->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Country Selection"), page1);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Country:"), gb);
	label->setMinimumSize(label->sizeHint());
	shbox->addWidget(label);

	general_country = new QComboBox(gb);
	label->setBuddy(general_country);
	for (i=0; i < 18; i++)
		general_country->insertItem(i18n(country_tab[i]), i);
	size = size.expandedTo(general_country->sizeHint());
	general_country->setFixedHeight(general_country->sizeHint().height());
	shbox->addWidget(general_country);

	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch();
	vbox->activate();

	/* Page 2: Calls */
	vbox = new QVBoxLayout(page2, 12);
	
	gb = new QGroupBox(i18n("Incoming Calls"), page2);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	general_ring = new QCheckBox(i18n("Modem rings with incoming call"), gb);
	general_ring->setMinimumSize(general_ring->sizeHint());
	svbox->addWidget(general_ring);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Accept calls:"), gb);
	label->setMinimumSize(label->sizeHint());
	shbox->addWidget(label);
	
	general_rings = new QComboBox(gb);
	label->setBuddy(general_rings);
	general_rings->insertItem(i18n("After 1 Ring"), 0);
	for (i=1; i < 9; i++) {
		sprintf(buf, i18n("After %d Rings"), i+1);
		general_rings->insertItem(buf, i);
	}
	size = size.expandedTo(general_rings->sizeHint());
	general_rings->setFixedHeight(general_rings->sizeHint().height());
	shbox->addWidget(general_rings);

	general_acceptearly = new QCheckBox(i18n("Accept calls after 2 rings"), gb);
	general_acceptearly->setMinimumSize(general_acceptearly->sizeHint());
	svbox->addWidget(general_acceptearly);

	general_acceptfull = new QCheckBox(i18n("Accept calls if memory full"), gb);
	general_acceptfull->setMinimumSize(general_acceptfull->sizeHint());
	svbox->addWidget(general_acceptfull);

	general_acceptdtr = new QCheckBox(i18n("Accept calls with running application"), gb);
	general_acceptdtr->setMinimumSize(general_acceptdtr->sizeHint());
	svbox->addWidget(general_acceptdtr);

	general_suppressri = new QCheckBox(i18n("Suppress ring indicator signal (RI/M3)"), gb);
	general_suppressri->setMinimumSize(general_suppressri->sizeHint());
	svbox->addWidget(general_suppressri);

	general_country->setFixedWidth(size.width());
	general_rings->setFixedWidth(size.width());

	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch();
	vbox->activate();

	/* Page 3: Answering Machine */
	size = QSize(150, 0);

	vbox = new QVBoxLayout(page3, 12);

	answer_enabled = new QCheckBox(i18n("Answering machine enabled"), page3);
	answer_enabled->setMinimumSize(answer_enabled->sizeHint());
	vbox->addWidget(answer_enabled);

	gb = new QGroupBox(i18n("Outgoing Message"), page3);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Message:"), gb);
	label->setMinimumSize(label->sizeHint());
	shbox->addWidget(label);
	
	answer_outgoing = new QComboBox(gb);
	label->setBuddy(answer_outgoing);
	answer_outgoing->insertItem(i18n("No. 1"), 0);
	answer_outgoing->insertItem(i18n("No. 2"), 1);
	answer_outgoing->insertItem(i18n("No. 3"), 2);
	answer_outgoing->insertItem(i18n("Memory full"), 3);
	size = size.expandedTo(answer_outgoing->sizeHint());
	answer_outgoing->setFixedHeight(answer_outgoing->sizeHint().height());
	shbox->addWidget(answer_outgoing);

	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Incoming Message"), page3);

	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	answer_record = new QCheckBox(i18n("Record incoming messages"), gb);
	answer_record->setMinimumSize(answer_record->sizeHint());
	svbox->addWidget(answer_record);

	grid = new QGridLayout(5, 2, 8);
	svbox->addLayout(grid);

	label = new QLabel(i18n("Recording volume:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 0);

	rec_recvol = new QSlider(0, 255, 128, 0, QSlider::Horizontal, gb);
	label->setBuddy(rec_recvol);
	rec_recvol->setTickmarks(QSlider::Below);
	size = size.expandedTo(rec_recvol->sizeHint());
	rec_recvol->setFixedHeight(rec_recvol->sizeHint().height());
	grid->addWidget(rec_recvol, 0, 1);

	label = new QLabel(i18n("Playback volume:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	rec_playvol = new QSlider(0, 255, 128, 0, QSlider::Horizontal, gb);
	label->setBuddy(rec_playvol);
	rec_playvol->setTickmarks(QSlider::Below);
	size = size.expandedTo(rec_playvol->sizeHint());
	rec_playvol->setFixedHeight(rec_playvol->sizeHint().height());
	grid->addWidget(rec_playvol, 1, 1);

	label = new QLabel(i18n("Recording quality:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 2, 0);
	
	rec_quality = new QComboBox(gb);
	label->setBuddy(rec_quality);
	rec_quality->insertItem(i18n("Low"), 0);
	rec_quality->insertItem(i18n("Medium"), 1);
	rec_quality->insertItem(i18n("High"), 2);
	size = size.expandedTo(rec_quality->sizeHint());
	rec_quality->setFixedHeight(rec_quality->sizeHint().height());
	grid->addWidget(rec_quality, 2, 1);

	label = new QLabel(i18n("Maximum duration:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 3, 0);
	
	rec_maxdur = new QComboBox(gb);
	label->setBuddy(rec_maxdur);
	rec_maxdur->insertItem(i18n("Unlimited"), 0);
	for (i=1; i < 21; i++) {
		sprintf(buf, i18n("%d Sec."), i*15);
		rec_maxdur->insertItem(buf, i);
	}
	size = size.expandedTo(rec_maxdur->sizeHint());
	rec_maxdur->setFixedHeight(rec_maxdur->sizeHint().height());
	grid->addWidget(rec_maxdur, 3, 1);

	label = new QLabel(i18n("Minimum duration:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 4, 0);
	
	rec_mindur = new QComboBox(gb);
	label->setBuddy(rec_mindur);
	for (i=0; i < 16; i++) {
		sprintf(buf, i18n("%d Sec."), i);
		rec_mindur->insertItem(buf, i);
	}
	size = size.expandedTo(rec_mindur->sizeHint());
	rec_mindur->setFixedHeight(rec_mindur->sizeHint().height());
	grid->addWidget(rec_mindur, 4, 1);

	rec_speaker = new QCheckBox(i18n("Enable speaker during recording"), gb);
	rec_speaker->setMinimumSize(rec_speaker->sizeHint());
	svbox->addWidget(rec_speaker);

	svbox->activate();
	
	vbox->addWidget(gb);

	answer_outgoing->setFixedWidth(size.width());
	rec_playvol->setFixedWidth(size.width());
	rec_recvol->setFixedWidth(size.width());
	rec_quality->setFixedWidth(size.width());
	rec_maxdur->setFixedWidth(size.width());
	rec_mindur->setFixedWidth(size.width());

	vbox->addStretch(1);
	vbox->activate();

	/* Page 4: Fax and Misc */
	vbox = new QVBoxLayout(page4, 12);

	gb = new QGroupBox(i18n("Fax"), page4);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	fax_operation = new QCheckBox(i18n("Fax reception enabled"), gb);
	fax_operation->setMinimumSize(fax_operation->sizeHint());
	svbox->addWidget(fax_operation);
	fax_takeover = new QCheckBox(i18n("Fax reception during phone calls"), gb);
	fax_takeover->setMinimumSize(fax_takeover->sizeHint());
	svbox->addWidget(fax_takeover);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Sender ID:"), gb);
	label->setMinimumSize(label->sizeHint());
	shbox->addWidget(label);
	
	fax_id = new QLineEdit(gb);
	label->setBuddy(fax_id);
	fax_id->setMaxLength(20);
	fax_id->setFixedWidth(fax_id->sizeHint().width() * 2);
	fax_id->setMinimumHeight(fax_id->sizeHint().height());
	shbox->addWidget(fax_id);
	shbox->addStretch(1);
	
	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Announcements"), page4);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	message_system = new QCheckBox(i18n("Announce system messages"), gb);
	message_system->setMinimumSize(message_system->sizeHint());
	svbox->addWidget(message_system);

	message_number = new QCheckBox(i18n("Announce caller ID"), gb);
	message_number->setMinimumSize(message_number->sizeHint());
	svbox->addWidget(message_number);

	message_time = new QCheckBox(i18n("Announce time of call"), gb);
	message_time->setMinimumSize(message_time->sizeHint());
	svbox->addWidget(message_time);

	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Speakerphone Mode"), page4);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	grid = new QGridLayout(2, 2, 8);
	svbox->addLayout(grid);

	label = new QLabel(i18n("Sound volume:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 0);

	speaker_volume = new QSlider(0, 255, 128, 0, QSlider::Horizontal, gb);
	label->setBuddy(speaker_volume);
	speaker_volume->setTickmarks(QSlider::Below);
	speaker_volume->setFixedWidth(150);
	speaker_volume->setFixedHeight(speaker_volume->sizeHint().height());
	grid->addWidget(speaker_volume, 0, 1);

	label = new QLabel(i18n("Microphone sensitivity:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	speaker_micro = new QSlider(0, 255, 128, 0, QSlider::Horizontal, gb);
	label->setBuddy(speaker_micro);
	speaker_micro->setTickmarks(QSlider::Below);
	speaker_micro->setFixedWidth(150);
	speaker_micro->setFixedHeight(speaker_micro->sizeHint().height());
	grid->addWidget(speaker_micro, 1, 1);

	speaker_monitor = new QCheckBox(i18n("Enable monitoring or recording of conversation"), gb);
	speaker_monitor->setMinimumSize(speaker_monitor->sizeHint());
	svbox->addWidget(speaker_monitor);

	svbox->activate();
	
	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();

	/* Page 5: Access */
	vbox = new QVBoxLayout(page5, 12);

	gb = new QGroupBox(i18n("Local Configuration"), page5);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	access_hook = new QCheckBox(i18n("Access by pushing hook flash"), gb);
	access_hook->setMinimumSize(access_hook->sizeHint());
	svbox->addWidget(access_hook);

	access_dtmf = new QCheckBox(i18n("Access by sending DTMF tones"), gb);
	access_dtmf->setMinimumSize(access_dtmf->sizeHint());
	svbox->addWidget(access_dtmf);

	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Remote Access"), page5);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	access_playback = new QCheckBox(i18n("Enable remote playback"), gb);
	access_playback->setMinimumSize(access_playback->sizeHint());
	svbox->addWidget(access_playback);

	access_config = new QCheckBox(i18n("Enable remote configuration"), gb);
	access_config->setMinimumSize(access_config->sizeHint());
	svbox->addWidget(access_config);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Password:"), gb);
	label->setMinimumSize(label->sizeHint());
	shbox->addWidget(label);
	
	access_password = new QLineEdit(gb);
	label->setBuddy(access_password);
	access_password->setMaxLength(4);
	access_password->setFixedWidth(access_password->sizeHint().width() / 2);
	access_password->setMinimumHeight(access_password->sizeHint().height());
	shbox->addWidget(access_password);
	shbox->addStretch();

	svbox->activate();
	
	vbox->addWidget(gb);
	vbox->addStretch();
	vbox->activate();

	setOKButton(i18n("&OK"));
	setDefaultButton(i18n("&Default"));
	setApplyButton(i18n("&Apply"));
	setCancelButton(i18n("&Cancel"));

	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), SLOT(timerSlot()));	
	timer->start(1000);

	connect(this, SIGNAL(defaultButtonPressed()), SLOT(reset()));
	connect(this, SIGNAL(applyButtonPressed()), SLOT(apply()));
	
	setCaption(i18n("Edit Configuration"));

	resize(sizeHint());
}


SetupDlg::~SetupDlg()
{
}


void SetupDlg::addToolTips(bool on)
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add(general_firmware, i18n("Some information about the modem's firmware"));
		QToolTip::add(general_time, i18n("The date and time of your modem's timer"));
		QToolTip::add(general_sync, i18n("Syncronize the modem's timer with your PC's timer"));
		QToolTip::add(general_country, i18n("The country in which you are located"));
		QToolTip::add(general_ring, i18n("Modem rings with incoming call"));
		QToolTip::add(general_rings, i18n("Number of rings before calls are accepted"));
		QToolTip::add(general_acceptearly, i18n("Accept calls after 2 rings"));
		QToolTip::add(general_acceptfull, i18n("Accept calls when modem memory is full"));
		QToolTip::add(general_acceptdtr, i18n("Accept calls when an application uses the modem"));
		QToolTip::add(general_suppressri, i18n("Suppress RI signal with incoming call"));
		QToolTip::add(answer_enabled, i18n("Use this option to enable the answering machine"));
		QToolTip::add(answer_record, i18n("Record messages of incoming calls"));
		QToolTip::add(answer_outgoing, i18n("Selection of the outgoing message"));
		QToolTip::add(rec_playvol, i18n("The volume with which recorded messages are played back"));
		QToolTip::add(rec_recvol, i18n("The volume with which incoming messages are recorded"));
		QToolTip::add(rec_quality, i18n("The quality with which incoming messages are recorded"));
		QToolTip::add(rec_maxdur, i18n("The maximum recording duration of incoming messages"));
		QToolTip::add(rec_mindur, i18n("The minimum recording duration of incoming messages"));
		QToolTip::add(rec_speaker, i18n("Turn on the speaker while incoming messages are recorded"));
		QToolTip::add(fax_operation, i18n("Use this option to enable fax reception"));
		QToolTip::add(fax_takeover, i18n("Automatic fax recognition during phone calls"));
		QToolTip::add(fax_id, i18n("Enter your fax ID here"));
		QToolTip::add(message_system, i18n("Announce the modem's system messages"));
		QToolTip::add(message_number, i18n("Announce the caller ID (CLIP) of recorded messages"));
		QToolTip::add(message_time, i18n("Announce the date and time of recorded messages"));
		QToolTip::add(speaker_volume, i18n("The volume of the modem's speaker"));
		QToolTip::add(speaker_micro, i18n("The sensitivity of the modem's microphone"));
		QToolTip::add(speaker_monitor, i18n("Enable monitoring and recording of conversations"));
		QToolTip::add(access_hook, i18n("Configure the modem by pushing the hook flash of the phone"));
		QToolTip::add(access_dtmf, i18n("Configure the modem by sending DTMF tones with the phone"));
		QToolTip::add(access_playback, i18n("Enable remote playback of recorded messages"));
		QToolTip::add(access_config, i18n("Enable remote configuration"));
		QToolTip::add(access_password, i18n("Enter the password for the remote playback"));
	} else {
		QToolTip::remove(general_firmware);
		QToolTip::remove(general_time);
		QToolTip::remove(general_sync);
		QToolTip::remove(general_country);
		QToolTip::remove(general_ring);
		QToolTip::remove(general_rings);
		QToolTip::remove(general_acceptearly);
		QToolTip::remove(general_acceptfull);
		QToolTip::remove(general_acceptdtr);
		QToolTip::remove(general_suppressri);
		QToolTip::remove(answer_enabled);
		QToolTip::remove(answer_record);
		QToolTip::remove(answer_outgoing);
		QToolTip::remove(rec_playvol);
		QToolTip::remove(rec_recvol);
		QToolTip::remove(rec_quality);
		QToolTip::remove(rec_maxdur);
		QToolTip::remove(rec_mindur);
		QToolTip::remove(rec_speaker);
		QToolTip::remove(fax_operation);
		QToolTip::remove(fax_takeover);
		QToolTip::remove(fax_id);
		QToolTip::remove(message_system);
		QToolTip::remove(message_number);
		QToolTip::remove(message_time);
		QToolTip::remove(speaker_volume);
		QToolTip::remove(speaker_micro);
		QToolTip::remove(speaker_monitor);
		QToolTip::remove(access_hook);
		QToolTip::remove(access_dtmf);
		QToolTip::remove(access_playback);
		QToolTip::remove(access_config);
		QToolTip::remove(access_password);
	}
}	


const mlo_config_t *SetupDlg::getConfig()
{
	return &c;
}


void SetupDlg::setConfig(const mlo_config_t *conf)
{
	c = *conf;
	reset();
}


void SetupDlg::timerSlot()
{
	c.date_time = c.date_time.addSecs(1);
	general_time->setText(KGlobal::locale()->formatDate(c.date_time.date(), true) + ", " + KGlobal::locale()->formatTime(c.date_time.time(), true));
}


void SetupDlg::reset()
{
	int i;

	general_firmware->setText(c.firmware);
	
	general_sync->setChecked(c.sync_timer);
	for (i=0; i < 18; i++)
		if (country_code_tab[i] == c.country)
			general_country->setCurrentItem(i);
	general_ring->setChecked(c.ring_signal);
	general_rings->setCurrentItem(c.ring_number-1);
	general_acceptearly->setChecked(c.ring_fixed);
	general_acceptfull->setChecked(c.call_accept);
	general_acceptdtr->setChecked(c.operation_control);
	general_suppressri->setChecked(c.ring_suppress_ri);

	answer_enabled->setChecked(c.voice_operation);
	answer_record->setChecked(c.voice_recording);
	if (c.outgoing_message < 3)
		answer_outgoing->setCurrentItem(c.outgoing_message);
	else
		answer_outgoing->setCurrentItem(3);

	rec_playvol->setValue(c.voice_play);
	rec_recvol->setValue(c.voice_rec);
	if (!strcmp(c.rec_quality,"ADPCM2-7"))
		rec_quality->setCurrentItem(0);
	else if (!strcmp(c.rec_quality,"ADPCM4-7"))
		rec_quality->setCurrentItem(1);
	else
		rec_quality->setCurrentItem(2);
	rec_maxdur->setCurrentItem(c.max_rec_time);
	rec_mindur->setCurrentItem(c.min_rec_time);
	rec_speaker->setChecked(c.rec_speaker);

	fax_operation->setChecked(c.fax_operation);
	fax_takeover->setChecked(c.fax_takeover);
	fax_id->setText(c.fax_id);

	message_system->setChecked(c.announce_system);
	message_number->setChecked(c.announce_number);
	message_time->setChecked(c.announce_time);

	speaker_volume->setValue(c.voice_speaker);
	speaker_micro->setValue(c.voice_micro);
	speaker_monitor->setChecked(c.rec_monitor);

	access_hook->setChecked(c.hook_config);
	access_dtmf->setChecked(c.keyboard_config);
	access_playback->setChecked(c.remote_query);
	access_config->setChecked(c.remote_config);
	access_password->setText(c.remote_pw);
}


void SetupDlg::apply()
{
	static const char *qualities[3] = { "ADPCM2-7", "ADPCM4-7", "PCM8L-7" };
	int i;

	c.sync_timer = general_sync->isChecked();
	c.country = country_code_tab[general_country->currentItem()];
	c.ring_signal = general_ring->isChecked();
	c.ring_number = general_rings->currentItem()+1;
	c.ring_fixed = general_acceptearly->isChecked();
	c.call_accept = general_acceptfull->isChecked();
	c.operation_control = general_acceptdtr->isChecked();
	c.ring_suppress_ri = general_suppressri->isChecked();

	c.voice_operation = answer_enabled->isChecked();
	c.voice_recording = answer_record->isChecked();
	i = answer_outgoing->currentItem();
	if (i < 3)
		c.outgoing_message = i;
	else
		c.outgoing_message = 255;

	c.voice_play = rec_playvol->value();
	c.voice_rec = rec_recvol->value();
	strcpy(c.rec_quality, qualities[rec_quality->currentItem()]);
	c.max_rec_time = rec_maxdur->currentItem();
	c.min_rec_time = rec_mindur->currentItem();
	c.rec_speaker = rec_speaker->isChecked();

	c.fax_operation = fax_operation->isChecked();
	c.fax_takeover = fax_takeover->isChecked();
	strcpy(c.fax_id, fax_id->text());

	c.announce_system = message_system->isChecked();
	c.announce_number = message_number->isChecked();
	c.announce_time = message_time->isChecked();

	c.voice_speaker = speaker_volume->value();
	c.voice_micro = speaker_micro->value();
	c.rec_monitor = speaker_monitor->isChecked();

	c.hook_config = access_hook->isChecked();
	c.keyboard_config = access_dtmf->isChecked();
	c.remote_query = access_playback->isChecked();
	c.remote_config = access_config->isChecked();
	strcpy(c.remote_pw, access_password->text());
	c.remote_pw[4] = 0;
	for (i=0; i < 4; i++)
		if ((c.remote_pw[i] < '0') || (c.remote_pw[i] > '9'))
			c.remote_pw[i] = '1';

	emit apply_clicked();
}
