/*
    KMLOFax
    
    A utility to process facsimiles received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <stdlib.h>

#include <qglobal.h>
#include <qscrollbar.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kmenubar.h>

#include "preview.h"
#include "mlofile.h"
#include "printdlg.h"
#include "exportdlg.h"
#include "maildlg.h"
#include "senderaliases.h"
#include "global.h"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>



FaxWidget *fw = 0;


char *shrinkPage(int w, int h, char *data)
{
	static const uchar shrink_tab[128] = {
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f
	};
	char *ndata;
	uchar *line, *nline;
	int nw, nh, i, j, k, bpl, nbpl;

	nw = w >> 1;
	nh = h >> 1;

	bpl = (((w-1) >> 4) + 1) << 1;
	nbpl = (((nw-1) >> 4) + 1) << 1;
	ndata = (char *)malloc(nbpl *nh);

	line = (uchar *)data;
	nline = (uchar *)ndata;

	for (i=0; i < nh; i++) {
		if (i & 1)
			line += bpl;
		for (j=0; j < nbpl; j++) {
			k = j << 1;
			if (j & 1)
				nline[j] = shrink_tab[line[k] >> 1] | (shrink_tab[line[k|1] >> 1] << 4);
			else
				nline[j] = shrink_tab[line[k] & 0x7f] | (shrink_tab[line[k|1] & 0x7f] << 4);
		}
		line += bpl;
		if (!(i & 1))
			line += bpl;
		nline += nbpl;
	}

	return ndata;
}


char *rotatePage(int w, int h, char *data, bool lsb_first)
{
	char *ndata;
	uchar *line, *nline, *nline2, buff[8];
	int nw, nh, i, j, k, l, max_lines, bpl, nbpl;

	nw = h;
	nh = w;
	
	bpl = (((w-1) >> 4) + 1) << 1;
	nbpl = (((nw-1) >> 4) + 1) << 1;
	ndata = (char *)malloc(nbpl * nh);
	
	for (i = 0; i < w; i += 8) { // bpl
		line = (uchar *)(data + bpl * h + (i >> 3));
		nline = (uchar *)(ndata + nbpl * i);
		max_lines = nh - i;
		if (max_lines > 8)
			max_lines = 8;

		for (j = 0; j < nw; j += 8) { // nbpl
			nline2 = nline;
			for (k = 0; k < max_lines; k++) {
				line -= bpl;
				buff[k] = *line;
			}
			for (k = 0; k < max_lines; k++) {
				if (lsb_first) {
					for (l = 0; l < 8; l++) {
						*nline2 = (uchar)((*nline2 << 1) | (buff[l] & 1));
						buff[l] >>= 1;
					}
				}
				else {
					for (l = 0; l < 8; l++) {
						*nline2 = (uchar)((*nline2 >> 1) | (buff[l] & 128));
						buff[l] <<= 1;
					}
				}
				*nline2 = xchg_endian[*nline2];
				nline2 += nbpl;
			}
			nline++;
		}
	}

	return ndata;
}



FaxWidget::FaxWidget(QWidget *parent, const char *name) : QWidget(parent, name)
{
	int screen;

	scrollx = 0;
	scrolly = 0;

	display = x11Display();
	screen = x11Screen();

	lsb_first = (BitmapBitOrder(display) == LSBFirst);
	
	window = XCreateSimpleWindow(display, winId(), 0, 0, 1, 1, 0,
		 BlackPixel(display, screen),
		 WhitePixel(display, screen));
	gc = XCreateGC(display, window, 0L, (XGCValues *)0);
	XSetForeground(display, gc, BlackPixel(display, screen));
	XSetBackground(display, gc, WhitePixel(display, screen));
	XSetFunction(display, gc, GXcopy);
	XSelectInput(display, window, ButtonPressMask | ButtonReleaseMask | Button1MotionMask | ExposureMask);
	XFlush(display);

	move_cursor = XCreateFontCursor(display, XC_fleur);
	moving = false;

	image = 0;

	hscroll = new QScrollBar(QScrollBar::Horizontal, this, "prehscroll");
	CHECK_PTR(hscroll);
	hscroll->setFixedHeight(hscroll->sizeHint().height());
	vscroll = new QScrollBar(QScrollBar::Vertical, this, "prevscroll");
	CHECK_PTR(vscroll);
	vscroll->setFixedWidth(vscroll->sizeHint().width());

	connect(hscroll, SIGNAL(valueChanged(int)), this, SLOT(setXScroll(int)));
	connect(vscroll, SIGNAL(valueChanged(int)), this, SLOT(setYScroll(int)));
}


FaxWidget::~FaxWidget()
{
	deleteImage();
	XFreeCursor(display, move_cursor);
	XFreeGC(display, gc);
	XDestroyWindow(display, window);
}


void FaxWidget::setImage(int w, int h, char *data)
{
	deleteImage();

	im_width = w;
	im_height = h;

	if ((w) && (h)) {
		image = XCreateImage(display, DefaultVisual(display, x11Screen()),
			1, XYBitmap, 0, data, im_width, im_height, 16, 0);
		XMapRaised(display, window);
	} else {
		XUnmapWindow(display, window);
		XResizeWindow(display, window, 1, 1);
	}

	updateGeometry();

	repaint(false);
}


void FaxWidget::deleteImage()
{
	if (image) {
		image->data = 0;
		XDestroyImage(image);
		image = 0;
	}
}


void FaxWidget::updateGeometry()
{
	int w, h, range;
	bool need_hscroll, need_vscroll;

	w = width();
	h = height();

	if ((need_hscroll = (im_width > w)))
		h -= hscroll->height();
	if ((need_vscroll = (im_height > h))) {
		w -= vscroll->width();
		if (!need_hscroll && (im_width > w)) {
			need_hscroll = true;
			h -= hscroll->height();
		}
	}
	
	if (need_hscroll) {
		range = im_width - w;
		hscroll->setRange(0, range);
		if (hscroll->value() > range)
			hscroll->setValue(range);
		hscroll->setSteps(12, w);
		hscroll->setGeometry(0, h, w, hscroll->height());
		hscroll->show();
	} else {
		hscroll->hide();
		hscroll->setRange(0, 0);
		hscroll->setValue(0);
	}
	
	if (need_vscroll) {
		range = im_height - h;
		vscroll->setRange(0, range);
		if (vscroll->value() > range)
			vscroll->setValue(range);
		vscroll->setSteps(12, h);
		vscroll->setGeometry(w, 0, vscroll->width(), h);
		vscroll->show();
	} else {
		vscroll->hide();
		vscroll->setRange(0, 0);
		vscroll->setValue(0);
	}
}


bool FaxWidget::x11EventFilter(XEvent *ev)
{
	bool refresh = false;
	bool move_it = false;
	int mv_x = 0, mv_y = 0, wheel_steps = 0, wheel_state = 0;
	int x1 = INT_MAX, y1 = INT_MAX, x2 = 0, y2 = 0, xb, yb;

	if (ev->xany.window != window)
		return false;
	
	do {
		switch(ev->type) {
			case ButtonPress:
				if (ev->xbutton.button == Button1) {
					XDefineCursor(display, window, move_cursor);
					moving = true;
					move_x = ev->xbutton.x;
					move_y = ev->xbutton.y;
				}
				if ((ev->xbutton.button == Button4) || (ev->xbutton.button == Button5)) {
					if (ev->xbutton.state & Button1Mask)
						wheel_state |= Qt::LeftButton;
					if (ev->xbutton.state & Button2Mask)
						wheel_state |= Qt::MidButton;
					if (ev->xbutton.state & Button3Mask)
						wheel_state |= Qt::RightButton;
					if (ev->xbutton.state & ShiftMask)
						wheel_state |= Qt::ShiftButton;
					if (ev->xbutton.state & ControlMask)
						wheel_state |= Qt::ControlButton;
					if (ev->xbutton.state & Mod1Mask)
						wheel_state |= Qt::AltButton;
					if (ev->xbutton.button == Button4)
						wheel_steps++;
					else
						wheel_steps--;
				}
				break;
			case ButtonRelease:
				if (ev->xbutton.button == Button1) {
					XUndefineCursor(display, window);
					moving = false;
				}
				break;
			case MotionNotify:
				if (moving) {
					move_it = true;
					mv_x = ev->xmotion.x;
					mv_y = ev->xmotion.y;
				}
				break;
			case Expose:
				refresh = true;
				xb = ev->xexpose.x + ev->xexpose.width;
				yb = ev->xexpose.y + ev->xexpose.height;
				if (ev->xexpose.x < x1)
					x1 = ev->xexpose.x;
				if (ev->xexpose.y < y1)
					y1 = ev->xexpose.y;
				if (xb > x2)
					x2 = xb;
				if (yb > y2)
					y2 = yb;
			default:
				break;
		}
	} while (XCheckWindowEvent(display, window, ButtonPressMask | ButtonReleaseMask | Button1MotionMask | ExposureMask, ev));

	if (refresh && image)
		XPutImage(display, window, gc, image, scrollx + x1, scrolly + y1, x1, y1, x2-x1, y2-y1);

	if (move_it) {
		hscroll->setValue(hscroll->value() + move_x - mv_x);
		vscroll->setValue(vscroll->value() + move_y - mv_y);
		move_x = mv_x;
		move_y = mv_y;
	}

	if (wheel_steps) {
		QWheelEvent e(QPoint(0, 0), 120 * wheel_steps, wheel_state);
		KApplication::sendEvent(vscroll, &e);
	}

	return true;
}	


void FaxWidget::setXScroll(int x)
{
	scrollx = x;
	repaint(false);
}


void FaxWidget::setYScroll(int y)
{
	scrolly = y;
	repaint(false);
}


void FaxWidget::keyLeft()
{
	hscroll->setValue(hscroll->value() - hscroll->lineStep());
}


void FaxWidget::keyRight()
{
	hscroll->setValue(hscroll->value() + hscroll->lineStep());
}


void FaxWidget::keyUp()
{
	vscroll->setValue(vscroll->value() - vscroll->lineStep());
}


void FaxWidget::keyDown()
{
	vscroll->setValue(vscroll->value() + vscroll->lineStep());
}


void FaxWidget::keyPageUp()
{
	vscroll->setValue(vscroll->value() - vscroll->pageStep());
}


void FaxWidget::keyPageDown()
{
	vscroll->setValue(vscroll->value() + vscroll->pageStep());
}


void FaxWidget::keyHome()
{
	vscroll->setValue(vscroll->minValue());
}


void FaxWidget::keyEnd()
{
	vscroll->setValue(vscroll->maxValue());
}


void FaxWidget::paintEvent(QPaintEvent *)
{
	int w, h, x = 0, y = 0;

	w = width();
	if (vscroll->isVisible())
		w -= vscroll->width();

	h = height();
	if (hscroll->isVisible())
		h -= hscroll->height();
	
	if (w > im_width) {
		x = (w - im_width) >> 1;
		w = im_width;
	}
	if (h > im_height) {
		y = (h - im_height) >> 1;
		h = im_height;
	}
	if (image) {
		XMoveResizeWindow(display, window, x, y, w, h);
		XPutImage(display, window, gc, image, scrollx, scrolly, 0, 0, w, h);
	}
}


void FaxWidget::resizeEvent(QResizeEvent *)
{
	updateGeometry();
}


Preview::Preview(QWidget *parent, const char *name) : KMainWindow(parent, name, WType_TopLevel)
{
	int i;

	config = kapp->config();
	
	for (i=0; i < 3; i++)
		page_datas[i] = 0;
	page = pages = 0;
	page_width = page_height = 0;
	zoom_width = zoom_height = 0;
	zoom = 1;
	angle = 0;

	initMenuBar();
	initToolBar();
	initStatusBar();

	view = new FaxWidget(this, "faxwidget");
	CHECK_PTR(view);
	fw = view;
	setCentralWidget(view);
	
	initKeyAccel();

	readOptions();

	setCaption(i18n("Preview"));
}


Preview::~Preview()
{
	deletePages();
}


void Preview::showFax(const QString& name)
{
	slotStatusName("");
	slotStatusSender("");

	file.setName(name);
	if (!file.open()) {
		KMessageBox::sorry(this, i18n("Cannot open facsimile file."), i18n("File Error"));
		return;
	}
	pages = file.pages();
	slotStatusName(name);
	slotStatusSender(SENDER_ALIAS(file.sender()));
	file.close();

	m_name = name;
	
	loadPage(1);

	if (isVisible())
		raise();
	else
		show();
}


void Preview::enableCommand(int id_, bool enable)
{
	menuBar()->setItemEnabled(id_, enable);
	toolBar()->setItemEnabled(id_, enable);
}


void Preview::readOptions()
{
	config->setGroup("Preview Options");

	bool toolBarOn = config->readBoolEntry("Show ToolBar", true);
	settingsMenu->setItemChecked(ID_PRE_SETTINGS_TOOLBAR, toolBarOn);
	if (!toolBarOn)
		toolBar()->hide();

	bool statusBarOn = config->readBoolEntry("Show StatusBar", true);
	settingsMenu->setItemChecked(ID_PRE_SETTINGS_STATUSBAR, statusBarOn);
	if (!statusBarOn)
		statusBar()->hide();

	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));
	
	QSize geoSize(650, 400);
	resize(config->readSizeEntry("Geometry", &geoSize));
}


void Preview::saveOptions()
{
	config->setGroup("Preview Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());
}


void Preview::initMenuBar()
{
	fileMenu = new QPopupMenu(0, "prefilemenu");
	CHECK_PTR(fileMenu);
	fileMenu->insertItem(SmallIcon("fileprint"), i18n("&Print..."), ID_PRE_FILE_PRINT);
	fileMenu->insertItem(SmallIcon("filesave"), i18n("&Export..."), ID_PRE_FILE_EXPORT);
	fileMenu->insertItem(SmallIcon("mail_send"), i18n("&Mail..."), ID_PRE_FILE_MAIL);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("fileclose"), i18n("&Close"), ID_PRE_FILE_CLOSE);

	viewMenu = new QPopupMenu(0, "previewmenu");
	CHECK_PTR(viewMenu);
	viewMenu->insertItem(SmallIcon("viewmag+"), i18n("Zoom &In"), ID_PRE_VIEW_ZOOM_IN);
	viewMenu->insertItem(SmallIcon("viewmag-"), i18n("Zoom &Out"), ID_PRE_VIEW_ZOOM_OUT);
	viewMenu->insertItem(SmallIcon("rotate"), i18n("&Rotate"), ID_PRE_VIEW_ROTATE);

	goMenu = new QPopupMenu(0, "pregomenu");
	CHECK_PTR(goMenu);
	goMenu->insertItem(SmallIcon("back"), i18n("&Previous Page"), ID_PRE_GO_PREVIOUS);
	goMenu->insertItem(SmallIcon("forward"), i18n("&Next Page"), ID_PRE_GO_NEXT);
	goMenu->insertItem(SmallIcon("start"), i18n("&First Page"), ID_PRE_GO_FIRST);
	goMenu->insertItem(SmallIcon("finish"), i18n("&Last Page"), ID_PRE_GO_LAST);
	
	settingsMenu = new QPopupMenu(0, "presettingsmenu");
	CHECK_PTR(settingsMenu);
	settingsMenu->setCheckable(true);
	settingsMenu->insertItem(i18n("Show &Toolbar"), ID_PRE_SETTINGS_TOOLBAR);
	settingsMenu->setItemChecked(ID_PRE_SETTINGS_TOOLBAR, true);
	settingsMenu->insertItem(i18n("Show &Statusbar"), ID_PRE_SETTINGS_STATUSBAR);
	settingsMenu->setItemChecked(ID_PRE_SETTINGS_STATUSBAR, true);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(i18n("Save &Options"), ID_PRE_SETTINGS_SAVEOPTIONS);

	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&View"), viewMenu);
	menuBar()->insertItem(i18n("&Go"), goMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	
	connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(goMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(viewMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(settingsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
}


void Preview::initToolBar()
{
	toolBar()->insertButton(BarIcon("fileprint"), ID_PRE_FILE_PRINT, true, i18n("Print Facsimile"));
	toolBar()->insertButton(BarIcon("filesave"), ID_PRE_FILE_EXPORT, true, i18n("Export Facsimile"));
	toolBar()->insertButton(BarIcon("mail_send"), ID_PRE_FILE_MAIL, true, i18n("Mail Facsimile"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("viewmag+"), ID_PRE_VIEW_ZOOM_IN, true, i18n("Zoom In"));
	toolBar()->insertButton(BarIcon("viewmag-"), ID_PRE_VIEW_ZOOM_OUT, true, i18n("Zoom Out"));
	toolBar()->insertButton(BarIcon("rotate"), ID_PRE_VIEW_ROTATE, true, i18n("Rotate"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("back"), ID_PRE_GO_PREVIOUS, true, i18n("Previous Page"));
	toolBar()->insertButton(BarIcon("forward"), ID_PRE_GO_NEXT, true, i18n("Next Page"));
	toolBar()->insertButton(BarIcon("start"), ID_PRE_GO_FIRST, true, i18n("First Page"));
	toolBar()->insertButton(BarIcon("finish"), ID_PRE_GO_LAST, true, i18n("Last Page"));
	toolBar()->insertButton(BarIcon("fileclose"), ID_PRE_FILE_CLOSE, true, i18n("Close"));
	toolBar()->alignItemRight(ID_PRE_FILE_CLOSE, true);
	
	connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
}


void Preview::initStatusBar()
{
	statusBar()->insertFixedItem(" 00000000000000.FAX ", ID_PRE_STATUS_NAME);
	statusBar()->insertFixedItem(i18n(" 000000000, Normal "), ID_PRE_STATUS_SIZE);
	statusBar()->insertFixedItem(i18n(" Page 000 of 000 "), ID_PRE_STATUS_PAGE);
	statusBar()->insertFixedItem(" 000 ", ID_PRE_STATUS_ANGLE);
	statusBar()->insertItem("", ID_PRE_STATUS_SENDER, 1);
	statusBar()->changeItem("", ID_PRE_STATUS_NAME);
	statusBar()->changeItem("", ID_PRE_STATUS_SIZE);
	statusBar()->changeItem("", ID_PRE_STATUS_PAGE);
	statusBar()->changeItem("", ID_PRE_STATUS_ANGLE);
}


void Preview::initKeyAccel()
{
	keyAccel = new KAccel(this);
	CHECK_PTR(keyAccel);

	keyAccel->insertItem(i18n("Export File"), "ExportFile", CTRL+Key_E);
	keyAccel->insertItem(i18n("Mail File"), "MailFile", CTRL+Key_M);
	
	keyAccel->insertItem(i18n("Go to Previous"), "GoPrevious", CTRL+Key_PageUp);
	keyAccel->insertItem(i18n("Go to Next"), "GoNext", CTRL+Key_PageDown);
	keyAccel->insertItem(i18n("Go to First"), "GoFirst", CTRL+Key_Home);
	keyAccel->insertItem(i18n("Go to Last"), "GoLast", CTRL+Key_End);
	
	keyAccel->insertItem(i18n("Zoom In"), "ZoomIn", CTRL+Key_Plus);
	keyAccel->insertItem(i18n("Zoom Out"), "ZoomOut", CTRL+Key_Minus);
	keyAccel->insertItem(i18n("Rotate"), "Rotate", CTRL+Key_R);
	
	keyAccel->insertItem(i18n("Show Toolbar"), "ShowToolbar", CTRL+Key_T);
	keyAccel->insertItem(i18n("Show Statusbar"), "ShowStatusbar", CTRL+Key_S);
	
	keyAccel->insertItem(i18n("Scroll Left"), "ScrollLeft", Key_Left);
	keyAccel->insertItem(i18n("Scroll Right"), "ScrollRight", Key_Right);
	keyAccel->insertItem(i18n("Scroll Up"), "ScrollUp", Key_Up);
	keyAccel->insertItem(i18n("Scroll Down"), "ScrollDown", Key_Down);
	keyAccel->insertItem(i18n("Page Up"), "PageUp", Key_PageUp);
	keyAccel->insertItem(i18n("Page Down"), "PageDown", Key_PageDown);
	keyAccel->insertItem(i18n("Home"), "Home", Key_Home);
	keyAccel->insertItem(i18n("End"), "End", Key_End);

	keyAccel->readSettings();

	keyAccel->connectItem(KStdAccel::Print, this, SLOT(slotFilePrint()));
	keyAccel->connectItem("ExportFile", this, SLOT(slotFileExport()));
	keyAccel->connectItem("MailFile", this, SLOT(slotFileMail()));
	keyAccel->connectItem(KStdAccel::Close, this, SLOT(hide()));
	
	keyAccel->connectItem("ZoomIn", this, SLOT(slotViewZoomIn()));
	keyAccel->connectItem("ZoomOut", this, SLOT(slotViewZoomOut()));
	keyAccel->connectItem("Rotate", this, SLOT(slotViewRotate()));
	
	keyAccel->connectItem("GoPrevious", this, SLOT(slotGoPreviousPage()));
	keyAccel->connectItem("GoNext", this, SLOT(slotGoNextPage()));
	keyAccel->connectItem("GoFirst", this, SLOT(slotGoFirstPage()));
	keyAccel->connectItem("GoLast", this, SLOT(slotGoLastPage()));
	
	keyAccel->connectItem("ShowToolbar", this, SLOT(slotSettingsShowToolbar()));
	keyAccel->connectItem("ShowStatusbar", this, SLOT(slotSettingsShowStatusbar()));
	
	keyAccel->connectItem(KStdAccel::Help, this, SLOT(appHelpActivated()));

	keyAccel->connectItem("ScrollLeft", view, SLOT(keyLeft()));
	keyAccel->connectItem("ScrollRight", view, SLOT(keyRight()));
	keyAccel->connectItem("ScrollUp", view, SLOT(keyUp()));
	keyAccel->connectItem("ScrollDown", view, SLOT(keyDown()));
	keyAccel->connectItem("PageUp", view, SLOT(keyPageUp()));
	keyAccel->connectItem("PageDown", view, SLOT(keyPageDown()));
	keyAccel->connectItem("Home", view, SLOT(keyHome()));
	keyAccel->connectItem("End", view, SLOT(keyEnd()));
	
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_PRINT, KStdAccel::Print);
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_EXPORT, "ExportFile");
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_MAIL, "MailFile");
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_CLOSE, KStdAccel::Close);
	
	keyAccel->changeMenuAccel(viewMenu, ID_PRE_VIEW_ZOOM_IN, "ZoomIn");
	keyAccel->changeMenuAccel(viewMenu, ID_PRE_VIEW_ZOOM_OUT, "ZoomOut");
	keyAccel->changeMenuAccel(viewMenu, ID_PRE_VIEW_ROTATE, "Rotate");

	keyAccel->changeMenuAccel(goMenu, ID_PRE_GO_PREVIOUS, "GoPrevious");
	keyAccel->changeMenuAccel(goMenu, ID_PRE_GO_NEXT, "GoNext");
	keyAccel->changeMenuAccel(goMenu, ID_PRE_GO_FIRST, "GoFirst");
	keyAccel->changeMenuAccel(goMenu, ID_PRE_GO_LAST, "GoLast");
	
	keyAccel->changeMenuAccel(settingsMenu, ID_PRE_SETTINGS_TOOLBAR, "ShowToolbar");
	keyAccel->changeMenuAccel(settingsMenu, ID_PRE_SETTINGS_STATUSBAR, "ShowStatusbar");
}


void Preview::commandCallback(int id_)
{
	switch (id_) {
		case ID_PRE_FILE_PRINT:
			slotFilePrint();
			break;
		case ID_PRE_FILE_EXPORT:
			slotFileExport();
			break;
		case ID_PRE_FILE_MAIL:
			slotFileMail();
			break;
		case ID_PRE_FILE_CLOSE:
			hide();
			break;
		case ID_PRE_VIEW_ZOOM_IN:
			slotViewZoomIn();
			break;
		case ID_PRE_VIEW_ZOOM_OUT:
			slotViewZoomOut();
			break;
		case ID_PRE_VIEW_ROTATE:
			slotViewRotate();
			break;
		case ID_PRE_GO_PREVIOUS:
			slotGoPreviousPage();
			break;
		case ID_PRE_GO_NEXT:
			slotGoNextPage();
			break;
		case ID_PRE_GO_FIRST:
			slotGoFirstPage();
			break;
		case ID_PRE_GO_LAST:
			slotGoLastPage();
			break;
		case ID_PRE_SETTINGS_TOOLBAR:
			slotSettingsShowToolbar();
			break;
		case ID_PRE_SETTINGS_STATUSBAR:
			slotSettingsShowStatusbar();
			break;
		case ID_PRE_SETTINGS_SAVEOPTIONS:
			saveOptions();
			break;
		default:
			break;
	}
}


void Preview::slotFilePrint()
{
	PrintDlg printdlg(this, "printdlg");
	printdlg.printFax(m_name);
	printdlg.exec();
}


void Preview::slotFileExport()
{
	ExportDlg exportdlg(this, "exportdlg");
	exportdlg.exportFax(m_name);
	exportdlg.exec();
}


void Preview::slotFileMail()
{
	MailDlg maildlg(this, "maildlg");
	maildlg.mailFax(m_name);
	maildlg.exec();
}


void Preview::slotViewZoomIn()
{
	if (zoom) {
		zoom--;
		showPage();
	}
}


void Preview::slotViewZoomOut()
{
	if (zoom < 2) {
		zoom++;
		showPage();
	}
}


void Preview::slotViewRotate()
{
	char *new_data;
	int i, w, h;

	if (angle & 1) {
		w = page_height;
		h = page_width;
	}
	else {
		w = page_width;
		h = page_height;
	}
	
	kapp->setOverrideCursor(waitCursor);

	new_data = rotatePage(w, h, page_datas[0], view->lsbFirst());
	
	for (i=0; i < 3; i++)
		free(page_datas[i]);
	
	page_datas[0] = new_data;
	page_datas[1] = shrinkPage(h, w, page_datas[0]);
	page_datas[2] = shrinkPage(h >> 1, w >> 1, page_datas[1]);

	kapp->restoreOverrideCursor();

	angle = (angle + 1) & 3;
	slotStatusAngle();

	showPage();
}


void Preview::slotGoPreviousPage()
{
	if (page > 1)
		loadPage(page-1);
}


void Preview::slotGoNextPage()
{
	if (page < pages)
		loadPage(page+1);
}


void Preview::slotGoFirstPage()
{
	if (page > 1)
		loadPage(1);
}


void Preview::slotGoLastPage()
{
	if (page < pages)
		loadPage(pages);
}


void Preview::slotSettingsShowToolbar()
{
	bool toolBarOn = !settingsMenu->isItemChecked(ID_PRE_SETTINGS_TOOLBAR);

	menuBar()->setItemChecked(ID_PRE_SETTINGS_TOOLBAR, toolBarOn);
	if (toolBarOn)
		toolBar()->show();
	else
		toolBar()->hide();
}


void Preview::slotSettingsShowStatusbar()
{
	bool statusBarOn = !settingsMenu->isItemChecked(ID_PRE_SETTINGS_STATUSBAR);

	menuBar()->setItemChecked(ID_PRE_SETTINGS_STATUSBAR, statusBarOn);
	if (statusBarOn)
		statusBar()->show();
	else
		statusBar()->hide();
}


void Preview::slotStatusName(const QString& text)
{
	statusBar()->changeItem(text, ID_PRE_STATUS_NAME);
}


void Preview::slotStatusSize(int width_, int height_, bool normal)
{
	if (width_)
		statusBar()->changeItem(QString("%1%2, %3").arg(width_).arg(height_).arg(normal ? i18n("Normal") : i18n("Fine")), ID_PRE_STATUS_SIZE);
	else
		statusBar()->changeItem("", ID_PRE_STATUS_SIZE);
}


void Preview::slotStatusPage(int page_, int pages_)
{
	if (page_)
		statusBar()->changeItem(i18n("Page %1 of %2").arg(page_).arg(pages_), ID_PRE_STATUS_PAGE);
	else
		statusBar()->changeItem("", ID_PRE_STATUS_PAGE);
}


void Preview::slotStatusAngle()
{
	static const int angles[4] = { 0, 90, 180, 270 };

	statusBar()->changeItem(QString("%1").arg(angles[angle]), ID_PRE_STATUS_ANGLE);
}


void Preview::slotStatusSender(const QString& text)
{
	statusBar()->changeItem(text, ID_PRE_STATUS_SENDER);
}


void Preview::deletePages()
{
	int i;

	page_width = page_height = 0;

	for (i=0; i < 3; i++)
		if (page_datas[i]) {
			free(page_datas[i]);
			page_datas[i] = 0;
		}
	view->deleteImage();
}


void Preview::showPage()
{
	enableCommand(ID_PRE_VIEW_ZOOM_IN, zoom > 0);
	enableCommand(ID_PRE_VIEW_ZOOM_OUT, zoom < 2);
	
	zoom_width = page_width >> zoom;
	zoom_height = page_height >> zoom;
	
	if (angle & 1)
		view->setImage(zoom_height, zoom_width, page_datas[zoom]);
	else
		view->setImage(zoom_width, zoom_height, page_datas[zoom]);
}


void Preview::loadPage(int p)
{
	page_info_t info;
	char *pd;
	int i, bpl;
	bool lsb_first = view->lsbFirst();

	angle = 0;

	slotStatusSize(0, 0, true);
	slotStatusPage(0, 0);
	slotStatusAngle();

	deletePages();

	page = p;

	enableCommand(ID_PRE_GO_PREVIOUS, page != 1);
	enableCommand(ID_PRE_GO_NEXT, page != pages);
	enableCommand(ID_PRE_GO_FIRST, page != 1);
	enableCommand(ID_PRE_GO_LAST, page != pages);

	if (!file.open()) {
		KMessageBox::sorry(this, i18n("Cannot open facsimile file."), i18n("File Error"));
		showPage();
		return;
	}
	if (!file.readPageInfo(p, &info)) {
		KMessageBox::sorry(this, i18n("Cannot read facsimile page info."), i18n("File Error"));
		file.close();
		showPage();
		return;
	}

	page_width = info.width;
	page_height = file.pageHeight(page);

	slotStatusSize(page_width, page_height, (info.lpi == 98));
	slotStatusPage(page, pages);
	
	kapp->setOverrideCursor(waitCursor);

	bpl = (((page_width-1) >> 4) + 1) << 1;
	if (info.lpi == 98)
		page_datas[0] = (char *)malloc(bpl * page_height * 2);
	else
		page_datas[0] = (char *)malloc(bpl * page_height);
	pd = page_datas[0];

	file.gotoPage(page);
	if (info.lpi == 98) {
		for (i=0; i < page_height; i++) {
			file.readImgLine((uchar *)pd, lsb_first);
			memcpy((void *)(pd + bpl), (void *)pd, bpl);
			pd += (bpl << 1);
		}
		page_height <<= 1;
	} else {
		for (i=0; i < page_height; i++) {
			file.readImgLine((uchar *)pd, lsb_first);
			pd += bpl;
		}
	}
	file.close();

	page_datas[1] = shrinkPage(page_width, page_height, page_datas[0]);
	page_datas[2] = shrinkPage(page_width >> 1, page_height >> 1, page_datas[1]);

	kapp->restoreOverrideCursor();

	showPage();
}
