/**
 * hostdata.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "hostdata.h"
#include "token.h"

#include <kurl.h>
#include <kdebug.h>

namespace KCfgCreator
{

HostData::HostData( const KURL& url )
    : m_url( url ), m_unfinishedCount( 0 ),
      m_numberOfWaiting( 0 ), m_done( false )
{
    m_kcfgFiles.setAutoDelete( true );
    m_kconfigFiles.setAutoDelete( true );
    kdDebug()<<"XXXXXXXXXXXXXXXXXXX Creating HostData with url == "
             <<url.prettyURL()<<endl;
}

HostData::~HostData()
{
    kdDebug()<<"XXXXXXXXXXXXXXXXXXXX Destroying HostData with url == "
             << url().prettyURL()<<endl;
    Q_ASSERT( m_kcfgFiles.isEmpty() );
    Q_ASSERT( m_kconfigFiles.isEmpty() );
}

bool
HostData::isDone() const
{
    return m_done;
}

void
HostData::setDone( bool d )
{
    m_done = d;
}

KURL
HostData::url() const
{
    return m_url;
}

void
HostData::add( FileData* data )
{
    if ( data->isKCfg() )
        m_kcfgFiles.append( data );
    else
        m_kconfigFiles.append( data );
    if ( !data->isFinished() )
        ++m_unfinishedCount;
}

void
HostData::addList( const QPtrList<FileData>& l )
{
    QPtrListIterator<FileData> itr( l );
    FileData *data;

    while ( (data = itr.current()) ) {
        data->setParent( this );
        add( data );
        ++itr;
    }

    checkFinished();
}

void
HostData::remove( FileData* data )
{
    QPtrList<FileData> &files = data->isKCfg() ? m_kcfgFiles : m_kconfigFiles;

    files.setAutoDelete( false );
    files.remove( data );
    files.setAutoDelete( true );

    if ( !data->isFinished() )
    {
        --m_unfinishedCount;
        checkFinished();
    }
}

void
HostData::fileFinished()
{
    --m_unfinishedCount;
    checkFinished();
}

QPtrList<FileData>&
HostData::kcfgFiles()
{
    return m_kcfgFiles;
}

QPtrList<FileData>&
HostData::kconfigFiles()
{
    return m_kconfigFiles;
}

void
HostData::addNumberOfWaitingUrls( int i )
{
    m_numberOfWaiting += i;
}

void
HostData::decrementNumberOfWaitingUrls()
{
    --m_numberOfWaiting;
    checkFinished();
}

bool
HostData::waitingForUrls() const
{
    return m_numberOfWaiting;
}

void
HostData::addKCfgPath( const KURL& url )
{
    m_kcfgPaths.append( url );
}

void
HostData::addKConfigPath( const KURL& url )
{
    m_kconfigPaths.append( url );
}

bool
HostData::isKCfgPath( const KURL& url ) const
{
    return ( m_kcfgPaths.find( url ) != m_kcfgPaths.end() );
}

bool
HostData::isKConfigPath( const KURL& url ) const
{
    return ( m_kconfigPaths.find( url ) != m_kconfigPaths.end() );
}

void
HostData::checkFinished()
{
    if ( m_unfinishedCount == 0 )
        emit done( this );
}

FileData*
HostData::popKCfg()
{
    FileData* fdata = m_kcfgFiles.take( 0 );
    return fdata;
}

FileData*
HostData::popKConfig()
{
    FileData* fdata = m_kconfigFiles.take( 0 );
    return fdata;
}

FileData::FileData( HostData* parent, const KURL& url, bool isKCfg )
    : m_parent( parent ), m_url( url ),
      m_finished( false ), m_kcfg( isKCfg )
{
    m_parent->add( this );
}

FileData::~FileData()
{
    if ( m_parent )
        m_parent->remove( this );
}

void
FileData::setParent( HostData* parent )
{
    m_parent = parent;
}

void
FileData::setReadOnly( bool r )
{
    m_readOnly = r;
}

bool
FileData::isFinished() const
{
    return m_finished;
}

QString
FileData::host() const
{
    return m_url.host();
}

QString
FileData::fileName() const
{
    return m_url.fileName();
}

KURL
FileData::url() const
{
    return m_url;
}

void
FileData::finished()
{
    m_finished = true;
    if ( m_parent )
        m_parent->fileFinished();
}

void
FileData::error()
{
    m_finished = true;
    if ( m_parent )
        m_parent->remove( this );
    delete this;
}

bool
FileData::isKCfg() const
{
    return m_kcfg;
}

}

#include "hostdata.moc"
