/**
 * kcfgcoptions.cpp
 *
 * Copyright (C)  2003  Zack Rusin <zack@kde.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
#include "kcfgcoptions.h"

#include "items.h"

#include "applicationtoken.h"
#include "kcfgparser.h"
#include "kconfigparser.h"

#include <klistview.h>
#include <kconfig.h>
#include <kstaticdeleter.h>
#include <kurl.h>
#include <kdebug.h>

#include <qdom.h>
#include <qfile.h>
#include <qdir.h>

using namespace KCfgCreator;
KCfgcOptions* KCfgcOptions::s_self = 0;
static KStaticDeleter<KCfgcOptions> sd;

KCfgcOptions*
KCfgcOptions::self()
{
    if ( !s_self )
        sd.setObject( s_self, new KCfgcOptions() );
    return s_self;
}

KCfgcOptions::KCfgcOptions()
    : m_lv( 0 ), m_singleton( false ),
      m_mutators( false ), m_itemAccessors( false ),
      m_userTexts( false ), m_globalEnums( false )
{
}

void
KCfgcOptions::setListView( KListView* lv )
{
    m_lv = lv;
}

void
KCfgcOptions::kcfgcFromFile( const QString& file )
{
    KConfig config( file );

    QString kfile = config.readEntry( "File", m_kcfgFile );
    if ( kfile.isEmpty() ) {
        kfile = file.mid( 0, file.findRev( '.' ) );
        kdDebug()<<"Kfile is "<<kfile<<endl;
        kfile += ".kcfg";
    }
    if ( m_kcfgFile != kfile && QFile::exists( kfile ) ) {
        m_kcfgFile = kfile;
        kcfgFromFile( m_kcfgFile );
    }

    m_namespace = config.readEntry( "NameSpace", QString::null );
    m_className = config.readEntry( "ClassName", QString::null );
    m_inherits = config.readEntry( "Inherits", "KConfigSkeleton" );
    m_singleton = config.readBoolEntry( "Singleton", false );
    m_memberVariables = config.readEntry( "MemberVariables", "private" );
    m_includeFiles = config.readListEntry( "IncludeFiles" );
    m_mutators = config.readBoolEntry( "Mutators", false );
    m_itemAccessors = config.readBoolEntry( "ItemAccessors", false );
    m_userTexts = config.readBoolEntry( "SetUserTexts", false );
    m_globalEnums = config.readBoolEntry( "GlobalEnums", false );
}

void
KCfgcOptions::fromRcFile( const QString& fileName )
{
    m_fileName        = fileName;
    m_namespace       = QString::null;
    m_className       = "Settings";
    m_inherits        = "KConfigSkeleton";
    m_singleton       = false;
    m_memberVariables = "private";
    m_includeFiles    = QStringList();
    m_mutators        = false;
    m_itemAccessors   = false;
    m_userTexts       = false;
    m_globalEnums     = false;


    KURL url; url.setPath( fileName );
    KConfigParser parser( 0 );
    ApplicationToken* app = parser.parseFileUrl( url, 0 );

    if ( app ) {
        QListViewItem* item = m_lv->firstChild();
        if ( item ) {
            m_lv->takeItem( item );
            delete item;
        }

        new AppItem( m_lv, app );
    }
}

void
KCfgcOptions::kcfgcToFile( const QString& file )
{
    kdDebug()<<"Saving kcfgc to "<<file<<endl;
    KConfig config( file );

    config.writeEntry( "File", m_kcfgFile );
    if ( !m_namespace.isEmpty() )
        config.writeEntry( "NameSpace", m_namespace );

    if ( !m_className.isEmpty() )
        config.writeEntry( "ClassName", m_className );

    if ( !m_inherits.isEmpty() )
        config.writeEntry( "Inherits", m_inherits );

    config.writeEntry( "Singleton", m_singleton );

    if ( !m_memberVariables.isEmpty() )
        config.writeEntry( "MemberVariables", m_memberVariables );

    if ( !m_includeFiles.empty() )
        config.writeEntry( "IncludeFiles", m_includeFiles );

    config.writeEntry( "Mutators", m_mutators );
    config.writeEntry( "ItemAccessors", m_itemAccessors );
    config.writeEntry( "SetUserTexts", m_userTexts );
    config.writeEntry( "GlobalEnums", m_globalEnums );
}

void
KCfgcOptions::kcfgFromFile( const QString& fileName )
{
    if ( m_kcfgFile.isEmpty() )
        m_kcfgFile = fileName;

    QDomDocument doc( "kcfg" );
    QFile file( fileName );
    if ( !file.open( IO_ReadOnly ) ) {
        return;
    }

    if ( !doc.setContent( &file ) ) {
        kdDebug()<<"Didn't set the content"<<endl;
        file.close();
        return;
    }
    file.close();

    QDomElement docElem = doc.documentElement();

    KURL url; url.setPath( fileName );
    KCfgParser parser( 0 );
    ApplicationToken* app = parser.parseFile( url, docElem,
                                              0, 0 );

    if ( app ) {
        QListViewItem* item = m_lv->firstChild();
        if ( item ) {
            m_lv->takeItem( item );
            delete item;
        }

        new AppItem( m_lv, app );
    }
}


void
KCfgcOptions::save()
{
    QString fileName = m_fileName;
    if ( fileName.isEmpty() ) {
        if ( !m_namespace.isEmpty() )
            fileName = QString( "%1_%2" )
                       .arg( m_namespace.lower() )
                       .arg( m_className.lower() );
        else
            fileName = m_className.lower();
        fileName = QString( "%1/%2.kcfgc" )
                   .arg( QDir::currentDirPath() )
                   .arg( fileName );
    }

    if ( !fileName.isEmpty() ) {
        kcfgcToFile( fileName );
    } else {
        kdDebug()<<"Skipping writting of the kcfgc file"<<endl;
    }
}

QString
KCfgcOptions::kcfgFile() const
{
    return m_kcfgFile;
}

void
KCfgcOptions::setKCfgFile( const QString& f )
{
    m_kcfgFile = f;
}

QString
KCfgcOptions::configName() const
{
    AppItem *item = dynamic_cast<AppItem*>( m_lv->firstChild() );
    if ( item )
        return item->name();
    return QString::null;
}

void
KCfgcOptions::setConfigName( const QString& name )
{
    AppItem *item = dynamic_cast<AppItem*>( m_lv->firstChild() );
    if ( item ) {
        item->setName( name );
    } else {
        new AppItem( m_lv, name );
    }
}

QString
KCfgcOptions::namespaceName() const
{
    return m_namespace;
}

void
KCfgcOptions::setNamespaceName( const QString& n )
{
    m_namespace = n;
}

QString
KCfgcOptions::className() const
{
    return m_className;
}

void
KCfgcOptions::setClassName( const QString& c )
{
    m_className = c;
}

QString
KCfgcOptions::inherits() const
{
    return m_inherits;
}

void
KCfgcOptions::setInherits( const QString& i )
{
    m_inherits = i;
}

bool
KCfgcOptions::singleton() const
{
    return m_singleton;
}

void
KCfgcOptions::setSingleton( bool b )
{
    m_singleton = b;
}

QString
KCfgcOptions::memberVariables() const
{
    return m_memberVariables;
}

void
KCfgcOptions::setMemberVariables( const QString& m )
{
    m_memberVariables = m;
}

QStringList
KCfgcOptions::includeFiles() const
{
    return m_includeFiles;
}

void
KCfgcOptions::setIncludeFiles( const QStringList& i )
{
    m_includeFiles = i;
}

bool
KCfgcOptions::mutators() const
{
    return m_mutators;
}

void
KCfgcOptions::setMutators( bool m )
{
    m_mutators = m;
}

bool
KCfgcOptions::itemAccessors() const
{
    return m_itemAccessors;
}

void
KCfgcOptions::setItemAccessors( bool i )
{
    m_itemAccessors = i;
}

bool
KCfgcOptions::userTexts() const
{
    return m_userTexts;
}

void
KCfgcOptions::setUserText( bool u )
{
    m_userTexts = u;
}

bool
KCfgcOptions::globalEnums() const
{
    return m_globalEnums;
}

void
KCfgcOptions::setGlobalEnums( bool g)
{
    m_globalEnums = g;
}
