/***************************************************************************
                          filemanager.cpp  -  description
                             -------------------
    begin                : Son Apr 7 2002
    copyright            : (C) 2002 by Martin Habbecke
    email                : M.Habbecke@gmx.de
    version              : $Revision: 1.4 $
    date                 : $Date: 2002/04/13 18:26:56 $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <iostream.h>
#include <stdio.h>

#include <qdom.h>
#include <qfile.h>
#include <qdatetime.h>

#include <kmessagebox.h>

#include "dive.h"
#include "logbook.h"
#include "filemanager.h"


FileManager::FileManager()
{
}


FileManager::~FileManager()
{
}



bool FileManager::writeLogBook( const QString &filename, LogBook *logBook )
{
    cout << "FileManager::writeLogBook" << endl;

    cout << "name: " << logBook->name() << endl;

    QDomDocument doc( "LOGBOOKDATA" );
    QDomElement root = doc.createElement( "logbook" );
    root.setAttribute( "name", logBook->name() );
    doc.appendChild( root );

    QDomElement diveElement, element;
    QDomText domText;
    QString txt;

    for( Dive *dive = logBook->first();
	 dive;
	 dive = logBook->next() )
    {
	// create the dive element with number as attribute
	diveElement = doc.createElement( "dive" );
	diveElement.setAttribute( "number", dive->number() );

	root.appendChild( diveElement );
	
	// create the "date" element for the current dive
	element = doc.createElement( "date" );
	diveElement.appendChild( element );

	domText = doc.createTextNode( dive->dateTime().date().toString( "dd.MM.yyyy" ) );
	element.appendChild( domText );

	// create the "time" element for the current dive
	element = doc.createElement( "time" );
	diveElement.appendChild( element );
	
	domText = doc.createTextNode( dive->dateTime().time().toString( "hh:mm" ) );
	element.appendChild( domText );

	// create the "duration" element for the current dive
	element = doc.createElement( "duration" );
	diveElement.appendChild( element );

	txt.sprintf( "%d", dive->duration() );
	domText = doc.createTextNode( txt );
	element.appendChild( domText );

	// create the "depth" element for the current dive
	element = doc.createElement( "depth" );
	diveElement.appendChild( element );
	
	txt.sprintf( "%d", dive->depth() );
	domText = doc.createTextNode( txt );
	element.appendChild( domText );

	// create the "place" element for the current dive
	element = doc.createElement( "place" );
	diveElement.appendChild( element );
	
	domText = doc.createTextNode( dive->place() );
	element.appendChild( domText );

    }

    QString str = doc.toString();

    QFile file( filename );
    if( file.open( IO_WriteOnly ) )
    {
	file.writeBlock( str.ascii(), str.length() );
	file.close();
    }
    else
    {
	str.sprintf( "Could not open file \"%s\" for writing.", filename.latin1() );
	KMessageBox::sorry( 0, str );
	return false;
    }

    return true;
}


bool FileManager::readLogBook( const QString &filename, LogBook *logBook )
{
    QDomDocument doc;
    QString error, str;
    int line, column;

    QFile file( filename );
    if( file.open( IO_ReadOnly ) )
    {
	if( doc.setContent( &file, false, &error, &line, &column ) )
	{
	    file.close();

	    // check if the doctype is "LOGBOOKDATA"
	    if( doc.doctype().name() != "LOGBOOKDATA" )
	    {
		str.sprintf( "Error: File has wrong DOCTYPE \"%s\" instead of \"LOGBOOKDATA\"",
			     doc.doctype().name().latin1() );
		KMessageBox::sorry( 0, str );
		return false;
	    }
	    
	    // check if the root-tag is "logbook"
	    QDomElement root = doc.documentElement();
	    if( root.tagName() != "logbook" )
	    {
		str.sprintf( "Error: File has bad root-tag \"%s\" instead of \"logbook\"",
			     root.tagName().latin1() );
		KMessageBox::sorry( 0, str );
		return false;
	    }
	    // read out the name of the logbook
	    QDomNode emptyNode;
	    QDomNode item = root.attributes().namedItem( "name" );
	    if( item == emptyNode )
	    {
		logBook->setName( "unnamed" );
	    }
	    else
	    {
		logBook->setName( item.nodeValue() );
	    }


	    // get a list of all "dive"-nodes
	    Dive *newDive;
	    QDomNodeList diveList;
	    QDomNode dive;
	    diveList = root.elementsByTagName( "dive" );
	    int hlp, number;
	    bool ok;
	    int d, m, y, h, min;


	    for( unsigned int i = 0; i < diveList.count(); i++ )
	    {
		newDive = new Dive;

		dive = diveList.item( i );

		// read out the number of the dive
		item = dive.attributes().namedItem( "number" );
		hlp = item.nodeValue().toInt( &ok );
		if( !ok )
		{
		    str.sprintf( "Error while parsing dive number" );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;	    
		}
		newDive->setNumber( hlp );
		number = hlp;


		// read out the date of the dive
		item = dive.namedItem( "date" );
		if( item == emptyNode )
		{
		    str.sprintf( "Error while parsing input-file:  date-tag is missing in dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;
		}
		if( sscanf( item.firstChild().nodeValue().latin1(), "%d.%d.%d", &d, &m, &y ) != 3 )
		{
		    str.sprintf( "Error while parsing the date of dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;	
		}


		// read out the time of the dive
		item = dive.namedItem( "time" );
		if( item == emptyNode )
		{
		    str.sprintf( "Error while parsing input-file:  time-tag is missing in dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;
		}
		if( sscanf( item.firstChild().nodeValue().latin1(), "%d:%d", &h, &min ) != 2 )
		{
		    str.sprintf( "Error while parsing the time of dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;	
		}
		newDive->setDateTime( QDateTime( QDate( y, m, d ), QTime( h, min ) ) );


		// read out the duration of the dive
		item = dive.namedItem( "duration" );
		if( item == emptyNode )
		{
		    str.sprintf( "Error while parsing input-file:  duration-tag is missing in dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;
		}
		if( sscanf( item.firstChild().nodeValue().latin1(), "%d", &d ) != 1 )
		{
		    str.sprintf( "Error while parsing the duration of dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;	
		}
		newDive->setDuration( d );


		// read out the depth of the dive
		item = dive.namedItem( "depth" );
		if( item == emptyNode )
		{
		    str.sprintf( "Error while parsing input-file:  depth-tag is missing in dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;
		}
		if( item == QDomNode() || sscanf( item.firstChild().nodeValue().latin1(), "%d", &d ) != 1 )
		{
		    str.sprintf( "Error while parsing the depth of dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;	
		}
		newDive->setDepth( d );


		// read out the place of the dive
		item = dive.namedItem( "place" );
		if( item == emptyNode )
		{
		    str.sprintf( "Error while parsing input-file:  place-tag is missing in dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;
		}
		if( item == QDomNode() )
		{
		    str.sprintf( "Error while parsing the place of dive %d", number );
		    KMessageBox::sorry( 0, str );
		    delete newDive;
		    return false;	
		}
		newDive->setPlace( item.firstChild().nodeValue().latin1() );


		logBook->append( newDive );
	    }
	    
	}
	else
	{
	    str.sprintf( "Error while parsing \"%s\" in line %d, column %d:\n%s",
			 filename.latin1(), line, column, error.latin1() );
	    KMessageBox::sorry( 0, str );
	    file.close();
	    return false;
	}
    }
    else
    {
	str.sprintf( "Could not open file \"%s\" for reading.", filename.latin1() );
	KMessageBox::sorry( 0, str );
	return false;
    }

    return true;
}
