/***************************************************************************
                          interfacealadin.cpp  -  description
                             -------------------
    begin                : Son Apr 7 2002
    copyright            : (C) 2002 by Martin Habbecke
    email                : M.Habbecke@gmx.de
    version              : $Revision: 1.3 $
    date                 : $Date: 2002/04/12 17:07:47 $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <iostream.h>
#include <iomanip.h>

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <termios.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <errno.h>
#include <string.h>

#include <kapplication.h>

#include "interfacealadin.h"


InterfaceAladin::InterfaceAladin()
    : InterfaceBase()
{
}


InterfaceAladin::~InterfaceAladin()
{
}


void InterfaceAladin::slotStopReading( void )
{
    _keepReading = false;
}




/*! All 2050 bytes of data are read from an aladin computer. This is done
  in non-blocking mode so that Qt-Events can be processed while waiting 
  for the user to switch into the logbook of his aladin.

  \param device Device to read from, something like "/dev/ttyS0"
  \return Result of the readout
*/
InterfaceBase::retVal InterfaceAladin::readData( const QString &device )
{
    struct termios term;

    // clear struct
    bzero( &term, sizeof( struct termios ) );

    // set attributes in term struct
    term.c_oflag = 0;
    term.c_iflag = 0;
    term.c_lflag = 0;
    term.c_cflag = CS8 | CLOCAL | CREAD;
    term.c_cc[VMIN]  = 1;
    term.c_cc[VTIME] = 0;
    
    // set input baud-rate
    cfsetispeed( &term, B19200 );

    // open device
    int fd = open( device.ascii(), O_RDONLY | O_NOCTTY | O_NDELAY );
    if( fd == -1 )
    {
	return ErrorOpen;
    }

    // set of device
    if( tcsetattr(fd, TCSANOW, &term) )
    {
	return ErrorIoctl;
    }

    // set DTR, clear RTS
    int status = TIOCM_RTS;

    if( ioctl( fd, TIOCMBIC, &status ) )
    {
	return ErrorIoctl;
    }


    // Read data from device
    _keepReading = true;


    // Read "UUU " as start sequence
    int anz;
    int num = 0;
    while( _keepReading )
    {
	anz = read( fd, _data + num, 1 );

	if( anz == -1 )
	{
	    if(  errno == EAGAIN )
	    {
		kapp->processEvents( 100 );
		continue;
	    }
	    else
	    {
		return ErrorRead;
	    }
	}

	if( num >= 0 && num <= 2 && *(_data + num) == 'U' )
	    num++;
	else if( num == 3 && *(_data + num) == 0 )
	    break;
	else
	    num = 0;
    }


    // Successfully read header, now read data
    num = 2046;
    while( _keepReading && num > 0 )
    {
	anz = read( fd, _data + 2050 - num, num );
	if( anz >= 0 )
	{
	    num -= anz;
	    emit( signalProgress( 2050 - num ) );
	}
	else
	{
	    if( errno == EAGAIN )
		continue;
	    else
		return ErrorRead;
	}
    }

    close( fd );

    // has the reading been aborted ?
    if( !_keepReading )
	return Aborted;


    // swap all bytes
    for( int i = 0; i < 2050; i++ )
	swapBits( _data + i );

    // calculate and check the checksum
    int checksum = 0;
    for( int i = 0; i < 2048; i++ )
	checksum += _data[ i ];
    checksum += 0xaa + 0xaa + 0xaa;
    
    if( (checksum & _data[ 2048 ]) &&
	((checksum >> 8) & _data[ 2049 ]) )
    {
	// Checksum correct
	return Ok;
    }
    else
    {
	// Bad checksum
	return ErrorAgain;
    }
}


/*! The number of dives is calculated by the total number of dives
  and the index of the last dive. If these two are the same the computer
  is rather new and not the whole ringbuffer is filled with logbooks.
  So the "index" of the latest dive is equal to the number of saved dives.
  If they are not the same the ringbuffer is full and exactly 37 dives
  are saved.

  \return Number of dives saved in the aladin computer
*/
int InterfaceAladin::getNumberOfDives( void )
{
    int latestDive = _data[ 0x7f4 + 4 ];

    int totalNumber = _data[ 0x7f2 + 4 ] * 256 + _data[ 0x7f3 + 4 ];

    if( totalNumber == latestDive )
  	return latestDive;
    else
  	return 37;
}



/*! The dive with index <tt>index</tt> is returned. The value of the index
  is not checked since it is forced to fit in the ringbuffer in the while-loop.
  The 12 byte of the logbook entry are decoded and the results are saved in the
  varialbe <tt>dive</tt>.

  \param index Index of the dive to be saved in <tt>dive</tt>. Index 0 is the latest dive
  \param dive Object of class Dive to save dive-data into
*/
bool InterfaceAladin::getDive( int index, Dive &dive )
{


    int latestDive = _data[ 0x7f4 + 4 ];
    int totalNumber = _data[ 0x7f2 + 4 ] * 256 + _data[ 0x7f3 + 4 ];

    if( index < 0 || index > 36 || (latestDive == totalNumber && index > (latestDive - 1)) )
	return false;

    int oldestDiveIndex;

    if( latestDive == totalNumber )
	oldestDiveIndex = 0;
    else
	oldestDiveIndex = latestDive % 37;


    // Set the number of the dive
    dive.setNumber( index + 1 );

    // calculate index in datablock
    unsigned int hlp;
    index = 0x604 + ((oldestDiveIndex + index) % 37) * 12;



    // depth of the dive
    hlp = (_data[ index + 2 ] * 256 + _data[ index + 3 ]) * 10 / 4096;
    dive.setDepth( hlp );

    // duration of the dive
    hlp = _data[ index + 1 ] & 0x0f;
    hlp += ((_data[ index + 1 ] & 0xf0) >> 4) * 10;
    if( _data[ index ] & 0x04 )
	hlp += 100;
    dive.setDuration( hlp );

    // date and time of the dive
    hlp = _data[ index + 7 ];
    hlp <<= 8;
    hlp += _data[ index + 8 ];
    hlp <<= 8;
    hlp += _data[ index + 9 ];
    hlp <<= 8;
    hlp += _data[ index + 10 ];

    QDateTime dateTime( QDate( 1994, 1, 1 ) );
    dive.setDateTime( dateTime.addSecs( hlp / 2 ) );

    // water temperature
    hlp = _data[ index + 11 ] / 4;
    dive.setWaterTemperature( hlp );

    // air consumption
    hlp = _data[ index + 6 ];
    dive.setAirConsumption( hlp );

    return true;
}



int InterfaceAladin::getProgressSteps( void )
{
    return 2050;
}




/*! Internal function to "swap" a byte. The aladin computer saves its
  data with all bits reversed (bit 7 is bit 0, bit 6 is bit 1 and so on).
  So all bytes have to be "swapped" befor they can be processed.
*/
void InterfaceAladin::swapBits( unsigned char *byte )
{
    char newByte = 0;
    char oldByte = *byte;
    int i;

    for( i = 0; i < 8; i++ )
    {
	newByte <<= 1;
	newByte |= oldByte & 1;
	oldByte >>= 1;
    }
    
    *byte = newByte;
}



