/*
 * KMLOCfg
 *
 * A utility to configure modems of the ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <stdio.h>
#include <string.h>

#include <qglobal.h>
#include <qwidget.h>
#include <qlabel.h>
#include <qframe.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qtooltip.h>
#include <qsize.h>
#include <qslider.h>

#include <kapplication.h>
#include <klocale.h>
#include <kglobal.h>
#include <kiconloader.h>

#include "setupdlg.h"
#include "global.h"



static const uchar country_code_tab[] = {
	0x0A, 0x0F, 0x31, 0x3C, 0x3D, 0x04, 0x59, 0x68, 0x69,
	0x74, 0x7B, 0x82, 0x8B, 0xA5, 0xA6, 0xA0, 0xB4, 0xB5
};


static const char *country_tab[] = {
	I18N_NOOP("Austria"),I18N_NOOP("Belgium"),I18N_NOOP("Denmark"),I18N_NOOP("Finland"),
	I18N_NOOP("France"),I18N_NOOP("Germany"),I18N_NOOP("Italy"),I18N_NOOP("Liechtenstein"),
	I18N_NOOP("Luxemburg"),I18N_NOOP("Monaco"),I18N_NOOP("Netherlands"),I18N_NOOP("Norway"),
	I18N_NOOP("Portugal"),I18N_NOOP("Sweden"),I18N_NOOP("Switzerland"),I18N_NOOP("Spain"),
	I18N_NOOP("United Kingdom"),I18N_NOOP("USA")
};


static const ushort isdn_country_code_tab[] = {
	1, 30, 31, 32, 33, 34, 39, 41, 43, 44, 45,
	46, 47, 49, 81, 90, 351, 352, 353, 357, 358
};


static const char *isdn_country_tab[] = {
	I18N_NOOP("North America"),I18N_NOOP("Greece"),I18N_NOOP("Netherlands"),I18N_NOOP("Belgium"),
	I18N_NOOP("France"),I18N_NOOP("Spain"),I18N_NOOP("Italy"),I18N_NOOP("Switzerland"),
	I18N_NOOP("Austria"),I18N_NOOP("United Kingdom"),I18N_NOOP("Denmark"),I18N_NOOP("Sweden"),
	I18N_NOOP("Norway"),I18N_NOOP("Germany"),I18N_NOOP("Japan"),I18N_NOOP("Turkey"),
	I18N_NOOP("Portugal"),I18N_NOOP("Luxemburg"),I18N_NOOP("Ireland"),I18N_NOOP("Cyprus"),
	I18N_NOOP("Finland")
};



CommonPageGeneral::CommonPageGeneral(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *svbox, *shbox, *shbox2, *svbox2;
	QLabel *label, *label1, *label2;
	QSize size;
	QGroupBox *gb;

	vbox = new QVBoxLayout(this, 10);

	gb = new QGroupBox(i18n("Device"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	thumb = new QLabel(gb);
	thumb->setPixmap(UserIcon("unknown"));
	thumb->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	thumb->setIndent(0);
	thumb->setFixedSize(thumb->sizeHint());

	shbox->addWidget(thumb);

	svbox2 = new QVBoxLayout(10);
	shbox->addLayout(svbox2);
	
	shbox2 = new QHBoxLayout(10);
	svbox2->addLayout(shbox2);

	label1 = new QLabel(i18n("Model:"), gb);
	shbox2->addWidget(label1);

	model = new QLabel(i18n("Unknown"), gb);
	model->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	model->setMinimumSize(model->sizeHint());
	shbox2->addWidget(model);

	shbox2 = new QHBoxLayout(10);
	svbox2->addLayout(shbox2);

	label2 = new QLabel(i18n("Firmware:"), gb);
	shbox2->addWidget(label2);

	firmware = new QLabel(" ", gb);
	firmware->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	firmware->setMinimumSize(firmware->sizeHint());
	shbox2->addWidget(firmware);

	size = label1->sizeHint().expandedTo(label2->sizeHint());
	label1->setFixedSize(size);
	label2->setFixedSize(size);

	svbox2->addStretch();

	svbox->activate();

	vbox->addWidget(gb);

	timer_box = new QGroupBox(i18n("Internal Modem Time"), this);
	svbox = new QVBoxLayout(timer_box, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Date, Time:"), timer_box);
	label->setFixedSize(label->sizeHint());
	shbox->addWidget(label);
	time = new QLabel(timer_box);
	time->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	setTime(QDateTime::currentDateTime());
	time->setFixedSize(time->sizeHint());
	shbox->addWidget(time);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	sync_time = new QCheckBox(i18n("Synchronize modem time"), timer_box);
	sync_time->setFixedSize(sync_time->sizeHint());
	shbox->addWidget(sync_time);
	shbox->addStretch();

	svbox->activate();

	vbox->addWidget(timer_box);

	vbox->addStretch();
	vbox->activate();

	timer_box->hide();

	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), SLOT(timerSlot()));	
	timer->start(1000);
}


CommonPageGeneral::~CommonPageGeneral()
{
}


void CommonPageGeneral::setToolTips(bool enable)
{
	if (enable) {
		QToolTip::add(model, i18n("The product name of this device"));
		QToolTip::add(firmware, i18n("Some information about the modem's firmware"));
		QToolTip::add(time, i18n("The date and time of your modem's timer"));
		QToolTip::add(sync_time, i18n("Syncronize the modem's timer with your PC's timer"));
	} else {
		QToolTip::remove(model);
		QToolTip::remove(firmware);
		QToolTip::remove(time);
		QToolTip::remove(sync_time);
	}
}


void CommonPageGeneral::setConfig(const mlo_config_t *conf)
{
	firmware->setText(QString(conf->firmware));
	setTime(conf->date_time);
	sync_time->setChecked(conf->sync_timer);
}


void CommonPageGeneral::getConfig(mlo_config_t *conf)
{
	conf->sync_timer = sync_time->isChecked();
}


void CommonPageGeneral::resetTime()
{
	if (sync_time->isChecked()) {
		setTime(QDateTime::currentDateTime());
		sync_time->setChecked(false);
	}
}


void CommonPageGeneral::setTime(const QDateTime &time_)
{
	m_time = time_;
	time->setText(KGlobal::locale()->formatDate(m_time.date(), true) + ", " + KGlobal::locale()->formatTime(m_time.time(), true));
}


void CommonPageGeneral::timerSlot()
{
	setTime(m_time.addSecs(1));
}



OfficePageGeneral::OfficePageGeneral(QWidget *parent, const char *name): CommonPageGeneral(parent, name)
{
	int i;
	QBoxLayout *svbox, *shbox;
	QLabel *label;
	QGroupBox *gb;

	thumb->setPixmap(UserIcon("office"));
	model->setText(ID_MODEL_OFFICE);
	model->setMinimumSize(model->sizeHint());

	timer_box->show();

	gb = new QGroupBox(i18n("Country Selection"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Country:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	shbox->addWidget(label);

	line_country = new QComboBox(gb);
//	label->setBuddy(line_country);
	for (i=0; i < 18; i++)
		line_country->insertItem(i18n(country_tab[i]), i);
	line_country->setFixedHeight(line_country->sizeHint().height());
	shbox->addWidget(line_country);

	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
}


OfficePageGeneral::~OfficePageGeneral()
{
}


void OfficePageGeneral::setToolTips(bool enable)
{
	CommonPageGeneral::setToolTips(enable);

	if (enable) {
		QToolTip::add(line_country, i18n("The country in which you are located"));
	} else {
		QToolTip::remove(line_country);
	}
}


void OfficePageGeneral::setConfig(const mlo_config_t *conf)
{
	int i;

	CommonPageGeneral::setConfig(conf);

	for (i=0; i < 18; i++)
		if (country_code_tab[i] == conf->mod.off.country)
			line_country->setCurrentItem(i);
}


void OfficePageGeneral::getConfig(mlo_config_t *conf)
{
	CommonPageGeneral::getConfig(conf);

	conf->mod.off.country = country_code_tab[line_country->currentItem()];
}


	
OfficePageCalls::OfficePageCalls(QWidget *parent, const char *name): QWidget(parent, name)
{
	int i;
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *label;
	QGroupBox *gb;

	vbox = new QVBoxLayout(this, 10);
	
	gb = new QGroupBox(i18n("Incoming Calls"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	
	general_ring = new QCheckBox(i18n("Modem rings with incoming call"), gb);
	general_ring->setMinimumSize(general_ring->sizeHint());
	shbox->addWidget(general_ring);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Accept calls:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	shbox->addWidget(label);
	
	general_rings = new QComboBox(gb);
//	label->setBuddy(general_rings);
	general_rings->insertItem(i18n("After 1 Ring"), 0);
	for (i=1; i < 9; i++) {
		general_rings->insertItem(i18n("After %1 Rings").arg(i+1), i);
	}
//	size = size.expandedTo(general_rings->sizeHint());
	general_rings->setFixedHeight(general_rings->sizeHint().height());
	shbox->addWidget(general_rings);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	general_acceptearly = new QCheckBox(i18n("Accept calls early"), gb);
	general_acceptearly->setFixedSize(general_acceptearly->sizeHint());
	shbox->addWidget(general_acceptearly);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	general_acceptfull = new QCheckBox(i18n("Accept calls if memory full"), gb);
	general_acceptfull->setFixedSize(general_acceptfull->sizeHint());
	shbox->addWidget(general_acceptfull);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	general_acceptdtr = new QCheckBox(i18n("Accept calls with running application"), gb);
	general_acceptdtr->setFixedSize(general_acceptdtr->sizeHint());
	shbox->addWidget(general_acceptdtr);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	general_suppressri = new QCheckBox(i18n("Suppress ring indicator signal (RI/M3)"), gb);
	general_suppressri->setFixedSize(general_suppressri->sizeHint());
	shbox->addWidget(general_suppressri);
	shbox->addStretch();

	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();
}


OfficePageCalls::~OfficePageCalls()
{
}


void OfficePageCalls::setToolTips(bool enable)
{
	if (enable) {
		QToolTip::add(general_ring, i18n("Modem rings with incoming call"));
		QToolTip::add(general_rings, i18n("Number of rings before calls are accepted"));
		QToolTip::add(general_acceptearly, i18n("Accept calls after 2 rings"));
		QToolTip::add(general_acceptfull, i18n("Accept calls when modem memory is full"));
		QToolTip::add(general_acceptdtr, i18n("Accept calls when an application uses the modem"));
		QToolTip::add(general_suppressri, i18n("Suppress RI signal with incoming call"));
	} else {
		QToolTip::remove(general_ring);
		QToolTip::remove(general_rings);
		QToolTip::remove(general_acceptearly);
		QToolTip::remove(general_acceptfull);
		QToolTip::remove(general_acceptdtr);
		QToolTip::remove(general_suppressri);
	}
}


void OfficePageCalls::setConfig(const mlo_config_t *conf)
{
	general_ring->setChecked(conf->mod.off.ring_signal);
	general_rings->setCurrentItem(conf->mod.off.ring_number-1);
	general_acceptearly->setChecked(conf->mod.off.accept_early);
	general_acceptfull->setChecked(conf->mod.off.accept_full);
	general_acceptdtr->setChecked(conf->mod.off.accept_dtr);
	general_suppressri->setChecked(conf->mod.off.suppress_ri);
}


void OfficePageCalls::getConfig(mlo_config_t *conf)
{
	conf->mod.off.ring_signal = general_ring->isChecked();
	conf->mod.off.ring_number = general_rings->currentItem()+1;
	conf->mod.off.accept_early = general_acceptearly->isChecked();
	conf->mod.off.accept_full = general_acceptfull->isChecked();
	conf->mod.off.accept_dtr = general_acceptdtr->isChecked();
	conf->mod.off.suppress_ri = general_suppressri->isChecked();
}



OfficePageAnswering::OfficePageAnswering(QWidget *parent, const char *name): QWidget(parent, name)
{
	int i;
	QBoxLayout *vbox, *svbox, *shbox;
	QGridLayout *grid;
	QLabel *label;
	QGroupBox *gb;
	
	vbox = new QVBoxLayout(this, 10);

	shbox = new QHBoxLayout(10);
	vbox->addLayout(shbox);

	answer_enabled = new QCheckBox(i18n("Answering machine enabled"), this);
	answer_enabled->setFixedSize(answer_enabled->sizeHint());
	shbox->addWidget(answer_enabled);
	shbox->addStretch();

	gb = new QGroupBox(i18n("Outgoing Message"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Message:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	shbox->addWidget(label);
	
	answer_outgoing = new QComboBox(gb);
//	label->setBuddy(answer_outgoing);
	answer_outgoing->insertItem(i18n("No. 1"), 0);
	answer_outgoing->insertItem(i18n("No. 2"), 1);
	answer_outgoing->insertItem(i18n("No. 3"), 2);
	answer_outgoing->insertItem(i18n("Memory full"), 3);
	answer_outgoing->setFixedHeight(answer_outgoing->sizeHint().height());
	shbox->addWidget(answer_outgoing);

	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Incoming Messages"), this);

	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	answer_record = new QCheckBox(i18n("Record incoming messages"), gb);
	answer_record->setFixedSize(answer_record->sizeHint());
	shbox->addWidget(answer_record);
	shbox->addStretch();

	grid = new QGridLayout(5, 2, 8);
	svbox->addLayout(grid);

	label = new QLabel(i18n("Recording volume:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	grid->addWidget(label, 0, 0);

	rec_recvol = new QSlider(0, 255, 128, 0, QSlider::Horizontal, gb);
//	label->setBuddy(rec_recvol);
	rec_recvol->setTickmarks(QSlider::Below);
	rec_recvol->setFixedHeight(rec_recvol->sizeHint().height());
	grid->addWidget(rec_recvol, 0, 1);

	label = new QLabel(i18n("Playback volume:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	grid->addWidget(label, 1, 0);

	rec_playvol = new QSlider(0, 255, 128, 0, QSlider::Horizontal, gb);
//	label->setBuddy(rec_playvol);
	rec_playvol->setTickmarks(QSlider::Below);
	rec_playvol->setFixedHeight(rec_playvol->sizeHint().height());
	grid->addWidget(rec_playvol, 1, 1);

	label = new QLabel(i18n("Recording quality:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	grid->addWidget(label, 2, 0);
	
	rec_quality = new QComboBox(gb);
//	label->setBuddy(rec_quality);
	rec_quality->insertItem(i18n("Low"), 0);
	rec_quality->insertItem(i18n("Medium"), 1);
	rec_quality->insertItem(i18n("High"), 2);
	rec_quality->setFixedHeight(rec_quality->sizeHint().height());
	grid->addWidget(rec_quality, 2, 1);

	label = new QLabel(i18n("Maximum duration:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	grid->addWidget(label, 3, 0);
	
	rec_maxdur = new QComboBox(gb);
//	label->setBuddy(rec_maxdur);
	rec_maxdur->insertItem(i18n("Unlimited"), 0);
	for (i=1; i < 21; i++) {
		rec_maxdur->insertItem(i18n("%1 Sec.").arg(i*15), i);
	}
	rec_maxdur->setFixedHeight(rec_maxdur->sizeHint().height());
	grid->addWidget(rec_maxdur, 3, 1);

	label = new QLabel(i18n("Minimum duration:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	grid->addWidget(label, 4, 0);
	
	rec_mindur = new QComboBox(gb);
//	label->setBuddy(rec_mindur);
	for (i=0; i < 16; i++) {
		rec_mindur->insertItem(i18n("%1 Sec.").arg(i), i);
	}
	rec_mindur->setFixedHeight(rec_mindur->sizeHint().height());
	grid->addWidget(rec_mindur, 4, 1);

	grid->setColStretch(0, 1);
	grid->setColStretch(1, 1);
	
	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	rec_speaker = new QCheckBox(i18n("Enable speaker during recording"), gb);
	rec_speaker->setFixedSize(rec_speaker->sizeHint());
	shbox->addWidget(rec_speaker);
	shbox->addStretch();

	svbox->activate();
	
	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();
}


OfficePageAnswering::~OfficePageAnswering()
{
}


void OfficePageAnswering::setToolTips(bool enable)
{
	if (enable) {
		QToolTip::add(answer_enabled, i18n("Use this option to enable the answering machine"));
		QToolTip::add(answer_record, i18n("Record messages of incoming calls"));
		QToolTip::add(answer_outgoing, i18n("Selection of the outgoing message"));
		QToolTip::add(rec_playvol, i18n("The volume with which recorded messages are played back"));
		QToolTip::add(rec_recvol, i18n("The volume with which incoming messages are recorded"));
		QToolTip::add(rec_quality, i18n("The quality with which incoming messages are recorded"));
		QToolTip::add(rec_maxdur, i18n("The maximum recording duration of incoming messages"));
		QToolTip::add(rec_mindur, i18n("The minimum recording duration of incoming messages"));
		QToolTip::add(rec_speaker, i18n("Turn on the speaker while incoming messages are recorded"));
	} else {
		QToolTip::remove(answer_enabled);
		QToolTip::remove(answer_record);
		QToolTip::remove(answer_outgoing);
		QToolTip::remove(rec_playvol);
		QToolTip::remove(rec_recvol);
		QToolTip::remove(rec_quality);
		QToolTip::remove(rec_maxdur);
		QToolTip::remove(rec_mindur);
		QToolTip::remove(rec_speaker);
	}
}


void OfficePageAnswering::setConfig(const mlo_config_t *conf)
{
	static const char *qualities[3] = { "ADPCM2-7", "ADPCM4-7", "PCM8L-7" };
	int i;

	answer_enabled->setChecked(conf->mod.off.voice_operation);
	answer_record->setChecked(conf->mod.off.voice_recording);
	answer_outgoing->setCurrentItem((conf->mod.off.outgoing_message < 3) ? conf->mod.off.outgoing_message : 3);
	rec_playvol->setValue(conf->mod.off.play_vol);
	rec_recvol->setValue(conf->mod.off.rec_vol);

	for (i=0; i < 3; i++)
		if (!strcmp(conf->mod.off.rec_quality, qualities[i]))
			rec_quality->setCurrentItem(i);

	rec_maxdur->setCurrentItem(conf->mod.off.max_rec_time);
	rec_mindur->setCurrentItem(conf->mod.off.min_rec_time);
	rec_speaker->setChecked(conf->mod.off.rec_speaker);
}


void OfficePageAnswering::getConfig(mlo_config_t *conf)
{
	static const char *qualities[3] = { "ADPCM2-7", "ADPCM4-7", "PCM8L-7" };
	int i;

	conf->mod.off.voice_operation = answer_enabled->isChecked();
	conf->mod.off.voice_recording = answer_record->isChecked();

	i = answer_outgoing->currentItem();
	conf->mod.off.outgoing_message = (i < 3) ? i : 255;
	
	conf->mod.off.play_vol = rec_playvol->value();
	conf->mod.off.rec_vol = rec_recvol->value();
	strcpy(conf->mod.off.rec_quality, qualities[rec_quality->currentItem()]);
	conf->mod.off.max_rec_time = rec_maxdur->currentItem();
	conf->mod.off.min_rec_time = rec_mindur->currentItem();
	conf->mod.off.rec_speaker = rec_speaker->isChecked();
}



OfficePageFax::OfficePageFax(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *svbox, *shbox;
	QGridLayout *grid;
	QLabel *label;
	QGroupBox *gb;
	FaxIDValidator *validator;
	
	validator = new FaxIDValidator(this);

	vbox = new QVBoxLayout(this, 10);

	gb = new QGroupBox(i18n("Fax"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	
	fax_operation = new QCheckBox(i18n("Fax reception enabled"), gb);
	fax_operation->setFixedSize(fax_operation->sizeHint());
	shbox->addWidget(fax_operation);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	fax_takeover = new QCheckBox(i18n("Fax reception during phone calls"), gb);
	fax_takeover->setFixedSize(fax_takeover->sizeHint());
	shbox->addWidget(fax_takeover);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Sender ID:"), gb);
	label->setFixedSize(label->sizeHint());
	shbox->addWidget(label);
	
	fax_id = new QLineEdit(gb);
//	label->setBuddy(fax_id);
	fax_id->setValidator(validator);
	fax_id->setMaxLength(20);
	fax_id->setFixedHeight(fax_id->sizeHint().height());
	shbox->addWidget(fax_id);
	
	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Announcements"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	message_system = new QCheckBox(i18n("Announce system messages"), gb);
	message_system->setMinimumSize(message_system->sizeHint());
	shbox->addWidget(message_system);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	message_number = new QCheckBox(i18n("Announce caller ID"), gb);
	message_number->setMinimumSize(message_number->sizeHint());
	shbox->addWidget(message_number);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	message_time = new QCheckBox(i18n("Announce time of call"), gb);
	message_time->setMinimumSize(message_time->sizeHint());
	shbox->addWidget(message_time);
	shbox->addStretch();

	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Speakerphone Mode"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	grid = new QGridLayout(2, 2, 8);
	svbox->addLayout(grid);

	label = new QLabel(i18n("Sound volume:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	grid->addWidget(label, 0, 0);

	speaker_volume = new QSlider(0, 255, 128, 0, QSlider::Horizontal, gb);
//	label->setBuddy(speaker_volume);
	speaker_volume->setTickmarks(QSlider::Below);
//	speaker_volume->setFixedWidth(150);
	speaker_volume->setFixedHeight(speaker_volume->sizeHint().height());
	grid->addWidget(speaker_volume, 0, 1);

	label = new QLabel(i18n("Microphone sensitivity:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	grid->addWidget(label, 1, 0);

	speaker_micro = new QSlider(0, 255, 128, 0, QSlider::Horizontal, gb);
//	label->setBuddy(speaker_micro);
	speaker_micro->setTickmarks(QSlider::Below);
//	speaker_micro->setFixedWidth(150);
	speaker_micro->setFixedHeight(speaker_micro->sizeHint().height());
	grid->addWidget(speaker_micro, 1, 1);

	grid->setColStretch(0, 1);
	grid->setColStretch(1, 1);
	
	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	speaker_monitor = new QCheckBox(i18n("Enable monitoring or recording of conversation"), gb);
	speaker_monitor->setMinimumSize(speaker_monitor->sizeHint());
	shbox->addWidget(speaker_monitor);
	shbox->addStretch();

	svbox->activate();
	
	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();
}


OfficePageFax::~OfficePageFax()
{
}


void OfficePageFax::setToolTips(bool enable)
{
	if (enable) {
		QToolTip::add(fax_operation, i18n("Use this option to enable fax reception"));
		QToolTip::add(fax_takeover, i18n("Automatic fax recognition during phone calls"));
		QToolTip::add(fax_id, i18n("Enter your fax ID here"));
		QToolTip::add(message_system, i18n("Announce the modem's system messages"));
		QToolTip::add(message_number, i18n("Announce the caller ID (CLIP) of recorded messages"));
		QToolTip::add(message_time, i18n("Announce the date and time of recorded messages"));
		QToolTip::add(speaker_volume, i18n("The volume of the modem's speaker"));
		QToolTip::add(speaker_micro, i18n("The sensitivity of the modem's microphone"));
		QToolTip::add(speaker_monitor, i18n("Enable monitoring and recording of conversations"));
	} else {
		QToolTip::remove(fax_operation);
		QToolTip::remove(fax_takeover);
		QToolTip::remove(fax_id);
		QToolTip::remove(message_system);
		QToolTip::remove(message_number);
		QToolTip::remove(message_time);
		QToolTip::remove(speaker_volume);
		QToolTip::remove(speaker_micro);
		QToolTip::remove(speaker_monitor);
	}
}


void OfficePageFax::setConfig(const mlo_config_t *conf)
{
	fax_operation->setChecked(conf->mod.off.fax_operation);
	fax_takeover->setChecked(conf->mod.off.fax_takeover);
	fax_id->setText(conf->mod.off.fax_id);
	message_system->setChecked(conf->mod.off.announce_system);
	message_number->setChecked(conf->mod.off.announce_number);
	message_time->setChecked(conf->mod.off.announce_time);
	speaker_volume->setValue(conf->mod.off.speaker_vol);
	speaker_micro->setValue(conf->mod.off.micro_vol);
	speaker_monitor->setChecked(conf->mod.off.rec_monitor);
}


void OfficePageFax::getConfig(mlo_config_t *conf)
{
	conf->mod.off.fax_operation = fax_operation->isChecked();
	conf->mod.off.fax_takeover = fax_takeover->isChecked();
	strcpy(conf->mod.off.fax_id, fax_id->text().latin1());
	conf->mod.off.announce_system = message_system->isChecked();
	conf->mod.off.announce_number = message_number->isChecked();
	conf->mod.off.announce_time = message_time->isChecked();
	conf->mod.off.speaker_vol = speaker_volume->value();
	conf->mod.off.micro_vol = speaker_micro->value();
	conf->mod.off.rec_monitor = speaker_monitor->isChecked();
}



OfficePageAccess::OfficePageAccess(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *label;
	QGroupBox *gb;
	NumberValidator *validator;

	validator = new NumberValidator(this, "validator");

	vbox = new QVBoxLayout(this, 10);

	gb = new QGroupBox(i18n("Local Configuration"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	access_hook = new QCheckBox(i18n("Access by pushing hook flash"), gb);
	access_hook->setFixedSize(access_hook->sizeHint());
	shbox->addWidget(access_hook);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	access_dtmf = new QCheckBox(i18n("Access by sending DTMF tones"), gb);
	access_dtmf->setFixedSize(access_dtmf->sizeHint());
	shbox->addWidget(access_dtmf);
	shbox->addStretch();

	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Remote Access"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	access_playback = new QCheckBox(i18n("Enable remote playback"), gb);
	access_playback->setFixedSize(access_playback->sizeHint());
	shbox->addWidget(access_playback);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	access_config = new QCheckBox(i18n("Enable remote configuration"), gb);
	access_config->setFixedSize(access_config->sizeHint());
	shbox->addWidget(access_config);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Password:"), gb);
	label->setFixedSize(label->sizeHint());
	shbox->addWidget(label);
	
	access_password = new QLineEdit(gb);
//	label->setBuddy(access_password);
	access_password->setValidator(validator);
	access_password->setMaxLength(4);
	access_password->setEchoMode(QLineEdit::Password);
	access_password->setText("8888");
	access_password->setFixedSize(access_password->sizeHint());
	access_password->clear();
	shbox->addWidget(access_password);
	shbox->addStretch();

	svbox->activate();
	
	vbox->addWidget(gb);
	vbox->addStretch(1);
	vbox->activate();
}


OfficePageAccess::~OfficePageAccess()
{
}


void OfficePageAccess::setToolTips(bool enable)
{
	if (enable) {
		QToolTip::add(access_hook, i18n("Configure the modem by pushing the hook flash of the phone"));
		QToolTip::add(access_dtmf, i18n("Configure the modem by sending DTMF tones with the phone"));
		QToolTip::add(access_playback, i18n("Enable remote playback of recorded messages"));
		QToolTip::add(access_config, i18n("Enable remote configuration"));
		QToolTip::add(access_password, i18n("Enter the password for the remote playback"));
	} else {
		QToolTip::remove(access_hook);
		QToolTip::remove(access_dtmf);
		QToolTip::remove(access_playback);
		QToolTip::remove(access_config);
		QToolTip::remove(access_password);
	}
}


void OfficePageAccess::setConfig(const mlo_config_t *conf)
{
	access_hook->setChecked(conf->mod.off.hook_config);
	access_dtmf->setChecked(conf->mod.off.keyboard_config);
	access_playback->setChecked(conf->mod.off.remote_query);
	access_config->setChecked(conf->mod.off.remote_config);
	access_password->setText(conf->mod.off.remote_pw);
	
}


void OfficePageAccess::getConfig(mlo_config_t *conf)
{
	conf->mod.off.hook_config = access_hook->isChecked();
	conf->mod.off.keyboard_config = access_dtmf->isChecked();
	conf->mod.off.remote_query = access_playback->isChecked();
	conf->mod.off.remote_config = access_config->isChecked();
	strcpy(conf->mod.off.remote_pw, access_password->text().latin1());
}



OfficeIIPageGeneral::OfficeIIPageGeneral(QWidget *parent, const char *name): CommonPageGeneral(parent, name)
{
	int i;
	QBoxLayout *svbox, *shbox;
	QLabel *label;
	QGroupBox *gb;

	thumb->setPixmap(UserIcon("office_ii"));
	model->setText(ID_MODEL_OFFICE_II);
	model->setMinimumSize(model->sizeHint());

	timer_box->show();

	gb = new QGroupBox(i18n("Country Selection"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Country:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	shbox->addWidget(label);

	line_country = new QComboBox(gb);
	label->setBuddy(line_country);
	for (i=0; i < 18; i++)
		line_country->insertItem(i18n(country_tab[i]), i);
	line_country->setFixedHeight(line_country->sizeHint().height());
	shbox->addWidget(line_country);

	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
}


OfficeIIPageGeneral::~OfficeIIPageGeneral()
{
}


void OfficeIIPageGeneral::setToolTips(bool enable)
{
	CommonPageGeneral::setToolTips(enable);

	if (enable) {
		QToolTip::add(line_country, i18n("The country in which you are located"));
	} else {
		QToolTip::remove(line_country);
	}
}


void OfficeIIPageGeneral::setConfig(const mlo_config_t *conf)
{
	int i;

	CommonPageGeneral::setConfig(conf);

	for (i=0; i < 18; i++)
		if (country_code_tab[i] == conf->mod.offii.country)
			line_country->setCurrentItem(i);
}


void OfficeIIPageGeneral::getConfig(mlo_config_t *conf)
{
	CommonPageGeneral::getConfig(conf);

	conf->mod.offii.country = country_code_tab[line_country->currentItem()];
}



OfficeIIPageCallerID::OfficeIIPageCallerID(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *labels[3];
	int i;
	QGroupBox *gb;
	QSize size;
	NumberValidator *validator;

	validator = new NumberValidator(this);

	vbox = new QVBoxLayout(this, 10);

	gb = new QGroupBox(i18n("Caller ID"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	for (i=0; i < 3; i++) {
		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);

		labels[i] = new QLabel(i18n("Incoming phone number:"), gb);
		shbox->addWidget(labels[i]);

		in_numbers[i] = new QLineEdit(gb);
		in_numbers[i]->setMaxLength(20);
		in_numbers[i]->setValidator(validator);
		in_numbers[i]->setFixedHeight(in_numbers[i]->sizeHint().height());
		shbox->addWidget(in_numbers[i]);
	}
	
	size = QSize(0,0);
	for (i=0; i < 3; i++)
		size = size.expandedTo(labels[i]->sizeHint());
	for (i=0; i < 3; i++)
		labels[i]->setFixedSize(size);

	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();
}


OfficeIIPageCallerID::~OfficeIIPageCallerID()
{
}


void OfficeIIPageCallerID::setToolTips(bool enable)
{
	int i;

	if (enable) {
		for (i=0; i < 3; i++)
			QToolTip::add(in_numbers[i], i18n("A call with this number will trigger a specific LED flashing pattern"));
	} else {
		for (i=0; i < 3; i++)
			QToolTip::remove(in_numbers[i]);
	}
}


void OfficeIIPageCallerID::setConfig(const mlo_config_t *conf)
{
	int i;

	for (i=0; i < 3; i++)
		in_numbers[i]->setText(conf->mod.offii.in_number[i]);
}


void OfficeIIPageCallerID::getConfig(mlo_config_t *conf)
{
	int i;

	for (i=0; i < 3; i++)
		strcpy(conf->mod.offii.in_number[i], in_numbers[i]->text().latin1());
}



OfficeIIPageCallerList::OfficeIIPageCallerList(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *labels[10];
	int i;
	QGroupBox *gb;
	QSize size;

	vbox = new QVBoxLayout(this, 10);

	gb = new QGroupBox(i18n("Caller List"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	for (i=0; i < 10; i++) {
		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);

		labels[i] = new QLabel(i18n("Number #%1:").arg(i+1), gb);
		shbox->addWidget(labels[i]);

		number_list[i] = new QLabel(" ", gb);
		number_list[i]->setFrameStyle(QFrame::Panel | QFrame::Sunken);
		number_list[i]->setFixedHeight(number_list[i]->sizeHint().height());
		shbox->addWidget(number_list[i]);
	}
	
	size = QSize(0,0);
	for (i=0; i < 10; i++)
		size = size.expandedTo(labels[i]->sizeHint());
	for (i=0; i < 10; i++)
		labels[i]->setFixedSize(size);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	reset_list = new QCheckBox(i18n("Reset"), gb);
	reset_list->setFixedSize(reset_list->sizeHint());
	shbox->addWidget(reset_list);
	shbox->addStretch();

	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();
}


OfficeIIPageCallerList::~OfficeIIPageCallerList()
{
}


void OfficeIIPageCallerList::setToolTips(bool enable)
{
	int i;

	if (enable) {
		for (i=0; i < 10; i++)
			QToolTip::add(number_list[i], i18n("Indicates the most recent calls that were not accepted"));
		QToolTip::add(reset_list, i18n("Use this option to reset the caller list"));
	} else {
		for (i=0; i < 10; i++)
			QToolTip::remove(number_list[i]);
		QToolTip::remove(reset_list);
	}
}


void OfficeIIPageCallerList::setConfig(const mlo_config_t *conf)
{
/*
	number_list[num]->setText(caller);
	reset_list->setChecked(enable);
*/
}


void OfficeIIPageCallerList::getConfig(mlo_config_t *conf)
{
/*
	number_list[num]->text();
	reset_list->isChecked();
*/
}



OfficeIIPageExtras::OfficeIIPageExtras(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *label;
	QGroupBox *gb;

	vbox = new QVBoxLayout(this, 10);

	gb = new QGroupBox(i18n("Extras"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	no_ring_signal = new QCheckBox(i18n("Acoustic call signaling off"), gb);
	no_ring_signal->setFixedSize(no_ring_signal->sizeHint());
	shbox->addWidget(no_ring_signal);
	shbox->addStretch();
	
	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	permit_ri = new QCheckBox(i18n("Permit call register management (RI/M3)"), gb);
	permit_ri->setFixedSize(permit_ri->sizeHint());
	shbox->addWidget(permit_ri);
	shbox->addStretch();
	
	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	delete_email = new QCheckBox(i18n("Delete the E-Mail LED"), gb);
	delete_email->setFixedSize(delete_email->sizeHint());
	shbox->addWidget(delete_email);
	shbox->addStretch();
	
	svbox->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Signaling of external Calls"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	external_signal = new QCheckBox(i18n("Signaling on"), gb);
	external_signal->setFixedSize(external_signal->sizeHint());
	shbox->addWidget(external_signal);
	shbox->addStretch();
	
	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	
	label = new QLabel(i18n("Ring delay [ms]:"), gb);
	label->setFixedSize(label->sizeHint());
	shbox->addWidget(label);
	
	ring_delay = new QSlider(1, 60, 4, 1, QSlider::Horizontal, gb);
	ring_delay->setTickmarks(QSlider::Below);
	ring_delay->setTickInterval(4);
	ring_delay->setFixedHeight(ring_delay->sizeHint().height());
	shbox->addWidget(ring_delay);
	
	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();
}


OfficeIIPageExtras::~OfficeIIPageExtras()
{
}


void OfficeIIPageExtras::setToolTips(bool enable)
{
	if (enable) {
		QToolTip::add(no_ring_signal, i18n("Switches the acoustic call signaling on or off"));
		QToolTip::add(permit_ri, i18n("Switches the call register management (for ATX PCs) on or off"));
		QToolTip::add(delete_email, i18n("Resets the E-Mail LED"));
		QToolTip::add(external_signal, i18n("Determines whether external calls are specially signaled"));
		QToolTip::add(ring_delay, i18n("Determines in which delay in milliseconds the ring signal occurs"));
	} else {
		QToolTip::remove(no_ring_signal);
		QToolTip::remove(permit_ri);
		QToolTip::remove(delete_email);
		QToolTip::remove(external_signal);
		QToolTip::remove(ring_delay);
	}
}


void OfficeIIPageExtras::setConfig(const mlo_config_t *conf)
{
/*
	no_ring_signal->setChecked(enable);
	permit_ri->setChecked(enable);
	delete_email->setChecked(enable);
	external_signal->setChecked(enable);
	ring_delay->setValue(ms);
*/
}


void OfficeIIPageExtras::getConfig(mlo_config_t *conf)
{
/*
	no_ring_signal->isChecked();
	permit_ri->isChecked();
	delete_email->isChecked();
	external_signal->isChecked();
	ring_delay->value();
*/
}



ISDNPageGeneral::ISDNPageGeneral(QWidget *parent, const char *name): CommonPageGeneral(parent, name)
{
	static const char *protocol_tab[] = {
		I18N_NOOP("Automatic"), I18N_NOOP("Euro DSS1"), I18N_NOOP("DSS1 Point to Point"), I18N_NOOP("National 1TR6"), I18N_NOOP("Japanese INS64")
	};
	QBoxLayout *svbox, *shbox;
	QLabel *label;
	QGroupBox *gb;
	int i;

	thumb->setPixmap(UserIcon("isdn_office"));
	model->setText(ID_MODEL_ISDN_OFFICE);
	model->setMinimumSize(model->sizeHint());

	timer_box->show();

	gb = new QGroupBox(i18n("D Channel Protocol"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Protocol:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	shbox->addWidget(label);

	protocol = new QComboBox(gb);
//	label->setBuddy(protocol);
	for (i=0; i < 5; i++)
		protocol->insertItem(i18n(protocol_tab[i]), i);
	protocol->setFixedHeight(protocol->sizeHint().height());
	shbox->addWidget(protocol);

	svbox->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("a/b Port Line Interface"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Country:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	shbox->addWidget(label);

	ab_country = new QComboBox(gb);
//	label->setBuddy(ab_country);
	for (i=0; i < 21; i++)
		ab_country->insertItem(i18n(isdn_country_tab[i]), i);
	ab_country->setFixedHeight(ab_country->sizeHint().height());
	shbox->addWidget(ab_country);

	svbox->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Call Accepting"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	disable_isdn = new QCheckBox(i18n("Disable call accepting on ISDN port"), gb);
	disable_isdn->setMinimumSize(disable_isdn->sizeHint());
	shbox->addWidget(disable_isdn);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	disable_ab1 = new QCheckBox(i18n("Disable call accepting on a/b port 1"), gb);
	disable_ab1->setMinimumSize(disable_ab1->sizeHint());
	shbox->addWidget(disable_ab1);
	shbox->addStretch();

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	disable_ab2 = new QCheckBox(i18n("Disable call accepting on a/b port 2"), gb);
	disable_ab2->setMinimumSize(disable_ab2->sizeHint());
	shbox->addWidget(disable_ab2);
	shbox->addStretch();

	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
}


ISDNPageGeneral::~ISDNPageGeneral()
{
}


void ISDNPageGeneral::setToolTips(bool enable)
{
	CommonPageGeneral::setToolTips(enable);

	if (enable) {
		QToolTip::add(protocol, i18n("The prefered D channel protocol"));
		QToolTip::add(ab_country, i18n("The country in which you are located"));
		QToolTip::add(disable_isdn, i18n("Disable call accepting on ISDN port"));
		QToolTip::add(disable_ab1, i18n("Disable call accepting on a/b port 1"));
		QToolTip::add(disable_ab2, i18n("Disable call accepting on a/b port 2"));
	} else {
		QToolTip::remove(protocol);
		QToolTip::remove(ab_country);
		QToolTip::remove(disable_isdn);
		QToolTip::remove(disable_ab1);
		QToolTip::remove(disable_ab2);
	}
}


void ISDNPageGeneral::setConfig(const mlo_config_t *conf)
{
	static const char *protocol_tab[] = { "AUTO", "DSS1", "DSS1PP", "1TR6", "INS64"	};
	int i;

	CommonPageGeneral::setConfig(conf);

	for (i=0; i < 5; i++)
		if (!strcmp(conf->mod.isdnoff.d_chn_prot, protocol_tab[i])) {
			protocol->setCurrentItem(i);
			break;
		}
	for (i=0; i < 21; i++)
		if (isdn_country_code_tab[i] == (ushort)conf->mod.isdnoff.country)
			ab_country->setCurrentItem(i);
	disable_isdn->setChecked(!conf->mod.isdnoff.accept_enable[CONF_PORT_ISDN]);
	disable_ab1->setChecked(!conf->mod.isdnoff.accept_enable[CONF_PORT_AB1]);
	disable_ab2->setChecked(!conf->mod.isdnoff.accept_enable[CONF_PORT_AB2]);
}


void ISDNPageGeneral::getConfig(mlo_config_t *conf)
{
	static const char *protocol_tab[] = { "AUTO", "DSS1", "DSS1PP", "1TR6", "INS64"	};

	CommonPageGeneral::getConfig(conf);

	strcpy(conf->mod.isdnoff.d_chn_prot, protocol_tab[protocol->currentItem()]);
	conf->mod.isdnoff.country = isdn_country_code_tab[ab_country->currentItem()];
	conf->mod.isdnoff.accept_enable[CONF_PORT_ISDN] = !disable_isdn->isChecked();
	conf->mod.isdnoff.accept_enable[CONF_PORT_AB1] = !disable_ab1->isChecked();
	conf->mod.isdnoff.accept_enable[CONF_PORT_AB2] = !disable_ab2->isChecked();
}



ISDNPageISDN::ISDNPageISDN(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *label, *label1, *label2, *labels[3];
	QString s;
	int i;
	QGroupBox *gb;
	QSize size;
	NumberValidator *validator;

	validator = new NumberValidator(this, "validator");

	vbox = new QVBoxLayout(this, 10);

	gb = new QGroupBox(i18n("Calling Numbers (DSS1)"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Outgoing calling number:"), gb);
	label->setFixedSize(label->sizeHint());
	shbox->addWidget(label);

	out_msn = new QLineEdit(gb);
	out_msn->setValidator(validator);
	out_msn->setMaxLength(20);
	out_msn->setFixedHeight(out_msn->sizeHint().height());
	shbox->addWidget(out_msn);
	
	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	suppress_msn = new QCheckBox(i18n("Suppress outgoing calling number"), gb);
	suppress_msn->setMinimumSize(suppress_msn->sizeHint());
	shbox->addWidget(suppress_msn);
	shbox->addStretch();
		
	for (i=0; i < 3; i++) {
		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);

		labels[i] = new QLabel(i18n("Calls accept on:"), gb);
		shbox->addWidget(labels[i]);

		accept_msn[i] = new QLineEdit(gb);
		accept_msn[i]->setValidator(validator);
		accept_msn[i]->setMaxLength(20);
		accept_msn[i]->setFixedHeight(accept_msn[i]->sizeHint().height());
		shbox->addWidget(accept_msn[i]);
	}
	
	size = label->sizeHint();
	for (i=0; i < 3; i++)
		size = size.expandedTo(labels[i]->sizeHint());
	label->setFixedSize(size);
	for (i=0; i < 3; i++)
		labels[i]->setFixedSize(size);

	svbox->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Calling number (1TR6)"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Terminal device digit:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	shbox->addWidget(label);

	device_digit = new QComboBox(gb);
	device_digit->insertItem(i18n("None"), 0);
	for (i=0; i < 20; i++) {
		s.setNum(i);
		device_digit->insertItem(s, i+1);
	}
	device_digit->setFixedHeight(device_digit->sizeHint().height());
	shbox->addWidget(device_digit);

	svbox->activate();

	vbox->addWidget(gb);
	
	gb = new QGroupBox(i18n("Extras"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label1 = new QLabel(i18n("ISDN data port call charge:"), gb);
	shbox->addWidget(label1);
	
	total_charge = new QLabel(" ", gb);
	total_charge->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	total_charge->setFixedHeight(total_charge->sizeHint().height());
	shbox->addWidget(total_charge);
	
	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label2 = new QLabel(i18n("Last call:"), gb);
	shbox->addWidget(label2);
	
	last_charge = new QLabel(" ", gb);
	last_charge->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	last_charge->setFixedHeight(last_charge->sizeHint().height());
	shbox->addWidget(last_charge);
	
	size = label1->sizeHint().expandedTo(label2->sizeHint());
	label1->setFixedSize(size);
	label2->setFixedSize(size);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	reset_charge = new QCheckBox(i18n("Reset call charge of the ISDN data port"), gb);
	reset_charge->setFixedSize(reset_charge->sizeHint());
	shbox->addWidget(reset_charge);
	shbox->addStretch();
	
	svbox->activate();

	vbox->addWidget(gb);
	
	vbox->addStretch(1);
	vbox->activate();
}


ISDNPageISDN::~ISDNPageISDN()
{
}


void ISDNPageISDN::setToolTips(bool enable)
{
	if (enable) {
		QToolTip::add(out_msn, i18n("The calling number to be transfered with outgoing calls"));
		QToolTip::add(suppress_msn, i18n("Suppress the outgoing calling number"));
		QToolTip::add(accept_msn[0], i18n("The first MSN to accept calls with"));
		QToolTip::add(accept_msn[1], i18n("The second MSN to accept calls with"));
		QToolTip::add(accept_msn[2], i18n("The third MSN to accept calls with"));
		QToolTip::add(device_digit, i18n("The calling digit (EAZ) to accept calls with"));
		QToolTip::add(total_charge, i18n("Total ISDN call charge display"));
		QToolTip::add(last_charge, i18n("Last call charge display"));
		QToolTip::add(reset_charge, i18n("Reset the call charge of the ISDN data port"));
	} else {
		QToolTip::remove(out_msn);
		QToolTip::remove(suppress_msn);
		QToolTip::remove(accept_msn[0]);
		QToolTip::remove(accept_msn[1]);
		QToolTip::remove(accept_msn[2]);
		QToolTip::remove(device_digit);
		QToolTip::remove(total_charge);
		QToolTip::remove(last_charge);
		QToolTip::remove(reset_charge);
	}
}


void ISDNPageISDN::setConfig(const mlo_config_t *conf)
{
	QString s;
	int i;
	
	out_msn->setText(conf->mod.isdnoff.msn_trans[CONF_PORT_ISDN]);
	suppress_msn->setChecked(!conf->mod.isdnoff.msn_do_trans[CONF_PORT_ISDN]);
	for (i=0; i < 3; i++)
		accept_msn[i]->setText(conf->mod.isdnoff.msn_acc[CONF_PORT_ISDN][i]);
	if (conf->mod.isdnoff.eaz[CONF_PORT_ISDN] > 19)
		device_digit->setCurrentItem(0);
	else
		device_digit->setCurrentItem(conf->mod.isdnoff.eaz[CONF_PORT_ISDN] + 1);
	s.setNum(conf->mod.isdnoff.tot_charge[CONF_PORT_ISDN]);
	total_charge->setText(s);
	s.setNum(conf->mod.isdnoff.last_charge[CONF_PORT_ISDN]);
	last_charge->setText(s);
}


void ISDNPageISDN::getConfig(mlo_config_t *conf)
{
	int i;

	strcpy(conf->mod.isdnoff.msn_trans[CONF_PORT_ISDN], out_msn->text().latin1());
	conf->mod.isdnoff.msn_do_trans[CONF_PORT_ISDN] = !suppress_msn->isChecked();
	for (i=0; i < 3; i++)
		strcpy(conf->mod.isdnoff.msn_acc[CONF_PORT_ISDN][i], accept_msn[i]->text().latin1());
	i = device_digit->currentItem();
	conf->mod.isdnoff.eaz[CONF_PORT_ISDN] = (i) ? (char)(i - 1) : 127;
	conf->mod.isdnoff.reset_charge[CONF_PORT_ISDN] = reset_charge->isChecked();
}



ISDNPageAB::ISDNPageAB(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *label, *labels[3];
	QString s;
	int i;
	QGroupBox *gb;
	QSize size;
	NumberValidator *validator;

	validator = new NumberValidator(this, "validator");

	vbox = new QVBoxLayout(this, 10);

	gb = new QGroupBox(i18n("Calling Numbers (DSS1)"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Outgoing calling number:"), gb);
	shbox->addWidget(label);

	out_msn = new QLineEdit(gb);
	out_msn->setValidator(validator);
	out_msn->setMaxLength(20);
	out_msn->setFixedHeight(out_msn->sizeHint().height());
	shbox->addWidget(out_msn);
	
	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	suppress_msn = new QCheckBox(i18n("Suppress outgoing calling number"), gb);
	suppress_msn->setMinimumSize(suppress_msn->sizeHint());
	shbox->addWidget(suppress_msn);
	shbox->addStretch();
		
	for (i=0; i < 3; i++) {
		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);

		labels[i] = new QLabel(i18n("Calls accept on:"), gb);
		shbox->addWidget(labels[i]);

		accept_msn[i] = new QLineEdit(gb);
		accept_msn[i]->setValidator(validator);
		accept_msn[i]->setMaxLength(20);
		accept_msn[i]->setFixedHeight(accept_msn[i]->sizeHint().height());
		shbox->addWidget(accept_msn[i]);
	}
	
	size = label->sizeHint();
	for (i=0; i < 3; i++)
		size = size.expandedTo(labels[i]->sizeHint());
	label->setFixedSize(size);
	for (i=0; i < 3; i++)
		labels[i]->setFixedSize(size);

	svbox->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Calling number (1TR6)"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Terminal device digit:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	shbox->addWidget(label);

	device_digit = new QComboBox(gb);
	device_digit->insertItem(i18n("None"), 0);
	for (i=0; i < 20; i++) {
		s.setNum(i);
		device_digit->insertItem(s, i+1);
	}
	device_digit->setFixedHeight(device_digit->sizeHint().height());
	shbox->addWidget(device_digit);

	svbox->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Public Exchange Access"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	access_hook = new QCheckBox(i18n("By picking up the phone"), gb);
	access_hook->setFixedSize(access_hook->sizeHint());
	shbox->addWidget(access_hook);
	shbox->addStretch();
	

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);
	access_flash = new QCheckBox(i18n("By pressing the flash key"), gb);
	access_flash->setMinimumSize(access_flash->sizeHint());
	shbox->addWidget(access_flash);
	shbox->addStretch();

	svbox->activate();

	vbox->addWidget(gb);
	
	gb = new QGroupBox(i18n("Service Attributes"), this);
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(12);

	shbox = new QHBoxLayout(10);
	svbox->addLayout(shbox);

	label = new QLabel(i18n("Outgoing indicator:"), gb);
	label->setFixedHeight(label->sizeHint().height());
	shbox->addWidget(label);

	out_ind = new QComboBox(gb);
//	label->setBuddy(outgoing_indicator);
	out_ind->insertItem(i18n("Telephone 3.1 kHz"), 0);
	out_ind->insertItem(i18n("Telephone analog"), 1);
	out_ind->insertItem(i18n("Fax group 2"), 2);
	out_ind->insertItem(i18n("Fax group 3"), 3);
	out_ind->setFixedHeight(out_ind->sizeHint().height());
	shbox->addWidget(out_ind);

	svbox->activate();

	vbox->addWidget(gb);
	
	vbox->addStretch(1);
	vbox->activate();
}


ISDNPageAB::~ISDNPageAB()
{
}


void ISDNPageAB::setToolTips(bool enable)
{
	if (enable) {
		QToolTip::add(out_msn, i18n("The calling number to be transfered with outgoing calls"));
		QToolTip::add(suppress_msn, i18n("Suppress the outgoing calling number"));
		QToolTip::add(accept_msn[0], i18n("The first MSN to accept calls with"));
		QToolTip::add(accept_msn[1], i18n("The second MSN to accept calls with"));
		QToolTip::add(accept_msn[2], i18n("The third MSN to accept calls with"));
		QToolTip::add(device_digit, i18n("The calling digit (EAZ) to accept calls with"));
		QToolTip::add(access_hook, i18n("Get public exchange access by picking up the phone"));
		QToolTip::add(access_flash, i18n("Get public exchange access by pressing the flash key"));
		QToolTip::add(out_ind, i18n("The outgoing service indicator"));
	} else {
		QToolTip::remove(out_msn);
		QToolTip::remove(suppress_msn);
		QToolTip::remove(accept_msn[0]);
		QToolTip::remove(accept_msn[1]);
		QToolTip::remove(accept_msn[2]);
		QToolTip::remove(device_digit);
		QToolTip::remove(access_hook);
		QToolTip::remove(access_flash);
		QToolTip::remove(out_ind);
	}
}


void ISDNPageAB::setConfig(int ab_port, const mlo_config_t *conf)
{
	int i, port = (ab_port == CONF_AB_PORT_1) ? CONF_PORT_AB1 : CONF_PORT_AB2;

	out_msn->setText(conf->mod.isdnoff.msn_trans[port]);
	suppress_msn->setChecked(!conf->mod.isdnoff.msn_do_trans[port]);
	for (i=0; i < 3; i++)
		accept_msn[i]->setText(conf->mod.isdnoff.msn_acc[port][i]);
	if (conf->mod.isdnoff.eaz[port] > 19)
		device_digit->setCurrentItem(0);
	else
		device_digit->setCurrentItem(conf->mod.isdnoff.eaz[port] + 1);
	access_hook->setChecked(conf->mod.isdnoff.exchange_hook[ab_port]);
	access_flash->setChecked(conf->mod.isdnoff.exchange_flash[ab_port]);
	out_ind->setCurrentItem((int)conf->mod.isdnoff.out_ind[port]);
}


void ISDNPageAB::getConfig(int ab_port, mlo_config_t *conf)
{
	int i, port = (ab_port == CONF_AB_PORT_1) ? CONF_PORT_AB1 : CONF_PORT_AB2;

	strcpy(conf->mod.isdnoff.msn_trans[port], out_msn->text().latin1());
	conf->mod.isdnoff.msn_do_trans[port] = !suppress_msn->isChecked();
	for (i=0; i < 3; i++)
		strcpy(conf->mod.isdnoff.msn_acc[port][i], accept_msn[i]->text().latin1());
	i = device_digit->currentItem();
	conf->mod.isdnoff.eaz[port] = (i) ? (char)(i - 1) : 127;
	conf->mod.isdnoff.exchange_hook[ab_port] = access_hook->isChecked();
	conf->mod.isdnoff.exchange_flash[ab_port] = access_flash->isChecked();
	conf->mod.isdnoff.out_ind[port] = out_ind->currentItem();
}



ISDNPageExtras::ISDNPageExtras(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *label1, *label2;
	int i;
	QGroupBox *gb;
	QSize size;
	NumberValidator *validator;

	validator = new NumberValidator(this, "validator");

	vbox = new QVBoxLayout(this, 10);

	for (i=0; i < 2; i++) {
		gb = new QGroupBox(i18n("a/b Port %1").arg(i+1), this);
		svbox = new QVBoxLayout(gb, 6);
		svbox->addSpacing(12);

		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);

		label1 = new QLabel(i18n("a/b port %1 call charge:").arg(i+1), gb);
		shbox->addWidget(label1);

		total_charge[i] = new QLabel(" ", gb);
		total_charge[i]->setFrameStyle(QFrame::Panel | QFrame::Sunken);
		total_charge[i]->setFixedHeight(total_charge[i]->sizeHint().height());
		shbox->addWidget(total_charge[i]);

		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);

		label2 = new QLabel(i18n("Last call:"), gb);
		shbox->addWidget(label2);

		last_charge[i] = new QLabel(" ", gb);
		last_charge[i]->setFrameStyle(QFrame::Panel | QFrame::Sunken);
		last_charge[i]->setFixedHeight(last_charge[i]->sizeHint().height());
		shbox->addWidget(last_charge[i]);

		size = label1->sizeHint().expandedTo(label2->sizeHint());
		label1->setFixedSize(size);
		label2->setFixedSize(size);

		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);
		reset_charge[i] = new QCheckBox(i18n("Reset call charge of a/b port %1").arg(i+1), gb);
		reset_charge[i]->setFixedSize(reset_charge[i]->sizeHint());
		shbox->addWidget(reset_charge[i]);
		shbox->addStretch();
		
		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);
		knocking_tone[i] = new QCheckBox(i18n("Let hear knocking tone on incoming call"), gb);
		knocking_tone[i]->setFixedSize(knocking_tone[i]->sizeHint());
		shbox->addWidget(knocking_tone[i]);
		shbox->addStretch();
		
		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);
		auto_dial[i] = new QCheckBox(i18n("By picking up the phone dial the following number:"), gb);
		auto_dial[i]->setFixedSize(auto_dial[i]->sizeHint());
		shbox->addWidget(auto_dial[i]);
		shbox->addStretch();
		
		shbox = new QHBoxLayout(10);
		svbox->addLayout(shbox);

		shbox->addSpacing(19);

		auto_dial_number[i] = new QLineEdit(gb);
		auto_dial_number[i]->setValidator(validator);
		auto_dial_number[i]->setMaxLength(20);
		auto_dial_number[i]->setFixedHeight(auto_dial_number[i]->sizeHint().height());
		auto_dial_number[i]->setEnabled(false);
		shbox->addWidget(auto_dial_number[i]);
		
		svbox->activate();

		vbox->addWidget(gb);
	}

	vbox->addStretch(1);
	vbox->activate();

	connect(auto_dial[0], SIGNAL(toggled(bool)), this, SLOT(autoDialAB1Toggled(bool)));
	connect(auto_dial[1], SIGNAL(toggled(bool)), this, SLOT(autoDialAB2Toggled(bool)));
}


ISDNPageExtras::~ISDNPageExtras()
{
}


void ISDNPageExtras::setToolTips(bool enable)
{
	int i;

	if (enable) {
		for (i=0; i < 2; i++) {
			QToolTip::add(total_charge[i], i18n("Total call charge display"));
			QToolTip::add(last_charge[i], i18n("Last call charge display"));
			QToolTip::add(reset_charge[i], i18n("Reset the call charge of the a/b port %1").arg(i+1));
			QToolTip::add(knocking_tone[i], i18n("Disable or enable the call waiting facility on a/b port %1").arg(i+1));
			QToolTip::add(auto_dial[i], i18n("Automatic dialing on a/b port %1").arg(i+1));
			QToolTip::add(auto_dial_number[i], i18n("Automatic dialing number for a/b port %1").arg(i+1));
		}
	} else {
		for (i=0; i < 2; i++) {
			QToolTip::remove(total_charge[i]);
			QToolTip::remove(last_charge[i]);
			QToolTip::remove(reset_charge[i]);
			QToolTip::remove(knocking_tone[i]);
			QToolTip::remove(auto_dial[i]);
			QToolTip::remove(auto_dial_number[i]);
		}
	}
}


void ISDNPageExtras::setConfig(const mlo_config_t *conf)
{
	QString s;
	int i, port;

	for (i=0; i < 2; i++) {
		port = (i == 0) ? CONF_PORT_AB1 : CONF_PORT_AB2;
		s.setNum(conf->mod.isdnoff.tot_charge[port]);
		total_charge[i]->setText(s);
		s.setNum(conf->mod.isdnoff.last_charge[port]);
		last_charge[i]->setText(s);
		knocking_tone[i]->setChecked(conf->mod.isdnoff.knock_enable[port]);
		auto_dial[i]->setChecked(conf->mod.isdnoff.auto_dial[port]);
		auto_dial_number[i]->setText(conf->mod.isdnoff.auto_dial_num[port]);
	}
}


void ISDNPageExtras::getConfig(mlo_config_t *conf)
{
	int i, port;

	for (i=0; i < 2; i++) {
		port = (i == 0) ? CONF_PORT_AB1 : CONF_PORT_AB2;
		conf->mod.isdnoff.reset_charge[port] = reset_charge[i]->isChecked();
		conf->mod.isdnoff.knock_enable[port] = knocking_tone[i]->isChecked();
		conf->mod.isdnoff.auto_dial[port] = auto_dial[i]->isChecked();
		strcpy(conf->mod.isdnoff.auto_dial_num[port], auto_dial_number[i]->text().latin1());
	}
}


void ISDNPageExtras::autoDialAB1Toggled(bool on)
{
	auto_dial_number[0]->setEnabled(on);
}


void ISDNPageExtras::autoDialAB2Toggled(bool on)
{
	auto_dial_number[1]->setEnabled(on);
}



SetupCtl::SetupCtl(int model, QWidget *parent, const char *name): KTabCtl(parent, name)
{
	int i;

	modem_model = model;

	has_tooltips = false;
	config = kapp->config();

	for (i=0; i < 5; i++)
		pages[i] = 0;

	if (modem_model == MODEL_UNKNOWN) {
		pages[0] = new CommonPageGeneral(this, "setuppage1");
		Q_CHECK_PTR(pages[0]);
		addTab(pages[0], i18n("General"));
	} else if (modem_model == MODEL_OFFICE) {
		pages[0] = new OfficePageGeneral(this, "setuppage1");
		Q_CHECK_PTR(pages[0]);
		addTab(pages[0], i18n("General"));
		pages[1] = new OfficePageCalls(this, "setuppage2");
		Q_CHECK_PTR(pages[1]);
		addTab(pages[1], i18n("Calls"));
		pages[2] = new OfficePageAnswering(this, "setuppage3");
		Q_CHECK_PTR(pages[2]);
		addTab(pages[2], i18n("Answering Machine"));
		pages[3] = new OfficePageFax(this, "setuppage4");
		Q_CHECK_PTR(pages[3]);
		addTab(pages[3], i18n("Fax && Misc"));
		pages[4] = new OfficePageAccess(this, "setuppage5");
		Q_CHECK_PTR(pages[4]);
		addTab(pages[4], i18n("Access"));
	} else if (modem_model == MODEL_OFFICE_II) {
		pages[0] = new OfficeIIPageGeneral(this, "setuppage1");
		Q_CHECK_PTR(pages[0]);
		addTab(pages[0], i18n("General"));
		pages[1] = new OfficeIIPageCallerID(this, "setuppage2");
		Q_CHECK_PTR(pages[1]);
		addTab(pages[1], i18n("Caller ID"));
		pages[2] = new OfficeIIPageCallerList(this, "setuppage3");
		Q_CHECK_PTR(pages[2]);
		addTab(pages[2], i18n("Caller List"));
		pages[3] = new OfficeIIPageExtras(this, "setuppage4");
		Q_CHECK_PTR(pages[3]);
		addTab(pages[3], i18n("Extras"));
	} else if (modem_model == MODEL_ISDN_OFFICE) {
		pages[0] = new ISDNPageGeneral(this, "setuppage1");
		Q_CHECK_PTR(pages[0]);
		addTab(pages[0], i18n("General"));
		pages[1] = new ISDNPageISDN(this, "setuppage2");
		Q_CHECK_PTR(pages[1]);
		addTab(pages[1], i18n("ISDN Adaptor"));
		pages[2] = new ISDNPageAB(this, "setuppage3");
		Q_CHECK_PTR(pages[2]);
		addTab(pages[2], i18n("a/b Port 1"));
		pages[3] = new ISDNPageAB(this, "setuppage4");
		Q_CHECK_PTR(pages[3]);
		addTab(pages[3], i18n("a/b Port 2"));
		pages[4] = new ISDNPageExtras(this, "setuppage5");
		Q_CHECK_PTR(pages[4]);
		addTab(pages[4], i18n("Extras"));
	}
	setToolTips();
}


SetupCtl::~SetupCtl()
{
}


void SetupCtl::setToolTips()
{
	config->setGroup(ID_PREF_GROUP_GENERAL);
	setToolTips(config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS));
}


void SetupCtl::setToolTips(bool enable)
{
	if (has_tooltips == enable)
		return;
	
	switch (modem_model) {
		case MODEL_UNKNOWN:
			((CommonPageGeneral *)pages[0])->setToolTips(enable);
			break;
		case MODEL_OFFICE:
			((OfficePageGeneral *)pages[0])->setToolTips(enable);
			((OfficePageCalls *)pages[1])->setToolTips(enable);
			((OfficePageAnswering *)pages[2])->setToolTips(enable);
			((OfficePageFax *)pages[3])->setToolTips(enable);
			((OfficePageAccess *)pages[4])->setToolTips(enable);
			break;
		case MODEL_OFFICE_II:
			((OfficeIIPageGeneral *)pages[0])->setToolTips(enable);
			((OfficeIIPageCallerID *)pages[1])->setToolTips(enable);
			((OfficeIIPageCallerList *)pages[2])->setToolTips(enable);
			((OfficeIIPageExtras *)pages[3])->setToolTips(enable);
			break;
		case MODEL_ISDN_OFFICE:
			((ISDNPageGeneral *)pages[0])->setToolTips(enable);
			((ISDNPageISDN *)pages[1])->setToolTips(enable);
			((ISDNPageAB *)pages[2])->setToolTips(enable);
			((ISDNPageAB *)pages[3])->setToolTips(enable);
			((ISDNPageExtras *)pages[4])->setToolTips(enable);
			break;
		default:
			break;
	}

	has_tooltips = enable;
}


const mlo_config_t *SetupCtl::getConfig()
{
	switch (modem_model) {
		case MODEL_OFFICE:
			((OfficePageGeneral *)pages[0])->getConfig(&m_conf);
			((OfficePageCalls *)pages[1])->getConfig(&m_conf);
			((OfficePageAnswering *)pages[2])->getConfig(&m_conf);
			((OfficePageFax *)pages[3])->getConfig(&m_conf);
			((OfficePageAccess *)pages[4])->getConfig(&m_conf);
			break;
		case MODEL_OFFICE_II:
			((OfficeIIPageGeneral *)pages[0])->getConfig(&m_conf);
			((OfficeIIPageCallerID *)pages[1])->getConfig(&m_conf);
			break;
		case MODEL_ISDN_OFFICE:
			((ISDNPageGeneral *)pages[0])->getConfig(&m_conf);
			((ISDNPageISDN *)pages[1])->getConfig(&m_conf);
			((ISDNPageAB *)pages[2])->getConfig(CONF_AB_PORT_1, &m_conf);
			((ISDNPageAB *)pages[3])->getConfig(CONF_AB_PORT_2, &m_conf);
			((ISDNPageExtras *)pages[4])->getConfig(&m_conf);
			break;
		default:
			break;
	}

	return &m_conf;
}


void SetupCtl::setConfig(const mlo_config_t *conf)
{
	if (conf->model != modem_model)
		return;

	m_conf = *conf;

	switch (modem_model) {
		case MODEL_OFFICE:
			((OfficePageGeneral *)pages[0])->setConfig(&m_conf);
			((OfficePageCalls *)pages[1])->setConfig(&m_conf);
			((OfficePageAnswering *)pages[2])->setConfig(&m_conf);
			((OfficePageFax *)pages[3])->setConfig(&m_conf);
			((OfficePageAccess *)pages[4])->setConfig(&m_conf);
			break;
		case MODEL_OFFICE_II:
			((OfficeIIPageGeneral *)pages[0])->setConfig(&m_conf);
			((OfficeIIPageCallerID *)pages[1])->setConfig(&m_conf);
			break;
		case MODEL_ISDN_OFFICE:
			((ISDNPageGeneral *)pages[0])->setConfig(&m_conf);
			((ISDNPageISDN *)pages[1])->setConfig(&m_conf);
			((ISDNPageAB *)pages[2])->setConfig(CONF_AB_PORT_1, &m_conf);
			((ISDNPageAB *)pages[3])->setConfig(CONF_AB_PORT_2, &m_conf);
			((ISDNPageExtras *)pages[4])->setConfig(&m_conf);
			break;
		default:
			break;
	}
}



SetupWidget::SetupWidget(int model, QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox;

	vbox = new QVBoxLayout(this, 4);
	setup_ctl = new SetupCtl(model, this, "setupdlg");
	vbox->addWidget(setup_ctl);
}


SetupWidget::~SetupWidget()
{
}
