/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#include <stdio.h>
#include <unistd.h>

#include <qglobal.h>
#include <qlayout.h>
#include <qkeycode.h>
#include <qpixmap.h>
#include <qsize.h>
#include <qwhatsthis.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <kstatusbar.h>
#include <ktoolbarbutton.h>
#include <krun.h>
#include <kpopupmenu.h>

#include "configure.h"
#include "folderview.h"
#include "printdlg.h"
#include "exportdlg.h"
#include "maildlg.h"
#include "global.h"
#include "senderaliases.h"
#include "kmlofax.h"




StatusProgress::StatusProgress(QWidget *parent = 0, const char *name = 0): QFrame(parent, name)
{
	QBoxLayout *box = new QHBoxLayout(this, 0, 0);

	abort = new QPushButton("X", this);
	Q_CHECK_PTR(abort);
	abort->setMinimumSize(20, fontMetrics().height());
	box->addWidget(abort);

	progress = new KProgress(this);
	Q_CHECK_PTR(progress);
	progress->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
	progress->setLineWidth(0);
	progress->setMargin(0);
	progress->setBackgroundMode(QWidget::PaletteBackground);
	progress->setFixedSize(80, fontMetrics().height());
	box->addWidget(progress);

	box->activate();

	connect(abort, SIGNAL(clicked()), SLOT(abortSlot()));

	setFixedSize(sizeHint());
}


StatusProgress::~StatusProgress()
{
}


void StatusProgress::setEnabled(bool enable)
{
	if (enable) {
		abort->show();
		progress->show();
	} else {
		abort->hide();
		progress->hide();
		setProgress(0);
	}
}


void StatusProgress::abortSlot()
{
	emit aborted();
}



KMLOFaxApp::KMLOFaxApp(): KUniqueApplication()
{
}


KMLOFaxApp::~KMLOFaxApp()
{
	Preview *p;
	for (p = previews.first(); p != 0; p = previews.next())
		delete p;
	previews.clear();
}


void KMLOFaxApp::addPreview(Preview *p)
{
	previews.append(p);
}


void KMLOFaxApp::removePreview(Preview *p)
{
	previews.remove(p);
}


Preview *KMLOFaxApp::getPreview()
{
	Preview *p = previews.first();

	if (p == 0) {
		p = new Preview(0, "preview");
		Q_CHECK_PTR(p);
	}

	return p;
}


bool KMLOFaxApp::x11EventFilter(XEvent *ev)
{
	if (KUniqueApplication::x11EventFilter(ev))
		return true;
		
	Preview *p;
	for (p = previews.first(); p != 0; p = previews.next())
		if (p->faxWidget()->x11EventFilter(ev))
			return false;

	return false;
}



KMLOFax::KMLOFax(QWidget *parent, const char *name): KMainWindow(parent, name)
{
	config = kapp->config();

	initActions();
	initMenuBar();
	initToolBar();
	initStatusBar();
	
	senderaliases = new SenderAliases(this, "senderaliases");
	Q_CHECK_PTR(senderaliases);

	config->setGroup(ID_PREF_GROUP_GENERAL);
	senderaliases->addToolTips(config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS));

	view = new FaxFiler(this, "faxfiler");
	Q_CHECK_PTR(view);
	setCentralWidget(view);
	
	initWhatsThis();
	
	loader = new MLOLoader(this, "mloloader");
	Q_CHECK_PTR(loader);
	
	readOptions();

	connect(loader, SIGNAL(totalSteps(int)), progress, SLOT(setTotalSteps(int)));
	connect(loader, SIGNAL(progress(int)), progress, SLOT(setProgress(int)));
	connect(loader, SIGNAL(message(const QString &)), SLOT(slotStatusMsg(const QString &)));
	connect(loader, SIGNAL(helpMessage(const QString &)), SLOT(slotStatusHelpMsg(const QString &)));
	connect(loader, SIGNAL(gotFile(const QString &)), SLOT(fetchFile(const QString &)));
	connect(loader, SIGNAL(done()), SLOT(fetchDone()));
	
	connect(view, SIGNAL(hasCurrentFolder(bool, bool)), SLOT(hasCurrentFolderSlot(bool, bool)));
	connect(view, SIGNAL(hasCurrentFax(bool)), SLOT(hasCurrentFaxSlot(bool)));
	connect(view, SIGNAL(command(int)), SLOT(commandCallback(int)));
	
	connect(progress, SIGNAL(aborted()), loader, SLOT(cancel()));

	show();

	view->setInboxFolder();
	view->updateFromPre21();
}


KMLOFax::~KMLOFax()
{
}


void KMLOFax::readOptions()
{
	config->setGroup("MainWindow Options");

	settingsShowToolbar->setChecked(config->readBoolEntry("Show ToolBar", true));
	slotSettingsShowToolbar();
	settingsShowStatusbar->setChecked(config->readBoolEntry("Show StatusBar", true));
	slotSettingsShowStatusbar();
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));
	QSize geoSize(596, 408);
	resize(config->readSizeEntry("Geometry", &geoSize));

	view->readOptions();
}



void KMLOFax::saveOptions()
{
	config->setGroup("MainWindow Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());

	view->saveOptions();
}


void KMLOFax::initActions()
{
	fileDownload = new KAction(i18n("&Download"), "fax_get", KShortcut(CTRL+Key_D), this, SLOT(slotFileDownload()), actionCollection(), "file_download");
	Q_CHECK_PTR(fileDownload);
	fileDownload->setWhatsThis(i18n("Click this button to download facsimiles from your modem's flash memory.<br><br>" \
		"You can also select the command <b>Download</b> from the \"File\" menu."));
	fileView = new KAction(i18n("&View..."), "viewmag", KShortcut(CTRL+Key_V), this, SLOT(slotFileView()), actionCollection(), "file_view");
	Q_CHECK_PTR(fileView);
	fileView->setWhatsThis(i18n("Click this button to view the facsimile that " \
		"is currently selected in the list below.<br><br>" \
		"You can also select the command <b>View</b> from the \"File\" menu."));
	filePrint = KStdAction::print(this, SLOT(slotFilePrint()), actionCollection());
	Q_CHECK_PTR(filePrint);
	filePrint->setWhatsThis(i18n("Click this button to print the facsimile that " \
		"is currently selected in the list below.<br><br>" \
		"You can also select the command <b>Print</b> from the \"File\" menu."));
	fileExport = new KAction(i18n("&Export..."), "filesave", KShortcut(CTRL+Key_E), this, SLOT(slotFileExport()), actionCollection(), "file_export");
	Q_CHECK_PTR(fileExport);
	fileExport->setWhatsThis(i18n("Click this button to export the facsimile " \
		"that is currently selected in the list below into a TIFF file.<br><br>" \
		"You can also select the command <b>Export</b> from the \"File\" menu."));
	fileMail = new KAction(i18n("&Mail..."), "mail_send", KShortcut(CTRL+Key_M), this, SLOT(slotFileMail()), actionCollection(), "file_mail");
	Q_CHECK_PTR(fileMail);
	fileMail->setWhatsThis(i18n("Click this button to send the facsimile that is " \
		"currently selected in the list below as email.<br><br>" \
		"You can also select the command <b>Mail</b> from the \"File\" menu."));
	fileTrash = new KAction(i18n("Move to &Trash"), "trashcan_empty", KShortcut(Key_Delete), this, SLOT(slotFileTrash()), actionCollection(), "file_trash");
	Q_CHECK_PTR(fileTrash);
	fileTrash->setWhatsThis(i18n("Click this button to move the facsimile that " \
		"is currently selected in the list below into the trash folder.<br><br>" \
		"You can also select the command <b>Move to Trash</b> from the \"File\" menu."));
	fileSenderAlias = new KAction(i18n("Sender &Alias..."), 0, 0, this, SLOT(slotFileSenderAlias()), actionCollection(), "file_sender_alias");
	Q_CHECK_PTR(fileSenderAlias);
	fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
	Q_CHECK_PTR(fileQuit);
	fileQuit->setWhatsThis(i18n("Click this button to quit KMLOFax.<br><br>" \
		"You can also select the command <b>Quit</b> from the \"File\" menu."));

	folderNew = new KAction(i18n("&New Folder..."), "folder_new", 0, this, SLOT(slotFolderNew()), actionCollection(), "folder_new");
	Q_CHECK_PTR(folderNew);
	folderRename = new KAction(i18n("&Rename..."), 0, 0, this, SLOT(slotFolderRename()), actionCollection(), "folder_rename");
	Q_CHECK_PTR(folderRename);
	folderEmpty = new KAction(i18n("&Empty"), 0, 0, this, SLOT(slotFolderEmpty()), actionCollection(), "folder_empty");
	Q_CHECK_PTR(folderEmpty);
	folderRemove = new KAction(i18n("Re&move"), "remove", 0, this, SLOT(slotFolderRemove()), actionCollection(), "folder_remove");
	Q_CHECK_PTR(folderRemove);

	settingsShowToolbar = KStdAction::showToolbar(this, SLOT(slotSettingsShowToolbar()), actionCollection());
	Q_CHECK_PTR(settingsShowToolbar);
	settingsShowStatusbar = KStdAction::showStatusbar(this, SLOT(slotSettingsShowStatusbar()), actionCollection());
	Q_CHECK_PTR(settingsShowStatusbar);
	settingsSaveOptions = new KAction(i18n("Save &Options"), 0, 0, this, SLOT(saveOptions()), actionCollection(), "settings_save_options");
	Q_CHECK_PTR(settingsSaveOptions);
	settingsSenderAliases = new KAction(i18n("Sender &Aliases..."), 0, 0, this, SLOT(slotSettingsSenderAliases()), actionCollection(), "settings_sender_aliases");
	Q_CHECK_PTR(settingsSenderAliases);
	settingsConfigure = new KAction(i18n("&Configure KMLOFax..."), "configure", 0, this, SLOT(slotSettingsConfigure()), actionCollection(), "settings_configure");
	Q_CHECK_PTR(settingsConfigure);
	settingsConfigure->setWhatsThis(i18n("Opens the configure dialog.<br><br>" \
		"You can also select the command <b>Configure KMLOFax</b> from the \"Settings\" menu."));
}


void KMLOFax::initMenuBar()
{
	QPopupMenu *fileMenu, *folderMenu, *settingsMenu;

	fileMenu = new QPopupMenu(0, "mainfilemenu");
	Q_CHECK_PTR(fileMenu);
	fileDownload->plug(fileMenu);
	fileMenu->insertSeparator();
	fileView->plug(fileMenu);
	filePrint->plug(fileMenu);
	fileExport->plug(fileMenu);
	fileMail->plug(fileMenu);
	fileTrash->plug(fileMenu);
	fileSenderAlias->plug(fileMenu);
	fileMenu->insertSeparator();
	fileQuit->plug(fileMenu);

	folderMenu = new QPopupMenu(0, "mainfoldermenu");
	Q_CHECK_PTR(folderMenu);
	folderNew->plug(folderMenu);
	folderRename->plug(folderMenu);
	folderMenu->insertSeparator();
	folderEmpty->plug(folderMenu);
	folderRemove->plug(folderMenu);

	settingsMenu = new QPopupMenu(0, "mainsettingsmenu");
	Q_CHECK_PTR(settingsMenu);
	settingsShowToolbar->plug(settingsMenu);
	settingsShowStatusbar->plug(settingsMenu);
	settingsMenu->insertSeparator();
	settingsSaveOptions->plug(settingsMenu);
	settingsMenu->insertSeparator();
	settingsSenderAliases->plug(settingsMenu);
	settingsConfigure->plug(settingsMenu);

	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("F&older"), folderMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), helpMenu());
}


void KMLOFax::initToolBar()
{
	fileDownload->plug(toolBar());
	toolBar()->insertSeparator();
	fileView->plug(toolBar());
	filePrint->plug(toolBar());
	fileExport->plug(toolBar());
	fileMail->plug(toolBar());
	fileTrash->plug(toolBar());
	toolBar()->insertSeparator();
	settingsConfigure->plug(toolBar());
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("help"), ID_HELP_CONTENTS, SIGNAL(clicked()), this, SLOT(appHelpActivated()), true, i18n("Help"));
	toolBar()->alignItemRight(fileQuit->itemId(fileQuit->plug(toolBar())));
	QWhatsThis::add(toolBar()->getButton(ID_HELP_CONTENTS),
		i18n("Invokes the KDE help system with the KMLOFax help pages displayed.<br><br>" \
		"You can also select the command <b>Contents</b> from the \"Help\" menu."));
}


void KMLOFax::initStatusBar()
{
	progress = new StatusProgress(statusBar());
	Q_CHECK_PTR(progress);
	
	statusBar()->addWidget(progress, 0, true);
	
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG, 1);
	statusBar()->setItemAlignment(ID_STATUS_MSG, AlignLeft | AlignVCenter);

	progress->setEnabled(false);
}


void KMLOFax::initWhatsThis()
{
	QWhatsThis::add(view,
		i18n("The list shows the facsimiles you have formerly " \
		"received. The columns of this list have the following meaning:<br>" \
		"<dl compact>"
		"<dt>Date (Time)<dd>Date and time when your modem received the facsimile." \
		"<dt>Sender<dd>Station id of the sender where the facsimile came from." \
		"<dt>Pages<dd>Number of pages in the facsimile document." \
		"<dt>Size<dd>File size in bytes of the facsimile file." \
		"</dl>"));
}


void KMLOFax::commandCallback(int id_)
{
	switch (id_) {
		case ID_FILE_DOWNLOAD:
			slotFileDownload();
			break;
		case ID_FILE_VIEW:
			slotFileView();
			break;
		case ID_FILE_PRINT:
			slotFilePrint();
			break;
		case ID_FILE_EXPORT:
			slotFileExport();
			break;
		case ID_FILE_MAIL:
			slotFileMail();
			break;
		case ID_FILE_TRASH:
			slotFileTrash();
			break;
		case ID_FILE_SENDER_ALIAS:
			slotFileSenderAlias();
			break;
		case ID_FILE_REBUILD_DIRECTORY:
			slotFileRebuildDir();
			break;
		case ID_FILE_QUIT:
			slotFileQuit();
			break;
		case ID_FOLDER_NEW:
			slotFolderNew();
			break;
		case ID_FOLDER_RENAME:
			slotFolderRename();
			break;
		case ID_FOLDER_EMPTY:
			slotFolderEmpty();
			break;
		case ID_FOLDER_REMOVE:
			slotFolderRemove();
			break;
		case ID_SETTINGS_TOOLBAR:
			slotSettingsShowToolbar();
			break;
		case ID_SETTINGS_STATUSBAR:
			slotSettingsShowStatusbar();
			break;
		case ID_SETTINGS_SAVEOPTIONS:
			saveOptions();
			break;
		case ID_SETTINGS_CONFIGURE:
			slotSettingsConfigure();
			break;
		case ID_SETTINGS_SENDER_ALIASES:
			slotSettingsSenderAliases();
			break;
		default:
			break;
	}
}


void KMLOFax::slotFileDownload()
{
	if (loader->startDownload()) {
		progress->setEnabled(true);
		fileDownload->setEnabled(false);
		slotStatusHelpMsg(i18n("Looking for new facsimiles..."));
	}
}


void KMLOFax::slotFileView()
{
	if (!view->currentFaxName().isEmpty()) {
		Preview *p;
		config->setGroup(ID_PREF_GROUP_GENERAL);
		switch (config->readNumEntry(ID_PREF_GEN_VIEWMODE, PREF_GEN_VIEWMODE)) {
			case 0:
				p = ((KMLOFaxApp *)kapp)->getPreview();
				p->showFax(view->currentPath() + "/" + view->currentFaxName());
				break;
			case 1:
				p = new Preview(0, "preview");
				Q_CHECK_PTR(p);
				p->showFax(view->currentPath() + "/" + view->currentFaxName());
				break;
			case 2:
				KRun::runURL(expandPath(view->currentPath() + "/" + view->currentFaxName()), "image/tiff");
				break;
			case 3:
				proc.clearArguments();
				proc << config->readEntry(ID_PREF_GEN_VIEWER, PREF_GEN_VIEWER);
				proc << expandPath(view->currentPath() + "/" + view->currentFaxName());
				proc.start(KProcess::DontCare);
				break;
			default:
				break;
		}
		view->setCurrentFaxNew(false);
	}
}


void KMLOFax::slotFilePrint()
{
	if (!view->currentFaxName().isEmpty()) {
		PrintDlg printdlg(this, "printdlg");
		printdlg.printFax(view->currentPath() + "/" + view->currentFaxName());
		if (printdlg.exec())
			view->setCurrentFaxNew(false);
	}
}


void KMLOFax::slotFileExport()
{
	if (!view->currentFaxName().isEmpty()) {
		ExportDlg exportdlg(this, "exportdlg");
		exportdlg.exportFax(view->currentPath() + "/" + view->currentFaxName());
		exportdlg.exec();
	}
}


void KMLOFax::slotFileMail()
{
	if (!view->currentFaxName().isEmpty()) {
		MailDlg maildlg(this, "maildlg");
		maildlg.mailFax(view->currentPath() + "/" + view->currentFaxName());
		maildlg.exec();
	}
}


void KMLOFax::slotFileTrash()
{
	view->deleteFax(false);
}


void KMLOFax::slotFileSenderAlias()
{
	if (!view->currentFaxName().isEmpty())
		senderaliases->setAlias(view->currentFaxSender());
}


void KMLOFax::slotFileRebuildDir()
{
}


void KMLOFax::slotFileQuit()
{
	close();
}


void KMLOFax::slotFolderNew()
{
	view->newFolder();
}


void KMLOFax::slotFolderRename()
{
	view->renameFolder();
}


void KMLOFax::slotFolderEmpty()
{
	view->emptyFolder();
}


void KMLOFax::slotFolderRemove()
{
	view->removeFolder();
}


void KMLOFax::slotSettingsShowToolbar()
{
	if (settingsShowToolbar->isChecked())
		toolBar()->show();
	else
		toolBar()->hide();
}


void KMLOFax::slotSettingsShowStatusbar()
{
	if (settingsShowStatusbar->isChecked())
		statusBar()->show();
	else
		statusBar()->hide();
}


void KMLOFax::slotSettingsConfigure()
{
	Configure configure(this, "configure");
	if (configure.exec()) {
		config->setGroup(ID_PREF_GROUP_GENERAL);
		senderaliases->addToolTips(config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS));
	}
}


void KMLOFax::slotSettingsSenderAliases()
{
	if (senderaliases->isVisible())
		senderaliases->raise();
	else
		senderaliases->show();
}


void KMLOFax::slotStatusMsg(const QString &text)
{
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KMLOFax::slotStatusHelpMsg(const QString &text)
{
	statusBar()->message(text, 2000);
}


void KMLOFax::hasCurrentFolderSlot(bool isRoot, bool isUser)
{
	folderNew->setEnabled(isRoot || isUser);
	folderRename->setEnabled(isUser);
	folderEmpty->setEnabled(!isRoot);
	folderRemove->setEnabled(isUser);
}


void KMLOFax::hasCurrentFaxSlot(bool yesno)
{
	fileView->setEnabled(yesno);
	filePrint->setEnabled(yesno);
	fileExport->setEnabled(yesno);
	fileMail->setEnabled(yesno);
	fileTrash->setEnabled(yesno);
	fileSenderAlias->setEnabled(yesno);
}


void KMLOFax::fetchFile(const QString &name)
{
	view->newFaxItem(name);
}


void KMLOFax::fetchDone()
{
	progress->setEnabled(false);

	slotStatusMsg(i18n("Ready."));

	fileDownload->setEnabled(true);
}
