// -*-Mode: C++;-*- 

#ifndef _MYSQL_H
#define _MYSQL_H

#include "config.h"

#ifdef HAVE_MYSQL_LIBS

#include "PatternPair.h"
#include "tabledescr.h"
#include "helpers/stringlist.h"
#include "sendmessage.h"
#include "change_type.h"
#include "my_sql_types.h"
#include "helpers/smart_pointer.h"

struct st_mysql;
struct st_mysql_res;

typedef st_mysql MYSQL;
typedef st_mysql_res MYSQL_RES;

#endif

//! Sql Wrapper class
/*!
 * This is the wrapper class for the mysql-c-lib
 *
 * It contains the most routins to connect and deal with a sql-server
 *
 * \author Rajko Albrecht
 * \version $Id: my_sql.h,v 2.32 2003/02/04 00:08:53 ral Exp $
 */
class CMySql:virtual public ref_count
{
#ifdef HAVE_MYSQL_LIBS
protected:
    //! internal fetch_row
    int _fetch_rows(const char*db,const char*table,const char*sort_by,
                   stringlist*which,unsigned int offset,unsigned int count,
                   PatternList*Pattern,bool errors_only,MYSQL_RES**Res);
    //! create a INSERT/REPLACE statment
    /*!
     * \sa insert_row, replace_row
     */
    int insert_replace_row(const std::string what, const char*db,const char*table, PatternList*Pattern);
    //! The connection itself
    MYSQL*Sql_Connection;
    //! a simple blocker
    int nothing_else;
    //! the callback for messages
    CSendMessage*message_call_back;
    //! send a message via the callback
    /*!
     * \param text a char pointer containing the message
     * \sa do_message(const std::string&text)
     */
    void do_message(const char*text)const{if (message_call_back) message_call_back->append_message(text);}
    //! send a message via the callback
    /*!
     * \param text a standart string object containing the message
     * \sa do_message(const char*text)
     */
    void do_message(const std::string&text)const{if (message_call_back) message_call_back->append_message(text);}
    //! Make a string sql clean
    std::string&do_escape(const std::string&what,std::string&target);
    //! refreshes the state of sql server
    /*!
     * \param What an integer selecting the part of sql to refresh.
     * this parameter is defined inside the sql lib from TCX
     */
    int do_refresh(int What, const char*ok_msg);
    //! Server version 
    /*!
     * the number before first dot
     */
    int Server_MasterVersion;
    //! Server version
    /*!
     * the number after the first dot
     */
    int Server_MinorVersion;
    //! Server version
    /*!
     * the number after the last dot
     */
    int Server_PatchLevel;
    //! array of messages
    /*!
     * here are often used strings stored
     */
    static const char* MessageArray[];
    //!Column types known
    static const my_sql_type f_arr[];
    //! size of MessageArray
    static int MessageCount;
    //! infos about connected server
    std::string connect_info;
    //! build sql command
    /*!
     * builds a sql descriptor of the columns given with desc, used
     * for alter table or create table
     * \params target string to store the description
     * \param desc List of column descriptors
     */
    int build_command(std::string&target,CColumnDescriptor&desc)const;
    //! build sql command
    /*!
     * builds a sql descripton of keys given with keys, used 
     * for alter table or create table
     * \param target string to store the description
     * \param keys key descriptors
     * \param add generate a ADD statement?
     */
    static int build_key_command(std::string&target,const KeyList*keys,bool add = false);
    //! the host name connected to
    std::string m_Host,
	//! the password used
	m_Pass,
	//! the username used
	m_User,
	//! on which tcp/ip port connected
	m_Port;

public:
    const std::string&Host()const{return m_Host;}
    const std::string&User()const{return m_User;}
    const std::string&Pass()const{return m_Pass;}
    const std::string&Port()const{return m_Port;}
    //! Select Database
    /*!
     * Selects a database of server to use, mostly used internal
     */
    int connect_db(const char*db)const;
    //! Reload grant table
    int refresh_grants();
    //! flush the cache of sql server
    int refresh_cache();
    //! reload tables
    int refresh_tables();
    //! refreshs the log of sql server
    int refresh_log();
    //! shut down the sql server
    int shutdown();
    //! convert the password
    void scramble_password(std::string&pass)const;
    //! the count of field descriptors
    static int FieldDescriptCounts() ;
    //! get a specific field descriptor
    static const char* FieldDescript(int i);
    //! get specific type name
    static const char* TypeDescript(int i);
    //! is the given db name a system database?
    static bool is_system_db(const char*name);
    //! retrieve type descriptor
    /*!
     * tries to retrieve a sql type which is represented with the parameter
     * \param i the index of type
     */
    static const my_sql_type&sql_type(int i);

    //! Constructor
    /*!
     * Constructor - constructs an empty SQL-class
     *
     * \author Rajko Albrecht
     */
    CMySql();
    //! Copy constructor
    /*!
     * \param old the object to be copied
     */
    CMySql(const CMySql&old);
    //! Constructor
    /*!
     * Constructs an SQL-class and connects to the server with 
     * the given parameters.
     * 
     * \sa sql_connect
     * \param host contains the host to connect to, if empty, localhost is used
     * \param user the user - if empty, current loginname is used
     * \param pass - password of the user at host
     * \param port - the port the class should use. If connect to the localhost
     * via unix-socket, unused
     */
    CMySql(const char*host,const char*user,const char*pass,const char*port);
    /**
     * Destructor - disconnects and destruct the object
     */
    virtual ~CMySql();
    //! Connect to sql server
    /*!
     * Connects to a mysql-server with the given parameter
     * \param host contains the host to connect to, if empty, localhost is used
     * \param user the user - if empty, current loginname is used
     * \param pass password of the user at host
     * \param port the port the class should use. If connect to the localhost  via unix-socket, unused
     * \sa success_connect, sql_disconnect
     */
    void sql_connect(const char*host,const char*user,const char*pass,const char*port);
    //! Connect to sql server
    /*!
     * Connects to a mysql-server with the given parameter
     * \param host contains the host to connect to, if empty, localhost is used
     * \param user the user - if empty, current loginname is used
     * \param pass password of the user at host
     * \param port the port the class should use. If connect to the localhost  via unix-socket, unused
     * \sa success_connect, sql_disconnect
     */
    void sql_connect(const std::string&host,const std::string&user,
		     const std::string&pass,const std::string&port);
    //! Disconnect
    /*!
      Disconnects from a server if connected
      \sa sql_connect, success_connect
    */
    void sql_disconnect();
    //! Retrieves infos
    /*!
     * Retrieves infos about the connected Sql-Server.
     *
     * \return A string, which contains the info
     */
    const std::string&get_info();
    //! Serverstatus
    /*!
     * Returns the current status of the server (threads, open tables etc.)
     * as String
     */
    std::string get_status();
    //! Get last error information
    /*!
      returns the last error statement. Most of them will printed out via the messagehandler.
     \sa set_message_call_back
    */
    std::string get_last_error()const;
    //! connection status
    /*!
     returns the connection status of the class
     \return 1 if connected, 0 if not connected
    */
    int success_connect()const{return (Sql_Connection != 0);}
    /**
     * returns a list of the threads of the server. Means, which user
     * are connected, which ids etc.
     *
     * @see #kill_thread
     */
    bool get_threads(stringlist&target)const;
    bool get_threads(list_vector&target)const;
    /**
     * kills the server thread with the id "pid". 
     *
     * If you kill your own, you've get an error-message!
     *
     * @param pid the id of the thread
     * @return 1 if successfull else 0
     * @see #get_threads
     */
    int kill_thread(unsigned long pid);
    /**
     * retrieves a list of databases in the server
     *
     * @param pattern a pattern to search for
     * @return a pointer to a list or 0, if no databases were found
     */
    bool get_db_list(const char*pattern,stringlist&target)const;
    //! Tablelist
    /**
      returns a list of tables in the given database
      @param pattern table matching pattern (% for all)
      @param db database to use
      @param target a stringlist, which will filled with the table information
      @return true if success else error
    */
    bool get_table_list(const char*pattern,const char*db,stringlist&target);
   //! Columnslist
    /*!
      returns a list of columns in a table
      \param db Database to use
      \param table table to use
      \param target target list to fill
      \return true if success or false if failure
    */
    bool get_table_fields(const char*db,const char*table,ColumnList&target)const;
    //! Key list
    /*!
      returns a list of keys of a table
      \param db Database to use (if db == 0, last used db will used again)
      \param table table the keys are wanted
      \param target where to store the key list
      \return true if success false if failure
    */
    bool get_table_keys(const char*db, const char*table, KeyList&target)const;
    /**
     * Creates a new database
     * @param db name of database
     * @return 0 if success, else errorcode
    */
    int create_db(const char*db);
    //!Create a table
    /*!
     * Creates a new table
     * \param db database to use
     * \param name name of the new table
     * \param columns a pointer to a list of CColumnDescriptor
     * \param keys a pointer to a list of CKeyDescriptor, may be NULL
     * \return true if success, false if failure
     */
    bool create_table(const char*db,const char*name,ColumnList*columns,KeyList*keys = 0);
    //! retrieves a list of rows
    /*!
     * This function retrieves a list of rows from a table. The entrys are justified
     * to fit into the largest returned value of a column.
     *
     * \return 0 if success, -1 if not success
     * \param db      the name of the database to use
     * \param table   the table to use
     * \param sort_by sort by which value
     * \param which   which of the tablefields should retrieved
     * \param offset  if 0 start by first+offset element with search
     * \param count   if 0 show all found, if > 0, do not show more then count elements
     * \param Pattern a list with entrys to search for. Every pattern the wildcard is %, if %% is supplied, 
     *                the function rebuild it to \% and search for a percent-sign.
     * \param errors_only if true, send only error messages to the protocol
     */
    int fetch_rows(const char*db,const char*table,
		   const char*sort_by,stringlist*which,
		   unsigned int offset = 0,unsigned int count = 0,
		   PatternList*Pattern = 0,bool errors_only = false);
    int fetch_rows(const char*db,const char*table,
                   const char*sort_by,stringlist*which,list_vector&target,
                   unsigned int offset = 0,unsigned int count = 0,
                   PatternList*Pattern = 0,bool errors_only = false);
    //!inserts a new row into table "table" in database db.
     /*!
      * \param db the database, if empty, current is used
      * \param table the table where to insert, can not be blank
      * \param Pattern a list of columnname-value pairs to insert into the table
      * \return 0 if success, -1 if error
      * \sa replace_row
      */
    int insert_row(const char*db,const char*table, PatternList*Pattern)
    { return insert_replace_row("INSERT",db,table,Pattern);}
    //!inserts a new row into table "table" in database db.
     /*!
      * It using the "REPLACE INTO" instead of "INSERT INTO" and will so replace existing entries
      * with the same key.
      * \param db the database, if empty, current is used
      * \param table the table where to insert, can not be blank
      * \param Pattern a list of columnname-value pairs to insert into the table
      * \return 0 if success, -1 if error
      * \sa insert_row
      */
    int replace_row(const char*db,const char*table, PatternList*Pattern)
    { return insert_replace_row("REPLACE",db,table,Pattern);}

    //! modifies a row
    /*!
     * \param db the database to use, if 0, current is used
     * \param table the table to work on, can not be 0
     * \param to_change list of values to modifiy. CPatternPair->entry the column to change, ->pattern the new value
     * \param identifiers a list of identifiers which rows are to change - they will with AND selected 
     *                    (means " where PairA->entry=PairA->pattern AND PairB->entry=PairB->pattern" )
     */
    int update_row(const char*db,const char*table,
		   PatternList*to_change,PatternList*identifiers);
    /**
     * doing a "DELETE" statement in a table. 
     *
     * @param db the database, if 0, current is used
     * @param table the table to use, cannot be null
     * @param identifiers list of identifiers to select the row(s) to delete. If 0, all rows in table will bedeleted
     *                    @see update_row
     */
    int delete_row(const char*db,const char*table,
		   PatternList*identifiers=0);
    //! Deleting table
    /*! 
      Deletes a table from a database
      \param db Database to use
      \param table Table to delete
      \return true if success false if error
      \sa create_table
    */
    bool delete_table(const char*db, const char*table);
    //! Renaming a table
    /*!
     * Renames a table in a given database
     *
     * \param db database to use
     * \param table table to rename
     * \param new_name the new name of the table
     * \return true if success else false
     */
    bool rename_table(const char*db,const char*table,const char*new_name)const;

    //! Alter tables
    /*!
     * Alter a table in a given database
     * \param db database to use
     * \param table table to alter
     * \param n_columns the columns of the table
     * \param n_keys the keys of the table
     * \param columns_change protocoll of changed columns
     * \param key_change protocoll of key_change
     * \return true if it was success, false if failure
     */
    bool alter_table(const char*db, const char*table,ColumnList&n_columns,const KeyList&n_keys,
		     change_list&columns_change,KeyList&key_orig)const;
    /**
     * Setup a callback handler where the class can store it
     * messages. 
     *
     * @param callback a Class based on CSendMessage or NULL if no message output wanted
     * @return none
     */
    void set_message_call_back(CSendMessage*callback){message_call_back=callback;}
    //! make a sql statement
    /*!
     * sends blank sql statements to the connected server
     * \param statement the statment(s) itself
     * \param target a list reference, where the results should be stored
     * \param sizes a list reference, where the maximum sizes of the columns are stored
     * \param headers a list reference where the column names are stored
     * \param log_error_only the kind of logging. If true, then only failures will be logged
     */    
    bool do_statement(const char*statement,stringlist&target,stringlist&sizes,stringlist&headers,
		      bool log_error_only = false);
#else
    CMySql(){}
    virtual ~CMySql(){};
#endif
};

#ifdef HAVE_MYSQL_LIBS
inline CMySql::CMySql(const char*host,const char*user,const char*pass,const char*port)
{
    Sql_Connection = 0L;
    message_call_back = 0;
    sql_connect(host,user,pass,port);
}
#endif

#endif

