/*
 * KHeiseReg
 *
 * A utility to search for articles within the Heise register.
 *
 * Copyright (C) 2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qglobal.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qtooltip.h>
#include <qstring.h>
#include <qpushbutton.h>

#include <kapplication.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kfiledialog.h>

#include "global.h"
#include "configure.h"



Configure::Configure(QWidget *parent, const char *name): KDialogBase(IconList, i18n("Configure"), Help|Default|Ok|Apply|Cancel, Ok, parent, name, true, true)
{
	QBoxLayout *vbox, *hbox, *svbox, *shbox;
	QGridLayout *grid;
	QGroupBox *gb;
	QFrame *page;
	QLabel *label;
	QPushButton *browse;

	config = kapp->config();

	setHelp(QString::fromLatin1("configure-help"));
	setIconListAllVisible(true);

	/* Page 1: General */
	page = addPage(i18n("General"), i18n("General Settings"), UserIcon(QString::fromLatin1("general")));

	vbox = new QVBoxLayout(page, spacingHint());

	gb = new QGroupBox(i18n("User Interface"), page);
	svbox = new QVBoxLayout(gb, spacingHint());
	svbox->addSpacing(fontMetrics().height());

	gen_tooltips = new QCheckBox(i18n("Show tooltips"), gb);
	gen_tooltips->setMinimumSize(gen_tooltips->sizeHint());
	svbox->addWidget(gen_tooltips);
	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();


	/* Page 2: Search */
	page = addPage(i18n("Search"), i18n("Search Settings"), UserIcon(QString::fromLatin1("search")));

	vbox = new QVBoxLayout(page, spacingHint());

	grid = new QGridLayout(6, 3, spacingHint());
	vbox->addLayout(grid);

	label = new QLabel(i18n("&Bywords:"), page);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 0, 0, AlignVCenter | AlignRight);

	sea_bywords = new QLineEdit(page);
	label->setBuddy(sea_bywords);
	sea_bywords->setMinimumSize(sea_bywords->sizeHint());
	grid->addMultiCellWidget(sea_bywords, 0, 0, 1, 5);

	label = new QLabel(i18n("&Authors:"), page);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 1, 0, AlignVCenter | AlignRight);

	sea_authors = new QLineEdit(page);
	label->setBuddy(sea_authors);
	sea_authors->setMinimumSize(sea_authors->sizeHint());
	grid->addMultiCellWidget(sea_authors, 1, 1, 1, 5);

	label = new QLabel(i18n("&Editors:"), page);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 2, 0, AlignVCenter | AlignRight);

	sea_editors = new QLineEdit(page);
	label->setBuddy(sea_editors);
	sea_editors->setMinimumSize(sea_editors->sizeHint());
	grid->addWidget(sea_editors, 2, 1);

	label = new QLabel(i18n("E&ditions:"), page);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 2, 2, AlignVCenter | AlignRight);

	sea_editions = new QLineEdit(page);
	label->setBuddy(sea_editions);
	sea_editions->setMinimumSize(sea_editions->sizeHint());
	grid->addWidget(sea_editions, 2, 3);

	label = new QLabel(i18n("&Years:"), page);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 2, 4, AlignVCenter | AlignRight);

	sea_years = new QLineEdit(page);
	label->setBuddy(sea_years);
	sea_years->setMinimumSize(sea_years->sizeHint());
	grid->addWidget(sea_years, 2, 5);

	grid->setColStretch(1, 1);
	grid->setColStretch(3, 1);
	grid->setColStretch(5, 1);

	hbox = new QHBoxLayout(spacingHint());
	vbox->addLayout(hbox);

	gb = new QGroupBox(i18n("Magazines"), page);

	svbox = new QVBoxLayout(gb, spacingHint());
	svbox->addSpacing(fontMetrics().height());

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	sea_ctmag = new QCheckBox(i18n("c'&t Magazine"), gb);
	sea_ctmag->setFixedSize(sea_ctmag->sizeHint());
	shbox->addWidget(sea_ctmag);
	shbox->addStretch();

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	sea_ixmag = new QCheckBox(i18n("&iX Magazine"), gb);
	sea_ixmag->setFixedSize(sea_ixmag->sizeHint());
	shbox->addWidget(sea_ixmag);
	shbox->addStretch();

	svbox->addStretch();

	svbox->activate();

	hbox->addWidget(gb);

	gb = new QGroupBox(i18n("Options"), page);

	svbox = new QVBoxLayout(gb, spacingHint());
	svbox->addSpacing(fontMetrics().height());

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	sea_case_sensitive = new QCheckBox(i18n("Cas&e sensitive"), gb);
	sea_case_sensitive->setFixedSize(sea_case_sensitive->sizeHint());
	shbox->addWidget(sea_case_sensitive);
	shbox->addStretch();

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	sea_fuzzy = new QCheckBox(i18n("&Fuzzy"), gb);
	sea_fuzzy->setFixedSize(sea_fuzzy->sizeHint());
	shbox->addWidget(sea_fuzzy);
	shbox->addStretch();

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	shbox->addSpacing(19);
	sea_min_threshold = new QLabel(QString::fromLatin1("0%"), gb);
	sea_min_threshold->setFixedSize(sea_min_threshold->sizeHint());
	sea_min_threshold->setEnabled(false);
	shbox->addWidget(sea_min_threshold);
	sea_threshold = new QSlider(0, 100, 10, 0, QSlider::Horizontal, gb);
	sea_threshold->setTickmarks(QSlider::Below);
	sea_threshold->setTickInterval(10);
	sea_threshold->setMinimumSize(sea_threshold->sizeHint());
	sea_threshold->setEnabled(false);
	shbox->addWidget(sea_threshold);
	sea_max_threshold = new QLabel(QString::fromLatin1("100%"), gb);
	sea_max_threshold->setFixedSize(sea_max_threshold->sizeHint());
	sea_max_threshold->setEnabled(false);
	shbox->addWidget(sea_max_threshold);

	svbox->addStretch();
	svbox->activate();

	hbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();


	/* Page 3: Network */
	page = addPage(i18n("Network"), i18n("Network Settings"), UserIcon(QString::fromLatin1("network")));

	vbox = new QVBoxLayout(page, spacingHint());

	gb = new QGroupBox(i18n("Register File FTP Directory URLs"), page);

	svbox = new QVBoxLayout(gb, spacingHint());
	svbox->addSpacing(fontMetrics().height());

	grid = new QGridLayout(2, 2, spacingHint());
	svbox->addLayout(grid);

	label = new QLabel(i18n("c'&t Magazine:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 0);

	net_cturl = new QLineEdit(gb);
	label->setBuddy(net_cturl);
	net_cturl->setMinimumSize(net_cturl->sizeHint());
	grid->addWidget(net_cturl, 0, 1);

	label = new QLabel(i18n("&iX Magazine:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	net_ixurl = new QLineEdit(gb);
	label->setBuddy(net_ixurl);
	net_ixurl->setMinimumSize(net_ixurl->sizeHint());
	grid->addWidget(net_ixurl, 1, 1);

	grid->setColStretch(1, 1);

	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();


	/* Page 4: c't-ROM */
	page = addPage(i18n("c't-ROM"), i18n("c't-ROM Settings"), UserIcon(QString::fromLatin1("ctrom")));

	vbox = new QVBoxLayout(page, spacingHint());

	gb = new QGroupBox(i18n("c't-ROM"), page);
	svbox = new QVBoxLayout(gb, spacingHint());
	svbox->addSpacing(fontMetrics().height());

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);

	label = new QLabel(i18n("&CD mount point:"), gb);
	label->setFixedSize(label->sizeHint());
	shbox->addWidget(label);

	ctrom_mountpt = new QLineEdit(gb);
	label->setBuddy(ctrom_mountpt);
	ctrom_mountpt->setMinimumSize(ctrom_mountpt->sizeHint());
	shbox->addWidget(ctrom_mountpt);

	browse = new QPushButton(gb);
	browse->setPixmap(SmallIcon("fileopen"));
	browse->setAutoDefault(false);
	browse->setFixedSize(browse->sizeHint()+QSize(4, 4));
	shbox->addWidget(browse);

	svbox->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Browser"), page);
	svbox = new QVBoxLayout(gb, spacingHint());
	svbox->addSpacing(fontMetrics().height());

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);

	ctrom_extbrowser = new QCheckBox(i18n("Use alternative browser"), gb);
	ctrom_extbrowser->setMinimumSize(ctrom_extbrowser->sizeHint());
	shbox->addWidget(ctrom_extbrowser);

	shbox->addStretch();

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);

	shbox->addSpacing(19);

	ctrom_browsercom = new QLineEdit(gb);
	ctrom_browsercom->setMinimumSize(ctrom_browsercom->sizeHint());
	ctrom_browsercom->setEnabled(false);
	shbox->addWidget(ctrom_browsercom);
	
	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();


	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_GENERAL));
	if (config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS)) {
		QToolTip::add(gen_tooltips, i18n("Show tooltips in dialogs"));
	}

	connect(sea_fuzzy, SIGNAL(toggled(bool)), this, SLOT(slotFuzzyToggled(bool)));
	connect(ctrom_extbrowser, SIGNAL(toggled(bool)), this, SLOT(slotExternalToggled(bool)));
	connect(browse, SIGNAL(clicked()), SLOT(slotBrowseMountPt()));

	reset();
}


Configure::~Configure()
{
}


void Configure::slotDefault()
{
	gen_tooltips->setChecked(PREF_GEN_TOOLTIPS);
	sea_bywords->setText(QString::fromLatin1(PREF_SEA_BYWORDS));
	sea_authors->setText(QString::fromLatin1(PREF_SEA_AUTHORS));
	sea_editors->setText(QString::fromLatin1(PREF_SEA_EDITORS));
	sea_editions->setText(QString::fromLatin1(PREF_SEA_EDITIONS));
	sea_years->setText(QString::fromLatin1(PREF_SEA_YEARS));
	sea_ctmag->setChecked(PREF_SEA_CTMAG);
	sea_ixmag->setChecked(PREF_SEA_IXMAG);
	sea_case_sensitive->setChecked(PREF_SEA_CASE);
	sea_fuzzy->setChecked(PREF_SEA_FUZZY);
	sea_threshold->setValue(PREF_SEA_THRESHOLD);
	net_cturl->setText(PREF_NET_CTURL);
	net_ixurl->setText(PREF_NET_IXURL);
	ctrom_mountpt->setText(QString::fromLatin1(PREF_CTROM_MOUNTPT));
	ctrom_extbrowser->setChecked(PREF_CTROM_ALTBROWSER);
	ctrom_browsercom->setText(QString::fromLatin1(PREF_CTROM_BROWSERCOM));
}


void Configure::slotApply()
{
	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_GENERAL));
	config->writeEntry(ID_PREF_GEN_TOOLTIPS, gen_tooltips->isChecked());

	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_SEARCH));
	config->writeEntry(ID_PREF_SEA_BYWORDS, sea_bywords->text());
	config->writeEntry(ID_PREF_SEA_AUTHORS, sea_authors->text());
	config->writeEntry(ID_PREF_SEA_EDITORS, sea_editors->text());
	config->writeEntry(ID_PREF_SEA_EDITIONS, sea_editions->text());
	config->writeEntry(ID_PREF_SEA_YEARS, sea_years->text());
	config->writeEntry(ID_PREF_SEA_CTMAG, sea_ctmag->isChecked());
	config->writeEntry(ID_PREF_SEA_IXMAG, sea_ixmag->isChecked());
	config->writeEntry(ID_PREF_SEA_CASE, sea_case_sensitive->isChecked());
	config->writeEntry(ID_PREF_SEA_FUZZY, sea_fuzzy->isChecked());
	config->writeEntry(ID_PREF_SEA_THRESHOLD, sea_threshold->value());

	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_NETWORK));
	config->writeEntry(ID_PREF_NET_CTURL, net_cturl->text());
	config->writeEntry(ID_PREF_NET_IXURL, net_ixurl->text());

	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_CTROM));
	config->writeEntry(ID_PREF_CTROM_MOUNTPT, ctrom_mountpt->text());
	config->writeEntry(ID_PREF_CTROM_ALTBROWSER, ctrom_extbrowser->isChecked());
	config->writeEntry(ID_PREF_CTROM_BROWSERCOM, ctrom_browsercom->text());

	config->sync();
}


void Configure::accept()
{
	slotApply();

	done(Accepted);
}


void Configure::reset()
{
	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_GENERAL));
	gen_tooltips->setChecked(config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS));

	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_SEARCH));
	sea_bywords->setText(config->readEntry(ID_PREF_SEA_BYWORDS, QString::fromLatin1(PREF_SEA_BYWORDS)));
	sea_authors->setText(config->readEntry(ID_PREF_SEA_AUTHORS, QString::fromLatin1(PREF_SEA_AUTHORS)));
	sea_editors->setText(config->readEntry(ID_PREF_SEA_EDITORS, QString::fromLatin1(PREF_SEA_EDITORS)));
	sea_editions->setText(config->readEntry(ID_PREF_SEA_EDITIONS, QString::fromLatin1(PREF_SEA_EDITIONS)));
	sea_years->setText(config->readEntry(ID_PREF_SEA_YEARS, QString::fromLatin1(PREF_SEA_YEARS)));
	sea_ctmag->setChecked(config->readBoolEntry(ID_PREF_SEA_CTMAG, PREF_SEA_CTMAG));
	sea_ixmag->setChecked(config->readBoolEntry(ID_PREF_SEA_IXMAG, PREF_SEA_IXMAG));
	sea_case_sensitive->setChecked(config->readBoolEntry(ID_PREF_SEA_CASE, PREF_SEA_CASE));
	sea_fuzzy->setChecked(config->readBoolEntry(ID_PREF_SEA_FUZZY, PREF_SEA_FUZZY));
	sea_threshold->setValue(config->readNumEntry(ID_PREF_SEA_THRESHOLD, PREF_SEA_THRESHOLD));

	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_NETWORK));
	net_cturl->setText(config->readEntry(ID_PREF_NET_CTURL, QString::fromLatin1(PREF_NET_CTURL)));
	net_ixurl->setText(config->readEntry(ID_PREF_NET_IXURL, QString::fromLatin1(PREF_NET_IXURL)));

	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_CTROM));
	ctrom_mountpt->setText(config->readEntry(ID_PREF_CTROM_MOUNTPT, QString::fromLatin1(PREF_CTROM_MOUNTPT)));
	ctrom_extbrowser->setChecked(config->readBoolEntry(ID_PREF_CTROM_ALTBROWSER, PREF_CTROM_ALTBROWSER));
	ctrom_browsercom->setText(config->readEntry(ID_PREF_CTROM_BROWSERCOM, QString::fromLatin1(PREF_CTROM_BROWSERCOM)));
}


void Configure::slotFuzzyToggled(bool on)
{
	sea_min_threshold->setEnabled(on);
	sea_threshold->setEnabled(on);
	sea_max_threshold->setEnabled(on);
}


void Configure::slotBrowseMountPt()
{
	QString s = ctrom_mountpt->text();

	if (s.isEmpty())
		s = QDir::currentDirPath();

	s = KFileDialog::getExistingDirectory(s, this);
	if (!s.isEmpty())
		ctrom_mountpt->setText(s);
}


void Configure::slotExternalToggled(bool on)
{
	ctrom_browsercom->setEnabled(on);
}
