/***************************************************************************
                          synk.cpp  -  description
                             -------------------
    begin                : Sat Aug 18 13:48:33 NZST 2001
    copyright            : (C) 2001 by Jonathon Sim
    email                : jonathonsim@iname.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "synk.h"
#include "kconnectionlist.h"
#include "connectioneditdialog.h"
#include "global.h"

#include <qpopupmenu.h>
#include <qsplitter.h>

#include <kaction.h>
#include <kapplication.h>
#include <kdebug.h>
#include <khelpmenu.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <ksimpleconfig.h>
#include <kstddirs.h>
#include <ktoolbarbutton.h>
#include <kurl.h>


SynK::SynK(QWidget *parent, const char *name)
    : KMainWindow(parent, name),
      m_isNewAccount( false )
{
    m_connectionEditDialog = 0L;
    initGUI(); // Creates the child widgets, actions etc.
    loadConnections();
}

SynK::~SynK()
{
}

/** Performs the synchronisation on the specified connection */
void SynK::slotSynchronise()
{
    const KSynkConnection * connection=ConnectionList->selectedConnection();
    if (transfer)
    {
        const QString currentfrom=transfer->getConnection().getRemoteURL().prettyURL();
        const QString currentto=transfer->getConnection().getLocalURL().prettyURL();
        const QString currentname=transfer->getConnection().getName();
        KMessageBox::detailedError (this,
                                    i18n("There is a transfer already running."
                                         "Try again when it has finished"),
                                    i18n("The connection being synchronised is:\nName: %1\nRemote URL: %2\nLocal URL: %3").arg(currentname).arg(currentfrom).arg(currentto)
                                    , i18n("Transfer already in progress")) ;
        //TODO:Give the option to cancel the transfer
    }
    else if (connection && connection->isValid() )
    {
        transfer=new KTransferKIO(*connection);
        TransferDetails->start(connection);
        TransferDetails->show();
        connect(transfer,SIGNAL(signalGettingFiles(const KURL::List &)) ,
                TransferDetails, SLOT(slotGettingFile(const KURL::List &)));
        connect(transfer,SIGNAL(signalComplete()) ,
                this, SLOT(slotTransferComplete()));
        connect(transfer,SIGNAL(signalComplete()) ,
                TransferDetails, SLOT(slotTransferComplete()));	
        connect(transfer,SIGNAL(signalPercent(unsigned long, unsigned long)),
                TransferDetails, SLOT(slotPercent(unsigned long, unsigned long)));

        connect(transfer,SIGNAL(signalFilesComplete(const KURL::List &)) ,
                TransferDetails, SLOT(slotFilesDone(const KURL::List &)));
        connect(transfer,SIGNAL(signalFilesCanceled(const KURL::List &)) ,
                TransferDetails, SLOT(slotCanceled(const KURL::List &)));
        connect(transfer,SIGNAL(signalFilesError(const KURL::List &)) ,
                TransferDetails, SLOT(slotError(const KURL::List &)));

        connect(TransferDetails,SIGNAL(signalCancelTransfer()) ,
                this, SLOT(slotCancelTransfer() ) );
        connect(transfer,SIGNAL(signalMessage(const QString &)) ,
                TransferDetails, SLOT(slotMessage(const QString &)));	
        transfer->transfer();
    }
    else if ( connection ) // invalid connection
    {
        KMessageBox::error( this,
                            i18n("The specified connection is improperly configured!"),
                            i18n("Improper Configuration") );
        editConnection( *connection );
    }
    else {
        kdDebug(DEBUG_AREA) << "No connection selected" << endl;
    }
}


void SynK::slotSelectionChanged()
{
    const KSynkConnection *conn = ConnectionList->selectedConnection();
    bool available = conn &&
                     (!m_connectionEditDialog || !m_connectionEditDialog->isVisible());
    m_editAct->setEnabled( available );
    m_removeAct->setEnabled( available );
    m_copyAct->setEnabled( available );
    m_syncAct->setEnabled( available && conn->isValid() );
}

/** Adds a new connection to the list,. */
void SynK::slotAddConnection()
{
    KSynkConnection newconnection;
    //See what to call the connection:
    int i=1;
    while (ConnectionList->getConnection(i18n("New Connection %1").arg(i)) )
        i++;
    newconnection.setName(i18n("New Connection %1").arg(i) );
    ConnectionList->addConnection( newconnection );
    m_isNewAccount = true;
    editConnection( newconnection );
}

void SynK::slotCopyConnection()
{
    if ( !ConnectionList->selectedConnection() )
        return; // shouldn't happen

    //See what to call the connection:
    int i=1;
    QString name = ConnectionList->selectedConnection()->getName();
    while (ConnectionList->getConnection( name + " " + QString::number(i)) )
        i++;
    KSynkConnection newconnection( *ConnectionList->selectedConnection() );
    newconnection.setName( name + " " + QString::number( i ) );
    ConnectionList->addConnection( newconnection );
}

/** Removes a connection from the list */
void SynK::slotRemoveConnection()
{
    ConnectionList->deleteSelectedConnection();
}

/** Edits the specified connection */
void SynK::slotEditConnection()
{
    if ( !ConnectionList->selectedConnection() ) // should never happen
        return;

    editConnection( *ConnectionList->selectedConnection() );
}

void SynK::editConnection( const KSynkConnection& connection )
{
    if ( !m_connectionEditDialog )
    {
        m_connectionEditDialog = new ConnectionEditDialog( this, "conn dlg" );
        connect( m_connectionEditDialog, SIGNAL( finished() ),
                 SLOT( slotConfigFinished() ));
        connect( m_connectionEditDialog, SIGNAL( cancelClicked() ),
                 SLOT( slotConfigCanceled() ));
    }

    // don't create more than one config dialog
    m_addAct->setEnabled( false );
    m_editAct->setEnabled( false );
    // prevent deletion of the current connection
    m_removeAct->setEnabled( false );

    m_connectionEditDialog->setConnection( connection );
    m_connectionEditDialog->show();
}

void SynK::slotConfigCanceled()
{
    // delete newly created account, if it was just added
    if ( m_isNewAccount )
    {
        m_isNewAccount = false;
        ConnectionList->deleteConnection( m_connectionEditDialog->connectionIdentifier() );
    }
}

/** When the connection editor commits its changes, this is called */
void SynK::slotConfigFinished()
{
    m_addAct->setEnabled( true );
    slotSelectionChanged();

    QString oldConnectionName = m_connectionEditDialog->connectionIdentifier();

    // connection may have been deleted in slotConfigCanceled
    if ( ConnectionList->hasConnection( oldConnectionName ) )
    {
        ConnectionList->changeConnection( oldConnectionName,
                                          m_connectionEditDialog->getConnection());
    }
}

/** Handles completion of the transfer */
void SynK::slotTransferComplete()
{
}

/** Saves the list of connections */
void SynK::saveConnections()
{
    KSimpleConfig config( locateLocal ("appdata", "connectionsrc"));
	
    QStringList grouplist = config.groupList() ;
    QStringList::ConstIterator it;
    for ( it = grouplist.begin(); it != grouplist.end(); ++it )	{
        config.deleteGroup( *it );
    }
    KConnectionList::List * list = ConnectionList->connectionList();
    KSynkConnection * current;
    for( current = list->first(); current != 0; current=list->next()) {
        kdDebug(DEBUG_AREA) << "SynK::saveConnections():" << current->getName() << endl;
        current->writeToConfig(&config);	
    }	
    config.sync();
}
/** Loads the list of connections */
void SynK::loadConnections()
{
    KSimpleConfig config( locateLocal("appdata", "connectionsrc"));
    QStringList list = config.groupList() ;
    KSynkConnection newconnection;
    QStringList::ConstIterator it;
    for ( it = list.begin(); it != list.end(); ++it )	{
        kdDebug(DEBUG_AREA) << "SynK::loadConnections():" << *it << endl;
        if ( (*it).mid(0, QString(CONNECTION_PREFIX).length()) == CONNECTION_PREFIX) 	{
            newconnection = KSynkConnection();
            QString name = *it;
            name=name.remove(0,QString(CONNECTION_PREFIX).length());
            newconnection.loadFromConfig(&config,name);
            ConnectionList->addConnection(newconnection);
        }
    }

    ConnectionList->setInitialSelection();
}

void SynK::closeEvent (QCloseEvent *qcloseevent)
{
    KConfigGroup settings( KGlobal::config(), "General" );
    settings.writeEntry( "Splitter", vsplit->sizes() );
    settings.sync();

    bool close=true;
    if (transfer)	{
        int result =  KMessageBox::warningContinueCancel(this,
            i18n("There is an active transfer. Closing will kill it.\n"
                 "Do you still want to exit?"),
            i18n("A transfer is running.  Exit?"),
            i18n("Close anyway"));
        if (result == KMessageBox::Cancel) {
            qcloseevent->ignore();
            close=false ;
        }
    }
    if (close)	{
        slotCancelTransfer();
        saveConnections();
        kapp->config()->sync();
        qcloseevent->accept();
        KMainWindow::closeEvent( qcloseevent );
    }
}

/** Sets up the toolbars, actions etc */
void SynK::initGUI(){
    //Create Child widgets
    vsplit = new QSplitter( this );
    vsplit->setOrientation(QSplitter::Vertical);
    ConnectionList=new KConnectionList(vsplit,"ConnectionList");
    TransferDetails=new KTransferDetailsWidgetImpl(vsplit,"TransferDetails");
    setCentralWidget(vsplit);

    KConfigGroup settings( KGlobal::config(), "General" );
    QValueList<int> spliterSizes = settings.readIntListEntry("Splitter");
    if ( !spliterSizes.isEmpty() )
        vsplit->setSizes( spliterSizes );
	
    //setup connections
    connect(ConnectionList,SIGNAL(signalSelectionChanged()),
            this, SLOT(slotSelectionChanged()));
	
    //Setup actions
    m_syncAct = new KAction(i18n("&Start Synchronization"),
                            "connect_creating",0,
                            this,SLOT(slotSynchronise()),
                            this, "syncconnection");

    m_addAct = new KAction(i18n("&Add Connection..."), "filenew",0,
                           this, SLOT(slotAddConnection()),
                           this,"addconnection");

    m_editAct = new KAction(i18n("&Configure Connection..."), "configure",0,
                           this, SLOT(slotEditConnection()),
                           this,"editconnection");

    m_copyAct = new KAction(i18n("&Copy Connection"), "editcopy",0,
                           this, SLOT(slotCopyConnection()),
                           this,"copyconnection");

    m_removeAct = new KAction(i18n("&Remove Connection"), "editdelete",0,
                              this,SLOT(slotRemoveConnection()),
                              this,"removeconnection");

    setAutoSaveSettings();

    // needs to be after setAutoSaveSettings()
    toolBar()->setIconSize( KIcon::SizeMedium );
    toolBar()->setIconText( KToolBar::IconTextBottom );

    m_syncAct->plug(toolBar());
    m_addAct->plug(toolBar());
    m_editAct->plug(toolBar());
    m_copyAct->plug(toolBar());
    m_removeAct->plug(toolBar());

    m_popup = new QPopupMenu( this );
    m_syncAct->plug( m_popup );
    m_popup->insertSeparator();
    m_editAct->plug( m_popup );
    m_copyAct->plug( m_popup );
    m_removeAct->plug( m_popup );

    connect( ConnectionList->ConnectionKListView,
             SIGNAL( contextMenuRequested(QListViewItem *, const QPoint&,int)),
             SLOT( slotContextMenu( QListViewItem *, const QPoint&, int )) );

    KHelpMenu *helpMenu = new KHelpMenu( this,
                                         KGlobal::instance()->aboutData(),
                                         false );
    // ### setDelayedPopup() when we have some documentation...
    toolBar()->insertButton( "help", 100, helpMenu->menu(), true, i18n("Help") );
//                        SIGNAL( clicked() ), this, SLOT( appHelpActivated() ));
     toolBar()->getButton(100)->setPopup( helpMenu->menu(), true );

     QSize tsize = toolBar()->sizeHint();
     QSize sh = sizeHint();
     if ( tsize.width() > sh.width() )
         sh.setWidth( tsize.width() );
     if ( sh.width() > width() )
         resize( sh );
}

/** Handles a request to cancel the current transfer */
void SynK::slotCancelTransfer()
{
    if (transfer) {
        transfer->kill();
        transfer = 0L;
    }
}

void SynK::slotContextMenu( QListViewItem *item, const QPoint& pos,
                            int /*col*/ )
{
    if ( item || ConnectionList->selectedConnection() )
        m_popup->popup( pos );
}

#include "synk.moc"
