#ifndef __KMTRACELEAK__H__
#define __KMTRACELEAK__H__

// Qt includes
#include <qobject.h>

// kmtraceviewer includes
#include "KMtraceCallStack.hh"
#include "KMtraceCall.hh"
#include "KMtraceLeaksViewItem.hh"

/**
 * The model of a memory leak. A memory leak has the following properties:
 * @li number of leaked bytes totally lost (still allocated at the end of the program)
 * @li number of times the program accessed this memory leak (indicates
 *     whether leaks are repeating or are just one time leaks)
 * @li number of leaked bytes in the first block
 * @li address in memory where the first block of leak starts
 *
 * These values can be only accessed via the getXXX methods.
 *
 * The model is currently used by one view/control element the
 * @ref KMtraceLeaksViewItem. Only a one to one relation is
 * implemented at the moment.
 *
 * Each memory leak has a function call stack, represented as a
 * @ref KMtraceCallStack.
 *
 * @short The model of a memory leak
 * @author Andreas Baumann <andreas_baumann@users.sourceforge.net>
 * @see KMtraceLeaksViewItem, KMtraceCallStack, KMtraceCall
 * @version $ProjectMajorVersion: 0 $
 */
class KMtraceLeak : public QObject
{
   Q_OBJECT
   
   public:
      // standard constructors
      KMtraceLeak( );
      KMtraceLeak( int blocks, int bytes, int firstBytes, QString firstAddress );
      
      // accesser functions
      int getBlocks( );
      int getBytes( );
      int getFirstBytes( );
      QString getFirstAddress( );
      
      /**
       * Adds a function call to the memory leak. Could also be seen as
       * being part of the construction process of a memory leak object.
       * Used by the @ref KMtraceModel::readModel method while parsing the kmtrace tree file
       * and add one call stack after the other.
       *
       * @param call the function to be added at the end of the current
       *        call stack of the object
       */
      void addCall( KMtraceCall *call );
      
      /**
       * Get the function call stack of the memory leak.
       *
       * @returns the function call stack of the memory leak
       */
      KMtraceCallStack *getCallStack( );

      /**
       * Suppress or show the memory leak. Suppressed leaks are normally
       * not shown to the user by the view object because they are e.g.
       * library functions the programmer is not willing to fix.
       *
       * @param suppressed set the state to suppressed or not suppressed
       */
      void setSuppressed( bool suppressed );
       
      /**
       * Returns whether the memory leak is currently suppressed or not.
       *
       * @returns suppression state of the leak
       */ 
      bool isSuppressed( );

      /**
       * Register the view/control object to this model
       *
       * @param kitem the view item to be registerd
       */
      void setView( KMtraceLeaksViewItem *kitem );
      
      /**
       * Get the registered view to this model. Especially useful to get
       * to the view which must be changed based on the signal
       * @ref suppressionChanged.
       *
       * @returns the view object to this model
       */
      KMtraceLeaksViewItem *getView( );

   signals:
      /**
       * The signal being emmited if the leak has been set in state
       * suppressed (i.e. the user marks the leak as not interresting
       * and it should be suppressed, see also method @ref setSuppressed.
       *
       * @param leak the model of the leak which emmits the signal
       * @param suppressed the flag indicating whether the leak is now
       *        suppressed or not.
       */
      void suppressionChanged( KMtraceLeak *leak );
      
   private:
      // blocks (how many calls to allocation routine)
      int m_blocks;
      
      // bytes (how many bytes make up the leak)
      int m_bytes;
      
      // how big is the first block
      int m_firstBytes;
      
      // address of first block
      QString m_firstAddress;
      
      // the backtrace (call stack in modules, lines, function names, addresses)
      KMtraceCallStack m_callStack;
      
      // whether this leak should be shown as suppressed leak (if ever)
      bool m_suppressed;

      // the view to the model
      KMtraceLeaksViewItem *m_kitem;
};

#endif
